/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  Author: Stéphane Démurget  <stephane.demurget@enst-bretagne.fr>
 */

#include "config.h"

#include "emifreq.h"
#include "ef-backend.h"
#include "ef-prefs.h"
#include "ef-popup.h"
#include "ef-utils.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gnome.h>
#include <panel-applet.h>
#include <glade/glade.h>

#define PATH_ICON_EMIFREQ 	PIXMAPSDIR"emifreq-applet.png"

#define PATH_ICON_NA 		PIXMAPSDIR"emifreq-icon-na.png"
#define PATH_ICON_POWERSAVE 	PIXMAPSDIR"emifreq-icon-powersave.png"
#define PATH_ICON_PERFORMANCE 	PIXMAPSDIR"emifreq-icon-performance.png"
#define PATH_ICON_ONDEMAND	PIXMAPSDIR"emifreq-icon-ondemand.png"
#define PATH_ICON_1 		PIXMAPSDIR"emifreq-icon-1.png"
#define PATH_ICON_2 		PIXMAPSDIR"emifreq-icon-2.png"
#define PATH_ICON_3 		PIXMAPSDIR"emifreq-icon-3.png"
#define PATH_ICON_4 		PIXMAPSDIR"emifreq-icon-4.png"

#define PATH_ICON_NA_V 		PIXMAPSDIR"emifreq-icon-na-v.png"
#define PATH_ICON_POWERSAVE_V   PIXMAPSDIR"emifreq-icon-powersave-v.png"
#define PATH_ICON_PERFORMANCE_V PIXMAPSDIR"emifreq-icon-performance-v.png"
#define PATH_ICON_ONDEMAND_V	PIXMAPSDIR"emifreq-icon-ondemand-v.png"
#define PATH_ICON_1_V		PIXMAPSDIR"emifreq-icon-1-v.png"
#define PATH_ICON_2_V		PIXMAPSDIR"emifreq-icon-2-v.png"
#define PATH_ICON_3_V		PIXMAPSDIR"emifreq-icon-3-v.png"
#define PATH_ICON_4_V		PIXMAPSDIR"emifreq-icon-4-v.png"

#define HBOX_SPACE 5

int applets_refcount = 0;

static GType
emifreq_applet_get_type (void)
{
	   static GType type = 0;

	   if (!type) {
			 static const GTypeInfo info = {
				    sizeof (PanelAppletClass),
				    NULL, NULL, NULL, NULL, NULL,
				    sizeof (EmiFreqApplet),
				    0, NULL, NULL
			 };

			 type = g_type_register_static (PANEL_TYPE_APPLET, "EmiFreqApplet", &info, 0);
	   }

	   return type;
}

static gboolean
applet_orient_is_horizontal (EmiFreqApplet *applet)
{
	switch (panel_applet_get_orient (PANEL_APPLET (applet))) {
		case PANEL_APPLET_ORIENT_UP:
		case PANEL_APPLET_ORIENT_DOWN:
			return TRUE;

		default:
			return FALSE;
	}
}

/* Note: shouldn't be used in fact. It prevents the applet from changing its size if the label text length change
	 but in our case, it should always be something like xx°, but it can helps for future extensions */
static void
label_size_request_cb (GtkWidget *widget, GtkRequisition *requisition, EmiFreqApplet *applet)
{
	if (!applet->show_temp) {
		requisition->width = 0;
		requisition->height = 0;

		return;
	}

	if (requisition->width <= applet->label_width)
		requisition->width = applet->label_width;
	else
		applet->label_width = requisition->width;

	if (requisition->height <= applet->label_height)
		requisition->height = applet->label_height;
	else
		applet->label_height = requisition->height;
}

static gboolean
applet_is_non_available (EmiFreqApplet *applet)
{
	g_return_val_if_fail (applet != NULL, FALSE);

	return ef_backend_get_governor_name () == NULL;
}

static void
update_applet_text (EmiFreqApplet *applet)
{
	int temp;
	char *text;

	g_return_if_fail (applet != NULL);

	if (applet_is_non_available (applet)) {
		gtk_label_set_text (GTK_LABEL (applet->label), _("N/A"));
		gtk_box_set_spacing (GTK_BOX (applet->box), HBOX_SPACE); /* Ditto */

		return;
	}

	temp = ef_backend_get_temperature (applet->enable_metric);

	if (!applet->show_temp || temp == -1) {
		gtk_label_set_text (GTK_LABEL (applet->label), "");
		gtk_box_set_spacing (GTK_BOX (applet->box), 0); /* Less error-prone than reparenting the label  */
		return;
	}

	text = g_strdup_printf ((applet->enable_metric ? _("%d°C") : _("%d°F")), temp);
	gtk_label_set_text (GTK_LABEL (applet->label), text);
	g_free (text);

	gtk_box_set_spacing (GTK_BOX (applet->box), HBOX_SPACE); /* Ditto */
}

static gboolean
horizontal_display (EmiFreqApplet *applet)
{
	gboolean horizontal =applet_orient_is_horizontal (applet);

	/* Special case to use the horizontal version on a vertical panel if we
	   have the necessary space. Note that it's potentially bad stuff because it
	   depends on the system label size but p_a_get_size is an _indication_, so
	   should just _suggest_ the orientation */
	if (!horizontal && applet->show_temp && panel_applet_get_size (PANEL_APPLET (applet)) >= 52)
		horizontal = TRUE;

	return horizontal;
}

GdkPixbuf *
get_pixbuf_from_usage (EmiFreqApplet *applet, float usage) {
	gboolean horizontal = horizontal_display (applet);
	char *path;

	if (applet_is_non_available (applet) || usage == -1)
		path = horizontal ? PATH_ICON_NA : PATH_ICON_NA_V;
	else if (usage == -2.0)
		path = horizontal ? PATH_ICON_ONDEMAND : PATH_ICON_ONDEMAND_V;
	else if (usage == 0.0)
		path = horizontal ? PATH_ICON_POWERSAVE : PATH_ICON_POWERSAVE_V;
	else if (usage == 100.0)
		path = horizontal ? PATH_ICON_PERFORMANCE : PATH_ICON_PERFORMANCE_V;
	else if (usage <= 25.0)
		path = horizontal ? PATH_ICON_1 : PATH_ICON_1_V;
	else if (usage <= 50.0)
		path = horizontal ? PATH_ICON_2 : PATH_ICON_2_V;
	else if (usage <= 75.0)
		path = horizontal ? PATH_ICON_3 : PATH_ICON_3_V;
	else
		path = horizontal ? PATH_ICON_4 : PATH_ICON_4_V;

	return gdk_pixbuf_new_from_file (path, NULL);
}

static gboolean
applet_button_cb (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	EmiFreqApplet *applet = (EmiFreqApplet *) data;

	if (!(event->type == GDK_BUTTON_PRESS && event->button == 1))
       		return FALSE;

	/* CpuFreq support is down, nothing to see here : it shouldn't work better
	   changing the CPU speed. */
	if (applet_is_non_available (applet))
		return FALSE;

	if (applet->msg_queue == NULL) {
		applet->msg_queue = ef_msg_queue_new (FALSE);

		/* Note: well, we won't present a message everytime the user click on the menu, or he'll want to kill us */
		if (applet->msg_queue == NULL) {
			display_error (_("Can't change the CPU speed."),
				       _("The daemon in charge of changing the CPU speed is not started. Please do so if you want to control your CPU speed, and then click on the applet."));

			return FALSE;
		}
	}

	/* Note: because the message queue could be not valid because the daemon has
		 been stopped. Okay, we could factorize the code with the previous test,
		 but that way the wording is better. */
	if (!ef_msg_queue_is_alive (applet->msg_queue)) {
		display_error (_("Can't change the CPU speed."),
			       _("The applet noticed that the connection to the daemon in charge of changing the CPU speed is broken. Please restart the daemon."));

		return FALSE;
	}

	ef_popup_show ((EmiFreqApplet *) data, event->time);

	return TRUE;
}

static void
update_applet_icon (EmiFreqApplet *applet)
{
	GdkPixbuf *pixbuf;
	float usage;

	g_return_if_fail (applet != NULL);
	g_return_if_fail (applet->icon != NULL);

	usage = ef_backend_get_usage ();

	pixbuf = get_pixbuf_from_usage (applet, usage);

	if (pixbuf != NULL) {
		gtk_image_set_from_pixbuf (GTK_IMAGE (applet->icon), pixbuf);
		g_object_unref (pixbuf);
	}	
}

static void
update_applet_tooltip (EmiFreqApplet *applet)
{
	char *text;
	int f;

	g_return_if_fail (applet != NULL);

	f = ef_backend_get_frequency ();

	/* Note: again, we're paranoid there ;) */
	if (applet_is_non_available (applet))
		text = g_strdup (_("CpuFreq support not available anymore"));
	else {
		/* Note: the init is important. Let this like that */
		char *temp_text = "";
		const char *governor = ef_backend_get_profile_name ();
		int temp = ef_backend_get_temperature (applet->enable_metric);

		if (!applet->show_temp && temp != -1) {
			/* this is added at the end of the next translated expression, it's just the
			   temperature in celsius degrees */
			gchar *format = applet->enable_metric ? _(", %d°C") : _(", %d°F");

			temp_text = g_strdup_printf (format, temp);
		}

		if (f < 1000000)
			/* the format is <profile name>, <frequency> and <the temperature field just above>
			   translators please tell me if it can't work like that for a given language */
			text = g_strdup_printf (_("%s, %d Mhz%s"), governor, f / 1000, temp_text);
		else
			text = g_strdup_printf (_("%s, %.1f Ghz%s"), governor, f / 1000000.0, temp_text);

		if (strlen (temp_text) > 0)
			g_free (temp_text);
	}

	gtk_tooltips_set_tip (applet->tooltips, GTK_WIDGET (applet), text, "");
	g_free (text);
}

static void create_msg_queue (EmiFreqApplet *applet)
{
	g_return_if_fail (applet != NULL);

	applet->msg_queue = ef_msg_queue_new (FALSE);

	if (applet->msg_queue == NULL)
		display_error (_("Can't change the CPU speed."),
			       _("The daemon in charge of changing the CPU speed is not started. Please do so if you want to control your CPU speed, and then click on the applet."));
}

static gboolean
timeout_cb (EmiFreqApplet *applet)
{
	if (!applet || !applet->timeout_id)
		return FALSE;

	ef_backend_reload ();

	if (!applet->read_only) {
		EFBackendGovernors avail_governors = ef_backend_get_avail_governors ();

		if (avail_governors != applet->last_avail_governors) {
			if (ef_backend_governors_count (avail_governors) == 1) {
				/* it's not useful to display the popup anymore */
				g_signal_handler_block (G_OBJECT (applet), applet->handler_id);
				ef_msg_queue_free (applet->msg_queue);
			} else {
				if (ef_backend_governors_count (applet->last_avail_governors) == 1) {
					/* we need to display the popup now */
					g_signal_handler_unblock (G_OBJECT (applet), applet->handler_id);
					create_msg_queue (applet);
				}

				/* when the popup menu will be displayed, ask for a refresh */
				applet->popup_dirty = TRUE;
			}

			applet->last_avail_governors = avail_governors;
		}
	}

	update_applet_text (applet);
	update_applet_icon (applet);
	update_applet_tooltip (applet);
	
	return TRUE;
}

void
force_refresh (EmiFreqApplet *applet, gboolean only_icon)
{
	/* we'd better do that before applying the changes,
	   that may take some time */
	reset_timeout (applet);

	if (!only_icon)
		timeout_cb (applet);
	else {
		/* Just to be faster ;) */
		ef_backend_reload_usage ();
		update_applet_icon (applet);
	}
}

void
reset_timeout (EmiFreqApplet *applet)
{
	g_return_if_fail (applet != NULL);

	if (applet->timeout_id != 0)
		gtk_timeout_remove (applet->timeout_id);

	applet->timeout_id = gtk_timeout_add (applet->refresh_time, (GtkFunction)timeout_cb, (gpointer)applet);
}

static void
applet_change_orient_cb (PanelApplet *a)
{
	EmiFreqApplet *applet = (EmiFreqApplet *) a;
	GtkWidget *new_box;
	gboolean horizontal = horizontal_display (applet);

	/* we update the icon to make the applet sexier */
	update_applet_icon (applet);

	/* creates the new holding box the same way the current is */
	new_box = horizontal ? gtk_hbox_new (FALSE, 0)
			     : gtk_vbox_new (FALSE, 0);

	gtk_box_set_spacing (GTK_BOX (new_box),
			     0);//gtk_box_get_spacing (GTK_BOX (applet->box)));
					
	/* We'll now move the widgets to their new container.
	   Note: we can only use gtk_widget_reparent once, because we use a box
		 and not a standard container */
	gtk_widget_reparent (applet->icon, new_box);

	g_object_ref (applet->label);
	gtk_container_remove (GTK_CONTAINER (applet->box), applet->label);
	gtk_box_pack_start (GTK_BOX (new_box), applet->label, TRUE, TRUE, 0);
	g_object_unref (applet->label);

	gtk_widget_destroy (applet->box);

	/* make the new box attached to the applet */
	applet->box = new_box;
	gtk_container_add (GTK_CONTAINER (applet), applet->box);
	gtk_widget_show (applet->box);

	/* schedule menu icons update */
	if (horizontal != applet->horizontal) {
		applet->horizontal = horizontal;

		g_object_unref (applet->popup);
		applet->popup = NULL;
	}
}

static void
applet_change_size (PanelApplet *applet_widget, int size, EmiFreqApplet *applet)
{
	g_return_if_fail (applet != NULL);

	applet_change_orient_cb (applet_widget);
}

static void
applet_destroy (PanelApplet *applet_widget, EmiFreqApplet *applet)
{
	g_return_if_fail (applet != NULL);

	gtk_timeout_remove (applet->timeout_id);
	applet->timeout_id = 0;

	if (--applets_refcount == 0)
		ef_backend_shutdown ();
}

static void
prefs_cb (BonoboUIComponent *uic, gpointer data, const gchar *verbname)
{
	ef_preferences_dialog_show ((EmiFreqApplet *) data);
}

static void
help_cb (BonoboUIComponent *uic, gpointer data, const gchar *verbname)
{
	display_help (NULL);
}

static void
about_cb (BonoboUIComponent *uic, gpointer data, const gchar *verbname)
{
	EmiFreqApplet *applet = (EmiFreqApplet*) data;
	GdkPixbuf *icon;

	const char *authors[] = 
	{
		"Stéphane Démurget <stephane.demurget@enst-bretagne.fr>", 
		NULL
	};

	if (applet->about)
	{
		gtk_window_present(GTK_WINDOW(applet->about));
		return;
	}

	icon = gdk_pixbuf_new_from_file (PATH_ICON_EMIFREQ, NULL);
	
	applet->about = GTK_DIALOG (gnome_about_new (
				"EmiFreq", VERSION,
				"Copyright 2004 Stéphane Démurget",
				/* translator note: ok, don't translate the second part, it's not necessary - this is french love ;) */
				_("A little GNOME applet that shows/control the CPU frequency and temperature thanks to the CpuFreq sysfs kernel interface.\n\nA cette fille extraordinaire comme on n'en fait plus,\nà mon Emi que j'aime plus que tout au monde..."),
				(const char **) authors, NULL, NULL,
				icon));

	if (icon != NULL)
		gdk_pixbuf_unref (icon);

	g_object_add_weak_pointer (G_OBJECT(applet->about), (gpointer*) &applet->about);

	gtk_widget_show (GTK_WIDGET (applet->about));
}

static const BonoboUIVerb
emifreq_applet_menu_verbs [] = 
{
		BONOBO_UI_VERB("EmiFreqAppletProperties", prefs_cb),
		BONOBO_UI_VERB("EmiFreqAppletHelp", help_cb),
		BONOBO_UI_VERB("EmiFreqAppletAbout", about_cb),
	
		BONOBO_UI_VERB_END
};

gboolean
emifreq_applet_factory (EmiFreqApplet *applet, const gchar *iid, gpointer data)
{
	GError *error = NULL;
	gboolean temp_available;

	if (strcmp (iid, "OAFIID:GNOME_EmiFreqApplet"))
		return FALSE;

	if (!applets_refcount && ef_backend_init (&error) == FALSE) {
		display_error ("Backend initialization error.",
			       error->message);

		g_error_free (error);

		return FALSE;
	}

	applets_refcount++;

	glade_gnome_init ();

	gnome_window_icon_set_default_from_file (PATH_ICON_EMIFREQ);

	applet->tooltips = gtk_tooltips_new ();
	applet->icon = gtk_image_new ();
	applet->label = gtk_label_new ("");

	applet->timeout_id = 0;
	applet->horizontal = TRUE;

	/* load preferences */
	ef_preferences_load (applet);

	applet->box = gtk_hbox_new (FALSE, HBOX_SPACE);
	gtk_box_pack_start (GTK_BOX (applet->box), applet->icon, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (applet->box), applet->label, TRUE, TRUE, 0);

	gtk_container_add (GTK_CONTAINER (applet), applet->box);

	/* apply preferences and force an update */
	timeout_cb (applet); /* force timeout to update label & icon */
	reset_timeout (applet);

	/* Note: well, we could not connect the signal and do nothing, but I find it better to check
		 just before showing the popup if the daemon has been started meanwhile */
	applet->handler_id = g_signal_connect (G_OBJECT (applet), "button-press-event",
					       G_CALLBACK (applet_button_cb), (gpointer) applet);

	/* If the CpuFreq support is here, we'll have at least the PERFORMANCE governor */
	applet->last_avail_governors = ef_backend_get_avail_governors ();

	if (ef_backend_governors_count (applet->last_avail_governors) > 1 &&
	    !applet->read_only)
		create_msg_queue (applet);
	else
		g_signal_handler_block (G_OBJECT (applet), applet->handler_id);

	gtk_widget_show_all (GTK_WIDGET (applet));

	/* Note: we pack the widgets in horizontal mode by default, the most 
		 common one, so we need to change the holding box if it's a 
		 vertical panel we are adding the applet to */ 
	if (!applet_orient_is_horizontal (applet))
		applet_change_orient_cb (PANEL_APPLET (applet));

	g_signal_connect (G_OBJECT (applet->label), "size_request",
                          G_CALLBACK (label_size_request_cb), (gpointer) applet);

	g_signal_connect (G_OBJECT (applet), "change_orient",
                          G_CALLBACK (applet_change_orient_cb), (gpointer) applet);

	g_signal_connect (G_OBJECT (applet), "change_size",
                          G_CALLBACK (applet_change_size), (gpointer) applet);

	g_signal_connect (G_OBJECT (applet), "destroy",
                          G_CALLBACK (applet_destroy), (gpointer) applet);

	panel_applet_setup_menu_from_file (PANEL_APPLET (applet),
					   NULL,
					   "GNOME_EmiFreqApplet.xml", NULL,
					   emifreq_applet_menu_verbs,
					   applet);

	temp_available = (ef_backend_get_temperature (TRUE) != -1);

	if (!temp_available) {
		BonoboUIComponent *popup_component = panel_applet_get_popup_component (PANEL_APPLET (applet));

		bonobo_ui_component_set_prop (popup_component,
					      "/commands/EmiFreqAppletProperties",
					      "hidden",
					      "1",
					      NULL);
	}

	return TRUE;
}

PANEL_APPLET_BONOBO_FACTORY("OAFIID:GNOME_EmiFreqApplet_Factory",
			    emifreq_applet_get_type () ,
			    PACKAGE, VERSION,
			    (PanelAppletFactoryCallback) emifreq_applet_factory, NULL)
