#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "progressdialog.h"
#include "pulist.h"

#include "edv_types.h"
#include "edv_date.h"
#include "edv_id.h"
#include "edv_obj.h"
#include "edv_recycled_obj.h"
#include "edv_archive_obj.h"
#include "obj_op_dlg.h"
#include "obj_op_dlg_op.h"
#include "endeavour2.h"
#include "edv_recbin_index.h"
#include "edv_obj_op.h"
#include "edv_recycled_obj_op.h"
#include "edv_cb.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"


void EDVObjOpDlgProcess(edv_obj_op_dlg_struct *d);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Performs the operation based on the current values set
 *	on the Object Operations Dialog.
 *
 *	The values from the Object Operations Dialog's widgets and the
 *	list of objects will be collected and the operation specified
 *	by the Object Operations Dialog will be performed on each
 *	object.
 *
 *	The progress dialog will be mapped as needed as the operation
 *	is performed and any errors that occure during or after
 *	the operation will be displayed.
 *
 *	The Object Operations Dialog will then be unmapped and its
 *	values will be reset.
 */
void EDVObjOpDlgProcess(edv_obj_op_dlg_struct *d)
{
	const gboolean show_progress = TRUE;
	gboolean	yes_to_all = FALSE,
			recursive = FALSE,
			dereference_links = FALSE;
	gint status;
	gchar *tar_path;
	GList *objs_list;
	GtkWidget *w, *toplevel;
	cfg_item_struct *cfg_list;
	edv_location_type location_type;
	edv_obj_op_dlg_op op;
	edv_core_struct *core;

	if(d == NULL)
	    return;

	toplevel = d->ref_toplevel;
	core = d->core;
	cfg_list = core->cfg_list;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	    return;

	/* Get the operation code that indicates which operation is
	 * to be performed and the location type
	 */
	op = d->op;
	location_type = d->location_type;

	/* Get the list of objects to operate on */
	objs_list = d->objs_list;

	/* Get the target value */
	w = d->target_entry;
	if((w != NULL) ? GTK_WIDGET_MAPPED(w) : FALSE)
	{
	    tar_path = EDVEvaluatePath(
		d->src_dir, gtk_entry_get_text(GTK_ENTRY(w))
	    );
	}
	else
	{
	    tar_path = NULL;
	}

	/* Get the options if they are specified */
	w = d->opt_parent;
	if((w != NULL) ? GTK_WIDGET_MAPPED(w) : FALSE)
	{
	    w = d->opt_recursive_check;
	    if(GTK_WIDGET_MAPPED(w) && GTK_WIDGET_SENSITIVE(w))
		recursive = GTK_TOGGLE_BUTTON_GET_ACTIVE(
		    d->opt_recursive_check
		);
	    w = d->opt_dereference_links_check;
	    if(GTK_WIDGET_MAPPED(w) && GTK_WIDGET_SENSITIVE(w))
		dereference_links = GTK_TOGGLE_BUTTON_GET_ACTIVE(
		    d->opt_dereference_links_check
		);
	}


	/* Unmap the Object Operations Dialog */
	EDVObjOpDlgUnmap(d);


	status = 0;

	/* Perform the operation
	 *
	 * Copy
	 */
	if(op == EDV_OBJ_OP_DLG_OP_COPY)
	{
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    struct stat lstat_buf;
		    const gchar *error_msg, *src_path;
		    gchar *new_path;
		    GList *glist;
		    edv_object_struct *obj;

		    /* Copy each object */
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			src_path = obj->full_path;
			if(STRISEMPTY(src_path))
			    continue;

			/* Copy this object */
			status = EDVObjectOPCopy(
			    core,
			    src_path,
			    tar_path,
			    &new_path,
			    toplevel,
			    show_progress,
			    TRUE,		/* Interactive */
			    &yes_to_all
			);

			/* Check for errors */
			error_msg = EDVObjectOPGetError(core);
			if(!STRISEMPTY(error_msg))
			{
			    /* Report the error */
			    EDVPlaySoundError(core);
			    EDVMessageError(
				"Copy Error",
				error_msg,
				NULL,
				toplevel
			    );
			}

			/* Get the source object's statistics */
			if(lstat((const char *)src_path, &lstat_buf))
			{
#ifdef ENOENT
			    /* Notify about the source object being removed */
			    const gint error_code = (gint)errno;
			    if(error_code == ENOENT)
				EDVObjectRemovedEmit(core, src_path);
#endif
			}

			/* Notify about the new object being added */
			if(new_path != NULL)
			{
			    if(!lstat((const char *)new_path, &lstat_buf))
				EDVObjectAddedEmit(
				    core,
				    new_path,
				    &lstat_buf
				);

			    /* Delete the new object path */
			    g_free(new_path);
			}

			/* User aborted? */
			if(status == -4)
			    break;
		    }
		}
		break;

	      case EDV_LOCATION_TYPE_RECBIN:
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	    }
	}
	/* Move */
	else if(op == EDV_OBJ_OP_DLG_OP_MOVE)
	{
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    struct stat lstat_buf;
		    const gchar *error_msg, *src_path;
		    gchar *new_path;
		    GList *glist;
		    edv_object_struct *obj;

		    /* Move each object */
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			src_path = obj->full_path;
			if(STRISEMPTY(src_path))
			    continue;

			/* Move this object */
			status = EDVObjectOPMove(
			    core,
			    src_path,
			    tar_path,
			    &new_path,
			    toplevel,
			    show_progress,
			    TRUE,		/* Interactive */
			    &yes_to_all
			);

			/* Check for errors */
			error_msg = EDVObjectOPGetError(core);
			if(!STRISEMPTY(error_msg))
			{
			    /* Report the error */
			    EDVPlaySoundError(core);
			    EDVMessageError(
				"Move Error",
				error_msg,
				NULL,
				toplevel
			    );
			}

			/* Get the source object's statistics */
			if(lstat((const char *)src_path, &lstat_buf))
			{
#ifdef ENOENT
			    /* Notify about the source object being removed */
			    const gint error_code = (gint)errno;
			    if(error_code == ENOENT)
				EDVObjectRemovedEmit(core, src_path);
#endif
			}

			/* Notify about the new object being added */
			if(new_path != NULL)
			{
			    if(!lstat((const char *)new_path, &lstat_buf))
				EDVObjectAddedEmit(
				    core,
				    new_path,
				    &lstat_buf
				);

			    /* Delete the new object path */
			    g_free(new_path);
			}

			/* User aborted? */
			if(status == -4)
			    break;
		    }
		}
		break;

	      case EDV_LOCATION_TYPE_RECBIN:
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	    }
	}
	/* Link */
	else if(op == EDV_OBJ_OP_DLG_OP_LINK)
	{
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    struct stat lstat_buf;
		    const gchar *error_msg, *src_path;
		    gchar *new_path;
		    GList *glist;
		    edv_object_struct *obj;

		    /* Link each object */
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			src_path = obj->full_path;
			if(STRISEMPTY(src_path))
			    continue;

			/* Link this object */
			status = EDVObjectOPLink(
			    core,
			    tar_path,		/* New link */
			    src_path,		/* Target */
			    &new_path,
			    toplevel,
			    show_progress,
			    TRUE,		/* Interactive */
			    &yes_to_all
			);

			/* Check for errors */
			error_msg = EDVObjectOPGetError(core);
			if(!STRISEMPTY(error_msg))
			{
			    /* Report the error */
			    EDVPlaySoundError(core);
			    EDVMessageError(
				"Link Error",
				error_msg,
				NULL,
				toplevel
			    );
			}

			/* Get the source object's statistics */
			if(lstat((const char *)src_path, &lstat_buf))
			{
#ifdef ENOENT
			    /* Notify about the source object being removed */
			    const gint error_code = (gint)errno;
			    if(error_code == ENOENT)
				EDVObjectRemovedEmit(core, src_path);
#endif
			}

			/* Notify about the new object being added */
			if(new_path != NULL)
			{
			    if(!lstat((const char *)new_path, &lstat_buf))
				EDVObjectAddedEmit(
				    core,
				    new_path,
				    &lstat_buf
				);

			    /* Delete the new object path */
			    g_free(new_path);
			}

			/* User aborted? */
			if(status == -4)
			    break;
		    }
		}
		break;

	      case EDV_LOCATION_TYPE_RECBIN:
		/* Recycled objects cannot be linked */
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		/* Archive objects cannot be linked */
		break;
	    }
	}
	/* Change Permissions */
	else if(op == EDV_OBJ_OP_DLG_OP_CHMOD)
	{
	    edv_permission_flags permissions = 0x00000000;

	    /* Get the permissions value from the widgets */
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_ur_check))
		permissions |= EDV_PERMISSION_UREAD;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_uw_check))
		permissions |= EDV_PERMISSION_UWRITE;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_ux_check))
		permissions |= EDV_PERMISSION_UEXECUTE;

	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_gr_check))
		permissions |= EDV_PERMISSION_GREAD;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_gw_check))
		permissions |= EDV_PERMISSION_GWRITE;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_gx_check))
		permissions |= EDV_PERMISSION_GEXECUTE;

	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_or_check))
		permissions |= EDV_PERMISSION_AREAD;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_ow_check))
		permissions |= EDV_PERMISSION_AWRITE;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_ox_check))
		permissions |= EDV_PERMISSION_AEXECUTE;

	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_suid_check))
		permissions |= EDV_PERMISSION_SETUID;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_sgid_check))
		permissions |= EDV_PERMISSION_SETGID;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_sticky_check))
		permissions |= EDV_PERMISSION_STICKY;

	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    const gchar *error_msg;
		    GList *glist, *paths_list, *modified_paths_list;
		    edv_object_struct *obj;

		    /* Create the paths list */
		    paths_list = NULL;
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			paths_list = g_list_append(
			    paths_list,
			    STRDUP(obj->full_path)
			);
		    }

		    /* Change the permissions */
		    status = EDVObjectOPChMod(
			core,
			paths_list,
			permissions,
			&modified_paths_list,
			toplevel,
			show_progress,
			TRUE,			/* Interactive */
			&yes_to_all,
			recursive,
			!dereference_links	/* Archive */
		    );

		    /* Check for errors */
		    error_msg = EDVObjectOPGetError(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Change Permissions Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Notify about the objects being modified */
		    if(modified_paths_list != NULL)
		    {
			struct stat lstat_buf;
			const gchar *path;
			GList *glist;

			for(glist = modified_paths_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    path = (const gchar *)glist->data;
			    if(path == NULL)
				continue;

			    if(!lstat((const char *)path, &lstat_buf))
				EDVObjectModifiedEmit(
				    core,
				    path,
				    path,
				    &lstat_buf
				);
			}

			/* Delete the modified paths list */
			g_list_foreach(modified_paths_list, (GFunc)g_free, NULL);
			g_list_free(modified_paths_list);
		    }

		    /* Delete the paths list */
		    if(paths_list != NULL)
		    {
			g_list_foreach(paths_list, (GFunc)g_free, NULL);
			g_list_free(paths_list);
		    }
		}
		break;

	      case EDV_LOCATION_TYPE_RECBIN:
		if(objs_list != NULL)
		{
		    const gchar *error_msg;
		    GList	*glist,
				*indicies_list = NULL,
				*modified_indicies_list;
		    edv_recycled_object_struct *obj;

		    /* Create the indicies list */
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_RECYCLED_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			indicies_list = g_list_append(
			    indicies_list,
			    (gpointer)obj->index
			);
		    }

		    /* Change the permissions */
		    status = EDVRecycledObjectOPChMod(
			core,
			indicies_list,
			permissions,
			&modified_indicies_list,
			toplevel,
			show_progress,
			TRUE,			/* Interactive */
			&yes_to_all
		    );

		    /* Check for errors */
		    error_msg = EDVRecycledObjectOPGetError(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Change Permissions Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Notify about the objects being modified */
		    if(modified_indicies_list != NULL)
		    {
			guint index;
			GList *glist;

			for(glist = modified_indicies_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    index = (guint)glist->data;
			    if(index == 0)
				continue;

			    EDVRecycledObjectModifiedEmit(core, index);
			}

			/* Delete the modified indicies list */
			g_list_free(modified_indicies_list);
		    }

		    /* Delete the indicies list */
		    g_list_free(indicies_list);
		}
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		/* Archive object permissions cannot be changed */
		break;
	    }
	}
	/* Change Ownership */
	else if(op == EDV_OBJ_OP_DLG_OP_CHOWN)
	{
	    gint owner_id = 0, group_id = 0;
	    GtkWidget *w;

	    /* Get the owner id value from the widget */
	    w = d->target_owner_entry;
	    if(w != NULL)
		owner_id = EDVUIDNameToUID(
		    core->uids_list,
		    gtk_entry_get_text(GTK_ENTRY(w))
		);

	    /* Get the group id value from the widget */
	    w = d->target_group_entry;
	    if(w != NULL)
		group_id = EDVGIDNameToGID(
		    core->gids_list,
		    gtk_entry_get_text(GTK_ENTRY(w))
		);

	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    const gchar *error_msg;
		    GList *glist, *paths_list, *modified_paths_list;
		    edv_object_struct *obj;

		    /* Create the paths list */
		    paths_list = NULL;
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			paths_list = g_list_append(
			    paths_list,
			    STRDUP(obj->full_path)
			);
		    }

		    /* Change the ownership */
		    status = EDVObjectOPChOwn(
			core,
			paths_list,
			owner_id, group_id,
			&modified_paths_list,
			toplevel,
			show_progress,
			TRUE,			/* Interactive */
			&yes_to_all,
			recursive,
			!dereference_links	/* Archive */
		    );

		    /* Check for errors */
		    error_msg = EDVObjectOPGetError(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Change Ownership Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Notify about the objects being modified */
		    if(modified_paths_list != NULL)
		    {
			struct stat lstat_buf;
			const gchar *path;
			GList *glist;

			for(glist = modified_paths_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    path = (const gchar *)glist->data;
			    if(path == NULL)
				continue;

			    if(!lstat((const char *)path, &lstat_buf))
				EDVObjectModifiedEmit(
				    core,
				    path,
				    path,
				    &lstat_buf
				);
			}

			/* Delete the modified paths list */
			g_list_foreach(modified_paths_list, (GFunc)g_free, NULL);
			g_list_free(modified_paths_list);
		    }

		    /* Delete the paths list */
		    if(paths_list != NULL)
		    {
			g_list_foreach(paths_list, (GFunc)g_free, NULL);
			g_list_free(paths_list);
		    }
		}
		break;

	      case EDV_LOCATION_TYPE_RECBIN:
		if(objs_list != NULL)
		{
		    const gchar *error_msg;
		    GList	*glist,
				*indicies_list = NULL,
				*modified_indicies_list;
		    edv_recycled_object_struct *obj;

		    /* Create the indicies list */
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_RECYCLED_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			indicies_list = g_list_append(
			    indicies_list,
			    (gpointer)obj->index
			);
		    }

		    /* Change the ownership */
		    status = EDVRecycledObjectOPChOwn(
			core,
			indicies_list,
			owner_id, group_id,
			&modified_indicies_list,
			toplevel,
			show_progress,
			TRUE,			/* Interactive */
			&yes_to_all
		    );

		    /* Check for errors */
		    error_msg = EDVRecycledObjectOPGetError(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Change Ownership Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Notify about the objects being modified */
		    if(modified_indicies_list != NULL)
		    {
			guint index;
			GList *glist;

			for(glist = modified_indicies_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    index = (guint)glist->data;
			    if(index == 0)
				continue;

			    EDVRecycledObjectModifiedEmit(core, index);
			}

			/* Delete the modified indicies list */
			g_list_free(modified_indicies_list);
		    }

		    /* Delete the indicies list */
		    g_list_free(indicies_list);
		}
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		/* Archive object ownership cannot be changed */
		break;
	    }
	}
	/* Change Time Stamps */
	else if(op == EDV_OBJ_OP_DLG_OP_CHTIME)
	{
	    struct tm *tm_buf;
	    gulong	access_time = 0l,
			modify_time = 0l,
			delete_time = 0l;

	    /* Get the time stamp values from the widgets
	     *
	     * Access time
	     */
	    tm_buf = (struct tm *)g_malloc0(sizeof(struct tm));
	    if(tm_buf != NULL)
	    {
		w = d->target_atime_year_spin;
		if(w != NULL)
		    tm_buf->tm_year = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    ) - 1900;
		tm_buf->tm_mon = (gint)PUListBoxGetSelected(
		    d->target_atime_month_pulistbox
		);
		w = d->target_atime_day_spin;
		if(w != NULL)
		    tm_buf->tm_mday = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );
		w = d->target_atime_hours_spin;
		if(w != NULL)
		    tm_buf->tm_hour = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );
		w = d->target_atime_minutes_spin;
		if(w != NULL)
		    tm_buf->tm_min = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );
		w = d->target_atime_seconds_spin;
		if(w != NULL)
		    tm_buf->tm_sec = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );

		tm_buf->tm_isdst = -1;		/* Not specified */

		access_time = (gulong)mktime(tm_buf);

		g_free(tm_buf);
	    }

	    /* Modify time */
	    tm_buf = (struct tm *)g_malloc0(sizeof(struct tm));
	    if(tm_buf != NULL)
	    {
		w = d->target_mtime_year_spin;
		if(w != NULL)
		    tm_buf->tm_year = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    ) - 1900;
		tm_buf->tm_mon = (gint)PUListBoxGetSelected(
		    d->target_mtime_month_pulistbox
		);
		w = d->target_mtime_day_spin;
		if(w != NULL)
		    tm_buf->tm_mday = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );
		w = d->target_mtime_hours_spin;
		if(w != NULL)
		    tm_buf->tm_hour = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );
		w = d->target_mtime_minutes_spin;
		if(w != NULL)
		    tm_buf->tm_min = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );
		w = d->target_mtime_seconds_spin;
		if(w != NULL)
		    tm_buf->tm_sec = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );

		tm_buf->tm_isdst = -1;		/* Not specified */

		modify_time = (gulong)mktime(tm_buf);

		g_free(tm_buf);
	    }

	    /* Delete time */
	    tm_buf = (struct tm *)g_malloc0(sizeof(struct tm));
	    if(tm_buf != NULL)
	    {
		w = d->target_dtime_year_spin;
		if(w != NULL)
		    tm_buf->tm_year = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    ) - 1900;
		tm_buf->tm_mon = (gint)PUListBoxGetSelected(
		    d->target_dtime_month_pulistbox
		);
		w = d->target_dtime_day_spin;
		if(w != NULL)
		    tm_buf->tm_mday = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );
		w = d->target_dtime_hours_spin;
		if(w != NULL)
		    tm_buf->tm_hour = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );
		w = d->target_dtime_minutes_spin;
		if(w != NULL)
		    tm_buf->tm_min = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );
		w = d->target_dtime_seconds_spin;
		if(w != NULL)
		    tm_buf->tm_sec = (gint)gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );

		tm_buf->tm_isdst = -1;		/* Not specified */

		delete_time = (gulong)mktime(tm_buf);

		g_free(tm_buf);
	    }

	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    const gchar *error_msg;
		    GList *glist, *paths_list, *modified_paths_list;
		    edv_object_struct *obj;

		    /* Create the paths list */
		    paths_list = NULL;
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			paths_list = g_list_append(
			    paths_list,
			    STRDUP(obj->full_path)
			);
		    }

		    /* Change the time stamps */
		    status = EDVObjectOPChTime(
			core,
			paths_list,
			access_time, modify_time,
			&modified_paths_list,
			toplevel,
			show_progress,
			TRUE,			/* Interactive */
			&yes_to_all,
			recursive,
			!dereference_links	/* Archive */
		    );

		    /* Check for errors */
		    error_msg = EDVObjectOPGetError(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Change Time Stamps Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Notify about the objects being modified */
		    if(modified_paths_list != NULL)
		    {
			struct stat lstat_buf;
			const gchar *path;
			GList *glist;

			for(glist = modified_paths_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    path = (const gchar *)glist->data;
			    if(path == NULL)
				continue;

			    if(!lstat((const char *)path, &lstat_buf))
				EDVObjectModifiedEmit(
				    core,
				    path,
				    path,
				    &lstat_buf
				);
			}

			/* Delete the modified paths list */
			g_list_foreach(modified_paths_list, (GFunc)g_free, NULL);
			g_list_free(modified_paths_list);
		    }

		    /* Delete the paths list */
		    if(paths_list != NULL)
		    {
			g_list_foreach(paths_list, (GFunc)g_free, NULL);
			g_list_free(paths_list);
		    }
		}
		break;

	      case EDV_LOCATION_TYPE_RECBIN:
		if(objs_list != NULL)
		{
		    const gchar *error_msg;
		    GList	*glist,
				*indicies_list = NULL,
				*modified_indicies_list;
		    edv_recycled_object_struct *obj;

		    /* Create the indicies list */
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_RECYCLED_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			indicies_list = g_list_append(
			    indicies_list,
			    (gpointer)obj->index
			);
		    }

		    /* Change the time stamps */
		    status = EDVRecycledObjectOPChTime(
			core,
			indicies_list,
			access_time, modify_time, delete_time,
			&modified_indicies_list,
			toplevel,
			show_progress,
			TRUE,			/* Interactive */
			&yes_to_all
		    );

		    /* Check for errors */
		    error_msg = EDVRecycledObjectOPGetError(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Change Time Stamps Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Notify about the objects being modified */
		    if(modified_indicies_list != NULL)
		    {
			guint index;
			GList *glist;

			for(glist = modified_indicies_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    index = (guint)glist->data;
			    if(index == 0)
				continue;

			    EDVRecycledObjectModifiedEmit(core, index);
			}

			/* Delete the modified indicies list */
			g_list_free(modified_indicies_list);
		    }

		    /* Delete the indicies list */
		    g_list_free(indicies_list);
		}
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		/* Archive object time stamps cannot be modified */
		break;
	    }
	}
	else
	{
	    gchar *msg = g_strdup_printf(
"EDVObjOpDlgProcess(): Operation \"%i\" not supported.",
		op
	    );
	    EDVPlaySoundError(core);
	    EDVMessageError(
		"Internal Error",
		msg,
		NULL,
		toplevel
	    );
	    g_free(msg);
	    status = -2;
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the completed sound on success */
	if(status == 0)
	    EDVPlaySoundCompleted(core);


	/* Delete the copy of the target path */
	g_free(tar_path);

	/* Delete the source objects list and the source directory
	 * on the object operations dialog
	 */
	EDVObjOpDlgReset(d);
}
