#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "guiutils.h"
#include "menubutton.h"
#include "pie_chart.h"
#include "cdialog.h"
#include "fb.h"
#include "progressdialog.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_date.h"
#include "edv_id.h"
#include "edv_obj.h"
#include "edv_recycled_obj.h"
#include "edv_archive_obj.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_recbin_stat.h"
#include "edv_recbin_index.h"
#include "edv_archive_stat.h"
#include "obj_op_dlg.h"
#include "prop_dlg.h"
#include "prop_dlg_op.h"
#include "endeavour2.h"
#include "edv_device_mount.h"
#include "edv_obj_op.h"
#include "edv_list_cb.h"
#include "edv_cb.h"
#include "edv_op.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_browse_20x20.xpm"
#include "images/icon_time_stamp_20x20.xpm"
#include "images/icon_mount_20x20.xpm"
#include "images/icon_unmount_20x20.xpm"
#include "images/icon_eject_20x20.xpm" 
#include "images/icon_fsck_32x32.xpm"
#include "images/icon_tools_32x32.xpm"
#include "images/icon_floppy_32x32.xpm"
#include "images/icon_properties2_48x48.xpm"


/* Callbacks */
static gint EDVPropDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVPropDlgSwitchPageCB(
	GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
	gpointer data
);

static void EDVPropDlgAnyChangedCB(GtkWidget *widget, gpointer data);

static void EDVPropDlgOKCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgApplyCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgCancelCB(GtkWidget *widget, gpointer data);

static void EDVPropDlgDateTouchCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgDateTouchSpecificCB(GtkWidget *widget, gpointer data);

static void EDVPropDlgDetailsVerboseToggledCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgDetailsRawToggledCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgDetailsListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void EDVPropDlgMountCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgEjectCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgRunCheckCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgRunToolsCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgRunFormatCB(GtkWidget *widget, gpointer data);

static void EDVPropDlgOwnerMapPUListCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgGroupMapPUListCB(GtkWidget *widget, gpointer data);

static void EDVPropDlgLinkChangedCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgLinkBrowseCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgLinkMakeRelativePathCB(GtkWidget *widget, gpointer data);
static void EDVPropDlgLinkTargetPropertiesCB(GtkWidget *widget, gpointer data);

void EDVPropDlgObjectAddedNotifyCB(
	edv_propdlg_struct *pd, const gchar *path,
	struct stat *lstat_buf
);
void EDVPropDlgObjectModifiedNotifyCB(
	edv_propdlg_struct *pd,
	const gchar *path, const gchar *new_path,
	struct stat *lstat_buf
);
void EDVPropDlgObjectRemovedNotifyCB(
	edv_propdlg_struct *pd, const gchar *path
);
void EDVPropDlgRecycledObjectAddedNotifyCB(
	edv_propdlg_struct *pd, const guint index
);
void EDVPropDlgRecycledObjectModifiedNotifyCB(
	edv_propdlg_struct *pd, const guint index
);
void EDVPropDlgRecycledObjectRemovedNotifyCB(
	edv_propdlg_struct *pd, const guint index
);
void EDVPropDlgMIMETypeAddedNotifyCB(
	edv_propdlg_struct *pd,
	const gint mt_num, edv_mime_type_struct *m
);
void EDVPropDlgMIMETypeModifiedNotifyCB(
	edv_propdlg_struct *pd,
	const gint mt_num, edv_mime_type_struct *m
);
void EDVPropDlgMIMETypeRemovedNotifyCB(
	edv_propdlg_struct *pd,
	const gint mt_num
);
void EDVPropDlgMountNotifyCB(
	edv_propdlg_struct *pd,
	const gint dev_num, edv_device_struct *dev,
	gboolean is_mounted
);
void EDVPropDlgWriteProtectChangedCB(
	edv_propdlg_struct *pd, const gboolean state
);
void EDVPropDlgReconfiguredNotifyCB(edv_propdlg_struct *pd);

/* Updating */
static void EDVPropDlgGeneralPageUpdate(
	edv_propdlg_struct *pd, struct stat *lstat_buf,
	const gint lstat_error_code,
	const gboolean update_settable_values
);
static void EDVPropDlgDetailsPageUpdate(
	edv_propdlg_struct *pd, struct stat *lstat_buf,
	GList *ext_props_list
);
static void EDVPropDlgLinkPageUpdate(edv_propdlg_struct *pd);
static void EDVPropDlgDeviceNodePageUpdate(
	edv_propdlg_struct *pd, struct stat *lstat_buf
);

/* Utilities */
static void EDVPropDlgSetWidgetChangedSignal(
	edv_propdlg_struct *pd, GtkWidget *w
);

/* Page Creation */
static void EDVPropDlgCreateGeneralPage(
	edv_propdlg_struct *pd,
	GtkWidget *parent,
	edv_object_struct *obj
);
static void EDVPropDlgCreateDetailsPage(
	edv_propdlg_struct *pd,
	GtkWidget *parent,
	edv_object_struct *obj
);
static void EDVPropDlgCreateLinkPage(
	edv_propdlg_struct *pd,
	GtkWidget *parent,
	edv_object_struct *obj
);
static void EDVPropDlgCreateDevicePage(
	edv_propdlg_struct *pd,
	GtkWidget *parent,
	edv_object_struct *obj,
	const gint dev_num, edv_device_struct *dev
);
static void EDVPropDlgCreateDeviceUsage(
	edv_propdlg_struct *pd,
	edv_device_struct *dev
);
static void EDVPropDlgCreateDeviceNodePage(
	edv_propdlg_struct *pd,
	GtkWidget *parent,
	edv_object_struct *obj
);

/* Properties Dialog */
edv_propdlg_struct *EDVPropDlgNew(
	edv_core_struct *core,
	const edv_location_type location_type,
	edv_object_struct *obj,
	const guint recycled_obj_index,
	const gchar *arch_path
);
void EDVPropDlgSetHasChanges(
	edv_propdlg_struct *pd, const gboolean has_changes
);
void EDVPropDlgUpdateMenus(edv_propdlg_struct *pd);
void EDVPropDlgSetBusy(edv_propdlg_struct *pd, const gboolean busy);
gboolean EDVPropDlgIsMapped(edv_propdlg_struct *pd);
void EDVPropDlgMap(edv_propdlg_struct *pd);
void EDVPropDlgUnmap(edv_propdlg_struct *pd);
void EDVPropDlgDelete(edv_propdlg_struct *pd);


#define PROPDLG_TITLE			"Properties"

#define PROPDLG_WIDTH			380
#define PROPDLG_HEIGHT			450

#define PROGDLG_USE_PERMISSION_CHECK_BUTTON	FALSE

#define PROPDLG_VALUE_NOT_AVAILABLE_STR	"*Not Available*"
#define PROPDLG_TIME_STAMP_NOT_SET_STR	"*Not Set*"


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	GtkWindow "delete_event" signal callback.
 */
static gint EDVPropDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return(TRUE);

	if(pd->freeze_count > 0)
	    return(TRUE);

	EDVPropDlgUnmap(pd);

	return(TRUE);
}

/*
 *      GtkNotebook "switch_page" signal callback.
 */
static void EDVPropDlgSwitchPageCB(
	GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
	gpointer data
)
{
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

}

/*
 *	Any GtkWidget "changed" signal callback.
 */
static void EDVPropDlgAnyChangedCB(GtkWidget *widget, gpointer data)
{
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	EDVPropDlgSetHasChanges(pd, TRUE);
}


/*
 *	OK callback.
 */
static void EDVPropDlgOKCB(GtkWidget *widget, gpointer data)
{
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	EDVPropDlgSetBusy(pd, TRUE);

	EDVPropDlgApply(pd);
	EDVPropDlgUnmap(pd);
	EDVPropDlgSetHasChanges(pd, FALSE);

	EDVPropDlgSetBusy(pd, FALSE);
}

/*
 *	Apply callback.
 */
static void EDVPropDlgApplyCB(GtkWidget *widget, gpointer data)
{
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	EDVPropDlgSetBusy(pd, TRUE);

	EDVPropDlgApply(pd);
	EDVPropDlgSetHasChanges(pd, FALSE);

	EDVPropDlgSetBusy(pd, FALSE);
}

/*
 *	Cancel callback.
 */
static void EDVPropDlgCancelCB(GtkWidget *widget, gpointer data)
{
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	EDVPropDlgUnmap(pd);
}


/*
 *	Date touch callback.
 */
static void EDVPropDlgDateTouchCB(GtkWidget *widget, gpointer data)
{
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	EDVPropDlgDateTouch(pd);
}

/*
 *	Date touch specific.
 */
static void EDVPropDlgDateTouchSpecificCB(GtkWidget *widget, gpointer data)
{
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	EDVPropDlgDateTouchSpecific(pd);
}


/*
 *	Details verbose GtkRadioButton "toggled" signal callback.
 */
static void EDVPropDlgDetailsVerboseToggledCB(
	GtkWidget *widget, gpointer data
)
{
	cfg_item_struct *cfg_list;
	edv_core_struct *core;
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	EDVPropDlgSetBusy(pd, TRUE);
	pd->freeze_count++;

	core = pd->core;
	cfg_list = core->cfg_list;

	/* Update the Details Page */
	EDVPropDlgDetailsPageUpdate(pd, NULL, pd->ext_props_list);

	pd->freeze_count--;
	EDVPropDlgSetBusy(pd, FALSE);
}

/*
 *	Details raw GtkRadioButton "toggled" signal callback.
 */
static void EDVPropDlgDetailsRawToggledCB(
	GtkWidget *widget, gpointer data
)
{
	cfg_item_struct *cfg_list;
	edv_core_struct *core;
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	EDVPropDlgSetBusy(pd, TRUE);
	pd->freeze_count++;

	core = pd->core;
	cfg_list = core->cfg_list;

	/* Update the Details Page */
	EDVPropDlgDetailsPageUpdate(pd, NULL, pd->ext_props_list);

	pd->freeze_count--;
	EDVPropDlgSetBusy(pd, FALSE);
}

/*
 *	Details GtkCList "select_row" signal callback.
 */
static void EDVPropDlgDetailsListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	/* Scroll to the selected row if it is not visible */
	if(gtk_clist_row_is_visible(clist, row) !=
	   GTK_VISIBILITY_FULL
	)
	    gtk_clist_moveto(
		clist,
		row, -1,		/* Row, column */
		0.5f, 0.0f		/* Row, column */
	    );

	/* Clear the DND icon */
	GUIDNDSetDragIcon(NULL, NULL, 0, 0);
}

/*
 *	Mount/Unmount Device callback.
 */
static void EDVPropDlgMountCB(GtkWidget *widget, gpointer data)
{
	gboolean original_mount_state;
	gint status, dev_num;
	GtkWidget *toplevel;
	edv_device_struct *dev;
	edv_core_struct *core;
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	toplevel = pd->toplevel;
	core = pd->core;

	dev_num = pd->dev_num;
	dev = ((dev_num >= 0) && (dev_num < core->total_devices)) ?
	    core->device[dev_num] : NULL;
	if(dev == NULL)
	    return;

	EDVPropDlgSetBusy(pd, TRUE);

	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Unmount or mount? */
 	if(EDV_DEVICE_IS_MOUNTED(dev))
	    status = EDVDeviceUnmount(
		core, dev,
		TRUE, TRUE,
		toplevel
	    );
	else
	    status = EDVDeviceMount(
		core, dev,
		TRUE, TRUE,
		toplevel
	    );

	/* Update all device mount states and stats */
	EDVDevicesListUpdateMountStates(
	    core->device, core->total_devices
	);
	EDVDevicesListUpdateStats(
	    core->device, core->total_devices
	);

	/* Mount/unmount error? */
	if(status != 0)
	{
	    /* Get the error message (if any) */
	    const gchar *last_error = EDVDeviceMountGetError(core);
	    if(!STRISEMPTY(last_error))
	    {
		EDVPlaySoundError(core);
		EDVMessageError(
		    original_mount_state ?
			"Unmount Failed" : "Mount Failed",
		    last_error,
		    NULL,
		    toplevel
		);
	    }
	}
	else
	{
	    /* Notify about this object being mounted/unmounted */
	    EDVObjectMountEmit(
		core, dev_num, dev, EDV_DEVICE_IS_MOUNTED(dev)
	    );

/* Warning, some widgets on the Propeties Dialog may be invalid at this
 * point if fetched earlier in this function
 */
	}

	EDVPropDlgSetBusy(pd, FALSE);
}

/*
 *	Eject Device callback.
 */
static void EDVPropDlgEjectCB(GtkWidget *widget, gpointer data)
{
	gboolean original_mount_state;
	gint status, dev_num;
	GtkWidget *toplevel;
	edv_device_struct *dev;
	edv_core_struct *core;
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	toplevel = pd->toplevel;
	core = pd->core;

	dev_num = pd->dev_num;
	dev = ((dev_num >= 0) && (dev_num < core->total_devices)) ?
	    core->device[dev_num] : NULL;
	if(dev == NULL)
	    return;

	EDVPropDlgSetBusy(pd, TRUE);

	original_mount_state = EDV_DEVICE_IS_MOUNTED(dev);

	/* Need to unmount first? */
	if(EDV_DEVICE_IS_MOUNTED(dev))
	    EDVDeviceUnmount(
		core, dev,
		TRUE, TRUE,
		toplevel
	    );

	/* Eject */
	status = EDVDeviceEject(
	    core, dev,
	    TRUE, TRUE,
	    toplevel
	);

	/* Update all device mount states and stats */
	EDVDevicesListUpdateMountStates(
	    core->device, core->total_devices
	);
	EDVDevicesListUpdateStats(
	    core->device, core->total_devices
	);

	/* Eject error? */
	if(status != 0)
	{
	    /* Get the error message (if any) */
	    const gchar *last_error = EDVDeviceMountGetError(core);
	    if(!STRISEMPTY(last_error))
	    {
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Eject Failed",
		    last_error,
		    NULL,
		    toplevel
		);
	    }
	}
	else
	{   
	    /* Notify about this object being ejected */
	    EDVObjectMountEmit(
		core, dev_num, dev, EDV_DEVICE_IS_MOUNTED(dev)
	    );
	}

	EDVPropDlgSetBusy(pd, FALSE);
}

/*
 *      Runs the file system check command.
 */
static void EDVPropDlgRunCheckCB(GtkWidget *widget, gpointer data)
{
	gint dev_num;
	edv_device_struct *dev;
	edv_core_struct *core;
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	core = pd->core;

	dev_num = pd->dev_num;
	dev = ((dev_num >= 0) && (dev_num < core->total_devices)) ?
	    core->device[dev_num] : NULL;
	if(dev == NULL)
	    return;

	EDVPropDlgSetBusy(pd, TRUE);

	EDVRunDeviceCheck(core, dev, pd->toplevel);

	EDVPropDlgSetBusy(pd, FALSE);
}

/*
 *	Runs the device tools command.
 */
static void EDVPropDlgRunToolsCB(GtkWidget *widget, gpointer data)
{
	gint dev_num;
	edv_device_struct *dev;
	edv_core_struct *core;
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	core = pd->core;

	dev_num = pd->dev_num;
	dev = ((dev_num >= 0) && (dev_num < core->total_devices)) ?
	    core->device[dev_num] : NULL;
	if(dev == NULL) 
	    return;

	EDVPropDlgSetBusy(pd, TRUE);

	EDVRunDeviceTools(core, dev, pd->toplevel);

	EDVPropDlgSetBusy(pd, FALSE);
}

/*
 *      Runs the device format command.
 */
static void EDVPropDlgRunFormatCB(GtkWidget *widget, gpointer data)
{
	gint dev_num;
	edv_device_struct *dev;
	edv_core_struct *core;
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	core = pd->core;

	dev_num = pd->dev_num;
	dev = ((dev_num >= 0) && (dev_num < core->total_devices)) ?
	    core->device[dev_num] : NULL;
	if(dev == NULL) 
	    return;

	EDVPropDlgSetBusy(pd, TRUE);

	EDVRunDeviceFormat(core, dev, pd->toplevel);

	EDVPropDlgSetBusy(pd, FALSE);
}


/*
 *	Maps the popup list for the owner entry.
 */
static void EDVPropDlgOwnerMapPUListCB(GtkWidget *widget, gpointer data)
{
	gint nitems, nitems_visible;
	gchar *old_value;
	const gchar *value;
	GtkEntry *entry;
	pulist_struct *pulist;
	edv_core_struct *core;
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	/* No General Page? */
	if(pd->gen_toplevel == NULL)
	    return;

	entry = GTK_ENTRY(pd->gen_owner_entry);
	core = pd->core;
	pulist = core->users_pulist;
	if(PUListIsQuery(pulist))
	    return;

	old_value = STRDUP(gtk_entry_get_text(entry));

	nitems = PUListGetTotalItems(pulist);
	nitems_visible = MIN(15, nitems);

	/* Block input and get value */
	value = PUListMapQuery(
	    pulist,			/* Popup List */
	    old_value,			/* Initial Value */
	    nitems_visible,
	    PULIST_RELATIVE_BELOW,	/* Popup Relativity */
	    GTK_WIDGET(entry),		/* Relative Widget */
	    widget			/* Map Trigger Widget */
	);
	if(value != NULL)
	{
	    /* Set new value if it is different from the old value  */
	    if((old_value != NULL) ? strcmp(old_value, value) : TRUE)
		gtk_entry_set_text(entry, value);
	}

	g_free(old_value);
}

/*
 *      Maps the popup list for the group entry.
 */
static void EDVPropDlgGroupMapPUListCB(GtkWidget *widget, gpointer data)
{
	gint nitems, nitems_visible;
	gchar *old_value;
	const gchar *value;
	GtkEntry *entry;
	pulist_struct *pulist;
	edv_core_struct *core;
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	/* No General Page? */
	if(pd->gen_toplevel == NULL)
	    return;

	entry = GTK_ENTRY(pd->gen_group_entry);
	core = pd->core;
	pulist = core->groups_pulist;
	if(PUListIsQuery(pulist))
	    return;

	old_value = STRDUP(gtk_entry_get_text(entry));

	nitems = PUListGetTotalItems(pulist);
	nitems_visible = MIN(15, nitems);

	/* Block input and get value */
	value = PUListMapQuery(
	    pulist,			/* Popup List */
	    old_value,			/* Initial Value */
	    nitems_visible,
	    PULIST_RELATIVE_BELOW,	/* Popup Relativity */
	    GTK_WIDGET(entry),		/* Relative Widget */
	    widget			/* Map Trigger Widget */
	);
	if(value != NULL)
	{
	    /* Set new value if it is different from the old value  */
	    if((old_value != NULL) ? strcmp(old_value, value) : TRUE)
		gtk_entry_set_text(entry, value);
	}

	g_free(old_value);
}

/*
 *	Link value GtkEntry "changed" callback.
 */
static void EDVPropDlgLinkChangedCB(GtkWidget *widget, gpointer data)
{
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	pd->freeze_count++;
	EDVPropDlgLinkPageUpdate(pd);
	pd->freeze_count--;
}

/*
 *	Link value browse callback.
 */
static void EDVPropDlgLinkBrowseCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL, *cur_path;
	const gchar *s, *path;
	GtkWidget *toplevel;
	GtkEntry *entry;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	edv_core_struct *core;
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	/* No Link Page? */
	if(pd->link_toplevel == NULL)
	    return;

	toplevel = pd->toplevel;
	core = pd->core;

	/* Get the starting location
	 *
	 * If the current link's target value is an absolute
	 * path then use its parent as the starting location,
	 * otherwise use the parent of the link
	 */
	entry = (GtkEntry *)pd->link_target_entry;
	if(entry == NULL)
	    return;

	EDVPropDlgSetBusy(pd, TRUE);
	pd->freeze_count++;

	path = pd->path;
	s = gtk_entry_get_text(entry);
	if(s != NULL)
	{
	    if(g_path_is_absolute(s))
		cur_path = g_dirname(s);
	    else if(path != NULL)
		cur_path = g_dirname(path);
	    else
		cur_path = NULL;
	}
	else if(path != NULL)
	{
	    cur_path = g_dirname(path);
	}
	else
	{
	    cur_path = NULL;
	}

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);
	     
	/* Query the user for the link's value */   
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Set Link Value",
	    "Set", "Cancel",
	    cur_path,
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	g_free(cur_path);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response)
	{
	    const gchar *new_path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(new_path != NULL)
	    {
		/* Query the user to use a relative path */
		gint response;
		gchar *msg = g_strdup_printf(
"Would you like to use a relative path from:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s",
		    path, new_path
		);
		EDVPlaySoundQuestion(core);
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
		    "Set Relative Path",
		    msg,
		    NULL,
		    CDIALOG_ICON_QUESTION,
		    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		    CDIALOG_BTNFLAG_YES
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);
		if(response == CDIALOG_RESPONSE_YES)
		{
		    /* Set the link's target as a relative path */
		    gchar *rel_path = EDVPlotRelativePath(
			path, new_path
		    );
		    gtk_entry_set_text(
			entry,
			(rel_path != NULL) ? rel_path : new_path
		    );
		    g_free(rel_path);
		    gtk_entry_set_position(entry, -1);
		}
		else
		{
		    /* Set the link's target */
		    gtk_entry_set_text(entry, new_path);
		    gtk_entry_set_position(entry, -1);
		}

		EDVPropDlgSetHasChanges(pd, TRUE);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	pd->freeze_count--;
	EDVPropDlgSetBusy(pd, FALSE);
}

/*
 *	Link Make Relative Path callback.
 */
static void EDVPropDlgLinkMakeRelativePathCB(GtkWidget *widget, gpointer data)
{
	const gchar *path;
	gchar *rel_path, *target;
	GtkWidget *toplevel;
	GtkEntry *entry;
	edv_core_struct *core;
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	/* No Link Page? */
	if(pd->link_toplevel == NULL)
	    return;

	path = pd->path;
	if(STRISEMPTY(path))
	    return;

	toplevel = pd->toplevel;
	core = pd->core;
	entry = (GtkEntry *)pd->link_target_entry;
	if(entry == NULL)
	    return;

	EDVPropDlgSetBusy(pd, TRUE);
	pd->freeze_count++;

	/* Get the link's current target value */
	target = STRDUP(gtk_entry_get_text(entry));
	if(target == NULL)
	{
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Make Relative Path Failed",
"Unable to obtain the current target value.",
		NULL,
		toplevel
	    );
	    pd->freeze_count--;
	    EDVPropDlgSetBusy(pd, FALSE);
	    return;
	}

	/* Check if the target value is already a relative path */
	if(!g_path_is_absolute(target))
	{
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Make Relative Path Failed",
"The target value is already a relative path.\n\
\n\
If you want to remake the relative path then set\n\
the target value to an absolute path and try again.",
		NULL,
		toplevel
	    );
	    g_free(target);
	    pd->freeze_count--;
	    EDVPropDlgSetBusy(pd, FALSE);
	    return;
	}

	/* Set the link's target as a relative path */
	rel_path = EDVPlotRelativePath(path, target);
	if(rel_path != NULL)
	{
	    gtk_entry_set_text(entry, rel_path);
	    g_free(rel_path);
	    gtk_entry_set_position(entry, -1);
	}
	else
	{
	    const gint error_code = (gint)errno;
	    gchar *msg = g_strdup_printf(
"Unable to plot a relative path from:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
		path, target, g_strerror(error_code)
	    );
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Make Relative Path Failed",
		msg,
		NULL,
		toplevel
	    );
	    g_free(msg);
	    g_free(target);
	    pd->freeze_count--;
	    EDVPropDlgSetBusy(pd, FALSE);
	    return;
	}

	g_free(target);

	EDVPropDlgSetHasChanges(pd, TRUE);

	pd->freeze_count--;
	EDVPropDlgSetBusy(pd, FALSE);
}

/*
 *	Link target properties callback.
 */
static void EDVPropDlgLinkTargetPropertiesCB(GtkWidget *widget, gpointer data)
{
	const gchar *value;
	GtkWidget *toplevel;
	GtkEntry *entry;
	edv_core_struct *core;
	edv_propdlg_struct *pd = EDV_PROPDLG(data);
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	/* No Link Page? */
	if(pd->link_toplevel == NULL)
	    return;

	toplevel = pd->toplevel;
	entry = (GtkEntry *)pd->link_target_entry;
	if(entry == NULL)
	    return;

	core = pd->core;

	EDVPropDlgSetBusy(pd, TRUE);
	pd->freeze_count++;

	/* Get the link's target value */
	value = gtk_entry_get_text(entry);
	if(STRISEMPTY(value))
	{
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Target Properties Failed",
"Unable to obtain the current target value.",
		NULL,
		toplevel
	    );
	}
	else if(pd->location_type == EDV_LOCATION_TYPE_VFS)
	{
	    gchar *full_tar_path;

	    /* Is the target value an absolute path? */
	    if(g_path_is_absolute(value))
	    {
		full_tar_path = STRDUP(value);
	    }
	    else
	    {
		/* Not an absolute path, prefix the parent path of
		 * the link to the target path
		 */
		const gchar *path = pd->path;
		gchar *parent_path = (path != NULL) ?
		    g_dirname(path) : STRDUP(path);
		full_tar_path = g_strconcat(
		    parent_path,
		    G_DIR_SEPARATOR_S,
		    value,
		    NULL
		);
		g_free(parent_path);
	    }

	    /* Got the full target path? */
	    if(full_tar_path != NULL)
	    {
		struct stat lstat_buf;

		/* Remove tailing deliminators and reduce any occurances
		 * of ../ and ./
		 */
		EDVSimplifyPath(full_tar_path);

		/* Check if the target path exists locally */
		if(lstat((const char *)full_tar_path, &lstat_buf))
		{
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
			g_strerror(error_code), full_tar_path
		    );
		    EDVPlaySoundWarning(core);
		    EDVMessageWarning(
			"Target Properties Failed",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
		else
		{
		    /* Got the target path's local stats
		     *
		     * Create an object to hold the target path's
		     * stats
		     */
		    edv_object_struct *tar_obj = EDVObjectNew();
		    if(tar_obj != NULL)
		    {
			EDVObjectSetPath(tar_obj, full_tar_path);
			EDVObjectSetStat(tar_obj, &lstat_buf);
			EDVObjectUpdateLinkFlags(tar_obj);

			/* Create a new Properties Dialog displaying
			 * the target object
			 */
			EDVNewPropertiesDialogVFS(
			    core,
			    tar_obj,
			    toplevel
			);

			EDVObjectDelete(tar_obj);
		    }
		}

		g_free(full_tar_path);
	    }
	    else
	    {
		const gchar *path = pd->path;
		gchar *parent_path = (path != NULL) ?
		    g_dirname(path) : STRDUP(path);
		gchar *msg = g_strdup_printf(
"Unable to form an absolute path to the target:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s",
		    value, parent_path
		);
		g_free(parent_path);
		EDVPlaySoundWarning(core);
 		EDVMessageWarning(
		    "Target Properties Failed",
		    msg,
		    NULL,
		    toplevel
		);
		g_free(msg);
	    }
	}

	pd->freeze_count--;
	EDVPropDlgSetBusy(pd, FALSE);
}


/*
 *	Object added callback.
 */
void EDVPropDlgObjectAddedNotifyCB(
	edv_propdlg_struct *pd, const gchar *path,
	struct stat *lstat_buf      
)
{
	const gchar *cur_path;
	edv_core_struct *core;

	if((pd == NULL) || STRISEMPTY(path))
	    return;

	core = pd->core;
	cur_path = pd->path;
	if(STRISEMPTY(cur_path))
	    return;

	if(pd->location_type == EDV_LOCATION_TYPE_VFS)
	{
	    /* Need to update the device usage? */
	    if(pd->dev_usage_frame != NULL)
	    {
		const gint dev_num = pd->dev_num;
		edv_device_struct *dev = ((dev_num >= 0) &&
		    (dev_num < core->total_devices)) ?
			core->device[dev_num] : NULL;
		if(dev != NULL)
		{
		    const gchar *mount_path = dev->mount_path;
		    if(mount_path != NULL)
		    {
			if(strpfx((const char *)path, (const char *)mount_path))
			    EDVPropDlgCreateDeviceUsage(pd, dev);
		    }
		}
	    }

	    /* Is the added object the object that we are displaying? */
	    if(!strcmp((const char *)path, (const char *)cur_path))
	    {
		/* Update the General, Details, and Device Node pages */
		EDVPropDlgGeneralPageUpdate(pd, lstat_buf, 0, TRUE);
		EDVPropDlgDetailsPageUpdate(pd, lstat_buf, pd->ext_props_list);
		EDVPropDlgDeviceNodePageUpdate(pd, lstat_buf);

		EDVPropDlgUpdateMenus(pd);
	    }
	}
}

/*
 *	Object modified callback.
 */
void EDVPropDlgObjectModifiedNotifyCB(
	edv_propdlg_struct *pd,
	const gchar *path, const gchar *new_path,
	struct stat *lstat_buf
)
{
	const gchar *cur_path;
	edv_core_struct *core;

	if((pd == NULL) || STRISEMPTY(path))
	    return;

	if(new_path == NULL)
	    new_path = path;

	core = pd->core;
	cur_path = pd->path;
	if(STRISEMPTY(cur_path))
	    return;

	if(pd->location_type == EDV_LOCATION_TYPE_VFS)
	{
	    /* Need to update the device usage? */
	    if(pd->dev_usage_frame != NULL)
	    {
		const gint dev_num = pd->dev_num;
		edv_device_struct *dev = ((dev_num >= 0) &&
		    (dev_num < core->total_devices)) ?
			core->device[dev_num] : NULL;
		if(dev != NULL)
		{
		    const gchar *mount_path = dev->mount_path;
		    if(mount_path != NULL)
		    {
			if(strpfx((const char *)path, (const char *)mount_path) ||
			   strpfx((const char *)new_path, (const char *)mount_path)
			)
			    EDVPropDlgCreateDeviceUsage(pd, dev);
		    }
		}
	    }

	    /* Is the modified object the object that we are displaying? */
	    if(!strcmp((const char *)path, (const char *)cur_path))
	    {
		/* Change in path name? */
		if(path != new_path)
		{
		    if(strcmp((const char *)path, (const char *)new_path))
		    {
			/* Set the new path on the dialog */
			g_free(pd->path);
			pd->path = STRDUP(new_path);
			cur_path = pd->path;
		    }
		}

		/* Update the General, Details, and Device Node pages */
		EDVPropDlgGeneralPageUpdate(pd, lstat_buf, 0, TRUE);
		EDVPropDlgDetailsPageUpdate(pd, lstat_buf, pd->ext_props_list);
		EDVPropDlgDeviceNodePageUpdate(pd, lstat_buf);

		EDVPropDlgUpdateMenus(pd);
	    }
	}
}

/*
 *	Object removed callback.
 */
void EDVPropDlgObjectRemovedNotifyCB(
	edv_propdlg_struct *pd, const gchar *path
)
{
	const gchar *cur_path;
	edv_core_struct *core;

	if((pd == NULL) || STRISEMPTY(path))
	    return;

	core = pd->core;
	cur_path = pd->path;
	if(STRISEMPTY(cur_path))
	    return;

	if(pd->location_type == EDV_LOCATION_TYPE_VFS)
	{
	    /* Need to update the device usage? */
	    if(pd->dev_usage_frame != NULL)
	    {
		const gint dev_num = pd->dev_num;
		edv_device_struct *dev = ((dev_num >= 0) &&
		    (dev_num < core->total_devices)) ?
			core->device[dev_num] : NULL;
		if(dev != NULL)
		{
		    const gchar *mount_path = dev->mount_path;
		    if(mount_path != NULL)
		    {
			if(strpfx((const char *)path, (const char *)mount_path))
			    EDVPropDlgCreateDeviceUsage(pd, dev);
		    }
		}
	    }

	    /* Is the removed object the object that we are displaying? */
	    if(!strcmp((const char *)path, (const char *)cur_path))
	    {
		/* Unmap the dialog since the object no longer exists */
		EDVPropDlgUnmap(pd);
	    }
	}
}

/*
 *	Recycled object added notify callback.
 */
void EDVPropDlgRecycledObjectAddedNotifyCB(
	edv_propdlg_struct *pd, const guint index
)
{

}

/*
 *	Recycled object modified notify callback.
 */
void EDVPropDlgRecycledObjectModifiedNotifyCB(
	edv_propdlg_struct *pd, const guint index
)
{
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	core = pd->core;
	cfg_list = core->cfg_list;

	if((pd->location_type == EDV_LOCATION_TYPE_RECBIN) &&
	   (index == pd->recycled_obj_index)
	)
	{
	    /* Get the recycled object's stats */
	    edv_recycled_object_struct *obj = EDVRecBinObjectStat(
		EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
		pd->recycled_obj_index
	    );
	    if(obj != NULL)
	    {
		struct stat lstat_buf;

		/* Convert the recycled object's stats to struct stat */
		EDVRecycledObjectSetToStat(&lstat_buf, obj);

		/* Delete the recycled object's stats */
		EDVRecycledObjectDelete(obj);

		/* Update the General, Details, and Device Node pages */
		EDVPropDlgGeneralPageUpdate(pd, &lstat_buf, 0, TRUE);
		EDVPropDlgDetailsPageUpdate(pd, &lstat_buf, pd->ext_props_list);
		EDVPropDlgDeviceNodePageUpdate(pd, &lstat_buf);
	    }
	}
}

/*
 *	Recycled object removed notify callback.
 */
void EDVPropDlgRecycledObjectRemovedNotifyCB(
	edv_propdlg_struct *pd, const guint index
)
{
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	core = pd->core;

	if((pd->location_type == EDV_LOCATION_TYPE_RECBIN) &&
	   (index == pd->recycled_obj_index)
	)
	{
	    /* Unmap the dialog since the object no longer exists */
	    EDVPropDlgUnmap(pd);
	}
}

/*
 *	MIME Type added notify callback.
 */
void EDVPropDlgMIMETypeAddedNotifyCB(
	edv_propdlg_struct *pd,
	const gint mt_num, edv_mime_type_struct *m
)
{
	const gchar *path, *mime_type;
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	mime_type = pd->mime_type;
	if((mime_type == NULL) || (m->type == NULL))
	    return;

	core = pd->core;
	path = pd->path;
	if(STRISEMPTY(path))
	    return;

	/* Is the modified MIME Type the same MIME Type used for
	 * this object?
	 */
	if(!g_strcasecmp(mime_type, m->type))
	{
	    GtkWidget *w = pd->gen_icon_pm;
	    if(w != NULL)
	    {
		const gchar *name = g_basename(path);
		edv_mime_type_icon_state state;
		GdkPixmap *pixmap;
		GdkBitmap *mask;

		EDVMimeTypeRealize(m, FALSE);

		/* Get the icon pixmap and mask of this MIME Type */
		if(EDVIsObjectNameHidden(name))
		    state = EDV_MIME_TYPE_ICON_STATE_HIDDEN;
		else
		    state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
		pixmap = m->medium_pixmap[state];
		mask = m->medium_mask[state];

		/* Update the icon */
		gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
	    }
	}
}

/*
 *	MIME Type modified notify callback.
 */
void EDVPropDlgMIMETypeModifiedNotifyCB(
	edv_propdlg_struct *pd,
	const gint mt_num, edv_mime_type_struct *m
)
{
	const gchar *path, *mime_type;
	edv_core_struct *core;

	if((pd == NULL) || (m == NULL))
	    return;

	mime_type = pd->mime_type;
	if((mime_type == NULL) || (m->type == NULL))
	    return;

	core = pd->core;
	path = pd->path;
	if(STRISEMPTY(path))
	    return;

	/* Is the modified MIME Type the same MIME Type used for
	 * this object?
	 */
	if(!g_strcasecmp(mime_type, m->type))
	{
	    GtkWidget *w = pd->gen_icon_pm;
	    if(w != NULL)
	    {
		const gchar *name = g_basename(path);
		edv_mime_type_icon_state state;
		GdkPixmap *pixmap;
		GdkBitmap *mask;

		EDVMimeTypeRealize(m, FALSE);

		/* Get the icon pixmap and mask of this MIME Type */
		if(EDVIsObjectNameHidden(name))
		    state = EDV_MIME_TYPE_ICON_STATE_HIDDEN;
		else
		    state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
		pixmap = m->medium_pixmap[state];
		mask = m->medium_mask[state];

		/* Update the icon */
		gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
	    }
	}
}

/*
 *	MIME Type removed notify callback.
 */
void EDVPropDlgMIMETypeRemovedNotifyCB(
	edv_propdlg_struct *pd,
	const gint mt_num
)
{
	if(pd == NULL)
	    return;

	if(pd->location_type == EDV_LOCATION_TYPE_VFS)
	{
	    /* Do nothing */
	}
}

/*
 *      Mount/unmount notify callback.
 */
void EDVPropDlgMountNotifyCB(
	edv_propdlg_struct *pd, 
	const gint dev_num, edv_device_struct *dev,
	const gboolean is_mounted
)
{
	if(pd == NULL)
	    return;

	if(pd->location_type == EDV_LOCATION_TYPE_VFS)
	{
	    /* Update device usage */
	    if((pd->dev_usage_frame != NULL) &&
	       (dev_num == pd->dev_num)
	    )
	    {
		if(dev != NULL)
		    EDVPropDlgCreateDeviceUsage(pd, dev);
	    }
	}
}

/*
 *	Write protect changed callback.
 */
void EDVPropDlgWriteProtectChangedCB(edv_propdlg_struct *pd, gboolean state)
{
	if(pd == NULL)
	    return;

	if(pd->freeze_count > 0)
	    return;

	EDVPropDlgUpdateMenus(pd);
}

/*
 *	Reconfigured notify callback.
 */
void EDVPropDlgReconfiguredNotifyCB(edv_propdlg_struct *pd)
{
	const gchar *path;
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkWidget *w;
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	core = pd->core;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;
	path = pd->path;

	if(pd->location_type == EDV_LOCATION_TYPE_VFS)
	{
	    /* Get the object's statistics */
	    struct stat lstat_buf;
	    const gint lstat_status = (path != NULL) ?
		lstat((const char *)path, &lstat_buf) : -1;
	    const gint lstat_error_code = (gint)errno;

	    /* Update the General Page */
	    if(lstat_status == 0)
		EDVPropDlgGeneralPageUpdate(pd, &lstat_buf, 0, FALSE);
	    else
		EDVPropDlgGeneralPageUpdate(pd, NULL, lstat_error_code, FALSE);

	    /* Update the Details Page */
	    if(lstat_status == 0)
		EDVPropDlgDetailsPageUpdate(pd, &lstat_buf, pd->ext_props_list);
	    else
		EDVPropDlgDetailsPageUpdate(pd, NULL, pd->ext_props_list);
	}

	/* Update the RC styles */
	if(standard_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		pd->toplevel, standard_rcstyle
	    );

	/* Does the General Page exist? */
	if(pd->gen_toplevel != NULL)
	{
	    w = MenuButtonGetMenu(pd->gen_date_touch_btn);
	    if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(
		    w, standard_rcstyle
		);
	}

	EDVPropDlgUpdateMenus(pd);
}


/*
 *	Updates the General Page.
 *
 *	Updates the name, location, type, size, and timestamps.
 *
 *	The lstat_buf specifies the object's values to update with. If
 *	lstat_buf is NULL then the values will reflect that the
 *	values are not available.
 *
 *	If lstat_buf is NULL then lstat_error_code will be used to
 *	indicate why the object's values were not available.
 *
 *	If update_settable_values is TRUE then the owner, group,
 *	and permissions will be updated as well.
 */
static void EDVPropDlgGeneralPageUpdate(
	edv_propdlg_struct *pd, struct stat *lstat_buf,
	const gint lstat_error_code,
	const gboolean update_settable_values
)
{
	const gchar *path, *name, *date_format;
	GtkWidget *w;
	cfg_item_struct *cfg_list;
	edv_date_relativity date_relativity;
	edv_location_type location_type;
	edv_object_type type;
	edv_recycled_object_struct *recobj = NULL;
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	/* No General Page? */
	if(pd->gen_toplevel == NULL)
	    return;

	pd->freeze_count++;

	type = pd->type;
	location_type = pd->location_type;
	if(pd->path != NULL)
	    path = g_path_is_absolute(pd->path) ?
		pd->path : g_basename(pd->path);
	else
	    path = NULL;
	name = (pd->path != NULL) ? g_basename(pd->path) : NULL;
	core = pd->core;
	cfg_list = core->cfg_list;

	date_relativity = (edv_date_relativity)EDV_GET_I(
	    EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);


	/* Get the object's statistics based on the location type */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    break;
	  case EDV_LOCATION_TYPE_RECBIN:
	    /* Get the recycled object values */
	    if(pd->recycled_obj_index > 0)
		recobj = EDVRecBinObjectStat(
		    EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
		    pd->recycled_obj_index
		);
	    break;
	  case EDV_LOCATION_TYPE_ARCHIVE:
#if 0
	    /* Get the archive object values */
	    if(!STRISEMPTY(pd->arch_path) && !STRISEMPTY(path))
		archobj = EDVArchStat(
		    core,
		    pd->arch_path,
		    path,
		    NULL
		);
#endif
	    break;
	}


	/* Icon */
	w = pd->gen_icon_pm;
	if(w != NULL)
	{
	    GdkPixmap *pixmap = NULL;
	    GdkBitmap *mask = NULL;
	    if(lstat_buf != NULL)
	    {
		GdkPixmap *pixmap_hid;
		GdkBitmap *mask_hid;
		EDVMatchObjectIcon(
		    core->device, core->total_devices,
		    core->mimetype, core->total_mimetypes,
		    type,
		    path,
		    TRUE,			/* Link valid */
		    EDVStatModeToEDVPermissions(lstat_buf->st_mode),
		    1,				/* Medium icons */
		    &pixmap, &mask,
		    NULL, NULL,
		    NULL, NULL,
		    &pixmap_hid, &mask_hid
		);
		if(EDVIsObjectNameHidden(name) && (pixmap_hid != NULL))
		{
		    pixmap = pixmap_hid;
		    mask = mask_hid;
		}
	    }
	    if(pixmap != NULL)
		gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
	}

	/* Name */
	w = pd->gen_name_label;
	if(w != NULL)
	{
	    gchar *s;
	    if(name != NULL)
		s = EDVShortenPath(name, 40);
	    else
		s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
	    gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
	    g_free(s);
	}

	/* Location */
	w = pd->gen_location_label;
	if(w != NULL)
	{
	    gchar *location = NULL;

	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(!STRISEMPTY(path))
		{
		    if(lstat_buf != NULL)
		    {
			location = g_dirname(path);
			if(location != NULL)
			{
			    gchar *s = EDVShortenPath(location, 45);
			    g_free(location);
			    location = s;
			}
			else
			    location = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		    }
		    else
			location = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		}
		else
		{
		    location = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		}
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		location = g_strdup_printf(
		    "#%u",
		    pd->recycled_obj_index
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		if(!STRISEMPTY(pd->path))
		{
		    const gchar *in_arch_path = pd->path;
		    gchar *s;

		    if(g_path_is_absolute(in_arch_path))
			s = STRDUP(in_arch_path);
		    else
			s = g_strconcat(
			    ".",
			    G_DIR_SEPARATOR_S,
			    in_arch_path,
			    NULL
			);
		    if(s != NULL)
		    {
			location = g_dirname(s);
			g_free(s);
			if(location != NULL)
			{
			    s = EDVShortenPath(location, 34);
			    g_free(location);
			    location = s;
			}
		    }
		    else
			location = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		}
		else
		{
		    location = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		}
		break;
	    }
	    gtk_label_set_text(GTK_LABEL(w), (location != NULL) ? location : "");
	    g_free(location);
	}

	/* Type */
	w = pd->gen_type_label;
	if(w != NULL)
	{
	    gchar *s;

	    if(pd->mime_type != NULL)
	    {
		s = EDVShortenPath(pd->mime_type, 20);
	    }
	    else if(lstat_buf != NULL)
	    {
		s = STRDUP(EDVGetObjectTypeName(
		    EDVStatModeToObjectType(lstat_buf->st_mode)
		));
	    }
	    else
	    {
		s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
	    }
	    gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
	    g_free(s);
	}

	/* Size */
	w = pd->gen_size_label;
	if(w != NULL)
	{
	    gchar *s;
	    if(lstat_buf != NULL)
		s = STRDUP(EDVSizeStrDelim((gulong)lstat_buf->st_size));
	    else
		s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
	    gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
	    g_free(s);
	}

	/* Last Accessed */
	w = pd->gen_date_access_label;
	if(w != NULL)
	{
	    gchar *s;
	    if(lstat_buf != NULL)
	    {
		const gulong t = (gulong)lstat_buf->st_atime;
		if(t > 0l)
		    s = STRDUP(EDVDateFormatString(
			t, date_format, date_relativity
		    ));
		else
		    s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
	    }
	    else
		s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
	    gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
	    g_free(s);
	}

	/* Last Modified */
	w = pd->gen_date_modify_label;
	if(w != NULL)
	{
	    gchar *s;
	    if(lstat_buf != NULL)
	    {
		const gulong t = (gulong)lstat_buf->st_mtime;
		if(t > 0l)
		    s = STRDUP(EDVDateFormatString(
			t, date_format, date_relativity
		    ));
		else
		    s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
	    }
	    else
		s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
	    gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
	    g_free(s);
	}

	/* Last Changed */
	w = pd->gen_date_change_label;
	if(w != NULL)
	{
	    gchar *s;
	    if(lstat_buf != NULL)
	    {
		const gulong t = (gulong)lstat_buf->st_ctime;
		if(t > 0l)
		    s = STRDUP(EDVDateFormatString(
			t, date_format, date_relativity
		    ));
		else
		    s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
	    }
	    else
		s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
	    gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
	    g_free(s);
	}

	/* Deleted On */
	w = pd->gen_date_deleted_label;
	if(w != NULL)
	{
	    if(location_type == EDV_LOCATION_TYPE_RECBIN)
	    {
		gchar *s;

		if(recobj != NULL)
		{
		    const gulong t = (gulong)recobj->deleted_time;
		    if(t > 0l)
			s = STRDUP(EDVDateFormatString(
			    t, date_format, date_relativity
			));
		    else
			s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
		}
		else
		    s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		gtk_label_set_text(GTK_LABEL(w), (s != NULL) ? s : "");
		g_free(s);
	    }
	}

	/* Owner */
	w = pd->gen_owner_entry;
	if((w != NULL) && update_settable_values)
	{
	    gchar *s;
	    if(lstat_buf != NULL)
		s = EDVUIDGetNameFromUID(
		    core->uids_list,
		    (gint)lstat_buf->st_uid, NULL
		);
	    else
		s = NULL;
	    gtk_entry_set_text(GTK_ENTRY(w), (s != NULL) ? s : "");
	    g_free(s);
	}

	/* Group */
	w = pd->gen_group_entry;
	if((w != NULL) && update_settable_values)
	{
	    gchar *s;
	    if(lstat_buf != NULL)
		s = EDVGIDGetNameFromGID(
		    core->gids_list,
		    (gint)lstat_buf->st_gid, NULL
		);
	    else
		s = NULL;
	    gtk_entry_set_text(GTK_ENTRY(w), (s != NULL) ? s : "");
	    g_free(s);
	}

	/* Permissions */
	w = pd->gen_ur_check;
	if((w != NULL) && update_settable_values)
	{
	    gboolean b;

	    if(type == EDV_OBJECT_TYPE_LINK)
		b = FALSE;
	    else if(lstat_buf != NULL)
		b = (lstat_buf->st_mode & S_IRUSR) ? TRUE : FALSE;
	    else
		b = FALSE;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = pd->gen_uw_check;
	if((w != NULL) && update_settable_values)
	{
	    gboolean b;

	    if(type == EDV_OBJECT_TYPE_LINK)
		b = FALSE;
	    else if(lstat_buf != NULL)
		b = (lstat_buf->st_mode & S_IWUSR) ? TRUE : FALSE;
	    else
		b = FALSE;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = pd->gen_ux_check;
	if((w != NULL) && update_settable_values)
	{
	    gboolean b;

	    if(type == EDV_OBJECT_TYPE_LINK)
		b = FALSE;
	    else if(lstat_buf != NULL)
		b = (lstat_buf->st_mode & S_IXUSR) ? TRUE : FALSE;
	    else
		b = FALSE;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}

	w = pd->gen_gr_check;
	if((w != NULL) && update_settable_values)
	{
	    gboolean b;

	    if(type == EDV_OBJECT_TYPE_LINK)
		b = FALSE;
	    else if(lstat_buf != NULL)
		b = (lstat_buf->st_mode & S_IRGRP) ? TRUE : FALSE;
	    else
		b = FALSE;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = pd->gen_gw_check;
	if((w != NULL) && update_settable_values)
	{
	    gboolean b;

	    if(type == EDV_OBJECT_TYPE_LINK)
		b = FALSE;
	    else if(lstat_buf != NULL)
		b = (lstat_buf->st_mode & S_IWGRP) ? TRUE : FALSE;
	    else
		b = FALSE;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = pd->gen_gx_check;
	if((w != NULL) && update_settable_values)
	{
	    gboolean b;

	    if(type == EDV_OBJECT_TYPE_LINK)
		b = FALSE;
	    else if(lstat_buf != NULL)
		b = (lstat_buf->st_mode & S_IXGRP) ? TRUE : FALSE;
	    else
		b = FALSE;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}

	w = pd->gen_or_check;
	if((w != NULL) && update_settable_values)
	{
	    gboolean b;

	    if(type == EDV_OBJECT_TYPE_LINK)
		b = FALSE;
	    else if(lstat_buf != NULL)
		b = (lstat_buf->st_mode & S_IROTH) ? TRUE : FALSE;
	    else
		b = FALSE;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = pd->gen_ow_check;
	if((w != NULL) && update_settable_values)
	{
	    gboolean b;

	    if(type == EDV_OBJECT_TYPE_LINK)
		b = FALSE;
	    else if(lstat_buf != NULL)
		b = (lstat_buf->st_mode & S_IWOTH) ? TRUE : FALSE;
	    else
		b = FALSE;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = pd->gen_ox_check;
	if((w != NULL) && update_settable_values)
	{
	    gboolean b;

	    if(type == EDV_OBJECT_TYPE_LINK)
		b = FALSE;
	    else if(lstat_buf != NULL)
		b = (lstat_buf->st_mode & S_IXOTH) ? TRUE : FALSE;
	    else
		b = FALSE;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}

	/* SetUID, SetGID, and Sticky */
	w = pd->gen_suid_check;
	if((w != NULL) && update_settable_values)
	{
	    gboolean b;

	    if(type == EDV_OBJECT_TYPE_LINK)
		b = FALSE;
	    else if(lstat_buf != NULL)
		b = (lstat_buf->st_mode & S_ISUID) ? TRUE : FALSE;
	    else
		b = FALSE;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = pd->gen_sgid_check;
	if((w != NULL) && update_settable_values)
	{
	    gboolean b;

	    if(type == EDV_OBJECT_TYPE_LINK)
		b = FALSE;
	    else if(lstat_buf != NULL)
		b = (lstat_buf->st_mode & S_ISGID) ? TRUE : FALSE;
	    else
		b = FALSE;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}
	w = pd->gen_sticky_check;
	if((w != NULL) && update_settable_values)
	{
	    gboolean b;

	    if(type == EDV_OBJECT_TYPE_LINK)
		b = FALSE;
	    else if(lstat_buf != NULL)
		b = (lstat_buf->st_mode & S_ISVTX) ? TRUE : FALSE;
	    else
		b = FALSE;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);
	}

	/* Delete the recycled object */
	EDVRecycledObjectDelete(recobj);

	pd->freeze_count--;
}

/*
 *	Updates the Details Page.
 *
 *	Clears and regets the detail values for the Details GtkCList.
 *
 *	The lstat_buf specifies the object's values. If lstat_buf is
 *	NULL then the object values will be obtained from the object
 *	specified by pd->path. If pd->location_type is not
 *	EDV_LOCATION_TYPE_VFS then the respective object's values
 *	will be obtained based on the location type, for example,
 *	if pd->location_type is EDV_LOCATION_TYPE_RECBIN then
 *	the object will be referenced by pd->recycled_obj_index
 *	with EDVRecBinObjectStat().
 *
 *	The ext_props_list specifies a GList of
 *	edv_object_prop_struct property values.
 */
static void EDVPropDlgDetailsPageUpdate(
	edv_propdlg_struct *pd, struct stat *lstat_buf,
	GList *ext_props_list
)
{
	const gint border_minor = 2;
	struct stat obj_lstat_buf;
	gboolean verbose;
	gint	i,
		ncolumns, row,
		value_pixmap_width, value_pixmap_height;
	gulong block_size;
	gfloat last_x, last_y;
	gchar *s, **strv;
	const gchar *path, *date_format;
	GtkStyle *style;
	GtkWidget *w;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	edv_location_type location_type;
	edv_date_relativity date_relativity;
	edv_recycled_object_struct *recobj = NULL;
	edv_archive_object_struct *archobj = NULL;
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	/* No Details Page? */
	if(pd->det_toplevel == NULL)
	    return;

	pd->freeze_count++;

	location_type = pd->location_type;
	path = pd->path;
	w = pd->det_clist;
	clist = GTK_CLIST(w);
	style = gtk_widget_get_style(w);
	ncolumns = MAX(clist->columns, 1);
	core = pd->core;
	cfg_list = core->cfg_list;

	/* Record the last scroll positions */
	last_x = GTK_ADJUSTMENT_GET_VALUE(clist->hadjustment);
	last_y = GTK_ADJUSTMENT_GET_VALUE(clist->vadjustment);

	/* Calculate the prefered size of the value pixmap */
	value_pixmap_width = 120;
	value_pixmap_height = GTK_CLIST_ROW_HEIGHT_SET(clist) ?
	    (clist->row_height - 4) : -1;

	/* Begin updating */
	gtk_clist_freeze(clist);

	/* Clear any existing details */
	gtk_clist_clear(clist);

	/* Get the display options */
	date_relativity = (edv_date_relativity)EDV_GET_I(
	    EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);
	block_size = EDV_GET_L(EDV_CFG_PARM_BLOCK_SIZE);
	if(block_size == 0l)
	    block_size = 1024l;

	/* Get the object's statistics based on the location type */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    /* Get the object's local stats if the specified
	     * lstat_buf was not given
	     */
	    if(lstat_buf == NULL)
	    {
		if(!STRISEMPTY(path))
		{
		    if(!lstat((const char *)path, &obj_lstat_buf))
			lstat_buf = &obj_lstat_buf;
		}
	    }
	    break;
	  case EDV_LOCATION_TYPE_RECBIN:
	    /* Get the recycled object values */
	    if(pd->recycled_obj_index > 0)
		recobj = EDVRecBinObjectStat(
		    EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
		    pd->recycled_obj_index
		);
	    break;
	  case EDV_LOCATION_TYPE_ARCHIVE:
	    /* Get the archive object values */
	    if(!STRISEMPTY(pd->arch_path) && !STRISEMPTY(path))
		archobj = EDVArchStat(
		    core,
		    pd->arch_path,
		    path,
		    NULL
		);
	    break;
	}

	/* Get the verbose state */
	verbose = GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    pd->det_verbose_radio
	);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

#define APPEND_LINE(_prop_,_val_)	{		\
 if(ncolumns >= 2) {					\
  strv[0] = ((_prop_) != NULL) ? (_prop_) : "";		\
  strv[1] = ((_val_) != NULL) ? (_val_) : "";		\
  row = gtk_clist_append(clist, strv);			\
  gtk_clist_set_selectable(clist, row, FALSE);		\
  gtk_clist_set_shift(					\
   clist, row, 0, 0, -border_minor			\
  );							\
  gtk_clist_set_shift(					\
   clist, row, 1, 0, 0					\
  );							\
 } else {						\
  row = -1;						\
 }							\
}
#define APPEND_LINE_PIXMAP(_prop_,_pixmap_,_mask_) {	\
 if(ncolumns >= 2) {					\
  strv[0] = ((_prop_) != NULL) ? (_prop_) : "";		\
  strv[1] = "";						\
  row = gtk_clist_append(clist, strv);			\
  if((_pixmap_) != NULL)				\
   gtk_clist_set_pixmap(				\
    clist, row, 1,					\
    (_pixmap_), (_mask_)				\
   );							\
  gtk_clist_set_selectable(clist, row, FALSE);		\
  gtk_clist_set_shift(					\
   clist, row, 0, 0, -border_minor			\
  );							\
  gtk_clist_set_shift(					\
   clist, row, 1, 0, 0					\
  );							\
 } else {						\
  row = -1;						\
 }							\
}

	/* Add the lines based on the location type */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    if(path != NULL)
	    {
		/* Name */
		APPEND_LINE("Name", g_basename(path));

		/* Location */
		s = g_dirname(path);
		if(s == NULL)
		    s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		APPEND_LINE("Location", s);
		g_free(s);
	    }

	    /* Object stats available? */
	    if(lstat_buf != NULL)
	    {
		/* Get the object's type and permissions */
		const mode_t m = lstat_buf->st_mode;
		const edv_object_type type = EDVStatModeToObjectType(m);
		edv_device_struct *d = EDVDevicesListMatchObject(
		    core->device, core->total_devices,
		    NULL, path
		);

		/* Device */
		if(verbose)
		{
		    if(d != NULL)
		    {
			if(!STRISEMPTY(d->name))
			    s = STRDUP(d->name);
			else if(!STRISEMPTY(d->device_path))
			    s = STRDUP(d->device_path);
			else
			    s = g_strdup_printf("#%ld", (gulong)lstat_buf->st_dev);
		    }
		    else
		    {
			s = g_strdup_printf("#%ld", (gulong)lstat_buf->st_dev);
		    }
		}
		else
		    s = g_strdup_printf("%ld", (gulong)lstat_buf->st_dev);
		APPEND_LINE("Device", s);
		g_free(s);

		/* Filesystem */
		if(d != NULL)
		{
		    if(verbose)
			s = STRDUP(EDVDeviceGetFSNameFromType(d->fs_type));
		    else
			s = g_strdup_printf("0x%.8X", d->fs_type);
		    APPEND_LINE("Filesystem", s);
		    g_free(s);
		}

		/* Index/INode */
		if(verbose)
		{
		    s = g_strdup_printf("#%ld", (gulong)lstat_buf->st_ino);
		    APPEND_LINE("Index", s);
		}
		else
		{
		    s = g_strdup_printf("%ld", (gulong)lstat_buf->st_ino);
		    APPEND_LINE("INode", s);
		}
		g_free(s);

		/* Type */
		if(verbose)
		    s = STRDUP(pd->mime_type);
		else
		    s = STRDUP(EDVGetObjectTypeName(type));
		APPEND_LINE("Type", s);
		g_free(s);

		/* Size */
		if(verbose)
		{
		    const gulong size = (gulong)lstat_buf->st_size;
		    s = g_strdup_printf(
			"%s %s",
			EDVSizeStrDelim(size),
			(size == 1l) ? "byte" : "bytes"
		    );
		}
		else
		    s = g_strdup_printf(
			"%ld",
			(gulong)lstat_buf->st_size
		    );
		APPEND_LINE("Size", s);
		g_free(s);

#if !defined(_WIN32)
		/* Permissions/Mode */
#if defined(S_IRUSR) && defined(S_IWUSR) && defined(S_IXUSR) && defined(S_IRGRP) && defined(S_IWGRP) && defined(S_IXGRP) && defined(S_IROTH) && defined(S_IWOTH) && defined(S_IXOTH) && defined(S_ISUID) && defined(S_ISGID) && defined(S_ISVTX)
		if(verbose)
		{
		    s = EDVGetPermissionsString(
			EDVStatModeToEDVPermissions(m)
		    );
		    APPEND_LINE("Permissions", s);
		}
		else
		{
		    s = g_strdup_printf("0x%.8X", (guint32)m);
		    APPEND_LINE("Mode", s);
		}
#else
		if(verbose)
		{
		    s = g_strdup_printf("0x%.8X", (guint32)m);
		    APPEND_LINE("Permissions", s);
		}
		else
		{
		    s = g_strdup_printf("0x%.8X", (guint32)m);
		    APPEND_LINE("Mode", s);
		}
#endif
		g_free(s);
#endif	/* !_WIN32 */

#if !defined(_WIN32)
		/* Hard Links */
		s = g_strdup_printf("%ld", (gulong)lstat_buf->st_nlink);
		APPEND_LINE("Hard Links", s);
		g_free(s);
#endif	/* !_WIN32 */

#if !defined(_WIN32)
		/* Owner */
		if(verbose)
		{
		    s = EDVUIDGetNameFromUID(
			core->uids_list,
			(gint)lstat_buf->st_uid, NULL
		    );
		    APPEND_LINE("Owner", s);
		}
		else
		{
		    s = g_strdup_printf("%i", (gint)lstat_buf->st_uid);
		    APPEND_LINE("UID", s);
		}
		g_free(s);
#endif	/* !_WIN32 */

#if !defined(_WIN32)
		/* Group */
		if(verbose)
		{
		    s = EDVGIDGetNameFromGID(
			core->gids_list,
			(gint)lstat_buf->st_gid, NULL
		    );
		    APPEND_LINE("Group", s);
		}
		else
		{
		    s = g_strdup_printf("%i", (gint)lstat_buf->st_gid);
		    APPEND_LINE("GID", s);
		}
		g_free(s);
#endif	/* !_WIN32 */

		/* Link Target */
		if(type == EDV_OBJECT_TYPE_LINK)
		{
		    s = EDVGetLinkTarget(path);
		    if(s == NULL)
			s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		    APPEND_LINE("Link Target", s);
		    g_free(s);
		}

#if !defined(_WIN32)
		/* Device Numbers */
		if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
		   (type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
		)
		{
		    if(verbose)
		    {
			gint major = 0, minor = 0;
			EDVGetDeviceNumbers(lstat_buf->st_rdev, &major, &minor);
			s = g_strdup_printf(
			    "%i, %i",
			    major, minor
			);
			APPEND_LINE("Device Numbers", s);
		    }
		    else
		    {
			s = g_strdup_printf("0x%.4X", (guint16)(lstat_buf->st_rdev & 0x0000FFFF));
			APPEND_LINE("Device Type", s);
		    }
		    g_free(s);
		}
#endif	/* !_WIN32 */

#if !defined(_WIN32)
		/* IO Block Size */
		if(verbose)
		{
		    const gulong io_block_size = (gulong)lstat_buf->st_blksize;
		    s = g_strdup_printf(
			"%s %s",
			EDVSizeStrDelim(io_block_size),
			(io_block_size == 1l) ? "byte" : "bytes"
		    );
		}
		else
		    s = g_strdup_printf(
			"%ld",
			(gulong)lstat_buf->st_blksize
		    );
		APPEND_LINE("Optimul IO Block Size", s);
		g_free(s);
#endif	/* !_WIN32 */

#if !defined(_WIN32)
		/* Bytes/Blocks Allocated */
		if(verbose)
		{
		    const gulong bytes_allocated = (gulong)lstat_buf->st_blocks * block_size;
		    s = g_strdup_printf(
			"%s %s",
			EDVSizeStrDelim(bytes_allocated),
			(bytes_allocated == 1l) ? "byte" : "bytes"
		    );
		    APPEND_LINE("Bytes Allocated", s);
		}
		else
		{
		    s = g_strdup_printf(
			"%ld",
			(gulong)lstat_buf->st_blocks
		    );
		    APPEND_LINE("Blocks Allocated", s);
		}
		g_free(s);
#endif	/* !_WIN32 */

		/* Last Accessed */
		if(verbose)
		{
		    if(lstat_buf->st_atime > 0l)
			s = STRDUP(EDVDateFormatString(
			    (gulong)lstat_buf->st_atime,
			    date_format, date_relativity
			));
		    else
			s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
		    APPEND_LINE("Last Accessed", s);
		}
		else
		{
		    s = g_strdup_printf("%ld", (gulong)lstat_buf->st_atime);
		    APPEND_LINE("Accessed Time", s);
		}
		g_free(s);

		/* Last Modified */
		if(verbose)
		{
		    if(lstat_buf->st_mtime > 0l)
			s = STRDUP(EDVDateFormatString(
			    (gulong)lstat_buf->st_mtime,
			    date_format, date_relativity
			));
		    else
			s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
		    APPEND_LINE("Last Modified", s);
		}
		else
		{
		    s = g_strdup_printf("%ld", (gulong)lstat_buf->st_mtime);
		    APPEND_LINE("Modified Time", s);
		}
		g_free(s);

		/* Last Changed */
		if(verbose)
		{
		    if(lstat_buf->st_ctime > 0l)
			s = STRDUP(EDVDateFormatString(
			    (gulong)lstat_buf->st_ctime,
			    date_format, date_relativity
			));
		    else
			s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
		    APPEND_LINE("Last Changed", s);
		}
		else
		{
		    s = g_strdup_printf("%ld", (gulong)lstat_buf->st_ctime);
		    APPEND_LINE("Changed Time", s);
		}
		g_free(s);
	    }
	    break;

	  case EDV_LOCATION_TYPE_RECBIN:
	    if(recobj != NULL)
	    {
		const gulong recbin_size_max = (gulong)EDV_GET_L(
		    EDV_CFG_PARM_RECBIN_SIZE_WARN
		);
		GdkBitmap *mask;
		GdkPixmap *pixmap;
		const edv_object_type type = recobj->type;
		const mode_t m = EDVEDVPermissionsToStatMode(
		    recobj->permissions
		);

		/* Name */
		s = STRDUP(recobj->name);
		if(s == NULL)
		    s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		APPEND_LINE("Name", s);
		g_free(s);

		/* Location */
		if(verbose)
		{
		    APPEND_LINE("Location", "Recycle Bin");
		}
		else
		{
		    gchar *location = EDVRecBinIndexGetRecycledObjectPath(
			EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
			recobj->index
		    );
		    APPEND_LINE("Location", location);
		    g_free(location);
		}

		/* Index */
		if(recobj->index > 0)
		{
		    const guint index = recobj->index;
		    if(verbose)
			s = g_strdup_printf("#%u", index);
		    else
			s = g_strdup_printf("%u", index);
		}
		else
		{
		    if(verbose)
			s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		    else
			s = g_strdup_printf("%u", recobj->index);
		}
		APPEND_LINE("Index", s);
		g_free(s);

		/* Original Location */
		s = STRDUP(recobj->original_path);
		if(s == NULL)
		    s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		APPEND_LINE("Original Location", s);
		g_free(s);

		/* Type */
		if(verbose)
		    s = STRDUP(pd->mime_type);
		else
		    s = STRDUP(EDVGetObjectTypeName(type));
		APPEND_LINE("Type", s);
		g_free(s);

		/* Size */
		if(verbose)
		{
		    const gulong size = recobj->size;
		    s = g_strdup_printf(
			"%s %s",
			EDVSizeStrDelim(size),
			(size == 1l) ? "byte" : "bytes"
		    );
		}
		else
		    s = g_strdup_printf(
			"%ld",
			recobj->size
		    );
		APPEND_LINE("Size", s);
		g_free(s);

		/* Capacity Used */
		if(recbin_size_max > 0l)
		{
		    const gfloat coeff = (gfloat)recobj->size / (gfloat)recbin_size_max;
		    if(verbose)
		    {
			pixmap = EDVNewProgressPixmap(
			    value_pixmap_width,
			    value_pixmap_height,
			    coeff,
			    TRUE,		/* Draw value */
			    GTK_ORIENTATION_HORIZONTAL,
			    FALSE,		/* Not reverse */
			    style,
			    &mask
			);
			APPEND_LINE_PIXMAP("Capacity Used", pixmap, mask);
			GDK_PIXMAP_UNREF(pixmap);
			GDK_BITMAP_UNREF(mask);
		    }
		    else
		    {
			s = g_strdup_printf(
			    "%.2f",
			    coeff
			);
			APPEND_LINE("Capacity Used", s);
			g_free(s);
		    }
		}

#if !defined(_WIN32)
		/* Permissions/Mode */
#if defined(S_IRUSR) && defined(S_IWUSR) && defined(S_IXUSR) && defined(S_IRGRP) && defined(S_IWGRP) && defined(S_IXGRP) && defined(S_IROTH) && defined(S_IWOTH) && defined(S_IXOTH) && defined(S_ISUID) && defined(S_ISGID) && defined(S_ISVTX)
		if(verbose)
		{
		    s = EDVGetPermissionsString(recobj->permissions);
		    APPEND_LINE("Permissions", s);
		}
		else
		{
		    s = g_strdup_printf("0x%.8X", (guint32)m);
		    APPEND_LINE("Mode", s);
		}
#else
		if(verbose)
		{
		    s = g_strdup_printf("0x%.8X", (guint32)m);
		    APPEND_LINE("Permissions", s);
		}
		else
		{
		    s = g_strdup_printf("0x%.8X", (guint32)m);
		    APPEND_LINE("Mode", s);
		}
#endif
		g_free(s);
#endif	/* !_WIN32 */

		/* Owner */
		if(verbose)
		{
		    s = EDVUIDGetNameFromUID(
			core->uids_list,
			recobj->owner_id, NULL
		    );
		    APPEND_LINE("Owner", s);
		}
		else
		{
		    s = g_strdup_printf("%i", recobj->owner_id);
		    APPEND_LINE("UID", s);
		}
		g_free(s);

		/* Group */
		if(verbose)
		{
		    s = EDVGIDGetNameFromGID(
			core->gids_list,
			recobj->group_id, NULL
		    );
		    APPEND_LINE("Group", s);
		}
		else
		{
		    s = g_strdup_printf("%i", recobj->group_id);
		    APPEND_LINE("GID", s);
		}
		g_free(s);

		/* Link Target */
		if(type == EDV_OBJECT_TYPE_LINK)
		{
		    s = STRDUP(recobj->link_target);
		    if(s == NULL)
			s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		    APPEND_LINE("Link Target", s);
		    g_free(s);
		}

		/* Last Accessed */
		if(verbose)
		{
		    if(recobj->access_time > 0l)
			s = STRDUP(EDVDateFormatString(
			    recobj->access_time,
			    date_format, date_relativity
			));
		    else
			s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
		    APPEND_LINE("Last Accessed", s);
		}
		else
		{
		    s = g_strdup_printf("%ld", recobj->access_time);
		    APPEND_LINE("Accessed Time", s);
		}
		g_free(s);

		/* Last Modified */
		if(verbose)
		{
		    if(recobj->modify_time > 0l)
			s = STRDUP(EDVDateFormatString(
			    recobj->modify_time,
			    date_format, date_relativity
			));
		    else
			s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
		    APPEND_LINE("Last Modified", s);
		}
		else
		{
		    s = g_strdup_printf("%ld", recobj->modify_time);
		    APPEND_LINE("Modified Time", s);
		}
		g_free(s);

		/* Last Changed */
		if(verbose)
		{
		    if(recobj->change_time > 0l)
			s = STRDUP(EDVDateFormatString(
			    recobj->change_time,
			    date_format, date_relativity
			));
		    else
			s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
		    APPEND_LINE("Last Changed", s);
		}
		else
		{
		    s = g_strdup_printf("%ld", recobj->change_time);
		    APPEND_LINE("Changed Time", s);
		}
		g_free(s);

		/* Deleted On/Deleted Time */
		if(verbose)
		{
		    if(recobj->deleted_time > 0l)
			s = STRDUP(EDVDateFormatString(
			    recobj->deleted_time,
			    date_format, date_relativity
			));
		    else
			s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
		    APPEND_LINE("Deleted On", s);
		}
		else
		{
		    s = g_strdup_printf("%ld", recobj->deleted_time);
		    APPEND_LINE("Deleted Time", s);
		}
		g_free(s);
	    }
	    break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
	    if(archobj != NULL)
	    {
		GdkBitmap *mask;
		GdkPixmap *pixmap;
		const edv_object_type type = archobj->type;
		const mode_t m = EDVEDVPermissionsToStatMode(
		    archobj->permissions
		);

		/* Name */
		s = STRDUP(archobj->name);
		if(s == NULL)
		    s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		APPEND_LINE("Name", s);
		g_free(s);

		/* Location */
		s = STRDUP(pd->arch_path);
		if(s == NULL)
		    s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		APPEND_LINE("Location", s);
		g_free(s);

		/* Index */
		if(archobj->index > 0)
		{
		    const guint index = archobj->index;
		    if(verbose)
			s = g_strdup_printf("#%u", index);
		    else
			s = g_strdup_printf("%u", index);
		}
		else
		{
		    if(verbose)
			s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		    else
			s = g_strdup_printf("%u", archobj->index);
		}
		APPEND_LINE("Index", s);
		g_free(s);

		/* Location In Archive */
		if(!STRISEMPTY(archobj->full_path))
		{
		    const gchar *in_arch_path = archobj->full_path;
		    gchar *location;

		    if(g_path_is_absolute(in_arch_path))
			location = STRDUP(in_arch_path);
		    else
			location = g_strconcat(
			    ".",
			    G_DIR_SEPARATOR_S,
			    in_arch_path,
			    NULL
			);
		    if(location != NULL)
		    {
			s = g_dirname(location);
			g_free(location);
		    }
		    else
			s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		}
		else
		{
		    s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		}
		APPEND_LINE("Location In Archive", s);
		g_free(s);

		/* Type */
		if(verbose)
		    s = STRDUP(pd->mime_type);
		else
		    s = STRDUP(EDVGetObjectTypeName(type));
		APPEND_LINE("Type", s);
		g_free(s);

		/* Size */
		if(verbose)
		{
		    const gulong size = archobj->size;
		    s = g_strdup_printf(
			"%s %s",
			EDVSizeStrDelim(size),
			(size == 1l) ? "byte" : "bytes"
		    );
		}
		else
		    s = g_strdup_printf(
			"%ld",
			archobj->size
		    );
		APPEND_LINE("Size", s);
		g_free(s);

		/* Compressed Size */
		if(((archobj->compressed_size > 0l) &&
		    (archobj->size > 0l)) ||
		   (archobj->size == 0l)
		)
		{
		    const gulong compressed_size = archobj->compressed_size;
		    if(verbose)
			s = g_strdup_printf(
			    "%s %s",
			    EDVSizeStrDelim(compressed_size),
			    (compressed_size == 1l) ? "byte" : "bytes"
			);
		    else
			s = g_strdup_printf(
			    "%ld",
			    compressed_size
			);
		}
		else
		{
		    if(verbose)
			s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		    else
			s = g_strdup_printf(
			    "%ld",
			    archobj->compressed_size
			);
		}
		APPEND_LINE("Compressed Size", s);
		g_free(s);

		/* Compression Ratio */
		s = NULL;
		pixmap = NULL;
		mask = NULL;
		if((type == EDV_OBJECT_TYPE_FILE) &&
		   (archobj->compression_ratio >= 0.0f)
		)
		{
		    const gfloat ratio = archobj->compression_ratio;

		    if(((archobj->compressed_size > 0l) &&
			(archobj->size > 0l)) ||
		       (archobj->size == 0l)
		    )
		    {
			if(verbose)
			    pixmap = EDVNewProgressPixmap(
				value_pixmap_width,
				value_pixmap_height,
				ratio,
				TRUE,		/* Draw value */
				GTK_ORIENTATION_HORIZONTAL,
				FALSE,		/* Not reverse */
				style,
				&mask
			    );
			else
			    s = g_strdup_printf(
				"%.2f",
				ratio
			    );
		    }
		    else
		    {
			if(verbose)
			    s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
			else
			    s = g_strdup_printf(
				"%.2f",
				ratio
			    );
		    }
		}
		else
		{
		    if(verbose)
			s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		    else
			s = g_strdup_printf(
			    "%.2f",
			    archobj->compression_ratio
			);
		}
		if(pixmap != NULL)
		{
		    APPEND_LINE_PIXMAP("Compression Ratio", pixmap, mask);
		    GDK_PIXMAP_UNREF(pixmap);
		    GDK_BITMAP_UNREF(mask);
		}
		else
		{
		    APPEND_LINE("Compression Ratio", s);
		}
		g_free(s);

		/* Encryption */
		s = STRDUP(archobj->encryption);
		if(s == NULL)
		    s = STRDUP("None");
		APPEND_LINE("Encryption", s);
		g_free(s);

#if !defined(_WIN32)
		/* Permissions/Mode */
#if defined(S_IRUSR) && defined(S_IWUSR) && defined(S_IXUSR) && defined(S_IRGRP) && defined(S_IWGRP) && defined(S_IXGRP) && defined(S_IROTH) && defined(S_IWOTH) && defined(S_IXOTH) && defined(S_ISUID) && defined(S_ISGID) && defined(S_ISVTX)
		if(verbose)
		{
		    s = EDVGetPermissionsString(archobj->permissions);
		    APPEND_LINE("Permissions", s);
		}
		else
		{
		    s = g_strdup_printf("0x%.8X", (guint32)m);
		    APPEND_LINE("Mode", s);
		}
#else
		if(verbose)
		{
		    s = g_strdup_printf("0x%.8X", (guint32)m);
		    APPEND_LINE("Permissions", s);
		}
		else
		{
		    s = g_strdup_printf("0x%.8X", (guint32)m);
		    APPEND_LINE("Mode", s);
		}
#endif
		g_free(s);
#endif	/* !_WIN32 */

		/* Owner */
		s = STRDUP(archobj->owner_name);
		if(s == NULL)
		    s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		APPEND_LINE("Owner", s);
		g_free(s);

		/* Group */
		s = STRDUP(archobj->group_name);
		if(s == NULL)
		    s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		APPEND_LINE("Group", s);
		g_free(s);

		/* Link Target */
		if(type == EDV_OBJECT_TYPE_LINK)
		{
		    s = STRDUP(archobj->link_target);
		    if(s == NULL)
			s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		    APPEND_LINE("Link Target", s);
		    g_free(s);
		}

		/* Device Numbers */
		if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
		   (type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
		)
		{
		    if(verbose)
		    {
			gint major = 0, minor = 0;
			EDVGetDeviceNumbers(archobj->device_type, &major, &minor);
			s = g_strdup_printf(
			    "%i, %i",
			    major, minor
			);
			APPEND_LINE("Device Numbers", s);
		    }
		    else
		    {
			s = g_strdup_printf("0x%.4X", (guint16)(archobj->device_type & 0x0000FFFF));
			APPEND_LINE("Device Type", s);
		    }
		    g_free(s);
		}

		/* Storage Method */
		s = STRDUP(archobj->method);
		if(s == NULL)
		    s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		APPEND_LINE("Storage Method", s);
		g_free(s);

		/* CRC */
		s = STRDUP(archobj->crc);
		if(s == NULL)
		    s = STRDUP(PROPDLG_VALUE_NOT_AVAILABLE_STR);
		APPEND_LINE("CRC", s);
		g_free(s);

		/* Last Accessed */
		if(verbose)
		{
		    if(archobj->access_time > 0l)
			s = STRDUP(EDVDateFormatString(
			    archobj->access_time,
			    date_format, date_relativity
			));
		    else
			s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
		    APPEND_LINE("Last Accessed", s);
		}
		else
		{
		    s = g_strdup_printf("%ld", archobj->access_time);
		    APPEND_LINE("Accessed Time", s);
		}
		g_free(s);

		/* Last Modified */
		if(verbose)
		{
		    if(archobj->modify_time > 0l)
			s = STRDUP(EDVDateFormatString(
			    archobj->modify_time,
			    date_format, date_relativity
			));
		    else
			s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
		    APPEND_LINE("Last Modified", s);
		}
		else
		{
		    s = g_strdup_printf("%ld", archobj->modify_time);
		    APPEND_LINE("Modified Time", s);
		}
		g_free(s);

		/* Last Changed */
		if(verbose)
		{
		    if(archobj->change_time > 0l)
			s = STRDUP(EDVDateFormatString(
			    archobj->change_time,
			    date_format, date_relativity
			));
		    else
			s = STRDUP(PROPDLG_TIME_STAMP_NOT_SET_STR);
		    APPEND_LINE("Last Changed", s);
		}
		else
		{
		    s = g_strdup_printf("%ld", archobj->change_time);
		    APPEND_LINE("Changed Time", s);
		}
		g_free(s);
	    }
	    break;
	}


	/* Object properties list available? */
	if(ext_props_list != NULL)
	{
	    GList *glist;
	    edv_object_prop_struct *prop;

	    for(glist = ext_props_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		prop = EDV_OBJECT_PROP(glist->data);
		if(prop == NULL)
		    continue;

		APPEND_LINE(prop->name, prop->value);
	    }
	}

	/* Delete the recycled object */
	EDVRecycledObjectDelete(recobj);

	/* Delete the archive object */
	EDVArchObjectDelete(archobj);

#undef APPEND_LINE_PIXMAP
#undef APPEND_LINE

	gtk_clist_columns_autosize(clist);

	/* Scroll back to the original position */
	EDVScrollCListToPosition(clist, last_x, last_y);

	gtk_clist_thaw(clist);

	g_free(strv);

	pd->freeze_count--;
}

/*
 *	Updates the Link Page.
 *
 *	Gets the current value from the Link Target GtkEntry and
 *	checks if the target object exists.
 *
 *	If the target object exists then its statistics will be obtained
 *	and the link icon will be updated along with the Target
 *	Properties GtkButton sensitivity updated.
 */
static void EDVPropDlgLinkPageUpdate(edv_propdlg_struct *pd)
{
	const gchar *value;
	GtkEntry *entry;
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	/* No Link Page? */
	if(pd->link_toplevel == NULL)
	    return;

	entry = (GtkEntry *)pd->link_target_entry;
	if(entry == NULL)
	    return;

	pd->freeze_count++;

	core = pd->core;

	/* Get the link's current target value */
	value = gtk_entry_get_text(entry);
	switch(pd->location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    if(!STRISEMPTY(value))
	    {
		gchar *full_tar_path;

		/* Is the target value an absolute path? */
		if(g_path_is_absolute(value))
		{
		    full_tar_path = STRDUP(value);
		    EDVSimplifyPath(full_tar_path);
		}
		else
		{
		    /* Not an absolute path, prefix the parent path
		     * of the link to the target path
		     */
		    const gchar *path = pd->path;
		    gchar *parent_path = (path != NULL) ?
			g_dirname(path) : STRDUP(path);
		    full_tar_path = g_strconcat(
			parent_path,
			G_DIR_SEPARATOR_S,
			value,
			NULL
		    );
		    g_free(parent_path);
		}

		/* Got the full target path? */
		if(full_tar_path != NULL)
		{
		    struct stat lstat_buf;
		    gboolean b, target_exists;
		    GtkWidget *w;
		    edv_object_struct *tar_obj;

		    /* Check if the target path exists locally */
		    if(lstat((const char *)full_tar_path, &lstat_buf))
		    {
			const gint error_code = (gint)errno;
#ifdef ENOENT
			if(error_code == ENOENT)
#else
			if(FALSE)
#endif
			    target_exists = FALSE;
			else
			    target_exists = FALSE;
		    }
		    else
		    {
			target_exists = TRUE;
		    }

		    /* If the target exists then create a tempory
		     * target object and get its statistics
		     */
		    if(target_exists)
		    {
			tar_obj = EDVObjectNew();
			if(tar_obj != NULL)
			{
			    EDVObjectSetPath(tar_obj, full_tar_path);
			    EDVObjectSetStat(tar_obj, &lstat_buf);
			    EDVObjectUpdateLinkFlags(tar_obj);
			}
		    }
		    else
		    {
			tar_obj = NULL;
		    }

		    /* Update the link icon */
		    w = pd->link_target_icon_pm;
		    if(w != NULL)
		    {
			/* Get the appropriate pixmap and mask */
			GdkPixmap *pixmap = NULL;
			GdkBitmap *mask = NULL;
			if(tar_obj != NULL)
			{
			    GdkPixmap *pixmap_ext, *pixmap_hid;
			    GdkBitmap *mask_ext, *mask_hid;
			    EDVMatchObjectIcon(
				core->device, core->total_devices,
				core->mimetype, core->total_mimetypes,
				tar_obj->type,
				tar_obj->full_path,
				TRUE,			/* Link valid */
				tar_obj->permissions,
				0,			/* Small icons */
				&pixmap, &mask,
				NULL, NULL,
				&pixmap_ext, &mask_ext,
				&pixmap_hid, &mask_hid
			    );
			    /* Hidden? */
			    if(EDVIsObjectHidden(tar_obj))
			    {
				if(pixmap_hid != NULL)
				{
				    pixmap = pixmap_hid;
				    mask = mask_hid;
				}
			    }
			    /* No access? */
			    if(!EDVIsObjectAccessable(core, tar_obj))
			    {
				if(pixmap_ext != NULL)
				{
				    pixmap = pixmap_ext;
				    mask = mask_ext;
				}
			    }
			}
			else
			{
			    edv_mime_type_struct *m = EDVMimeTypesListMatchType(
				core->mimetype, core->total_mimetypes,
				NULL,
				EDV_MIME_TYPE_TYPE_INODE_LINK,
				FALSE
			    );
			    if(m != NULL)
			    {
				const edv_mime_type_icon_state state =
				    EDV_MIME_TYPE_ICON_STATE_EXTENDED;
				pixmap = m->small_pixmap[state];
				mask = m->small_mask[state];
			    }
			}

			/* Set the icon */
			if(pixmap != NULL)
			    gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
		    }

		    /* Update the target properties button sensitivity */
		    b = target_exists;
		    GTK_WIDGET_SET_SENSITIVE(
			pd->link_target_properties_btn, b
		    );

		    EDVObjectDelete(tar_obj);

		    g_free(full_tar_path);
		}
	    }
	    break;

	  case EDV_LOCATION_TYPE_RECBIN:
	  case EDV_LOCATION_TYPE_ARCHIVE:
	    if(TRUE)
	    {
		GtkWidget *w;

		/* Update the link icon */
		w = pd->link_target_icon_pm;
		if(w != NULL)
		{
		    GdkPixmap *pixmap = NULL;
		    GdkBitmap *mask = NULL;
		    edv_mime_type_struct *m = EDVMimeTypesListMatchType(
			core->mimetype, core->total_mimetypes,
			NULL,
			EDV_MIME_TYPE_TYPE_INODE_LINK,
			FALSE
		    );
		    if(m != NULL)
		    {
			const edv_mime_type_icon_state state =
			    EDV_MIME_TYPE_ICON_STATE_STANDARD;
			pixmap = m->small_pixmap[state];
			mask = m->small_mask[state];
		    }

		    /* Set the icon */
		    if(pixmap != NULL)
			gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
		}
	    }
	    break;
	}

	pd->freeze_count--;
}

/*
 *	Updates the device node page.
 *
 *	Sets the device number GtkEntries.
 */
static void EDVPropDlgDeviceNodePageUpdate(
	edv_propdlg_struct *pd, struct stat *lstat_buf
)
{
	gint major, minor;
	GtkEntry *entry;
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	/* No Device Node Page? */
	if(pd->dev_node_toplevel == NULL)
	    return;

	pd->freeze_count++;

	core = pd->core;

	/* Get the device major and minor numbers */
	if(lstat_buf != NULL)
	{
	    EDVGetDeviceNumbers(lstat_buf->st_rdev, &major, &minor);
	}
	else
	{
	    major = 0;
	    minor = 0;
	}

	/* Set the major number GtkEntry */
	entry = (GtkEntry *)pd->dev_node_major_entry;
	if(entry != NULL)
	{
	    gchar *s = g_strdup_printf("%i", major);
	    gtk_entry_set_text(entry, s);
	    g_free(s);
	}

	/* Set the minor number GtkEntry */
	entry = (GtkEntry *)pd->dev_node_minor_entry;
	if(entry != NULL)
	{
	    gchar *s = g_strdup_printf("%i", minor);
	    gtk_entry_set_text(entry, s);
	    g_free(s);
	}

	pd->freeze_count--;
}

/*
 *	Connects the GtkWidget to the EDVPropDlgAnyChangedCB signal
 *	callback.
 */
static void EDVPropDlgSetWidgetChangedSignal(
	edv_propdlg_struct *pd, GtkWidget *w
)
{
	void (*func_cb)(GtkWidget *, gpointer) = EDVPropDlgAnyChangedCB;
	gpointer data = pd;

	if(w == NULL)
	    return;

	if(GTK_IS_EDITABLE(w))
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(func_cb), data
	    );
	else if(GTK_IS_TOGGLE_BUTTON(w))
	    gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(func_cb), data
	    );

}


/*
 *	Creates the General Page.
 *
 *	The pd specifies the Properties Dialog.
 *
 *	The parent specifies the parent GtkVBox that is to be the
 *	parent for all the child widgets on this page.
 *
 *	The obj specifies the object, if obj is NULL then certain
 *	widgets will not be created and certain values will not be set.
 */
static void EDVPropDlgCreateGeneralPage(
	edv_propdlg_struct *pd,
	GtkWidget *parent,
	edv_object_struct *obj
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint font_size;
	const gchar	*font_encoding,
			*mime_type = pd->mime_type;
	gchar *s, *font_name_h1_bold;
	const edv_location_type location_type = pd->location_type;
	GdkFont *font;
	GdkPixmap *pixmap, *pixmap_hid;
	GdkBitmap *mask, *mask_hid;
	GtkRcStyle *rcstyle;
	GtkWidget *toplevel = pd->toplevel;
	GtkStyle *style = gtk_widget_get_style(toplevel);
	GtkAccelGroup *accelgrp = pd->accelgrp;
	GtkWidget	*w, *menu,
			*parent2, *parent3, *parent4, *parent5;
	edv_object_type type;
	edv_permission_flags permissions;
	edv_core_struct *core = pd->core;

	if(obj == NULL)
	    return;

	pd->freeze_count++;

	/* Get the base font size */
        font = style->font;
        font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
        if(font_size < 3)
            font_size = 3;

	/* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
        font_encoding = "iso8859-2";
#else
        font_encoding = "iso8859-1";
#endif

        font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
            font_size + 2,
            font_encoding
        );

	type = obj->type;
	permissions = obj->permissions;

	/* Record the parent */
	pd->gen_toplevel = parent;

	/* Get the pixmap and mask for this object */
	EDVMatchObjectIcon(
	    core->device, core->total_devices,
	    core->mimetype, core->total_mimetypes,
	    type,
	    obj->full_path,
	    TRUE,
	    permissions,
	    1,			/* Medium icons */
	    &pixmap, &mask,
	    NULL, NULL,
	    NULL, NULL,
	    &pixmap_hid, &mask_hid
	);

	if(EDVIsObjectHidden(obj) && (pixmap_hid != NULL))
	{
	    pixmap = pixmap_hid;
	    mask = mask_hid;
	}


	/* Identification GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Identificacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Identification"
#elif defined(PROG_LANGUAGE_GERMAN)
"Identifikation"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Identificazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Identificatie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Identificao"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Identification"
#else
"Identification"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Name GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Icon */
	if(pixmap != NULL)
	{
	    pd->gen_icon_pm = w = gtk_pixmap_new(pixmap, mask);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}
	else
	{
	    pd->gen_icon_pm = NULL;
	}
	/* Label */
	pd->gen_name_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h1_bold);
	gtk_widget_modify_style(w, rcstyle);
	GTK_RC_STYLE_UNREF(rcstyle);
	gtk_widget_show(w);


	/* Location GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Label */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"La Ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ort"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plaats"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Plassering"
#else
"Location"
#endif
		":"
	    );
	    break;
	  case EDV_LOCATION_TYPE_RECBIN:
	    w = gtk_label_new("Index:");
	    break;
	  case EDV_LOCATION_TYPE_ARCHIVE:
	    w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"La Ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ort"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plaats"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Plassering"
#else
"Location In Archive"
#endif
		":"
	    );
	    break;
	}
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Label */
	pd->gen_location_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Type & Size GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Type GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Label */
	w = gtk_label_new("Type:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Type label */
	if(mime_type != NULL)
	    s = EDVShortenPath(mime_type, 20);
	else
	    s = STRDUP(EDVGetObjectTypeName(type));
	pd->gen_type_label = w = gtk_label_new(
	    (s != NULL) ? s : ""
	);
	g_free(s);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Size */
	if(type == EDV_OBJECT_TYPE_FILE)
	{
	    /* Size GtkHBox */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Label */
	    w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Tamao"
#elif defined(PROG_LANGUAGE_FRENCH)
"Taille"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gre"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Misura"
#elif defined(PROG_LANGUAGE_DUTCH)
"Maat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Tamanho"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Strrelse"
#else
"Size"
#endif
		":"
	    );
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    /* Size Label */
	    pd->gen_size_label = w = gtk_label_new("");
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    /* Units Label */
	    w = gtk_label_new("bytes");
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}
	else
	{
	    /* Size GtkHBox */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Label */
	    w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Tamao"
#elif defined(PROG_LANGUAGE_FRENCH)
"Taille"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gre"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Misura"
#elif defined(PROG_LANGUAGE_DUTCH)
"Maat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Tamanho"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Strrelse"
#else
"Size"
#endif
		":"
	    );
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    GTK_WIDGET_SET_SENSITIVE(w, FALSE);
	    gtk_widget_show(w);
	}

	/* Time Stamps GtkFrame */
	pd->gen_timestamps_frame = w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El tiempo Estampa"
#elif defined(PROG_LANGUAGE_FRENCH)
"Horodateur"
#elif defined(PROG_LANGUAGE_GERMAN)
"Zeit Stempelt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Tempo Timbra"
#elif defined(PROG_LANGUAGE_DUTCH)
"Tijd Stempelt"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Tempo Sela"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tid Stamps"
#else
"Time Stamps"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	/* Date labels & touch button GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Date labels GtkVBox */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Last Accessed GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Prefix label */
	w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 100, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Dure Accesado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dernier Accs"
#elif defined(PROG_LANGUAGE_GERMAN)
"Leisten Zugegriffen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Durare Accesso"
#elif defined(PROG_LANGUAGE_DUTCH)
"Leest Had Toegang Tot"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Dure Acedido"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Siste Accessed"
#else
"Last Accessed"
#endif
	    ":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_show(w);
	/* Date label */
	pd->gen_date_access_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Last Modified GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Prefix label */
	w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 100, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Dure Modificado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dernire Modification"
#elif defined(PROG_LANGUAGE_GERMAN)
"Dauern Modifiziert"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Durare Modificato"
#elif defined(PROG_LANGUAGE_DUTCH)
"Duur Wijzigde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Dure Modificado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Siste Modified"
#else
"Last Modified"
#endif
	    ":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_show(w);
	/* Date label */
	pd->gen_date_modify_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Last Changed GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Prefix label */
	w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 100, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Dure Cambiado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dernier changement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Dauern Gendert"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Durare Cambiato"
#elif defined(PROG_LANGUAGE_DUTCH)
"Duur Veranderde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Dure Mudado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Siste Changed"
#else
"Last Changed"
#endif
	    ":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_show(w);
	/* Date label */
	pd->gen_date_change_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	if(location_type == EDV_LOCATION_TYPE_RECBIN)
	{
	    /* Deleted On GtkHBox */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Prefix GtkLabel */
	    w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	    gtk_widget_set_usize(w, 100, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent5 = w;
	    w = gtk_label_new("Deleted On:");
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_container_add(GTK_CONTAINER(parent5), w);
	    gtk_widget_show(w);
	    /* Date GtkLabel */
	    pd->gen_date_deleted_label = w = gtk_label_new("");
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}

	/* Touch button GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Touch button */
	pd->gen_date_touch_btn = w = MenuButtonNewH(
#if defined(PROG_LANGUAGE_SPANISH)
"Toque"
#elif defined(PROG_LANGUAGE_FRENCH)
"Maintenant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Berhrung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Tocco"
#elif defined(PROG_LANGUAGE_DUTCH)
"Aanraking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Toque"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Berring"
#else
"Touch"
#endif
	    ,
	    (guint8 **)icon_time_stamp_20x20_xpm,
	    &menu
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVPropDlgDateTouchCB), pd
	);
	MenuButtonSetMapTrigger(w, MENU_BUTTON_MAP_TYPE_PRESSED_DRAG);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Modifique Sellos De Tiempo (el clic y el obstculo para el men)"
#elif defined(PROG_LANGUAGE_FRENCH)
"Modifier la datation (dclic et trane pour le menu)"
#elif defined(PROG_LANGUAGE_GERMAN)
"Modifizieren Sie Zeit Briefmarken (klicken und widerstand fr men)"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Modificare I Francobolli Di Tempo (lo scatto e trascinare per il menu)"
#elif defined(PROG_LANGUAGE_DUTCH)
"Wijziig Tijd Stempels (klik en ruk voor menu)"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Modifique Selos De Tempo (estalido e arrasta para cardpio)"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Modifiser Time Stamps (klikk og hindring for meny)"
#else
"Modify Time Stamps (click and drag for menu)"
#endif
	);
	gtk_widget_show(w);
	/* Touch menu */
	if(menu != NULL)
	{
	    guint accel_key, accel_mods;
	    const gchar *label;
	    guint8 **icon;
	    gpointer mclient_data = pd;
	    void (*func_cb)(GtkWidget *w, gpointer);

#define DO_ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
  icon, label, accel_key, accel_mods, NULL,	\
  mclient_data, func_cb				\
 );						\
}
#define DO_ADD_MENU_SEP		{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}
	    icon = NULL;
	    label =
#if defined(PROG_LANGUAGE_SPANISH)
"La Fecha Actual & Tiempo"
#elif defined(PROG_LANGUAGE_FRENCH)
"La Date & heure courante"
#elif defined(PROG_LANGUAGE_GERMAN)
"Jetziges Datum & Zeit"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Data Attuale & Il Tempo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Huidig Datum & Tijd"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Data Atual & Tempo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nvrende Date & Time"
#else
"Current Date & Time"
#endif
	    ;
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVPropDlgDateTouchCB;
	    DO_ADD_MENU_ITEM_LABEL

	    icon = NULL;
	    label =
#if defined(PROG_LANGUAGE_SPANISH)
"La Fecha Fija & Tiempo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Rgler ~La date & l'heure"
#elif defined(PROG_LANGUAGE_GERMAN)
"Festes Datum & Zei"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ha Regolato La Data & Il Tempo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vast Datum & Tijd"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Data fixa & Tempo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fast Date & Time"
#else
"Set Date & Time"
#endif
	    "...";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVPropDlgDateTouchSpecificCB;
	    DO_ADD_MENU_ITEM_LABEL

#undef DO_ADD_MENU_SEP
#undef DO_ADD_MENU_ITEM_LABEL
	}


	/* Ownership GtkFrame */
	pd->gen_ownership_frame = w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Propiedad"
#elif defined(PROG_LANGUAGE_FRENCH)
"Propritaire"
#elif defined(PROG_LANGUAGE_GERMAN)
"Eigentumsrecht"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Propriet"
#elif defined(PROG_LANGUAGE_DUTCH)
"Eigendomsrecht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Posse"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Eiendomsrett"
#else
"Ownership"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Owner & Group GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Owner GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Label */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Dueo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Propritaire"
#elif defined(PROG_LANGUAGE_GERMAN)
"Eigentmer"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Proprietario"
#elif defined(PROG_LANGUAGE_DUTCH)
"Eigenaar"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Proprietrio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Eier"
#else
"Owner"
#endif
	    ":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Entry and map button GtkHBox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Entry */
	pd->gen_owner_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent5), w, TRUE, TRUE, 0);
	EDVPropDlgSetWidgetChangedSignal(pd, w);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);
	/* Popup list map button */
	pd->gen_owner_btn = w = PUListNewMapButtonArrow(
	    GTK_ARROW_DOWN, GTK_SHADOW_OUT,
	    EDVPropDlgOwnerMapPUListCB, pd
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Group GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Label */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Grupo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Groupe de propritaire"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gruppe"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Gruppo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Groep"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Grupo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gruppe"
#else
"Group"
#endif
	    ":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Entry and map button GtkHBox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Entry */
	pd->gen_group_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent5), w, TRUE, TRUE, 0);
	EDVPropDlgSetWidgetChangedSignal(pd, w);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);
	/* Popup list map button */
	pd->gen_group_btn = w = PUListNewMapButtonArrow(
	    GTK_ARROW_DOWN, GTK_SHADOW_OUT,
	    EDVPropDlgGroupMapPUListCB, pd
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Permissions GtkFrame */
	pd->gen_permissions_frame = w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Los Permisos"
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Permissions"
#else
	    "Permissions"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* GtkVBox for standard permission labels and toggle buttons */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Labels GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
	    "El Dueo"
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Propritaire"
#else
	    "Owner"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_CENTER);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
	    "El Grupo"
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Groupe"
#else
	    "Group"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_CENTER);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Otro"
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Autre"
#else
	    "Other"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_CENTER);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);

	/* Toggle buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

#define NEW_PERMISSION_TOGGLE_BUTTON(_label_,_tip_,_parent_)	{\
 if(PROGDLG_USE_PERMISSION_CHECK_BUTTON) {			\
  /* Create permission button as a GtkCheckButton */		\
  w = gtk_check_button_new_with_label(_label_);			\
 } else {							\
  /* Create permission button as a GtkToggleButton */		\
  GdkFont *font = style->font;					\
  const gint font_height = (font != NULL) ?			\
   (font->ascent + font->descent) : 0;				\
  w = gtk_toggle_button_new_with_label(_label_);		\
  if(font_height > 0)						\
   gtk_widget_set_usize(					\
    w, MIN(font_height + 5, 20), MIN(font_height + 5, 20)	\
   );								\
 }								\
 gtk_box_pack_start(GTK_BOX(_parent_), w, FALSE, FALSE, 0);	\
 EDVPropDlgSetWidgetChangedSignal(pd, w);			\
 GUISetWidgetTip(w, (_tip_));					\
 gtk_widget_show(w);						\
}
	/* Owner GtkToggleButton set GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Read GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "R", "Read",
	    parent5
	);
	pd->gen_ur_check = w;
	/* Write GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "W", "Write",
	    parent5
	);
	pd->gen_uw_check = w;
	/* Execute GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "X", "Execute",
	    parent5
	);
	pd->gen_ux_check = w;

	/* Group GtkToggleButton set GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Read GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "R", "Read",
	    parent5
	);
	pd->gen_gr_check = w;
	/* Write GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "W", "Write",
	    parent5
	);
	pd->gen_gw_check = w;
	/* Execute GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "X", "Execute",
	    parent5
	);
	pd->gen_gx_check = w;

	/* Other/Anonymous GtkToggleButton set GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Read GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "R", "Read",
	    parent5
	);
	pd->gen_or_check = w;
	/* Write GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "W", "Write",
	    parent5
	);
	pd->gen_ow_check = w;
	/* Execute GtkToggleButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "X", "Execute",
	    parent5
	);
	pd->gen_ox_check = w;

#undef NEW_PERMISSION_TOGGLE_BUTTON

	/* SetUID, SetGID, and Sticky GtkHBox */
	w = gtk_hbox_new(TRUE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* SetUID GtkCheckButton */
	pd->gen_suid_check = w = gtk_check_button_new_with_label("SetUID");
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	EDVPropDlgSetWidgetChangedSignal(pd, w);
	GUISetWidgetTip(w, "Set owner's permission on use");
	gtk_widget_show(w);

	/* SetGID GtkCheckButton */
	pd->gen_sgid_check = w = gtk_check_button_new_with_label("SetGID");
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	EDVPropDlgSetWidgetChangedSignal(pd, w);
	GUISetWidgetTip(w, "Set group's permissions on use");
	gtk_widget_show(w);

	/* Sticky GtkCheckButton */
	pd->gen_sticky_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Pegajoso"
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Collant"
#else
	    "Sticky"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	EDVPropDlgSetWidgetChangedSignal(pd, w);
	GUISetWidgetTip(w, "Set user's permissions on use");
	gtk_widget_show(w);


	g_free(font_name_h1_bold);

	pd->freeze_count--;
}

/*
 *	Creates the Details Page.
 *
 *	The pd specifies the Properties Dialog.
 *
 *	The parent specifies the parent GtkVBox that is to be the
 *	parent for all the child widgets on this page.
 *
 *	The obj specifies the object, if obj is NULL then certain
 *	widgets will not be created and certain values will not be set.
 */
static void EDVPropDlgCreateDetailsPage(
	edv_propdlg_struct *pd,
	GtkWidget *parent,
	edv_object_struct *obj
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gchar *heading[2];
	GSList *gslist;
	GtkWidget *w, *parent2, *parent3;
	GtkCList *clist;
	edv_core_struct *core = pd->core;

	if(obj == NULL)
	    return;

	pd->freeze_count++;

	/* Record the parent */
	pd->det_toplevel = parent;

	/* Details GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Los Detalles"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dtails"
#elif defined(PROG_LANGUAGE_GERMAN)
"Statistik"
#elif defined(PROG_LANGUAGE_ITALIAN)
"I Dettagli"
#elif defined(PROG_LANGUAGE_DUTCH)
"Statistieken"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Os Detalhes"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Detaljer"
#else
"Details"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Radios GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Despliegue"
#elif defined(PROG_LANGUAGE_FRENCH)
"Affichage"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ausstellung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Mostra"
#elif defined(PROG_LANGUAGE_DUTCH)
"Tentoonstelling"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Exposio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Utstilling"
#else
"Display"
#endif
	    ":"
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Verbose Radio */
	gslist = NULL;
	pd->det_verbose_radio = w = gtk_radio_button_new_with_label(
	    gslist,
#if defined(PROG_LANGUAGE_SPANISH)
"Detallado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dtaill"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verbos"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Dettagliato"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verbose"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Verbose"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Verbose"
#else
"Verbose"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(EDVPropDlgDetailsVerboseToggledCB), pd
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	/* Raw Radio */
	pd->det_raw_radio = w = gtk_radio_button_new_with_label(
	    gslist,
#if defined(PROG_LANGUAGE_SPANISH)
"Crudo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Brute"
#elif defined(PROG_LANGUAGE_GERMAN)
"Roh"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Crudo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Rauw"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cru"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"R"
#else
"Raw"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(EDVPropDlgDetailsRawToggledCB), pd
	);
	gtk_widget_show(w);


	/* Create the GtkScrolledWindow for the GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Details GtkCList */
	heading[0] = "Property";
	heading[1] = "Value";
	pd->det_clist = w = gtk_clist_new_with_titles(
	    sizeof(heading) / sizeof(gchar *),
	    heading
	);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(EDVCListMotionEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(EDVPropDlgDetailsListSelectRowCB), pd
	);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_column_titles_show(clist);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_RIGHT
	);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, TRUE);
	gtk_clist_set_column_justification(
	    clist, 1, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 1, TRUE);
	gtk_clist_set_column_resizeable(clist, 1, TRUE);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_widget_show(w);


	/* Set the initial value to verbose */
	gtk_toggle_button_set_active(
	    GTK_TOGGLE_BUTTON(pd->det_verbose_radio),
	    TRUE
	);

	pd->freeze_count--;
}

/*
 *      Creates the link page.
 *
 *	The pd specifies the Properties Dialog.
 *
 *	The parent specifies the parent GtkVBox that is to be the
 *	parent for all the child widgets on this page.
 *
 *	The obj specifies the object, if obj is NULL then certain
 *	widgets will not be created and certain values will not be set.
 */
static void EDVPropDlgCreateLinkPage(
	edv_propdlg_struct *pd,
	GtkWidget *parent,
	edv_object_struct *obj
)
{
	const gint	border_major = 5,
			border_minor = 2;
	GtkWidget *w, *parent2, *parent3;
	edv_mime_type_struct *m;
	edv_core_struct *core = pd->core;

	if(obj == NULL)
	    return;

	pd->freeze_count++;

	/* Record the parent */
	pd->link_toplevel = parent;

	/* Target GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
	    "El Destino"
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Destination"
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Destinaion"
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Destinaion"
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Destinaion"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Destinaion"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Destinaion"
#else
	    "Target"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Target GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Link icon GtkPixmap */
	m = EDVMimeTypesListMatchType(
	    core->mimetype, core->total_mimetypes,
	    NULL,
	    EDV_MIME_TYPE_TYPE_INODE_LINK,
	    FALSE
	);
	if(m != NULL)
	{
	    const edv_mime_type_icon_state state =
		EDV_MIME_TYPE_ICON_STATE_EXTENDED;
	    pd->link_target_icon_pm = w = gtk_pixmap_new(
		m->small_pixmap[state],
		m->small_mask[state]
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}

	/* Target GtkEntry */
	pd->link_target_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	EDVEntrySetDND(core, w);
	EDVEntrySetCompletePath(core, w);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVPropDlgLinkChangedCB), pd
	);
	EDVPropDlgSetWidgetChangedSignal(pd, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Entre el valor del destino de la conexin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Entrer la valeur de destination du lien"
#elif defined(PROG_LANGUAGE_GERMAN)
"Tragen Sie das Reiseziel der valueEnter von dem Kettenglied\
 das Reiseziel der Wert von dem Kettenglied ein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Entrare il valueEnter di destinazione della maglia il valore di\
 destinazione della maglia"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga de bestemming van de schakel valueenter de bestemming van de\
 schakel waarde binnen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Entre o valueEnter de destino do elo o valor de destino do elo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"G inn i leddets destinasjon valueenter leddets\
 destinasjonsverdi"
#else
"Enter the link's target value"
#endif
	);
	gtk_widget_show(w);

	if(pd->location_type == EDV_LOCATION_TYPE_VFS)
	{
	    /* Browse button */
	    w = GUIButtonPixmap(
		(guint8 **)icon_browse_20x20_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVPropDlgLinkBrowseCB), pd
	    );
	    GUISetWidgetTip(w, "Browse");
	    gtk_widget_show(w);

	    /* Make Relative Path & Link Target Properties GtkHBox */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Make Relative Path button */
	    pd->link_target_make_relative_path_btn = w = gtk_button_new_with_label(
		"Make Relative Path"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVPropDlgLinkMakeRelativePathCB), pd
	    );
	    GUISetWidgetTip(
		w,
"Click this to convert the target value from an absolute path\
 into a relative path"
	    );
	    gtk_widget_show(w);


	    /* Link Target Properties button */
	    pd->link_target_properties_btn = w = gtk_button_new_with_label(
		"Target Properties"
	    );
	    gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVPropDlgLinkTargetPropertiesCB), pd
	    );
	    GUISetWidgetTip(
		w,
"Click this to view the properties of the target object"
	    );
	    gtk_widget_show(w);
	}

	/* Set the link value */
	if(obj->link_target != NULL)
	    gtk_entry_set_text(
		GTK_ENTRY(pd->link_target_entry), obj->link_target
	    );

	EDVPropDlgLinkPageUpdate(pd);

	pd->freeze_count--;
}

/*
 *	Creates the Device Page.
 *
 *	The pd specifies the Properties Dialog.
 *
 *	The parent specifies the parent GtkVBox that is to be the
 *	parent for all the child widgets on this page.
 *
 *	The obj specifies the object, if obj is NULL then certain
 *	widgets will not be created and certain values will not be set.
 *
 *	The dev_num and dev specifies the device.
 */
static void EDVPropDlgCreateDevicePage(
	edv_propdlg_struct *pd,
	GtkWidget *parent,
	edv_object_struct *obj,
	const gint dev_num, edv_device_struct *dev
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint font_size;
	const gchar *font_encoding;
	gchar *s, *font_name_h1_bold;
	GdkFont *font;
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;
	GtkRcStyle *rcstyle;
	GtkStyle *style = gtk_widget_get_style(pd->toplevel);
	GtkWidget	*w,
			*parent2, *parent3;

	if(obj == NULL)
	    return;

	/* Record the parent */
	pd->dev_toplevel = parent;

	pd->freeze_count++;

	/* Get the base font size */
        font = style->font;
        font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
        if(font_size < 3)
            font_size = 3;

	/* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
        font_encoding = "iso8859-2";
#else
        font_encoding = "iso8859-1";
#endif

        font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
            font_size + 2,
            font_encoding
        );

	/* Record the device reference number */
	pd->dev_num = dev_num;

	/* Get icon for this device (if any) */
	if(TRUE)
	{
	    const gint n = EDV_DEVICE_TOTAL_ICON_STATES;
	    gint i;

	    for(i = 0; i < n; i++)
	    {
		if(pixmap != NULL)
		    break;

		pixmap = dev->medium_pixmap[i];
		mask = dev->medium_mask[i];
	    }
	    for(i = 0; i < n; i++)
	    {
		if(pixmap != NULL)
		    break;

		pixmap = dev->large_pixmap[i];
		mask = dev->large_mask[i];
	    }
	    for(i = 0; i < n; i++)
	    {
		if(pixmap != NULL)
		    break;

		pixmap = dev->small_pixmap[i];
		mask = dev->small_mask[i];
	    }
	}


	/* Identification GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Identificacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Identification"
#elif defined(PROG_LANGUAGE_GERMAN)
"Identifikation"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Identificazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Identificatie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Identificao"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Identification"
#else
"Identification"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Device GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Icon */
	if(pixmap != NULL)
	{
	    pd->dev_icon_pm = w = gtk_pixmap_new(pixmap, mask);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}
	/* Name */
	if(dev->name != NULL)
	{
	    if(dev->device_path != NULL)
		s = g_strconcat(
		    dev->name,
		    " (",
		    dev->device_path,
		    ")",
		    NULL
		);
	    else
		s = STRDUP(dev->name);
	}
	else if(dev->device_path != NULL)
	{
	    s = STRDUP(dev->device_path);
	}
	else
	{
	    s = STRDUP("");
	}
	pd->dev_device_path_label = w = gtk_label_new(s);
	g_free(s);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h1_bold);
	gtk_widget_modify_style(w, rcstyle);
	GTK_RC_STYLE_UNREF(rcstyle);
	gtk_widget_show(w);

	/* Mount Path GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Prefix label */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Monte Sendero:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Point de montage:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Stellen Sie Pfad Auf:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Montare Il Sentiero:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bestijg Pad:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Monte Caminho:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Monter Path:"
#else
"Mount Path:"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Mount Path label */
	pd->dev_mount_path_label = w = gtk_label_new(
	    (dev->mount_path != NULL) ?
		dev->mount_path : ""
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Filesystem GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Prefix label */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Archive El De Sistema:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Systme de fichier:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Dateisystem:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Schedare Il Di Sistema:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Archiveer Systeem:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquive De Sistema:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiver System:"
#else
"Filesystem:"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Filesystem label */
	w = gtk_label_new(
	    EDVDeviceGetFSNameFromType(dev->fs_type)
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Usage GtkFrame */
	pd->dev_usage_frame = w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Uso"
#elif defined(PROG_LANGUAGE_FRENCH)
"Usage"
#elif defined(PROG_LANGUAGE_GERMAN)
"Brauch"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Uso"
#elif defined(PROG_LANGUAGE_DUTCH)
"Gebruik"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Uso"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Praksis"
#else
"Usage"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	/* Create the device usage widgets */
	pd->dev_usage_vbox = NULL;
	EDVPropDlgCreateDeviceUsage(pd, dev);


	/* Tools GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Mantenga"
#elif defined(PROG_LANGUAGE_FRENCH)
"Maintenance"
#elif defined(PROG_LANGUAGE_GERMAN)
"Maintainance"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Maintainance"
#elif defined(PROG_LANGUAGE_DUTCH)
"Maintainance"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Maintainance"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Maintainance"
#else
"Maintainance"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(TRUE, 0);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;


	/* Filesystem check button */
	pd->dev_check_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_fsck_32x32_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVPropDlgRunCheckCB), pd
	);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Archive Cheque De Sistema"
#elif defined(PROG_LANGUAGE_FRENCH)
"Controler le systme de fichier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Dateisystem Kontrolle"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Schedare L'Assegno Di Sistema"
#elif defined(PROG_LANGUAGE_DUTCH)
"Archiveer Systeemcontrole"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquive Sistema Verificar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiver Systemkontroll"
#else
"File System Check"
#endif
	);
	gtk_widget_show(w);
	if(STRISEMPTY(dev->command_check))
	    GTK_WIDGET_SET_SENSITIVE(w, FALSE);


	/* Tools button */
	pd->dev_tools_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_tools_32x32_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVPropDlgRunToolsCB), pd
	);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Las Herramientas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Outils"
#elif defined(PROG_LANGUAGE_GERMAN)
"Werkzeuge"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Gli Attrezzi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Werktuigen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"As Ferramentas"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Redskaper"
#else
"Tools"
#endif
	);
	gtk_widget_show(w);
	if(STRISEMPTY(dev->command_tools))
	    GTK_WIDGET_SET_SENSITIVE(w, FALSE);


	/* Format button */
	pd->dev_format_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_floppy_32x32_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVPropDlgRunFormatCB), pd
	);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Formatear Medios"
#elif defined(PROG_LANGUAGE_FRENCH)
"Formatter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Formatieren Sie Medien"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Stampa Di Formato"
#elif defined(PROG_LANGUAGE_DUTCH)
"Formatteer Media"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Imprensa De Formato"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Formater Media"
#else
"Format Media"
#endif
	);
	gtk_widget_show(w);
	if(STRISEMPTY(dev->command_format))
	    GTK_WIDGET_SET_SENSITIVE(w, FALSE);


	g_free(font_name_h1_bold);

	pd->freeze_count--;
}

/*
 *	Recreates the device usage on the Device Page.
 *
 *	The device page must already have been created by
 *	EDVPropDlgCreateDevicePage().
 *
 *	The pd specifies the Properties Dialog.
 *
 *	The dev specifies the Device.
 */
static void EDVPropDlgCreateDeviceUsage(
	edv_propdlg_struct *pd,
	edv_device_struct *dev
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint font_size;
	const gchar *font_encoding;
	gchar *font_name_h1_bold;
	GdkFont *font;
	GtkRcStyle *standard_rcstyle;
	GtkStyle *style = gtk_widget_get_style(pd->toplevel);
	GtkWidget *w, *parent, *parent2, *parent3;
	edv_core_struct *core = pd->core;
	cfg_item_struct *cfg_list = core->cfg_list;

	/* No Device Page? */
	if(pd->dev_toplevel == NULL)
	    return;

	pd->freeze_count++;

	/* Destroy any existing device usage widget s*/
	PieChartDelete(pd->dev_usage_pc);
	pd->dev_usage_pc = NULL;
	GTK_WIDGET_DESTROY(pd->dev_mount_btn);
	pd->dev_mount_btn = NULL;
	GTK_WIDGET_DESTROY(pd->dev_eject_btn);
	pd->dev_eject_btn = NULL;
	GTK_WIDGET_DESTROY(pd->dev_usage_vbox);
	pd->dev_usage_vbox = NULL;


	standard_rcstyle = core->standard_rcstyle;

	/* Get the usage parent GtkContainer */
	parent = pd->dev_usage_frame;

	/* Get the base font size */
        font = style->font;
        font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
        if(font_size < 3)
            font_size = 3;

	/* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
        font_encoding = "iso8859-2";
#else
        font_encoding = "iso8859-1";
#endif

        font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
            font_size + 2,
            font_encoding
        );

	/* Usage GtkVBox */
	pd->dev_usage_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent = w;

	/* Create the Usage Pie Chart if mounted */
	if(EDV_DEVICE_IS_MOUNTED(dev))
	{
	    const gulong	adj_division = 10l,
				block_size = EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE),
				total_kb = dev->blocks_total,
				free_kb = dev->blocks_free,
				available_kb = dev->blocks_available,
				used_kb = MAX(total_kb - free_kb, 0l);
	    gchar *ss_kb, *ss_mb, *text, *text2;
	    GdkColor c;
	    GtkAdjustment *adj;
	    pie_chart_struct *pc;

	    /* GtkHBox for multiple columns */
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* GtkHBox to center the Usage Pie Chart */
	    w = gtk_hbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent3 = w;


	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 0.0f,
		(gfloat)(total_kb / adj_division),
		0.0f, 0.0f, 0.0f
	    );
	    GDK_COLOR_SET_COEFF(&c, 0.0f, 0.0f, 1.0f);
	    ss_kb = STRDUP(EDVSizeStrDelim(total_kb));
	    ss_mb = STRDUP(EDVSizeStrDelim(
		(block_size > 0l) ?
		    (total_kb / block_size) : (total_kb / 1024l)
	    ));
	    text = g_strdup_printf(
		"%s: %s kb (%s mb)",
#if defined(PROG_LANGUAGE_SPANISH)
"La Capacidad Total"
#elif defined(PROG_LANGUAGE_FRENCH)
"Capacit Totale"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gesamte Kapazitt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Capacit Totale"
#elif defined(PROG_LANGUAGE_DUTCH)
"Totale Capaciteit"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Capacidade Total"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Total Capacity"
#else
"Total Capacity"
#endif
		, ss_kb, ss_mb   
	    );
	    g_free(ss_kb);
	    g_free(ss_mb);

	    ss_kb = STRDUP(EDVSizeStrDelim(free_kb));
	    ss_mb = STRDUP(EDVSizeStrDelim(
		(block_size > 0l) ?
		    (free_kb / block_size) : (free_kb / 1024l)
	    ));
	    text2 = g_strdup_printf(
		"%s kb (%s mb)",
		ss_kb, ss_mb
	    );
	    g_free(ss_kb);
	    g_free(ss_mb);

	    pd->dev_usage_pc = pc = PieChartNew(
		110, 70,
		adj, &c,
		NULL, text,
#if defined(PROG_LANGUAGE_SPANISH)
"Libre:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Libre:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Frei:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Libero:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vrij:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Livre:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fri:"
#else
"Free:"
#endif
		, text2
	    );
	    GTK_OBJECT_UNREF(adj);
	    g_free(text);
	    g_free(text2);
	    w = pc->toplevel;
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    PieChartMap(pc);

	    GDK_COLOR_SET_COEFF(&c, 0.0f, 1.0f, 1.0f);
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 0.0f,
		(gfloat)(available_kb / adj_division),
		0.0f, 0.0f, 0.0f
	    );
	    ss_kb = STRDUP(EDVSizeStrDelim(available_kb));
	    ss_mb = STRDUP(EDVSizeStrDelim(
		(block_size > 0l) ?
		    (available_kb / block_size) : (available_kb / 1024l)
	    ));
	    text = g_strdup_printf(
		"%s kb (%s mb)",
		ss_kb, ss_mb
	    );
	    g_free(ss_kb);
	    g_free(ss_mb);
	    PieChartValueAdd(
		pc, adj, &c,
#if defined(PROG_LANGUAGE_SPANISH)
"Libre & Disponible:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Libre & Disponible:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Frei & Verfgbar:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Libero & Disponibile:"                           
#elif defined(PROG_LANGUAGE_DUTCH)
"Vrij & Verkrijgbare:"              
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Livre & Disponvel:"            
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fri & Available:"             
#else           
"Free & Available:"         
#endif
		, text 
	    );
	    GTK_OBJECT_UNREF(adj);
	    g_free(text);

	    GDK_COLOR_SET_COEFF(&c, 1.0f, 0.0f, 1.0f);
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 0.0f,
		(gfloat)(used_kb / adj_division),
		0.0f, 0.0f, 0.0f
	    );
	    ss_kb = STRDUP(EDVSizeStrDelim(used_kb));
	    ss_mb = STRDUP(EDVSizeStrDelim(
		(block_size > 0l) ?
		    (used_kb / block_size) : (used_kb / 1024l)
	    ));
	    text = g_strdup_printf(
		"%s kb (%s mb)",
		ss_kb, ss_mb
	    );
	    g_free(ss_kb);
	    g_free(ss_mb);
	    PieChartValueAdd(
		pc, adj, &c,
#if defined(PROG_LANGUAGE_SPANISH)
"Usado:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Utilis:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Benutzt:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Usato:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Gebruikt:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Usado:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Brukt:"
#else           
"Used:"
#endif
		, text 
	    );
	    GTK_OBJECT_UNREF(adj);
	    g_free(text);


	    /* Right column vbox for buttons */
	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Unmount button */
	    pd->dev_mount_btn = w = GUIButtonPixmap(
		(guint8 **)icon_unmount_20x20_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVPropDlgMountCB), pd
	    );
	    GUISetWidgetTip(w, "Unmount");
	    GTK_WIDGET_SET_SENSITIVE(w, !EDV_DEVICE_IS_NO_UNMOUNT(dev));
	    if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	    gtk_widget_show(w);

	    /* Eject button */
	    pd->dev_eject_btn = w = GUIButtonPixmap(
		(guint8 **)icon_eject_20x20_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVPropDlgEjectCB), pd
	    );
	    GUISetWidgetTip(w, "Eject");
	    GTK_WIDGET_SET_SENSITIVE(
		w,
		!STRISEMPTY(dev->command_eject) ? TRUE : FALSE
	    );
	    if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	    gtk_widget_show(w);
	}
	else
	{
	    GtkRcStyle *rcstyle;

	    w = gtk_hbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Device not mounted label */
	    w = gtk_label_new("(Device Not Mounted)");
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_CENTER);
	    gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	    rcstyle = gtk_rc_style_new();
	    rcstyle->font_name = STRDUP(font_name_h1_bold);
	    gtk_widget_modify_style(w, rcstyle);
	    GTK_RC_STYLE_UNREF(rcstyle);
	    gtk_widget_show(w);

	    w = gtk_hbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Mount button */
	    pd->dev_mount_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_mount_20x20_xpm, "Mount", NULL
	    );
	    gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVPropDlgMountCB), pd
	    );
	    GUISetWidgetTip(w, "Click to mount the device");
	    GTK_WIDGET_SET_SENSITIVE(w, !EDV_DEVICE_IS_NO_UNMOUNT(dev));
	    if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	    gtk_widget_show(w);

	    /* Eject button */
	    pd->dev_eject_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_eject_20x20_xpm, "Eject", NULL
	    );
	    gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVPropDlgEjectCB), pd
	    );
	    GUISetWidgetTip(w, "Click to eject the media from the device");
	    GTK_WIDGET_SET_SENSITIVE(
		w,
		!STRISEMPTY(dev->command_eject) ? TRUE : FALSE
	    );
	    if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	    gtk_widget_show(w);
	}


	g_free(font_name_h1_bold);

	pd->freeze_count--;
}

/*
 *	Creates the Device Node Page.
 *
 *	The pd specifies the Properties Dialog.
 *
 *	The parent specifies the parent GtkVBox that is to be the
 *	parent for all the child widgets on this page.
 *
 *	The obj specifies the object, if obj is NULL then certain
 *	widgets will not be created and certain values will not be set.
 */
static void EDVPropDlgCreateDeviceNodePage(
	edv_propdlg_struct *pd,
	GtkWidget *parent,
	edv_object_struct *obj
)
{
	const gint	border_major = 5,
			border_minor = 2;
	GtkWidget *w, *parent2, *parent3;

	if(obj == NULL)
	    return;

	pd->freeze_count++;

	/* Record the parent */
	pd->dev_node_toplevel = parent;

	/* Device numbers GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Artefacto Numeros"
#elif defined(PROG_LANGUAGE_FRENCH)
"Numro de composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Vorrichtung Zhlt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"I Numeri Di Congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Apparaat Nummert"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Artifcio Numera"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innretning Numbers"
#else
"Device Numbers"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Major number GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Label */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Mayor:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Majeur:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Major:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Maggiore:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Majoor:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Importante:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Major:"
#else
"Major:"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Entry */
	pd->dev_node_major_entry = w = gtk_entry_new();
	gtk_widget_set_usize(w, 80, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	EDVPropDlgSetWidgetChangedSignal(pd, w);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);

	/* Minor number GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Label */
	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Menor:"
#elif defined(PROG_LANGUAGE_FRENCH)
"Mineur:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Minderjhriger:"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Minore:"
#elif defined(PROG_LANGUAGE_DUTCH)
"Minderjarige:"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Menor:"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Mindrerig:"
#else
"Minor:"
#endif
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Entry */
	pd->dev_node_minor_entry = w = gtk_entry_new();
	gtk_widget_set_usize(w, 80, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	EDVPropDlgSetWidgetChangedSignal(pd, w);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);

	pd->freeze_count--;
}


/*
 *	Creates a new Properties Dialog.
 */
edv_propdlg_struct *EDVPropDlgNew(
	edv_core_struct *core,
	const edv_location_type location_type,
	edv_object_struct *obj,
	const guint recycled_obj_index,
	const gchar *arch_path
)
{
	const gint border_major = 5;
	gint dev_num;
	const gchar *path = NULL;
	GdkWindow *window;
	GtkStyle *style;
	GtkAccelGroup *accelgrp;
	GtkRcStyle	*standard_rcstyle,
			*lists_rcstyle;
	GtkWidget	*w,
			*parent, *parent2,
			*toplevel;
	const cfg_item_struct *cfg_list;
	edv_object_type type;
	edv_device_struct *dev;
	edv_propdlg_struct *pd;

	if(core == NULL)
	    return(NULL);

	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	pd = EDV_PROPDLG(g_malloc0(sizeof(edv_propdlg_struct)));
	if(pd == NULL)
	    return(NULL);

	pd->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	pd->accelgrp = accelgrp = gtk_accel_group_new();
	pd->busy_count = 0;
	pd->freeze_count = 0;
	pd->has_changes = FALSE;
	pd->title = STRDUP(PROPDLG_TITLE);
	pd->core = core;
	pd->path = NULL;
	pd->type = type = EDV_OBJECT_TYPE_UNKNOWN;
	pd->mime_type = NULL;
	pd->recycled_obj_index = 0;
	pd->arch_path = NULL;
	pd->ext_props_list = NULL;

	pd->freeze_count++;

	/* Get the object's path, type, and MIME Type */
	pd->location_type = location_type;
	if(obj != NULL)
	{
	    const gchar *name;
	    gchar *mime_type;

	    /* Get the path */
	    g_free(pd->path);
	    pd->path = STRDUP(obj->full_path);
	    path = pd->path;

	    /* Get the name */
	    name = (path != NULL) ? g_basename(path) : NULL;

	    /* Get the type */
	    pd->type = type = obj->type;

	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		/* Get the MIME Type */
		if(!EDVMatchObjectTypeString(
		    core->mimetype, core->total_mimetypes,
		    type,
		    obj->permissions,
		    path,
		    &mime_type
		))
		{
		    g_free(pd->mime_type);
		    pd->mime_type = STRDUP(mime_type);
		}
		break;

	      case EDV_LOCATION_TYPE_RECBIN:
		/* Get the index */
		pd->recycled_obj_index = recycled_obj_index;

		/* Get the MIME Type */
		if(!EDVMatchObjectTypeString(
		    core->mimetype, core->total_mimetypes,
		    type,
		    obj->permissions,
		    path,
		    &mime_type
		))
		{
		    g_free(pd->mime_type);
		    pd->mime_type = STRDUP(mime_type);
		}
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		/* Get the archive path */
		g_free(pd->arch_path);
		pd->arch_path = STRDUP(arch_path);

		/* Get the MIME Type */
		if(!EDVMatchObjectTypeString(
		    core->mimetype, core->total_mimetypes,
		    type,
		    obj->permissions,
		    name,			/* Use the name and not the
						 * full path for archive
						 * objects */
		    &mime_type
		))
		{
		    g_free(pd->mime_type);
		    pd->mime_type = STRDUP(mime_type);
		}
		break;
	    }

	    /* Get the properties list */
	    if(obj->ext_props_list != NULL)
	    {
		GList *glist;
		for(glist = obj->ext_props_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		    pd->ext_props_list = g_list_append(
			pd->ext_props_list,
			EDVObjectPropCopy(EDV_OBJECT_PROP(glist->data))
		    );
	    }
	}

	/* Refresh all device mount states and device stats, then get
	 * the device path or mount path that matches the object
	 */
	EDVDevicesListUpdateMountStates(
	    core->device, core->total_devices
	);
	EDVDevicesListUpdateStats(
	    core->device, core->total_devices
	);
	dev = EDVDevicesListMatchMountPath(
	   core->device, core->total_devices,
	   &dev_num, path
	);
	if(dev == NULL)
	   dev = EDVDevicesListMatchDevicePath(
		core->device, core->total_devices,
		&dev_num, path
	   );


	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_widget_set_usize(w, PROPDLG_WIDTH, PROPDLG_HEIGHT);
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "propertiesdialog", PROG_NAME
	);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
		GDK_DECOR_MINIMIZE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_properties2_48x48_xpm);
	}
	gtk_widget_add_events( 
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVPropDlgDeleteEventCB), pd
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	style = gtk_widget_get_style(w);
	parent = w;


	/* Main GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Main notebook GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Main GtkNotebook */
	pd->notebook = w = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(w), TRUE);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w), TRUE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(w), TRUE);
/*      gtk_notebook_set_page(GTK_NOTEBOOK(w), 0); */
	gtk_signal_connect(
	    GTK_OBJECT(w), "switch_page",
	    GTK_SIGNAL_FUNC(EDVPropDlgSwitchPageCB), pd
	);
	gtk_widget_show(w);
	parent2 = w;


	/* Begin creating the pages on the main GtkNotebook */

	/* Create the General Page (always the first page) */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_notebook_append_page(
	    GTK_NOTEBOOK(parent2), w, gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"General"
#elif defined(PROG_LANGUAGE_FRENCH)
"Gnral"
#elif defined(PROG_LANGUAGE_GERMAN)
"General"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Generale"
#elif defined(PROG_LANGUAGE_DUTCH)
"Generaal"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O General"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"General"
#else
"General"
#endif
	    )
	);
	gtk_widget_show(w);
	EDVPropDlgCreateGeneralPage(pd, w, obj);


	/* Is the object a link? */
	if(type == EDV_OBJECT_TYPE_LINK)
	{
	    /* Create Link Page */
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_notebook_append_page(
		GTK_NOTEBOOK(parent2), w, gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Enlace"
#elif defined(PROG_LANGUAGE_FRENCH)
"Lien"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kettenglied"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Maglia"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schakel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Elo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ledd"
#else
"Link"
#endif
		)
	    );
	    gtk_widget_show(w);
	    EDVPropDlgCreateLinkPage(pd, w, obj);
	}


	/* Create the Device Page
	 *
	 * Check if a device structure for this object was matched
	 * (if this object is a mount point for a device)
	 */
	if((location_type == EDV_LOCATION_TYPE_VFS) &&
	   (dev != NULL)
	)
	{
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_notebook_append_page(
		GTK_NOTEBOOK(parent2), w, gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Artefacto"
#elif defined(PROG_LANGUAGE_FRENCH)
"Composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innretning"
#else
"Device"
#endif
		)
	    );
	    gtk_widget_show(w);
	    EDVPropDlgCreateDevicePage(pd, w, obj, dev_num, dev);
	}


	/* Create the Device Node Page
	 *
	 * Check if the object is a device node
	 */
	if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
	   (type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
	{
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_notebook_append_page(
		GTK_NOTEBOOK(parent2), w, gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Artefacto Puerto"
#elif defined(PROG_LANGUAGE_FRENCH)
"Port du composant"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gerteknoten"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Nodo Di Congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Apparaat Knoest"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Node De Artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innretning Node"
#else
"Device Node"
#endif
		)
	    );
	    gtk_widget_show(w);
	    EDVPropDlgCreateDeviceNodePage(pd, w, obj);
	}

	/* Create the Details Page */
	if((location_type == EDV_LOCATION_TYPE_VFS) ||
	   (location_type == EDV_LOCATION_TYPE_RECBIN) ||
	   (location_type == EDV_LOCATION_TYPE_ARCHIVE)
	)
	{
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_notebook_append_page(
		GTK_NOTEBOOK(parent2), w, gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Los Detalles"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dtails"
#elif defined(PROG_LANGUAGE_GERMAN)
"Statistik"
#elif defined(PROG_LANGUAGE_ITALIAN)
"I Dettagli"
#elif defined(PROG_LANGUAGE_DUTCH)
"Statistieken"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Os Detalhes"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Detaljer"
#else
"Details"
#endif
		)
	    );
	    gtk_widget_show(w);
	    EDVPropDlgCreateDetailsPage(pd, w, obj);
	}


	/* GtkSeparator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* OK button */
	pd->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "OK", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVPropDlgOKCB), pd
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);
	gtk_widget_show(w);

	/* Apply button */
	pd->apply_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_select_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique"
#elif defined(PROG_LANGUAGE_FRENCH)
"Appliquer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toepas"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld"
#else
"Apply"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVPropDlgApplyCB), pd
	);
	gtk_accel_group_add(
	    accelgrp, GDK_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_a);
	gtk_widget_show(w);

	/* Cancel button */
	pd->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cancele"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Heben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Annullare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Annuleer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cancelamento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kanseller"
#else
"Cancel"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVPropDlgCancelCB), pd
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);

	/* Close button */
	pd->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Quitter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVPropDlgCancelCB), pd
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);


	/* Set the initial values for the General, Details, and
	 * Device Node pages
	 */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    if(path != NULL)
	    {
		struct stat lstat_buf;
		if(lstat((const char *)path, &lstat_buf))
		{
		    const gint error_code = (gint)errno;
		    EDVPropDlgGeneralPageUpdate(pd, NULL, error_code, TRUE);
		    EDVPropDlgDeviceNodePageUpdate(pd, NULL);
		    EDVPropDlgDetailsPageUpdate(pd, NULL, pd->ext_props_list);
		}
		else
		{
		    EDVPropDlgGeneralPageUpdate(pd, &lstat_buf, 0, TRUE);
		    EDVPropDlgDeviceNodePageUpdate(pd, &lstat_buf);
		    EDVPropDlgDetailsPageUpdate(pd, &lstat_buf, pd->ext_props_list);
		}
	    }
	    break;
	  case EDV_LOCATION_TYPE_RECBIN:
	    if(obj != NULL)
	    {
		struct stat lstat_buf;

		EDVObjectSetToStat(&lstat_buf, obj);

		EDVPropDlgGeneralPageUpdate(pd, &lstat_buf, 0, TRUE);
		EDVPropDlgDeviceNodePageUpdate(pd, &lstat_buf);
		EDVPropDlgDetailsPageUpdate(pd, &lstat_buf, pd->ext_props_list);
	    }
	    break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
	    if(obj != NULL)
	    {
		struct stat lstat_buf;

		EDVObjectSetToStat(&lstat_buf, obj);

		EDVPropDlgGeneralPageUpdate(pd, &lstat_buf, 0, TRUE);
		EDVPropDlgDeviceNodePageUpdate(pd, &lstat_buf);
		EDVPropDlgDetailsPageUpdate(pd, &lstat_buf, pd->ext_props_list);
	    }
	    break;
	}

	/* Set initial the RC styles */
	if(standard_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		pd->toplevel, standard_rcstyle
	    );

	if(pd->gen_toplevel != NULL)
	{
	    w = MenuButtonGetMenu(pd->gen_date_touch_btn);
	    if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(
		    w, standard_rcstyle
		);
	}


	EDVPropDlgUpdateMenus(pd);

	pd->freeze_count--;

	return(pd);
}


/*
 *	Sets the Properties Dialog has changes marker.
 */
void EDVPropDlgSetHasChanges(
	edv_propdlg_struct *pd, const gboolean has_changes
)
{
	if(pd == NULL)
	    return;

	if(pd->has_changes == has_changes)
	    return;

	pd->has_changes = has_changes;
	EDVPropDlgUpdateMenus(pd);
}

/*
 *	Updates the Properties Dialog's widgets to reflect current
 *	values.
 */
void EDVPropDlgUpdateMenus(edv_propdlg_struct *pd)
{
	gboolean	sensitive,
			write_protect,
			has_changes,
			read_only;
	gchar *s;
	const gchar *path, *name;
	const cfg_item_struct *cfg_list;
	edv_object_type type;
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	pd->freeze_count++;

	core = pd->core;
	cfg_list = core->cfg_list;
	write_protect = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);
	type = pd->type;
	path = pd->path;
	name = (path != NULL) ? g_basename(path) : NULL;
	has_changes = pd->has_changes;

	/* Determine if the object is read only */
	read_only = TRUE;
	switch(pd->location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    read_only = write_protect;
	    break;
	  case EDV_LOCATION_TYPE_RECBIN:
	    read_only = write_protect;
	    break;
	  case EDV_LOCATION_TYPE_ARCHIVE:
	    read_only = TRUE;
	    break;
	}

#define MAP(_w_)	{		\
 if((_w_) != NULL)			\
  gtk_widget_show(_w_);			\
}
#define UNMAP(_w_)	{		\
 if((_w_) != NULL)			\
  gtk_widget_hide(_w_);			\
}

	/* Title */
	if(path != NULL)
	    s = g_strconcat(
		pd->title,
		": ",
		(name != NULL) ? name : path,
		(has_changes) ? " (*)" : "",
		NULL
	    );
	else
	    s = STRDUP(pd->title);
	gtk_window_set_title(GTK_WINDOW(pd->toplevel), s);
	g_free(s);

	/* Touch Button */
	sensitive = !read_only;
	GTK_WIDGET_SET_SENSITIVE(pd->gen_date_touch_btn, sensitive);

	/* Set the Permissions GtkFrame insensitive for object
	 * types that do not have permissions
	 */
	sensitive = (type == EDV_OBJECT_TYPE_LINK) ? FALSE : TRUE;
	GTK_WIDGET_SET_SENSITIVE(pd->gen_permissions_frame, sensitive);

	/* OK, Apply, Cancel, Close Buttons */
	sensitive = (read_only) ? FALSE : has_changes;
	GTK_WIDGET_SET_SENSITIVE(pd->ok_btn, sensitive);
	GTK_WIDGET_SET_SENSITIVE(pd->apply_btn, sensitive);
	sensitive = TRUE;
	GTK_WIDGET_SET_SENSITIVE(pd->cancel_btn, sensitive);
	GTK_WIDGET_SET_SENSITIVE(pd->close_btn, sensitive);
	if(has_changes)
	{
	    MAP(pd->ok_btn);
	    MAP(pd->apply_btn);
	    MAP(pd->cancel_btn);
	    UNMAP(pd->close_btn);
	}
	else
	{
	    MAP(pd->ok_btn);
	    MAP(pd->apply_btn);
	    UNMAP(pd->cancel_btn);
	    MAP(pd->close_btn);
	}

#undef UNMAP
#undef MAP

	pd->freeze_count--;
}

/*
 *	Sets the Properties Dialog as busy or ready.
 */
void EDVPropDlgSetBusy(edv_propdlg_struct *pd, const gboolean busy)
{
	GdkCursor *cursor;
	GtkWidget *w;
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	core = pd->core;

	w = pd->toplevel;
	if(w != NULL)
	{
	    if(busy)
	    {
		/* Increase the busy count */
		pd->busy_count++;

		/* If already busy then don't change anything */
		if(pd->busy_count > 1)
		    return;

		cursor = EDVGetCursor(core, EDV_CURSOR_CODE_BUSY);
	    }
	    else
	    {
		/* Reduce the busy count */
		pd->busy_count--;
		if(pd->busy_count < 0)
		    pd->busy_count = 0;

		/* If still busy then do not change anything */
		if(pd->busy_count > 0)
		    return;

		cursor = NULL;		/* Use default cursor */
	    }

	    /* Update toplevel window's cursor */
	    if(w->window != NULL)
	    {
		gdk_window_set_cursor(w->window, cursor);
		gdk_flush();
	    }
	}
}

/*
 *	Checks if the Properties Dialog is mapped.
 */
gboolean EDVPropDlgIsMapped(edv_propdlg_struct *pd)
{
	if(pd == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(pd->toplevel));
}

/*
 *	Maps the Properties Dialog.
 */
void EDVPropDlgMap(edv_propdlg_struct *pd)
{
	GtkWidget *w;

	if(pd == NULL)
	    return;

	gtk_widget_show_raise(pd->toplevel);

	w = pd->close_btn;
	gtk_widget_grab_focus(w);
	gtk_widget_grab_default(w);
}

/*
 *	Unmaps the Properties Dialog.
 */
void EDVPropDlgUnmap(edv_propdlg_struct *pd)
{
	if(pd == NULL)
	    return;

	gtk_widget_hide(pd->toplevel);
}

/*
 *	Deletes the Properties Dialog.
 */
void EDVPropDlgDelete(edv_propdlg_struct *pd)
{
	if(pd == NULL)
	    return;

	EDVPropDlgUnmap(pd);

	pd->freeze_count++;

	/* Details Page */
	if(pd->det_toplevel != NULL)
	{
	    if(pd->det_clist != NULL)
	    {
		GtkCList *clist = GTK_CLIST(pd->det_clist);
		gtk_clist_freeze(clist);
		gtk_clist_clear(clist);
		gtk_clist_thaw(clist);
	    }
	}

	/* Device Page */
	if(pd->dev_toplevel != NULL)
	{
	    PieChartDelete(pd->dev_usage_pc);
	}

	/* Destroy the toplevel GtkWidget */
	gtk_widget_destroy(pd->toplevel);
	gtk_accel_group_unref(pd->accelgrp);

	if(pd->ext_props_list != NULL)
	{
	    g_list_foreach(
		pd->ext_props_list,
		(GFunc)EDVObjectPropDelete,
		NULL
	    );
	    g_list_free(pd->ext_props_list);
	}

	g_free(pd->title);
	g_free(pd->path);
	g_free(pd->mime_type);
	g_free(pd->arch_path);

	pd->freeze_count--;

	g_free(pd);
}
