#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "pdialog.h"
#include "progressdialog.h"
#include "toolbar.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_obj.h"
#include "edv_recycled_obj.h"
#include "edv_recbin_index.h"
#include "edv_recbin_purge.h"
#include "edv_recbin_recover.h"
#include "edv_find_bar.h"
#include "edv_status_bar.h"
#include "edv_confirm.h"
#include "edv_utils.h"
#include "endeavour2.h"
#include "edv_recbin_sync.h"
#include "recbin.h"
#include "recbin_cb.h"
#include "recbin_op_cb.h"
#include "recbin_contents_list.h"
#include "recbin_desktop_icon.h"
#include "edv_cb.h"
#include "edv_help.h"
#include "edv_op.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_wildcards_32x32.xpm"


static gint EDVRecBinRMkDir(
	edv_core_struct *core, const gchar *path
);

void EDVRecBinOPCB(
	toolbar_item_struct *item, gint id, gpointer data
);
void EDVRecBinOPEnterCB(
	toolbar_item_struct *item, gint id, gpointer data
);
void EDVRecBinOPLeaveCB(
	toolbar_item_struct *item, gint id, gpointer data
);

void EDVRecBinOPClose(edv_recbin_struct *recbin);
void EDVRecBinOPExit(edv_recbin_struct *recbin);

void EDVRecBinOPSyncRecycleBin(edv_recbin_struct *recbin);

void EDVRecBinOPSyncDisks(edv_recbin_struct *recbin);
void EDVRecBinOPWriteProtect(edv_recbin_struct *recbin);
void EDVRecBinOPDeleteMethodRecycle(edv_recbin_struct *recbin);
void EDVRecBinOPDeleteMethodPurge(edv_recbin_struct *recbin);

void EDVRecBinOPRecover(edv_recbin_struct *recbin);
void EDVRecBinOPPurge(edv_recbin_struct *recbin);
void EDVRecBinOPPurgeAll(edv_recbin_struct *recbin);
void EDVRecBinOPRename(edv_recbin_struct *recbin);
void EDVRecBinOPChMod(edv_recbin_struct *recbin);
void EDVRecBinOPChOwn(edv_recbin_struct *recbin);
void EDVRecBinOPChTime(edv_recbin_struct *recbin);
void EDVRecBinOPProperties(edv_recbin_struct *recbin);

void EDVRecBinOPSelectAll(edv_recbin_struct *recbin);
void EDVRecBinOPUnselectAll(edv_recbin_struct *recbin);
void EDVRecBinOPInvertSelection(edv_recbin_struct *recbin);

void EDVRecBinOPRefresh(edv_recbin_struct *recbin);
void EDVRecBinOPRefreshAll(edv_recbin_struct *recbin);

void EDVRecBinContentsFilter(edv_recbin_struct *recbin);

void EDVRecBinMIMETypes(edv_recbin_struct *recbin);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates the directory and all parent directories as needed and
 *	then reports them.
 *
 *	The path specifies the directory, if it does not exist then
 *	it and all its parent directories will be created as needed.
 */
static gint EDVRecBinRMkDir(
	edv_core_struct *core, const gchar *path
)
{
	struct stat stat_buf;
	gchar *dpath;
	GList *paths_list = NULL;

	if(STRISEMPTY(path))
	    return(-2);

	dpath = STRDUP(path);
	if(dpath == NULL)
	    return(-3);

	/* Path does not exist? */
	if(stat(dpath, &stat_buf))
	{ 
	    const guint m = EDVGetUMask();
	    const gchar *parent;
	    GList *glist;

	    /* Seek to first deliminator */
	    gchar *s = strchr(
		dpath + STRLEN("/"),
		G_DIR_SEPARATOR
	    );

	    /* Iterate through each directory compoent in the
	     * path, by tempory setting the deliminator to '\0'
	     * and checking for its existance
	     */
	    while(s != NULL)
	    {
		/* Tempory set deliminator to '\0' */
		*s = '\0';

		/* Does this compoent not exist? */
		if(stat(dpath, &stat_buf))
		    paths_list = g_list_append(
			paths_list, STRDUP(dpath)
		    );

		/* Restore deliminator */
		*s = G_DIR_SEPARATOR;

		/* Seek to next deliminator (if any) */
		s = strchr(s + 1, G_DIR_SEPARATOR);
	    }

	    /* Last directory compoent does not exist (from the
	     * very first check), so add it to the list of new
	     * objects
	     */
	    paths_list = g_list_append(
		paths_list, STRDUP(dpath)
	    );

	    /* Create the directories */
	    if(rmkdir(
		dpath,
		(~m) &
		    (S_IRUSR | S_IWUSR | S_IXUSR |
		     S_IRGRP | S_IWGRP | S_IXGRP |
		     S_IROTH | S_IWOTH | S_IXOTH)
	    ))
	    {
		g_list_foreach(paths_list, (GFunc)g_free, NULL);
		g_list_free(paths_list);
		g_free(dpath);
		return(-1);
	    }

	    /* Report the added directories */
	    for(glist = paths_list; glist != NULL; glist = g_list_next(glist))
	    {
		parent = (const gchar *)glist->data;
		if(parent == NULL)
		    continue;

		if(!lstat((const char *)parent, &stat_buf))
		    EDVObjectAddedEmit(
			core, parent, &stat_buf
		    );
	    }
	}

	g_list_foreach(paths_list, (GFunc)g_free, NULL);
	g_list_free(paths_list);
	g_free(dpath);

	return(0);
}


/*
 *	Operation id callback nexus.
 *
 *	The given client data must be a edv_recbin_opid_struct *.
 */
void EDVRecBinOPCB(
	toolbar_item_struct *item, gint id, gpointer data  
)
{
	GtkWidget *toplevel;
	cfg_item_struct *cfg_list;
	edv_recbin_struct *recbin;
	edv_core_struct *core;
	edv_recbin_opid_struct *opid = EDV_RECBIN_OPID(data);
	if(opid == NULL)
	    return;

	recbin = opid->recbin;
	if(recbin == NULL)
	    return;

	if(recbin->processing || (recbin->freeze_count > 0))
	    return;

	toplevel = recbin->toplevel;
	core = recbin->core;
	cfg_list = core->cfg_list;

	recbin->freeze_count++;

	/* Handle by operation id code */
	switch(opid->op)
	{
	  case EDV_RECBIN_OP_NONE:
	  case EDV_RECBIN_OP_SEPARATOR:
	    break;

	  case EDV_RECBIN_OP_CLOSE:
	    EDVRecBinOPClose(recbin);
	    break;

	  case EDV_RECBIN_OP_EXIT:
	    EDVRecBinOPExit(recbin);
	    break;


	  case EDV_RECBIN_OP_SYNC_DISKS:
	    EDVRecBinOPSyncDisks(recbin);
	    break;

	  case EDV_RECBIN_OP_HISTORY:
	    EDVMapHistoryListWin(core, toplevel);
	    break;

	  case EDV_RECBIN_OP_RUN:
	    EDVMapRunDialogCommand(
		core,
		NULL,
		NULL,
		toplevel
	    );
	    break;

	  case EDV_RECBIN_OP_RUN_TERMINAL:
	    EDVRunTerminal(core, NULL, NULL, toplevel);
	    break;

	  case EDV_RECBIN_OP_WRITE_PROTECT:
	    EDVRecBinOPWriteProtect(recbin);
	    break;

	  case EDV_RECBIN_OP_DELETE_METHOD_RECYCLE:
	    EDVRecBinOPDeleteMethodRecycle(recbin);
	    break;

	  case EDV_RECBIN_OP_DELETE_METHOD_PURGE:
	    EDVRecBinOPDeleteMethodPurge(recbin);
	    break;


	  case EDV_RECBIN_OP_SYNC_RECYCLE_BIN:
	    EDVRecBinOPSyncRecycleBin(recbin);
	    break;


	  case EDV_RECBIN_OP_RECOVER:
	    EDVRecBinOPRecover(recbin);
	    break;

	  case EDV_RECBIN_OP_PURGE:
	    EDVRecBinOPPurge(recbin);
	    break;

	  case EDV_RECBIN_OP_PURGE_ALL:
	    EDVRecBinOPPurgeAll(recbin);
	    break;

	  case EDV_RECBIN_OP_RENAME:
	    EDVRecBinOPRename(recbin);
	    break;

	  case EDV_RECBIN_OP_CHMOD:
	    EDVRecBinOPChMod(recbin);
	    break;

	  case EDV_RECBIN_OP_CHOWN:
	    EDVRecBinOPChOwn(recbin);
	    break;

	  case EDV_RECBIN_OP_CHTIME:
	    EDVRecBinOPChTime(recbin);
	    break;

	  case EDV_RECBIN_OP_PROPERTIES:
	    EDVRecBinOPProperties(recbin);
	    break;


	  case EDV_RECBIN_OP_SELECT_ALL:
	    EDVRecBinOPSelectAll(recbin);
	    break;

	  case EDV_RECBIN_OP_UNSELECT_ALL:
	    EDVRecBinOPUnselectAll(recbin);
	    break;

	  case EDV_RECBIN_OP_INVERT_SELECTION:
	    EDVRecBinOPInvertSelection(recbin);
	    break;


	  case EDV_RECBIN_OP_FIND:
	    EDVMapRecBinFindWin(core, recbin);
	    break;


	  case EDV_RECBIN_OP_REFRESH:
	    EDVRecBinOPRefresh(recbin);
	    break;

	  case EDV_RECBIN_OP_REFRESH_ALL:
	    EDVRecBinOPRefreshAll(recbin);
	    break;

	  case EDV_RECBIN_OP_SHOW_TOOL_BAR:
	    if(core != NULL)
	    {
		const gboolean state = !EDV_GET_B(
		    EDV_CFG_PARM_RECBIN_SHOW_TOOL_BAR
		);
		EDV_SET_B(
		    EDV_CFG_PARM_RECBIN_SHOW_TOOL_BAR,
		    state
		);
		EDVReconfiguredEmit(core);
	    }
	    break;

	  case EDV_RECBIN_OP_SHOW_FIND_BAR:
	    if(core != NULL)
	    {
		const gboolean state = !EDV_GET_B(
		    EDV_CFG_PARM_RECBIN_SHOW_FIND_BAR
		);
		EDV_SET_B(
		    EDV_CFG_PARM_RECBIN_SHOW_FIND_BAR,
		    state
		);
		EDVReconfiguredEmit(core);
	    }
	    break;

	  case EDV_RECBIN_OP_SHOW_STATUS_BAR:
	    if(core != NULL)
	    {
		const gboolean state = !EDV_GET_B(
		    EDV_CFG_PARM_RECBIN_SHOW_STATUS_BAR
		);
		EDV_SET_B(
		    EDV_CFG_PARM_RECBIN_SHOW_STATUS_BAR,
		    state
		);
		EDVReconfiguredEmit(core);
	    }
	    break;


	  case EDV_RECBIN_OP_CONTENTS_LIST_FILTER:
	    EDVRecBinContentsFilter(recbin);
	    break;

	  case EDV_RECBIN_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS:
	    if(core != NULL)
	    {
		const gboolean state = !EDV_GET_B(
		    EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS
		);
		EDV_SET_B(
		    EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS,
		    state
		);
		EDVReconfiguredEmit(core);
	    }
	    break;


	  case EDV_RECBIN_OP_MIME_TYPES:
	    EDVRecBinMIMETypes(recbin);
	    break;


	  case EDV_RECBIN_OP_NEW_BROWSER:
	    EDVNewBrowser(core, NULL);
	    break;

	  case EDV_RECBIN_OP_NEW_IMBR:
	    EDVNewImbr(core, NULL);
	    break;

	  case EDV_RECBIN_OP_NEW_ARCHIVER:
	    EDVNewArchiver(core, NULL, NULL);
	    break;


	  case EDV_RECBIN_OP_OPTIONS:
	    EDVMapOptionsWin(core, toplevel);
	    break;

	  case EDV_RECBIN_OP_CUSTOMIZE:
	    EDVMapCustomizeWin(core, toplevel);
	    break;


	  case EDV_RECBIN_OP_HELP_CONTENTS:
	    EDVHelp(core, "Contents", toplevel);
	    break;
	  case EDV_RECBIN_OP_HELP_FILE_BROWSER:
	    EDVHelp(core, "File Browser", toplevel);
	    break;
	  case EDV_RECBIN_OP_HELP_IMAGE_BROWSER:
	    EDVHelp(core, "Image Browser", toplevel);
	    break;
	  case EDV_RECBIN_OP_HELP_ARCHIVER:
	    EDVHelp(core, "Archiver", toplevel);
	    break;
	  case EDV_RECBIN_OP_HELP_RECYCLE_BIN:
	    EDVHelp(core, "Recycle Bin", toplevel);
	    break;
	  case EDV_RECBIN_OP_HELP_KEYS_LIST:
	    EDVHelp(core, "Keys List", toplevel);
	    break;
	  case EDV_RECBIN_OP_HELP_COMMON_OPERATIONS:
	    EDVHelp(core, "Common Operations", toplevel);
	    break;
	  case EDV_RECBIN_OP_HELP_ABOUT:
	    EDVAbout(core, toplevel);
	    break;
	}

	recbin->freeze_count--;
}

/*
 *	Operation id enter notify callback nexus.
 *
 *	The given client data must be a edv_recbin_opid_struct *.
 */
void EDVRecBinOPEnterCB(
	toolbar_item_struct *item, gint id, gpointer data  
)
{
	const gchar *tooltip;
	edv_recbin_opid_struct *opid = EDV_RECBIN_OPID(data);
	edv_recbin_struct *recbin = (opid != NULL) ? opid->recbin : NULL;
	if(recbin == NULL)
	    return;

	tooltip = opid->tooltip;
	if(tooltip != NULL)
	    EDVStatusBarMessage(recbin->status_bar, tooltip, FALSE);
}

/*
 *	Operation id leave notify callback nexus.
 */
void EDVRecBinOPLeaveCB(
	toolbar_item_struct *item, gint id, gpointer data  
)
{
	edv_recbin_opid_struct *opid = EDV_RECBIN_OPID(data);
	edv_recbin_struct *recbin = (opid != NULL) ? opid->recbin : NULL;
	if(recbin == NULL)
	    return;

	EDVStatusBarMessage(recbin->status_bar, NULL, FALSE);
}


/*
 *	Close.
 */
void EDVRecBinOPClose(edv_recbin_struct *recbin)
{
	if(recbin == NULL)
	    return;

	EDVRecBinSyncConfiguration(recbin);
	EDVRecBinUnmap(recbin);
}

/*
 *	Close all windows.
 */
void EDVRecBinOPExit(edv_recbin_struct *recbin)
{
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	core = recbin->core;

	/* Record the Recycle Bin's current configuration */
	EDVRecBinSyncConfiguration(recbin);

	/* Unmap the Recycle Bin */
	EDVRecBinUnmap(recbin);

	/* Schedual a new pending operation on the core to close all
	 * the windows
	 */
	core->pending_flags |= EDV_CORE_PENDING_CLOSE_ALL_WINDOWS;
}


/*
 *	Sync Disks.
 */
void EDVRecBinOPSyncDisks(edv_recbin_struct *recbin)
{
	if(recbin == NULL)
	    return;

	EDVStatusBarMessage(
	    recbin->status_bar,
	    "Syncing disks...",
	    TRUE
	);

	EDVRecBinSetBusy(recbin, TRUE);

	EDVSyncDisks(recbin->core);

	EDVRecBinSetBusy(recbin, FALSE);

	EDVStatusBarMessage(
	    recbin->status_bar,
	    "Disk sync done",
	    FALSE
	);
	EDVStatusBarProgress(recbin->status_bar, 0.0f, FALSE);
}

/*
 *	Write Protect Toggle.
 */
void EDVRecBinOPWriteProtect(edv_recbin_struct *recbin)
{
	gboolean write_protect;
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Get current write protect state */
	write_protect = (gboolean)CFGItemListGetValueI(
	    cfg_list, EDV_CFG_PARM_WRITE_PROTECT
	);

	/* Toggle write protect */
	write_protect = !write_protect;

	/* Set new write protect state */
	CFGItemListSetValueI(
	    cfg_list, EDV_CFG_PARM_WRITE_PROTECT,
	    write_protect, FALSE
	);

	/* Emit write protect changed signal */
	EDVWriteProtectChangedEmit(core, write_protect);
}

/*
 *	Delete Method: Recycle.
 */
void EDVRecBinOPDeleteMethodRecycle(edv_recbin_struct *recbin)
{
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	core = recbin->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
	    EDV_CFG_PARM_DELETE_METHOD,
	    EDV_DELETE_METHOD_RECYCLE
	);
	EDVReconfiguredEmit(core);
}

/*
 *	Delete Method: Purge.
 */
void EDVRecBinOPDeleteMethodPurge(edv_recbin_struct *recbin)
{
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	core = recbin->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
	    EDV_CFG_PARM_DELETE_METHOD,
	    EDV_DELETE_METHOD_PURGE
	);
	EDVReconfiguredEmit(core);
}


/*
 *	Sync Recycle Bin.
 *
 *	Compacts recycled objects and fix any errors.
 */
void EDVRecBinOPSyncRecycleBin(edv_recbin_struct *recbin)
{
	gboolean yes_to_all = FALSE;
	gint status;
	GtkWidget *toplevel;
	edv_status_bar_struct *sb;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	toplevel = recbin->toplevel;
	core = recbin->core;
	sb = recbin->status_bar;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	    return;

	EDVStatusBarMessage(
	    sb,
	    "Syncing recycle bin...",
	    TRUE
	);

	EDVRecBinSetBusy(recbin, TRUE);

	/* Sync the recycle bin */
	status = EDVRecBinSync(
	    core,
	    toplevel,
	    TRUE,
	    TRUE,
	    &yes_to_all,
	    EDVRecBinStatusMessageCB,
	    recbin,
	    EDVRecBinStatusProgressCB,
	    recbin
	);

	/* Refresh listing of recycled objects */
	EDVRecBinOPRefresh(recbin);

	EDVRecBinSetBusy(recbin, FALSE);

	EDVStatusBarMessage(
	    sb,
	    "Recycle bin sync done",
	    FALSE
	);
	EDVStatusBarProgress(sb, 0.0f, FALSE);
}


/*
 *	Recover callback.
 */
void EDVRecBinOPRecover(edv_recbin_struct *recbin)
{
	gboolean yes_to_all = FALSE;
	struct stat stat_buf;
	gint row, status, nobjs, nobjs_recovered = 0;
	guint index;
	gchar *new_path;
	const gchar *error_msg;
	GList *glist, *obj_list = NULL;
	GtkWidget *toplevel;
	GtkCList *clist;
	edv_recycled_object_struct *obj = NULL;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	toplevel = recbin->toplevel;
	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	    return;

#define CLEANUP_RETURN	{				\
 if(obj_list != NULL) {					\
  g_list_foreach(					\
   obj_list, (GFunc)EDVRecycledObjectDelete, NULL	\
  );							\
  g_list_free(obj_list);				\
 }							\
							\
 return;						\
}

	/* Get the list of selected recycled objects */
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    row = (gint)glist->data;
	    obj = EDV_RECYCLED_OBJECT(
		gtk_clist_get_row_data(clist, row)
	    );
	    if(obj != NULL)
		obj_list = g_list_append(
		    obj_list,
		    EDVRecycledObjectCopy(obj)
		);
	}

	/* No recycled objects to recover? */
	if(obj_list == NULL)
	{
	    CLEANUP_RETURN;
	}

	nobjs = g_list_length(obj_list);

	EDVRecBinSetBusy(recbin, TRUE);

	/* Confirm recover */
	status = EDVConfirmRecover(
	    core, toplevel,
	    (obj != NULL) ? obj->name : NULL,
	    nobjs,
	    ((obj != NULL) && (nobjs == 1)) ? obj->original_path : NULL
	);
	switch(status)
	{
	  case CDIALOG_RESPONSE_YES:
	  case CDIALOG_RESPONSE_YES_TO_ALL:
	    break;

	  default:
	    EDVRecBinSetBusy(recbin, FALSE);
	    CLEANUP_RETURN;
	    break;
	}

	/* Iterate through list of recycled objects and recover each
	 * one
	 */
	status = 0;
	for(glist = obj_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    obj = EDV_RECYCLED_OBJECT(glist->data);
	    if(obj == NULL)
		continue;

	    index = obj->index;

	    /* Check if this object's original path no longer exists */
	    if((obj->original_path != NULL) ?
		stat((const char *)obj->original_path, &stat_buf) : FALSE
	    )
	    {
		gint response;
		const gchar *parent = obj->original_path;
		gchar *s = g_strdup_printf(
"The original location:\n\
\n\
    %s\n\
\n\
Of:\n\
\n\
    %s\n\
\n\
No longer exists, create it?",
		    parent,
		    obj->name
		);
		EDVPlaySoundQuestion(core);
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
		    "Create Directory",
		    s,
		    NULL,
		    CDIALOG_ICON_QUESTION,
		    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO | CDIALOG_BTNFLAG_CANCEL,
		    CDIALOG_BTNFLAG_YES
		);
		CDialogSetTransientFor(NULL);
		g_free(s);

		/* Create directory? */
		if(response == CDIALOG_RESPONSE_YES)
		{
		    if(EDVRecBinRMkDir(core, parent))
		    {
			s = g_strdup_printf(
"Unable to create directory:\n\
\n\
    %s",
			    parent
			);
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Create Directory Error",
			    s,
			    NULL,
			    toplevel
			);
			g_free(s);
		    }
		}
		/* User aborted? */
		else if(response == CDIALOG_RESPONSE_CANCEL)
		{
		    status = -4;
		    break;
		}
	    }

	    /* Recover this recycled object */
	    new_path = NULL;
	    status = EDVRecBinRecover(
		core,
		index,		/* Recycled object to recover */
		NULL,		/* Use the original location as the
				 * recovery path */
		&new_path,
		toplevel,
		TRUE, TRUE, &yes_to_all
	    );

	    /* Check for errors */
	    error_msg = EDVRecBinRecoverGetError(core);
	    if(!STRISEMPTY(error_msg))
	    {
		/* Report the error */
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Recover Recycled Object Error",
		    error_msg,
		    NULL,
		    toplevel
		);
	    }

	    /* Object recovered? */
	    if(new_path != NULL)
	    {
		struct stat lstat_buf;

		nobjs_recovered++;

		/* Report this recycled object removed */
		EDVRecycledObjectRemovedEmit(core, index);

		/* Report the recovered object */
		if(!lstat(new_path, &lstat_buf))
		    EDVObjectAddedEmit(
			core,
			new_path,
			&lstat_buf
		    );

		g_free(new_path);
	    }

	    /* User aborted? */
	    if(status == -4)
		break;
	}

	/* Unmap the progress dialog, it may have been mapped during
	 * the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the "completed" sound on success */
	if(status == 0)
	    EDVPlaySoundCompleted(core);

	/* Update the status bar message */
	if(TRUE)
	{
	    gchar *msg;
	    if(status == -4)
		msg = STRDUP(
		    "Recover operation canceled"
		);
	    else if(nobjs_recovered > 0)
		msg = g_strdup_printf(
		    "Recovered %i %s",
		    nobjs_recovered,
		    (nobjs_recovered == 1) ? "recycled object" : "recycled objects" 		);
	    else
		msg = g_strdup_printf(
		    "Unable to recover %s",
		    (nobjs == 1) ? "recycled object" : "recycled objects"
		);
	    EDVStatusBarMessage(recbin->status_bar, msg, FALSE);
	    g_free(msg);
	}

	EDVRecBinSetBusy(recbin, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Purge callback.
 */
void EDVRecBinOPPurge(edv_recbin_struct *recbin)
{
	gboolean yes_to_all = FALSE;
	gint row, status, nobjs, nobjs_purged = 0;
	guint index;
	const gchar *error_msg;
	GtkWidget *toplevel;
	GList *glist, *obj_list = NULL;
	GtkCList *clist;
	edv_recycled_object_struct *obj = NULL;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	toplevel = recbin->toplevel;
	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	    return;

#define CLEANUP_RETURN	{				\
 if(obj_list != NULL) {					\
  g_list_foreach(					\
   obj_list, (GFunc)EDVRecycledObjectDelete, NULL	\
  );							\
  g_list_free(obj_list);				\
 }							\
							\
 return;						\
}

	/* Get a list of selected recycled objects indices */
	glist = clist->selection;
	while(glist != NULL)
	{
	    row = (gint)glist->data;
	    obj = EDV_RECYCLED_OBJECT(
		gtk_clist_get_row_data(clist, row)
	    );
	    if(obj != NULL)
		obj_list = g_list_append(
		    obj_list,
		    EDVRecycledObjectCopy(obj)
		);

	    glist = g_list_next(glist);
	}

	/* No recycled objects to purge? */
	if(obj_list == NULL)
	{
	    CLEANUP_RETURN;
	}

	nobjs = g_list_length(obj_list);

	EDVRecBinSetBusy(recbin, TRUE);

	/* Confirm purge */
	status = EDVConfirmPurge(
	    core, toplevel,
	    (obj != NULL) ? obj->name : NULL,
	    nobjs
	);
	switch(status)
	{
	  case CDIALOG_RESPONSE_YES:
	  case CDIALOG_RESPONSE_YES_TO_ALL:
	    break;

	  default:
	    EDVRecBinSetBusy(recbin, FALSE);
	    CLEANUP_RETURN;
	    break;
	}

	/* Do not reference the last selected recycled object after
	 * this point
	 */
	obj = NULL;

	/* Iterate through list of recycled objects and purge each
	 * one
	 */
	status = 0;
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    obj = EDV_RECYCLED_OBJECT(glist->data);
	    if(obj == NULL)
		continue;

	    index = obj->index;

	    /* Purge this recycled object */
	    status = EDVRecBinPurge(
		core,
		index,
		toplevel,
		(nobjs > 0) ?
		    ((gfloat)nobjs_purged / (gfloat)nobjs) : -1.0f,
		TRUE, TRUE, &yes_to_all
	    );

	    /* Check for errors */
	    error_msg = EDVRecBinPurgeGetError(core);
	    if(!STRISEMPTY(error_msg))
	    {
		/* Report the error */
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Purge Recycled Object Error",
		    error_msg,
		    NULL,
		    toplevel
		);
	    }

	    /* Was the recycled object purged successfully? */
	    if(status == 0)
	    {
		nobjs_purged++;
		EDVRecycledObjectRemovedEmit(core, index);
	    }

	    /* User aborted? */
	    if(status == -4)
		break;
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the "completed" sound on success */
	if(status == 0) 
	    EDVPlaySoundCompleted(core);

	/* Update the status bar message */
	if(TRUE)
	{
	    gchar *msg;
	    if(status == -4)
		msg = STRDUP(
		    "Purge operation canceled"
		);
	    else if(nobjs_purged > 0)
		msg = g_strdup_printf(
		    "Purged %i %s",
		    nobjs_purged,
		    (nobjs_purged == 1) ? "recycled object" : "recycled objects"
		);
	    else
		msg = g_strdup_printf(
		    "Unable to purge %s",
		    (nobjs == 1) ? "recycled object" : "recycled objects"
		);
	    EDVStatusBarMessage(recbin->status_bar, msg, FALSE);
	    g_free(msg);
	}

	EDVRecBinSetBusy(recbin, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Purge all.
 */
void EDVRecBinOPPurgeAll(edv_recbin_struct *recbin)
{
	gboolean yes_to_all = FALSE;
	gint status, nobjs, nobjs_purged = 0;
	guint index;
	const gchar *error_msg, *index_path;
	GList *glist, *index_list = NULL;
	GtkWidget *toplevel;
	cfg_item_struct *cfg_list;
	edv_recbin_index_struct *rbi_ptr;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	toplevel = recbin->toplevel;
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	    return;

	/* Get the path to the recycled objects index file */
	index_path = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(index_path == NULL)
	    return;

#define CLEANUP_RETURN	{	\
 g_list_free(index_list);	\
				\
 return;			\
}

	/* Get the list of recycled objects */
	rbi_ptr = EDVRecBinIndexOpen(index_path);
	if(rbi_ptr != NULL)
	{
	    while(!EDVRecBinIndexNext(rbi_ptr))
	    {
		if(rbi_ptr->index != 0)
		    index_list = g_list_append(
			index_list,
			(gpointer)rbi_ptr->index
		    );
	    }
	    EDVRecBinIndexClose(rbi_ptr);
	}

	/* No recycled objects to purge? */
	if(index_list == NULL)
	{
	    CLEANUP_RETURN;
	}

	nobjs = g_list_length(index_list);

	EDVRecBinSetBusy(recbin, TRUE);

	/* Confirm purge */
	status = EDVConfirmPurge(
	    core, toplevel,
	    NULL,
	    nobjs
	);
	switch(status)
	{
	  case CDIALOG_RESPONSE_YES:
	  case CDIALOG_RESPONSE_YES_TO_ALL:
	    break;

	  default:
	    EDVRecBinSetBusy(recbin, FALSE);
	    CLEANUP_RETURN;
	    break;
	}

	/* Iterate through the list of recycled objects and purge
	 * each one
	 *
	 * Note that EDVRecBinPurgeAll() is not used here because
	 * it does not allow the reporting of each recycled object
	 * purged during the purge
	 */
	status = 0;
	for(glist = index_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    index = (guint)glist->data;

	    /* Purge this recycled object */
	    status = EDVRecBinPurge(
		core,
		index,			/* Recycled object to purge */
		toplevel,
		(nobjs > 0) ?
		    ((gfloat)nobjs_purged / (gfloat)nobjs) : -1.0f,
		TRUE,			/* Show progress */
		TRUE,			/* Interactive */
		&yes_to_all
	    );

	    /* Check for errors */
	    error_msg = EDVRecBinPurgeGetError(core);
	    if(!STRISEMPTY(error_msg))
	    {
		/* Report the error */
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Purge Recycled Object Error",
		    error_msg,
		    NULL,
		    toplevel
		);
	    }

	    /* Was the recycled object purged successfully? */
	    if(status == 0)
	    {
		/* Notify about this recycled object being purged */
		EDVRecycledObjectRemovedEmit(core, index);
		nobjs_purged++;
	    }

	    /* User aborted? */
	    if(status == -4)
		break;
	}

	/* If no errors occured or no used abort then call
	 * EDVRecBinPurgeAll() to remove any stray recycled object
	 * files and the recycled objects index file
	 */
	if(status == 0)
	{
	    /* Remove any stray recycled object files and the recycled
	     * objects index file
	     */
	    status = EDVRecBinPurgeAll(
		core,
		toplevel,
		FALSE,			/* Do not show progress */
		TRUE,			/* Interactive */
		&yes_to_all
	    );

	    /* Check for errors */
	    error_msg = EDVRecBinPurgeGetError(core);
	    if(!STRISEMPTY(error_msg))
	    {
		/* Report the error */
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Purge Recycled Object Error",
		    error_msg,
		    NULL,
		    toplevel
		);
	    }
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the "completed" sound on success */
	if(status == 0)
	    EDVPlaySoundCompleted(core);

	/* Update the status bar message */
	if(TRUE)
	{
	    gchar *msg;
	    if(status == -4)
		msg = STRDUP(
		    "Purge operation canceled"
		);
	    else if(nobjs_purged > 0)
		msg = g_strdup_printf(
		    "Purged %i %s",
		    nobjs_purged,
		    (nobjs_purged == 1) ? "recycled object" : "recycled objects"
		);
	    else
		msg = g_strdup_printf(
		    "Unable to purge %s",
		    (nobjs == 1) ? "recycled object" : "recycled objects"
		);
	    EDVStatusBarMessage(recbin->status_bar, msg, FALSE);
	    g_free(msg);
	}

	EDVRecBinSetBusy(recbin, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Rename.
 */
void EDVRecBinOPRename(edv_recbin_struct *recbin)
{
	gint row;
	GtkWidget *toplevel;
	GtkCList *clist;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	toplevel = recbin->toplevel;
	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, recbin->toplevel))
	    return;

	/* Get the last selected row */
	row = EDVCListGetSelectedLast(clist, NULL);
	if(row > -1)
	{
	    /* Map the rename prompt */
	    EDVRecBinContentsPromptRename(
		recbin,
		row, -1
	    );
	}
}

/*
 *	Change Permissions.
 */
void EDVRecBinOPChMod(edv_recbin_struct *recbin)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	EDVRecBinSetBusy(recbin, TRUE);

	toplevel = recbin->toplevel;
	core = recbin->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVRecBinSetBusy(recbin, FALSE);
	    return;
	}

	/* Get the object operations dialog and create it as needed */
	d = EDVGetObjOpDlg(core);
	if(d == NULL)
	{
	    EDVRecBinSetBusy(recbin, FALSE);
	    return;
	}

	/* Sync data to ensure that current values to operate on */
	EDVRecBinSyncData(recbin);

	/* Get the selected objects list */
	sel_objs_list = EDVRecBinGetSelectedObjectsList(recbin, FALSE);
	if(sel_objs_list == NULL)
	{
	    EDVRecBinSetBusy(recbin, FALSE);
	    return;
	}

	/* Map the object operations dialog to change permissions */
	EDVObjOpDlgMapValues(
	    d,
	    EDV_OBJ_OP_DLG_OP_CHMOD,
	    EDV_LOCATION_TYPE_RECBIN,
	    sel_objs_list,
	    NULL,				/* No srouce directory */
	    toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	EDVRecBinSetBusy(recbin, FALSE);
}

/*
 *	Change Ownership.
 */
void EDVRecBinOPChOwn(edv_recbin_struct *recbin)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	EDVRecBinSetBusy(recbin, TRUE);

	toplevel = recbin->toplevel;
	core = recbin->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVRecBinSetBusy(recbin, FALSE);
	    return;
	}

	/* Get the object operations dialog and create it as needed */
	d = EDVGetObjOpDlg(core);
	if(d == NULL)
	{
	    EDVRecBinSetBusy(recbin, FALSE);
	    return;
	}

	/* Sync data to ensure that current values to operate on */
	EDVRecBinSyncData(recbin);

	/* Get the selected objects list */
	sel_objs_list = EDVRecBinGetSelectedObjectsList(recbin, FALSE);
	if(sel_objs_list == NULL)
	{
	    EDVRecBinSetBusy(recbin, FALSE);
	    return;
	}

	/* Map the object operations dialog to change ownership */
	EDVObjOpDlgMapValues(
	    d,
	    EDV_OBJ_OP_DLG_OP_CHOWN,
	    EDV_LOCATION_TYPE_RECBIN,
	    sel_objs_list,
	    NULL,				/* No srouce directory */
	    toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	EDVRecBinSetBusy(recbin, FALSE);
}

/*
 *	Change Time Stamps.
 */
void EDVRecBinOPChTime(edv_recbin_struct *recbin)
{
	GList *sel_objs_list;
	GtkWidget *toplevel;
	edv_obj_op_dlg_struct *d;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	EDVRecBinSetBusy(recbin, TRUE);

	toplevel = recbin->toplevel;
	core = recbin->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVRecBinSetBusy(recbin, FALSE);
	    return;
	}

	/* Get the object operations dialog and create it as needed */
	d = EDVGetObjOpDlg(core);
	if(d == NULL)
	{
	    EDVRecBinSetBusy(recbin, FALSE);
	    return;
	}

	/* Sync data to ensure that current values to operate on */
	EDVRecBinSyncData(recbin);

	/* Get the selected objects list */
	sel_objs_list = EDVRecBinGetSelectedObjectsList(recbin, FALSE);
	if(sel_objs_list == NULL)
	{
	    EDVRecBinSetBusy(recbin, FALSE);
	    return;
	}

	/* Map the object operations dialog to change time stamps */
	EDVObjOpDlgMapValues(
	    d,
	    EDV_OBJ_OP_DLG_OP_CHTIME,
	    EDV_LOCATION_TYPE_RECBIN,
	    sel_objs_list,
	    NULL,				/* No srouce directory */
	    toplevel
	);

	/* Delete the selected objects list */
	g_list_free(sel_objs_list);

	EDVRecBinSetBusy(recbin, FALSE);
}

/*
 *	Properties.
 */
void EDVRecBinOPProperties(edv_recbin_struct *recbin)
{
	GtkWidget *toplevel;
	edv_recycled_object_struct *obj = NULL;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	EDVRecBinSetBusy(recbin, TRUE);

	toplevel = recbin->toplevel;
	core = recbin->core;

	EDVRecBinSyncData(recbin);

	/* Get the selected object */
	if(recbin->contents_clist_selected_row > -1)
	{
	    GtkCList *clist = GTK_CLIST(recbin->contents_clist);
	    GList *glist = clist->selection_end;
	    if(glist != NULL)
		obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(
		    clist,
		    (gint)glist->data
		));
	}

	/* No selected object? */
	if(obj == NULL)
	{
	    EDVRecBinSetBusy(recbin, FALSE);
	    return;
	}

	/* Create a new properties dialog displaying the object */
	EDVNewPropertiesDialogRecycledPage(
	    core,
	    obj,
	    NULL,				/* Default page */
	    toplevel
	);

	EDVRecBinSetBusy(recbin, FALSE);
}


/*
 *	Select all.
 */
void EDVRecBinOPSelectAll(edv_recbin_struct *recbin)
{
	GtkCList *clist;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	EDVRecBinSetBusy(recbin, TRUE);

	/* Select all rows on clist */
	gtk_clist_freeze(clist);
	gtk_clist_select_all(clist);
	gtk_clist_thaw(clist);

	/* Assume highest row index as the last selected row */
	recbin->contents_clist_selected_row = clist->rows - 1;

	EDVStatusBarMessage(
	    recbin->status_bar, "All objects selected", FALSE
	);

	EDVRecBinSetBusy(recbin, FALSE);
}

/*
 *	Unselect all.
 */
void EDVRecBinOPUnselectAll(edv_recbin_struct *recbin)
{
	GtkCList *clist;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	EDVRecBinSetBusy(recbin, TRUE);

	/* Unselect all rows on clist */
	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);

	/* Mark contents clist's row as unselected */
	recbin->contents_clist_selected_row = -1;

	EDVStatusBarMessage(
	    recbin->status_bar, "All objects unselected", FALSE
	);

	EDVRecBinSetBusy(recbin, FALSE);
}

/*
 *	Invert Selection.
 */
void EDVRecBinOPInvertSelection(edv_recbin_struct *recbin)
{
	gint row, total_rows;
	GList *glist, *selection;
	GtkCList *clist;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;

	EDVRecBinSetBusy(recbin, TRUE);
	gtk_clist_freeze(clist);

	/* Get copy of selected rows list from clist */
	selection = (clist->selection != NULL) ?
	    g_list_copy(clist->selection) : NULL;

	for(row = 0, total_rows = clist->rows;
	    row < total_rows;
	    row++
	)
	{
	    for(glist = selection;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		if(row == (gint)glist->data)
		{
		    gtk_clist_unselect_row(clist, row, 0);
		    break;
		}
	    }
	    /* Row not selected? */
	    if(glist == NULL)
		gtk_clist_select_row(clist, row, 0);
	}

	g_list_free(selection);

	gtk_clist_thaw(clist);
	EDVStatusBarMessage(
	    recbin->status_bar, "Selection inverted", FALSE
	);
	EDVRecBinSetBusy(recbin, FALSE);
}


/*
 *	Refresh.
 */
void EDVRecBinOPRefresh(edv_recbin_struct *recbin)
{
	GtkWidget *toplevel;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	edv_status_bar_struct *sb;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	toplevel = recbin->toplevel;
	core = recbin->core;
	cfg_list = core->cfg_list;
	sb = recbin->status_bar;

	EDVRecBinSetBusy(recbin, TRUE);
	GUIBlockInput(toplevel, TRUE);

	/* Update the Contents GtkCList */
	clist = GTK_CLIST(recbin->contents_clist);
	if(clist != NULL)
	{
	    /* Record the last scroll position */
	    const gfloat	last_x = GTK_ADJUSTMENT_GET_VALUE(clist->hadjustment),
				last_y = GTK_ADJUSTMENT_GET_VALUE(clist->vadjustment);

	    gtk_clist_freeze(clist);

	    /* Reget the listing */
	    EDVRecBinContentsGetListing(
		recbin,
		EDV_GET_B(EDV_CFG_PARM_LISTS_ANIMATE_UPDATES)
	    );

	    gtk_clist_thaw(clist);

	    /* Scroll back to the original position */
	    EDVScrollCListToPosition(clist, last_x, last_y);
	}

	EDVRecBinUpdateMenus(recbin);
	EDVStatusBarMessage(sb, "Refreshed contents listing", FALSE);

	GUIBlockInput(toplevel, FALSE);
	EDVRecBinSetBusy(recbin, FALSE);
}

/*
 *	Refresh All.
 */
void EDVRecBinOPRefreshAll(edv_recbin_struct *recbin)
{
	const gchar *index_path;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Get the path to the recycled objects index file */
	index_path = CFGItemListGetValueS(
	    cfg_list, EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX
	);

	/* Update the number of recycled objects on the core */
	core->last_recbin_items = EDVRecBinIndexGetTotal(
	    index_path
	);

	/* Refresh the Recycle Bin Desktop Icon */
	EDVRecBinDeskIconUpdate(core->recbin_deskicon);

	/* Refresh the Recycle Bin */
	EDVRecBinOPRefresh(recbin);
}


/*
 *	Sets the contents list filter.
 */
void EDVRecBinContentsFilter(edv_recbin_struct *recbin)
{
	gchar **strv;
	gint strc;
	GtkWidget *toplevel;

	if((recbin == NULL) || PDialogIsQuery())
	    return;

	toplevel = recbin->toplevel;
	EDVRecBinSetBusy(recbin, TRUE);

	PDialogDeleteAllPrompts();
	PDialogSetTransientFor(toplevel);
	PDialogAddPrompt(NULL, "Filter:", recbin->contents_list_filter);
	PDialogSetSize(320, -1);
	strv = PDialogGetResponseIconData(
	    "Set Filter",
	    NULL, NULL,
	    (guint8 **)icon_wildcards_32x32_xpm,
	    "Set", "Cancel",
	    PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
	    PDIALOG_BTNFLAG_SUBMIT,
	    &strc
	);
	PDialogSetTransientFor(NULL);
	if((strv != NULL) && (strc > 0))  
	{                                 
	    if(strc > 0)
	    {
		g_free(recbin->contents_list_filter);
		recbin->contents_list_filter = STRDUP(strv[0]);
	    }

	    EDVRecBinOPRefresh(recbin);
	}

	PDialogDeleteAllPrompts();

	EDVRecBinSetBusy(recbin, FALSE);
}

/*
 *      MIME Types.
 */
void EDVRecBinMIMETypes(edv_recbin_struct *recbin)
{
	gchar *type_str = NULL;
	gint i;
	GtkWidget *toplevel;
	GtkCList *clist;
	edv_core_struct *core;
	if(recbin == NULL)
	    return;

	toplevel = recbin->toplevel;
	clist = (GtkCList *)recbin->contents_clist;
	core = recbin->core;
	if((clist == NULL) || (core == NULL))
	    return;

	i = EDVCListGetSelectedLast(clist, NULL);
	if(i > -1)
	{
	    edv_recycled_object_struct *obj = EDV_RECYCLED_OBJECT(
		gtk_clist_get_row_data(clist, i)
	    );
	    if(obj != NULL)
		EDVMatchObjectTypeString(
		    core->mimetype, core->total_mimetypes,
		    obj->type,
		    obj->permissions,
		    obj->name,
		    &type_str
	    );
	}

	EDVMapMIMETypesListWin(core, type_str, toplevel);
}
