#include <sys/time.h>
#include <gtk/gtk.h>
#if defined(_WIN32)
# include <gdk/gdkprivate.h>
# include <gdk/gdkwin32.h>
#else
# include <gdk/gdkx.h>
# include <unistd.h>
#endif

#include "guiutils.h"
#include "guirgbimg.h"

#include "splash.h"


/* Utils */
static gulong SplashCurrentMS(void);

/* Callbacks */
static gint SplashObjectExposeCB(gpointer data);
static gint SplashObjectConfigureCB(gpointer data);
static gint SplashObjectShowCB(gpointer data);
static gint SplashObjectMapCB(gpointer data);
static gint SplashObjectRealizeCB(gpointer data);
static gint SplashObjectDrawCB(gpointer data);
static gint SplashDrawIdleCB(gpointer data);

void SplashDraw(splash_struct *splash);
void SplashQueueDraw(splash_struct *splash);

static GdkPixmap *SplashCreateNoSuchFilePixmap(
	const gchar *path,
	GdkWindow *window,
	GtkStyle *style,
	GdkBitmap **mask_rtn
);

/* Splash */
splash_struct *SplashNew(void);
void SplashSetXPMFile(
	splash_struct *splash,
	const gchar *xpm_file
);
void SplashSetXPMData(
	splash_struct *splash,
	const guint8 **xpm_data
);
void SplashSetMessageFont(
	splash_struct *splash,
	GdkFont *font
);
void SplashSetMessageColor(
	splash_struct *splash,
	GdkColor *fg_color,
	GdkColor *bg_color
);
void SplashSetMessageJustification(
	splash_struct *splash,
	const GtkJustification justify
);
void SplashSetMessagePosition(
	splash_struct *splash,
	const GtkPositionType position
);
void SplashUpdateMessage(
	splash_struct *splash,
	const gfloat v,
	const gchar *msg
);
void SplashUpdate(splash_struct *splash);
void SplashMap(
	splash_struct *splash,
	const splash_effects effects,
	const gulong duration_ms,
	const GtkWindowPosition position
);
void SplashUnmap(
	splash_struct *splash,
	const splash_effects effects,
	const gulong duration_ms
);
void SplashClearBuffers(splash_struct *splash);
void SplashDelete(splash_struct *splash);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


static gulong SplashCurrentMS(void)
{
	struct timeval tv[1];

	if(gettimeofday(tv, NULL) < 0)
	    return(0l);

	return((tv->tv_sec % 86400l) * 1000l + tv->tv_usec / 1000l);
}


/*
 *	Splash object "expose_event" signal callback.
 */
static gint SplashObjectExposeCB(gpointer data)
{
	splash_struct *splash = SPLASH(data);
	if(splash == NULL)
	    return(FALSE);

	if(splash->freeze_count > 0)
	    return(FALSE);

	SplashDraw(splash);

	return(TRUE);
}

/*
 *	Splash object "configure_event" signal callback.
 */
static gint SplashObjectConfigureCB(gpointer data)
{
	splash_struct *splash = SPLASH(data);
	if(splash == NULL)
	    return(FALSE);

	if(splash->freeze_count > 0)
	    return(FALSE);

/*	SplashDraw(splash); */

	return(TRUE);
}

/*
 *	Splash object "show" signal callback.
 */
static gint SplashObjectShowCB(gpointer data)
{
	splash_struct *splash = SPLASH(data);
	if(splash == NULL)
	    return(FALSE);

	if(splash->freeze_count > 0)
	    return(FALSE);

/*	SplashDraw(splash); */

	return(TRUE);
}

/*
 *	Splash object "map" signal callback.
 */
static gint SplashObjectMapCB(gpointer data)
{
	splash_struct *splash = SPLASH(data);
	if(splash == NULL)
	    return(FALSE);

	if(splash->freeze_count > 0)
	    return(FALSE);

/*	SplashDraw(splash); */

	return(TRUE);
}

/*
 *	Splash object "realize" signal callback.
 */
static gint SplashObjectRealizeCB(gpointer data)
{
	GdkColormap *colormap;
	GdkWindow *window;
	GdkGC *gc;
	GtkStyle *style;
	GtkWidget *w;
	splash_struct *splash = SPLASH(data);
	if(splash == NULL)
	    return(FALSE);

	if(splash->freeze_count > 0)
	    return(FALSE);

	/* Get the splash's realized GdkWindow */
	w = splash->toplevel;
	window = w->window;
	colormap = gtk_widget_get_colormap(w);
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
	    return(FALSE);

	/* Create the GC as needed */
	if(splash->gc == NULL)
	    splash->gc = gdk_gc_new(window);
	gc = splash->gc;

	/* If the message font was not set then use the style's font */
	if(splash->message_font == NULL)
	{
	    GdkFont *font = style->font;
	    splash->message_font = font;
	    if(font != NULL)
		gdk_font_ref(font);
	}

	/* Allocate the colors as needed */
	if(colormap != NULL)
	{
#define ALLOC_COLOR(_c_)	{	\
 if((_c_) != NULL) {			\
  gdk_colormap_alloc_color(		\
   colormap, (_c_), TRUE, TRUE		\
  );					\
 }					\
}

	    if(splash->message_fg_color == NULL)
	    {
		GdkColor color = {0l, 0xffff, 0xffff, 0xffff};
		splash->message_fg_color = g_memdup(&color, sizeof(GdkColor));
	    }
	    ALLOC_COLOR(splash->message_fg_color);

	    if(splash->message_bg_color == NULL)
	    {
		GdkColor color = {0l, 0x0000, 0x0000, 0x0000};
		splash->message_bg_color = g_memdup(&color, sizeof(GdkColor));
	    }
	    ALLOC_COLOR(splash->message_bg_color);

#undef ALLOC_COLOR
	}

	/* Set the background GdkPixmap for the GdkWindow */
	if(splash->bg_rgba != NULL)
	{
	    const gint	width = splash->bg_rgba_width,
			height = splash->bg_rgba_height;
	    GdkPixmap *pixmap = gdk_pixmap_new(window, width, height, -1);
	    if(pixmap != NULL)
	    {
		gdk_draw_rgb_32_image(
		    (GdkDrawable *)pixmap,
		    gc,
		    0, 0,
		    width, height,
		    GDK_RGB_DITHER_NONE,
		    splash->bg_rgba,
		    splash->bg_rgba_bpl
		);
		gdk_window_set_back_pixmap(window, pixmap, FALSE);
		gdk_pixmap_unref(pixmap);
	    }
	}
	else if(splash->rgba != NULL)
	{
	    const gint	width = splash->rgba_width,
			height = splash->rgba_height;
	    GdkPixmap *pixmap = gdk_pixmap_new(window, width, height, -1);
	    if(pixmap != NULL)
	    {
		gdk_draw_rgb_32_image(
		    (GdkDrawable *)pixmap,
		    gc,
		    0, 0,
		    width, height,
		    GDK_RGB_DITHER_NONE,
		    splash->rgba,
		    splash->rgba_bpl
		);
		gdk_window_set_back_pixmap(window, pixmap, FALSE);
		gdk_pixmap_unref(pixmap);
	    }
	}

	return(TRUE);
}

/*
 *	Splash object "draw" signal callback.
 */
static gint SplashObjectDrawCB(gpointer data)
{
	splash_struct *splash = SPLASH(data);
	if(splash == NULL)
	    return(FALSE);

	if(splash->freeze_count > 0)
	    return(FALSE);

	SplashDraw(splash);

	return(TRUE);
}

/*
 *	Draw idle callback.
 */
static gint SplashDrawIdleCB(gpointer data)
{
	splash_struct *splash = SPLASH(data);
	if(splash == NULL)
	    return(FALSE);

	splash->draw_idle_id = 0;

	SplashDraw(splash);

	return(FALSE);
}


/*
 *	Draws the background image on to the splash toplevel GtkWindow.
 */
void SplashDraw(splash_struct *splash)
{
	const gint border_major = 5;
	gint width, height;
	GdkDrawable *drawable;
	GdkGC *gc;
	GtkStateType state;
	GtkWidget *w;
	GtkStyle *style;

	if(splash == NULL)
	    return;

	w = splash->toplevel;
	gc = splash->gc;
	drawable = (GdkDrawable *)w->window;
	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	if((drawable == NULL) || (gc == NULL) || (style == NULL))
	    return;

	gdk_window_get_size((GdkWindow *)drawable, &width, &height);

	/* Mapped? (not unmapped or in the mapping/unmapping stage */
	if(splash->map_state)
	{
	    /* Draw the splash image */
	    if(splash->rgba != NULL)
		gdk_draw_rgb_32_image(
		    drawable,
		    gc,
		    0, 0,
		    splash->rgba_width, splash->rgba_height,
		    GDK_RGB_DITHER_NONE,
		    splash->rgba,
		    splash->rgba_bpl
		);
	}
	else
	{
	    /* Draw the background image */
	    if(splash->bg_rgba != NULL)
		gdk_draw_rgb_32_image(
		    drawable,
		    gc,
		    0, 0,
		    splash->bg_rgba_width, splash->bg_rgba_height,
		    GDK_RGB_DITHER_NONE,
		    splash->bg_rgba,
		    splash->bg_rgba_bpl
		);
	    else if(splash->rgba != NULL)
		gdk_draw_rgb_32_image(
		    drawable,
		    gc,
		    0, 0,
		    splash->rgba_width, splash->rgba_height,
		    GDK_RGB_DITHER_NONE,
		    splash->rgba,
		    splash->rgba_bpl
		);
	}

	/* Draw the message */
	if((splash->message != NULL) && (splash->message_font != NULL))
	{
	    gint x = 0, y = 0;
	    const gchar *msg = splash->message;
	    GdkFont *font = splash->message_font;
	    const gint font_height = font->ascent + font->descent;
	    GdkTextBounds *b = &splash->message_bounds;

	    gdk_gc_set_font(gc, font);

	    switch(splash->message_position)
	    {
	      case GTK_POS_TOP:
		switch(splash->message_justify)
		{
		  case GTK_JUSTIFY_LEFT:
		    x = (2 * border_major) - b->lbearing;
		    y = (2 * border_major) + font->ascent;
		    break;
		  case GTK_JUSTIFY_RIGHT:
		    x = width - b->width - (2 * border_major) - b->lbearing;
		    y = (2 * border_major) + font->ascent;
		    break;
		  case GTK_JUSTIFY_CENTER:
		  case GTK_JUSTIFY_FILL:
		    x = (width - b->width) / 2;
		    y = (2 * border_major) + font->ascent;
		    break;
		}
		break;

	      case GTK_POS_BOTTOM:
		switch(splash->message_justify)
		{
		  case GTK_JUSTIFY_LEFT:
		    x = (2 * border_major) - b->lbearing;
		    y = height - font_height - (2 * border_major) + font->ascent;
		    break;
		  case GTK_JUSTIFY_RIGHT:
		    x = width - b->width - (2 * border_major) - b->lbearing;
		    y = height - font_height - (2 * border_major) + font->ascent;
		    break;
		  case GTK_JUSTIFY_CENTER:
		  case GTK_JUSTIFY_FILL:
		    x = (width - b->width) / 2;
		    y = height - font_height - (2 * border_major) + font->ascent;
		    break;
		}
		break;

	      case GTK_POS_LEFT:
	      case GTK_POS_RIGHT:
		switch(splash->message_justify)
		{
		  case GTK_JUSTIFY_LEFT:
		    x = (2 * border_major) - b->lbearing;
		    y = ((height - font_height) / 2);
		    break;
		  case GTK_JUSTIFY_RIGHT:
		    x = width - b->width - (2 * border_major) - b->lbearing;
		    y = ((height - font_height) / 2);
		    break;
		  case GTK_JUSTIFY_CENTER:
		  case GTK_JUSTIFY_FILL:
		    x = (width - b->width) / 2;
		    y = ((height - font_height) / 2);
		    break;
		}
		break;
	    }

#define DRAW_STRING_ITERATE(_x_,_y_)	{	\
 gdk_draw_string(				\
  drawable, font, gc,				\
  (_x_), (_y_),					\
  msg						\
 );						\
}
	    /* Draw the message background */
	    if(splash->message_bg_color != NULL)
	    {
		gdk_gc_set_foreground(
		    gc,
		    splash->message_bg_color
		);
		DRAW_STRING_ITERATE(x - 1, y);
		DRAW_STRING_ITERATE(x, y - 1);
		DRAW_STRING_ITERATE(x + 1, y);
		DRAW_STRING_ITERATE(x, y + 1);
	    }

	    /* Draw the message foreground */
	    if(splash->message_fg_color != NULL)
		gdk_gc_set_foreground(
		    gc,
		    splash->message_fg_color
		);
	    DRAW_STRING_ITERATE(x, y);

#undef DRAW_STRING_ITERATE
	}
}

/*
 *	Queues the splash to draw.
 */
void SplashQueueDraw(splash_struct *splash)
{
	if(splash == NULL)
	    return;

	if(splash->draw_idle_id != 0)
	    return;

	splash->draw_idle_id = gtk_idle_add_priority(
	    GTK_PRIORITY_REDRAW,
	    SplashDrawIdleCB, splash
	);
}

/*
 *	Creates a "no such file" GdkPixmap.
 */
static GdkPixmap *SplashCreateNoSuchFilePixmap(
	const gchar *path,
	GdkWindow *window,
	GtkStyle *style,
	GdkBitmap **mask_rtn
)
{
	const gint border_major = 5;
	gint x, y, width, height, font_height;
	gchar *s;
	GdkTextBounds b;
	GdkFont *font;
	GdkPixmap *pixmap;
	GdkDrawable *drawable;
	GtkStateType state;

	if(mask_rtn != NULL)
	    *mask_rtn = NULL;

	if((path == NULL) || (window == NULL) || (style == NULL))
	    return(NULL);

	font = style->font;
	if(font == NULL)
	    return(NULL);

	state = GTK_STATE_NORMAL;
	font_height = font->ascent + font->descent;
	gdk_string_bounds(font, path, &b);
	width = 25 + b.width + (border_major * 2);
	height = (font_height * 3) + (border_major * 2);

	pixmap = gdk_pixmap_new(window, width, height, -1);
	if(pixmap == NULL)
	    return(NULL);

	drawable = (GdkDrawable *)pixmap;

	gdk_draw_rectangle(
	    drawable, style->bg_gc[state], TRUE,
	    0, 0, width, height
	);

	gtk_draw_box(
	    style, drawable, state,
	    GTK_SHADOW_OUT,
	    0, 0, width, height
	);

	x = border_major;
	y = border_major;
	s = STRDUP("No such file:");
	gdk_string_bounds(font, s, &b);
	gdk_draw_string(
	    drawable, font, style->text_gc[state],
	    x - b.lbearing,
	    y + font->ascent,
	    s
	);
	g_free(s);

	y += font_height;
	x = border_major;
	/* Empty line */

	y += font_height;
	x = border_major + 25;
	gdk_string_bounds(font, path, &b);
	gdk_draw_string(
	    drawable, font, style->text_gc[state],
	    x - b.lbearing,
	    y + font->ascent,
	    path
	);

	return(pixmap);
}


/*
 *	Create a new splash.
 */
splash_struct *SplashNew(void)
{
	splash_struct *splash;

	splash = SPLASH(g_malloc0(sizeof(splash_struct)));
	if(splash == NULL)
	    return(NULL);

	splash->toplevel = NULL;
	splash->map_state = FALSE;
	splash->freeze_count = 0;
	splash->gc = NULL;
	splash->draw_idle_id = 0;

	splash->value = -1.0f;

	splash->message = NULL;
	splash->message_font = NULL;
	splash->message_fg_color = NULL;
	splash->message_bg_color = NULL;
	splash->message_justify = GTK_JUSTIFY_CENTER;
	splash->message_position = GTK_POS_BOTTOM;

	return(splash);
}

/*
 *	Set the XPM file.
 */
void SplashSetXPMFile(
	splash_struct *splash,
	const gchar *xpm_file
)
{
	if(splash == NULL)
	    return;

	g_free(splash->xpm_file);
	splash->xpm_file = STRDUP(xpm_file);

	splash->xpm_data = NULL;	/* Unset the XPM data */
}

/*
 *	Set the XPM data.
 */
void SplashSetXPMData(
	splash_struct *splash,
	const guint8 **xpm_data
)
{
	if(splash == NULL)
	    return;

	g_free(splash->xpm_file);	/* Unset the XPM file */
	splash->xpm_file = NULL;

	splash->xpm_data = xpm_data;
}

/*
 *	Sets the message font.
 */
void SplashSetMessageFont(
	splash_struct *splash,
	GdkFont *font
)
{
	if((splash == NULL) || (font == NULL))
	    return;

	GDK_FONT_UNREF(splash->message_font);
	gdk_font_ref(font);
	splash->message_font = font;

	/* Update the message's string bounds as needed */
	if(splash->message != NULL)
	    gdk_string_bounds(
		font,
		splash->message,
		&splash->message_bounds
	    );
}

/*
 *	Sets the message's color.
 *
 *	The fg_color specifies the message's foreground GdkColor.
 *	The GdkColor does not need to be allocated.
 *
 *	The bg_color specifies the message's background GdkColor,
 *	The GdkColor does not need to be allocated.
 */
void SplashSetMessageColor(
	splash_struct *splash,
	GdkColor *fg_color,
	GdkColor *bg_color
)
{
	GdkColormap *colormap;
	GtkWidget *w;

	if(splash == NULL)
	    return;

	w = splash->toplevel;
	colormap = (w != NULL) ? gtk_widget_get_colormap(w) : NULL;
	if(colormap != NULL)
	{
#define SET_COLOR(_cp_,_cv_)	{				\
 if((_cv_) != NULL) {						\
  if((_cp_) != NULL)						\
   gdk_colormap_free_colors(colormap, (_cp_), 1);		\
  else								\
   (_cp_) = (GdkColor *)g_malloc(sizeof(GdkColor));		\
  if((_cp_) != NULL) {						\
   memcpy((_cp_), (_cv_), sizeof(GdkColor));			\
   gdk_colormap_alloc_color(colormap, (_cp_), TRUE, TRUE);	\
  }								\
 }								\
}
	    SET_COLOR(splash->message_fg_color, fg_color);
	    SET_COLOR(splash->message_bg_color, bg_color);
#undef SET_COLOR
	}
	else
	{
#define SET_COLOR(_cp_,_cv_)	{				\
 g_free(_cp_);							\
 (_cp_) = ((_cv_) != NULL) ?					\
  g_memdup((_cv_), sizeof(GdkColor)) : NULL;			\
}
	    SET_COLOR(splash->message_fg_color, fg_color);
	    SET_COLOR(splash->message_bg_color, bg_color);
#undef SET_COLOR
	}
}

/*
 *	Sets the message justification.
 */
void SplashSetMessageJustification(
	splash_struct *splash,
	const GtkJustification justify
)
{
	if(splash == NULL)
	    return;

	splash->message_justify = justify;
}

/*
 *	Sets the message position.
 *
 *	The position specifies the message position, if position is
 *	either GTK_POS_LEFT or GTK_POS_RIGHT then the position is
 *	center vertically. SplashSetMessageJustification() controls
 *	whether the position will actually be left or right.
 */
void SplashSetMessagePosition(
	splash_struct *splash,
	const GtkPositionType position
)
{
	if(splash == NULL)
	    return;

	splash->message_position = position;
}

/*
 *	Updates the splash by setting a new value and message.
 *
 *	The splash will be redrawn as needed.
 */
void SplashUpdateMessage(
	splash_struct *splash,
	const gfloat v,
	const gchar *msg
)
{
	gboolean value_changed;

	if(splash == NULL)
	    return;

	value_changed = FALSE;

	if(v != splash->value)
	{
	    splash->value = v;
	    value_changed = TRUE;
	}

	if(splash->message != msg)
	{
	    g_free(splash->message);
	    splash->message = STRDUP(msg);

	    /* Update the message's string bounds if a new message
	     * is set
	     */
	    if((msg != NULL) && (splash->message_font != NULL))
		gdk_string_bounds(
		    splash->message_font,
		    msg,
		    &splash->message_bounds
		);

	    value_changed = TRUE;
	}

	if(splash->map_state && value_changed)
	{
	    SplashDraw(splash);
	    gdk_flush();
	}
}

/*
 *	Updates and queues the splash to draw.
 */
void SplashUpdate(splash_struct *splash)
{
	if(splash == NULL)
	    return;

	if(splash->map_state)
	    SplashQueueDraw(splash);
}

/*
 *	Maps the splash and performs any mapping effects.
 */
void SplashMap(
	splash_struct *splash,
	const splash_effects effects,
	const gulong duration_ms,
	const GtkWindowPosition position
)
{
	const gint bpp = 4 * sizeof(guint8);
	gint	x, y, width, height,
		root_width, root_height,
		bpl;
	guint8 *frame_rgba;
	GdkModifierType mask;
	GdkWindow *root = (GdkWindow *)GDK_ROOT_PARENT();
	GdkDrawable *drawable;
	GdkGC *gc;
	GtkWidget *w;
	GtkStyle *style;

	if((splash == NULL) || (root == NULL))
	    return;

	/* Get the size of the root window */
	gdk_window_get_size(root, &root_width, &root_height);

	/* Clear all the image datas */
	SplashClearBuffers(splash);

	/* Get the splash image data
	 *
	 * Open from XPM file?
	 */
	if(splash->xpm_file != NULL)
	{
	    GdkBitmap *mask;
	    GdkPixmap *pixmap = gdk_pixmap_create_from_xpm(
		root,
		&mask,
		NULL,
		splash->xpm_file
	    );
	    if(pixmap == NULL)
	    {
		GtkWidget *w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_widget_realize(w);
		pixmap = SplashCreateNoSuchFilePixmap(
		    splash->xpm_file,
		    root,
		    gtk_widget_get_style(w),
		    &mask
		);
		gtk_widget_destroy(w);
	    }
	    if(pixmap != NULL)
	    {
		splash->rgba = gdk_get_rgba_image(
		    pixmap,
		    NULL,
		    &splash->rgba_width,
		    &splash->rgba_height,
		    &splash->rgba_bpl
		);
		GDK_PIXMAP_UNREF(pixmap);
		GDK_BITMAP_UNREF(mask);
	    }
	}
	/* Open from XPM data? */
	else if(splash->xpm_data != NULL)
	{
	    GdkBitmap *mask;
	    GdkPixmap *pixmap = gdk_pixmap_create_from_xpm_d(
		root,
		&mask,
		NULL,
		(gchar **)splash->xpm_data
	    );
	    if(pixmap != NULL)
	    {
		splash->rgba = gdk_get_rgba_image(
		    pixmap,
		    NULL,
		    &splash->rgba_width,
		    &splash->rgba_height,
		    &splash->rgba_bpl
		);
		GDK_PIXMAP_UNREF(pixmap);
		GDK_BITMAP_UNREF(mask);
	    }
	}

	width = splash->rgba_width;
	height = splash->rgba_height;
	bpl = splash->rgba_bpl;

	if((splash->rgba == NULL) || (width <= 0) || (height <= 0))
	    return;

	/* Calculate the position */
	switch(position)
	{
	  case GTK_WIN_POS_NONE:
	    x = 0;
	    y = 0;
	    break;
	  case GTK_WIN_POS_CENTER:
	    x = (root_width - width) / 2;
	    y = (root_height - height) / 2;
	    break;
	  case GTK_WIN_POS_MOUSE:
	    (void)gdk_window_get_pointer(root, &x, &y, &mask);
	    break;
	  case GTK_WIN_POS_CENTER_ALWAYS:
	    x = (root_width - width) / 2;
	    y = (root_height - height) / 2;
	    break;
	}

	/* Get/create the splash toplevel GtkWindow as needed */
	if(splash->toplevel != NULL)
	{
	    w = splash->toplevel;
	    x = w->allocation.x;
	    y = w->allocation.y;
	}
	else
	{
	    /* Create the toplevel GtkWindow */
	    gtk_widget_push_visual(gdk_rgb_get_visual());
	    gtk_widget_push_colormap(gdk_rgb_get_cmap());
	    splash->toplevel = w = gtk_window_new(GTK_WINDOW_POPUP);
	    gtk_widget_pop_visual();
	    gtk_widget_pop_colormap();
	    if(w == NULL)
		return;

	    gtk_widget_set_uposition(w, x, y);
	    gtk_widget_set_usize(w, width, height);
	    gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	    gtk_widget_set_app_paintable(w, TRUE);
	    gtk_signal_connect_object(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(SplashObjectExposeCB), (GtkObject *)splash
	    );
	    gtk_signal_connect_object(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(SplashObjectConfigureCB), (GtkObject *)splash
	    );
	    gtk_signal_connect_object(
		GTK_OBJECT(w), "show",
		GTK_SIGNAL_FUNC(SplashObjectShowCB), (GtkObject *)splash
	    );
	    gtk_signal_connect_object(
		GTK_OBJECT(w), "map",
		GTK_SIGNAL_FUNC(SplashObjectMapCB), (GtkObject *)splash
	    );
	    gtk_signal_connect_object(
		GTK_OBJECT(w), "realize",
		GTK_SIGNAL_FUNC(SplashObjectRealizeCB), (GtkObject *)splash
	    );
	    gtk_signal_connect_object(
		GTK_OBJECT(w), "draw",
		GTK_SIGNAL_FUNC(SplashObjectDrawCB), (GtkObject *)splash
	    );
	}
	if(w == NULL)
	    return;

	/* Get the background image data as needed */
	switch(effects)
	{
	  case SPLASH_EFFECTS_NONE:
	    break;
	  case SPLASH_EFFECTS_FADE_WHITE:
	    if((width > 0) && (height > 0))
	    {
		splash->bg_rgba_width = width;
		splash->bg_rgba_height = height;
		splash->bg_rgba_bpl = bpl;
		splash->bg_rgba = (guint8 *)g_malloc(bpl * height);
		if(splash->bg_rgba != NULL)
		{
		    const guint8 pixel[4] = {0xff, 0xff, 0xff, 0xff};
		    guint8 *line_ptr;
		    gint x, y;

		    for(y = 0; y < height; y++)
		    {
			line_ptr = splash->bg_rgba + (bpl * y);
			for(x = 0; x < width; x++)
			{
			    memcpy(line_ptr, pixel, bpp);
			    line_ptr += bpp;
			}
		    }
		}
	    }
	    break;
	  case SPLASH_EFFECTS_FADE_BLACK:
	    if((width > 0) && (height > 0))
	    {
		splash->bg_rgba_width = width;
		splash->bg_rgba_height = height;
		splash->bg_rgba_bpl = bpl;
		splash->bg_rgba = (guint8 *)g_malloc(bpl * height);
		if(splash->bg_rgba != NULL)
		{
		    const guint8 pixel[4] = {0x00, 0x00, 0x00, 0xff};
		    guint8 *line_ptr;
		    gint x, y;

		    for(y = 0; y < height; y++)
		    {
			line_ptr = splash->bg_rgba + (bpl * y);
			for(x = 0; x < width; x++)
			{
			    memcpy(line_ptr, pixel, bpp);
			    line_ptr += bpp;
			}
		    }
		}
	    }
	    break;
	  case SPLASH_EFFECTS_FADE_BACKGROUND:
	    if((width > 0) && (height > 0))
	    {
		GdkRectangle rect;
		rect.x = x;
		rect.y = y;
		rect.width = width;
		rect.height = height;
		splash->bg_rgba = gdk_get_rgba_image(
		    root,
		    &rect,
		    &splash->bg_rgba_width,
		    &splash->bg_rgba_height,
		    &splash->bg_rgba_bpl
		);
	    }
	    break;
	}

	/* Allocate the rendering image buffer */
	frame_rgba = (guint8 *)g_malloc(bpl * height);
	if(frame_rgba == NULL)
	    return;

	/* Map and realize the splash */
	gtk_widget_show_raise(w);
	splash->map_state = TRUE;

	drawable = (GdkDrawable *)w->window;
	gc = splash->gc;
	style = gtk_widget_get_style(w);
	if((drawable == NULL) || (gc == NULL) || (style == NULL))
	{
	    g_free(frame_rgba);
	    return;
	}

	/* This is required so that the mapping effect starts off
	 * without any skipped "frames" due to previous unprocessed
	 * GDK operations
	 */
	gdk_flush();

	/* Perform the mapping effect */
	switch(effects)
	{
	  case SPLASH_EFFECTS_NONE:
	    break;

	  case SPLASH_EFFECTS_FADE_WHITE:
	  case SPLASH_EFFECTS_FADE_BLACK:
	  case SPLASH_EFFECTS_FADE_BACKGROUND:
	    if(splash->bg_rgba != NULL)
	    {
		/* Fade the splash image in with the background image */
		const gulong	start_ms = SplashCurrentMS(),
				end_ms = start_ms + duration_ms;
		gulong cur_ms = start_ms;
		gint x, y;
		const guint8 *bg_line, *src_line;
		guint8 *tar_line;
		gfloat coeff, inv_coeff;

		splash->freeze_count++;	/* Prevent event handling */

		while(cur_ms <= end_ms)
		{
		    coeff = (gfloat)(cur_ms - start_ms) / (gfloat)duration_ms;
		    inv_coeff = 1.0f - coeff;

		    /* Blend the splash image on to the background image */
		    for(y = 0; y < height; y++)
		    {
			bg_line = splash->bg_rgba + (splash->bg_rgba_bpl * y);
			src_line = splash->rgba + (bpl * y);
			tar_line = frame_rgba + (bpl * y);

			for(x = 0; x < width; x++)
			{
			    tar_line[0] = (guint8)(
				(bg_line[0] * inv_coeff) +
				(src_line[0] * coeff)
			    );
			    tar_line[1] = (guint8)(
				(bg_line[1] * inv_coeff) +
				(src_line[1] * coeff)
			    );
			    tar_line[2] = (guint8)(
				(bg_line[2] * inv_coeff) +
				(src_line[2] * coeff)
			    );
			    tar_line[3] = 0xff;

			    bg_line += bpp;
			    src_line += bpp;
			    tar_line += bpp;
			}
		    }

		    /* Put the rendering image buffer on to the GtkDrawable */
		    gdk_draw_rgb_32_image(
			drawable,
			gc,
			0, 0, width, height,
			GDK_RGB_DITHER_NONE,
			frame_rgba,
			bpl
		    );

		    /* Update the current time */
		    cur_ms = SplashCurrentMS();
		}

		/* Do not draw the last frame, it will be drawn
		 * outside of this case
		 */

		splash->freeze_count--;
	    }
	    break;
	}

	/* Delete the rendering image buffer */
	g_free(frame_rgba);

	/* Draw the final foreground image */
	SplashDraw(splash);

	/* This is required to ensure that the final draw is completed
	 * after the mapping effect so that we do not see any partial
	 * mapping effect "frames" left over after this call
	 */
	gdk_flush();
}

/*
 *	Unmaps the splash and performs any mapping effects.
 */
void SplashUnmap(
	splash_struct *splash,
	const splash_effects effects,
	const gulong duration_ms
)
{
	const gint bpp = 4 * sizeof(guint8);
	gint width, height, bpl;
	guint8 *frame_rgba;
	GdkGC *gc;
	GtkWidget *w;

	if(splash == NULL)
	    return;

	w = splash->toplevel;
	if(w == NULL)
	    return;

	width = splash->rgba_width;
	height = splash->rgba_height;
	bpl = splash->rgba_bpl;

	gc = splash->gc;

	if((width <= 0) || (height <= 0) || (gc == NULL))
	{
	    gtk_widget_hide(w);
	    return;
	}

	/* Allocate the rendering image buffer */
	frame_rgba = (guint8 *)g_malloc(bpl * height);
	if(frame_rgba == NULL)
	{
	    gtk_widget_hide(w);
	    return;
	}

	/* This is required so that the unmapping effect starts off
	 * without any skipped "frames" due to previous unprocessed
	 * GDK operations
	 */
	gdk_flush();

	/* Perform the unmapping effect */
	switch(effects)
	{
	  case SPLASH_EFFECTS_NONE:
	    break;

	  case SPLASH_EFFECTS_FADE_WHITE:
	  case SPLASH_EFFECTS_FADE_BLACK:
	  case SPLASH_EFFECTS_FADE_BACKGROUND:
	    if(splash->bg_rgba != NULL)
	    {
		/* Fade the splash image in with the background image */
		const gulong	start_ms = SplashCurrentMS(),
				end_ms = start_ms + duration_ms;
		gulong cur_ms = start_ms;
		gint x, y;
		const guint8 *bg_line, *src_line;
		guint8 *tar_line;
		gfloat coeff, inv_coeff;
		GdkDrawable *drawable = (GdkDrawable *)w->window;

		splash->freeze_count++;	/* Prevent event handling */

		while(cur_ms <= end_ms)
		{
		    inv_coeff = (gfloat)(cur_ms - start_ms) / (gfloat)duration_ms;
		    coeff = 1.0f - inv_coeff;

		    /* Blend the splash image on to the background image */
		    for(y = 0; y < height; y++)
		    {
			bg_line = splash->bg_rgba + (splash->bg_rgba_bpl * y);
			src_line = splash->rgba + (bpl * y);
			tar_line = frame_rgba + (bpl * y);

			for(x = 0; x < width; x++)
			{
			    tar_line[0] = (guint8)(
				(bg_line[0] * inv_coeff) +
				(src_line[0] * coeff)
			    );
			    tar_line[1] = (guint8)(
				(bg_line[1] * inv_coeff) +
				(src_line[1] * coeff)
			    );
			    tar_line[2] = (guint8)(
				(bg_line[2] * inv_coeff) +
				(src_line[2] * coeff)
			    );
			    tar_line[3] = 0xff;

			    bg_line += bpp;
			    src_line += bpp;
			    tar_line += bpp;
			}
		    }

		    /* Put the rendering image buffer on to the GtkDrawable */
		    gdk_draw_rgb_32_image(
			drawable,
			gc,
			0, 0, width, height,
			GDK_RGB_DITHER_NONE,
			frame_rgba,
			bpl
		    );

		    /* Update the current time */
		    cur_ms = SplashCurrentMS();
		}

		/* Draw the last frame */
		gdk_draw_rgb_32_image(
		    drawable,
		    gc,
		    0, 0,
		    splash->bg_rgba_width, splash->bg_rgba_height,
		    GDK_RGB_DITHER_NONE,
		    splash->bg_rgba,
		    splash->bg_rgba_bpl
		);

		splash->freeze_count--;
	    }
	    break;
	}

	/* Delete the rendering image buffer */
	g_free(frame_rgba);

	/* This is required so that the unmapping effect finishes
	 * with the last frame before the splash is unmapped
	 */
	gdk_flush();

	/* Unmap the splash */
	gtk_widget_hide(splash->toplevel);
	splash->map_state = FALSE;

	/* Clear all the image datas */
	SplashClearBuffers(splash);

	/* This is required to ensure that the splash has been actually
	 * unmapped before the end of this call
	 */
	gdk_flush();
}

/*
 *	Deletes all the image buffers on the splash.
 */
void SplashClearBuffers(splash_struct *splash)
{
	if(splash == NULL)
	    return;

	/* Clear all the image datas */
	g_free(splash->rgba);
	splash->rgba = NULL;
	splash->rgba_width = 0;
	splash->rgba_height = 0;
	splash->rgba_bpl = 0;

	g_free(splash->bg_rgba);
	splash->bg_rgba = NULL;
	splash->bg_rgba_width = 0;
	splash->bg_rgba_height = 0;
	splash->bg_rgba_bpl = 0;
}

/*
 *	Deletes the splash.
 */
void SplashDelete(splash_struct *splash)
{
	GdkColormap *colormap;
	GtkWidget *w;

	if(splash == NULL)
	    return;

	if(splash->draw_idle_id != 0)
	{
	    gtk_timeout_remove(splash->draw_idle_id);
	    splash->draw_idle_id = 0;
	}

	splash->freeze_count++;

	/* Delete the colors */
	w = splash->toplevel;
	colormap = (w != NULL) ? gtk_widget_get_colormap(w) : NULL;
	if(colormap != NULL)
	{
#define DELETE_COLOR(_cp_)	{				\
 if((_cp_) != NULL) {						\
  gdk_colormap_free_colors(colormap, (_cp_), 1);		\
  g_free(_cp_);							\
  (_cp_) = NULL;						\
 }								\
}
	    DELETE_COLOR(splash->message_fg_color);
	    DELETE_COLOR(splash->message_bg_color);
#undef DELETE_COLOR
	}

	GTK_WIDGET_DESTROY(splash->toplevel);
	GDK_GC_UNREF(splash->gc);
	GDK_FONT_UNREF(splash->message_font);
	g_free(splash->xpm_file);
	g_free(splash->rgba);
	g_free(splash->bg_rgba);
	g_free(splash->message);

	splash->freeze_count--;

	g_free(splash);
}
