#include <stdlib.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include "../splash.h"


static void print_help(const gchar *prog_name);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


static void print_help(const gchar *prog_name)
{
	g_print(
"Usage: %s [splash_xpm_path] [options]\n",
	    prog_name
	);
	g_print(
"\n\
    The [splash_xpm_path] specifies the path to the splash XPM\n\
    image file.\n\
\n\
    The [options] can be any of the following:\n\
\n\
        --effects <effects>     Specifies the effects, <effects> can be:\n\
                                    none\n\
                                    fade_white\n\
                                    fade_black\n\
                                    fade_background\n\
        -e                      Same as --effects.\n\
        --map-int <ms>          Specifies the map and unmap intervals\n\
                                in milliseconds.\n\
        -i                      Same as --map-int.\n\
        --message <message>     Specifies the splash message.\n\
        -m                      Same as --message.\n\
        --delay <seconds>       Specifies the delay to keep the splash\n\
                                mapped before it is unmapped in seconds.\n\
        -d                      Same as --delay.\n\
\n\
        --help                  Prints this help screen and exits.\n\
\n"
	);
}


int main(int argc, char *argv[])
{
	const gchar *splash_xpm_path =
"/usr/share/endeavour2/images/endeavour_splash.xpm";
        gulong	delay = 3l,
		map_int_ms = SPLASH_DEF_MAP_DURATION_MS;
	gint i;
	const gchar *arg, *message = NULL;
	GdkFont *font;
	splash_struct *splash;
	splash_effects effects = SPLASH_EFFECTS_FADE_BACKGROUND;


	for(i = 1; i < argc; i++)
	{
	    arg = argv[i];
	    if(arg == NULL)
		continue;

	    if(!g_strcasecmp(arg, "--help") ||
	       !g_strcasecmp(arg, "-help") ||
	       !g_strcasecmp(arg, "--h") ||
	       !g_strcasecmp(arg, "-h") ||
	       !g_strcasecmp(arg, "-?")
	    )
	    {
		print_help(argv[0]);
		return(0);
	    }
	    else if(!g_strcasecmp(arg, "--version") ||
	            !g_strcasecmp(arg, "-version")
	    )
	    {
		return(0);
	    }
	    /* Effects */
	    else if(!g_strcasecmp(arg, "--effects") ||
	            !g_strcasecmp(arg, "-e")
	    )
	    {
		i++;
		if(i < argc)
		{
		    arg = argv[i];
		    if(arg == NULL)
			break;

		    if(!g_strcasecmp(arg, "none"))
			effects = SPLASH_EFFECTS_NONE;
		    else if(!g_strcasecmp(arg, "fade_white"))
			effects = SPLASH_EFFECTS_FADE_WHITE;
		    else if(!g_strcasecmp(arg, "fade_black"))
			effects = SPLASH_EFFECTS_FADE_BLACK;
		    else if(!g_strcasecmp(arg, "fade_background"))
			effects = SPLASH_EFFECTS_FADE_BACKGROUND;
		}
		else
		{
		    g_printerr(
"%s: Requires argument.\n",
			arg
		    );
		    return(2);
		}
	    }
	    /* Map Interval */
	    else if(!g_strcasecmp(arg, "--map-int") ||
	            !g_strcasecmp(arg, "-i")
	    )
	    {
		i++;
		if(i < argc)
		{
		    map_int_ms = ATOL(argv[i]);
		}
		else
		{
		    g_printerr(
"%s: Requires argument.\n",
			arg
		    );
		    return(2);
		}
	    }
	    /* Message */
	    else if(!g_strcasecmp(arg, "--message") ||
	            !g_strcasecmp(arg, "-m")
	    )
	    {
		i++;
		if(i < argc)
		{
		    message = argv[i];
		}
		else
		{
		    g_printerr(
"%s: Requires argument.\n",
			arg
		    );
		    return(2);
		}
	    }
	    /* Delay */
	    else if(!g_strcasecmp(arg, "--delay") ||
	            !g_strcasecmp(arg, "-d")
	    )
	    {
		i++;
		if(i < argc)
		{
		    delay = ATOL(argv[i]);
		}
		else
		{
		    g_printerr(
"%s: Requires argument.\n",
			arg
		    );
		    return(2);
		}
	    }
	    /* Splash XPM Path */
	    else if((*arg != '+') && (*arg != '-'))
	    {
		splash_xpm_path = arg;
	    }
	    else
	    {
		g_printerr(
"%s: Unsupported argument.\n",
		    arg
		);
		return(2);
	    }
	}


	/* Initialize GTK */
	gtk_init_check(&argc, &argv);

	/* Initialize the GDK RGB buffers needed by the splash */
	gdk_rgb_init();

	/* Create a new splash */
	splash = SplashNew();
	SplashSetXPMFile(
	    splash,
	    splash_xpm_path
	);
	font = gdk_font_load(
"-adobe-helvetica-bold-r-normal-*-24-*-*-*-p-*-iso8859-1"
	);
	if(font != NULL)
	{
	    SplashSetMessageFont(splash, font);
	    gdk_font_unref(font);
	}
	SplashSetMessageJustification(splash, GTK_JUSTIFY_CENTER);
	SplashSetMessagePosition(splash, GTK_POS_BOTTOM);

	/* Map the splash */
	SplashMap(
	    splash,
	    effects,
	    map_int_ms,
	    GTK_WIN_POS_CENTER
	);

	if(message != NULL)
	    SplashUpdateMessage(
		splash,
		-1.0f,
		message
	    );

	/* Wait for the delay */
	if(delay > 0l)
	    sleep(delay);

	SplashUpdate(splash);

	/* Unmap the splash */
	SplashUnmap(
	    splash,
	    effects,
	    map_int_ms
	);

	/* Delete the splash */
	SplashDelete(splash);

	/* Manage any pending GTK events */
	while(gtk_events_pending() > 0)
	    gtk_main_iteration();

	return(0);
}
