#!/usr/bin/env python2.4

import apt_pkg
import gobject
import pygtk
pygtk.require('2.0')
import gtk
import sys
import commands
import tempfile
import subprocess

import utils

from gettext import gettext as _
from UpdateManager.Common.SimpleGladeApp import SimpleGladeApp
from softwareproperties.gtk.DialogAptKey import DialogAddKey as apt_key
from aptsources.sourceslist import SourceEntry, SourcesList

from gapti.SubscriptionConfirmDialog import SubscriptionConfirmDialog

class GAptI:

    def __init__(self, datadir=None):  

        if (datadir == None):
            datadir = '/usr/share/gapti/'
        self.datadir = datadir

        # Continue processing on the main loop.
        gobject.timeout_add(0, self.main)
        
    def main(self):
        """Invoked on the main loop. Runs once."""
        if (len(sys.argv) < 2):
            utils.dialog_error(None, 'GAptI Error', 'GAptI should not be run manually.')
        else:
            for uri in sys.argv[1:]:
                self.handle_uri(uri)
                break
                
        gtk.main_quit()
        return False

    def handle_uri(self, uri):
        """Completes the entire lifecycle for one uri to one .apt file."""
        
        # Unpack the stanza structure and public key from the file.
        apt_file, key_file, rng_file = self.unpack_apt_file(uri)
        if (apt_file == None or key_file == None or rng_file == None):
            return
        
        # Parse the file out into a structure.
        apt = self.parse_apt_file(apt_file)
        if (apt == None):
            return
        
        # Eliminate stanzas in the apt structure that don't match the current
        # platform.
        apt = self.filter_apt_file(apt)
        
        # Compiles a complete list of archives and packages which will be
        # installed as a result of running this .apt file.
        (archives, packages) = self.process_apt_file(apt)
        if (archives == None or packages == None):
            return
        
        # Gets the textual value of the GPG key.
        keys_txt = self.extract_key_name(rng_file)
        if (keys_txt == None):
            return
        
        # If no matches at all, show a dialog to the user which describes what
        # is going on.
        if (len(packages) == 0):
            utils.dialog_error(None, 'No available packages.',
                               'The Apt file opened described no applicable packages.' +
                               'There may be no packages available for your combination of OS and architecture.')
            return
        
        self.confirm_dialog = SubscriptionConfirmDialog(self.datadir + '/gapti.glade', 'confirm_dialog')

        # Populate the confirm_dialog.
        if (self.populate_confirm_dialog(keys_txt, archives) == False):
            return
        
        if (self.confirm_dialog.run() != gtk.RESPONSE_ACCEPT):
            return
        
        if (self.install(key_file, archives, packages) == False):
            return
        
        apt_file.close()    
        key_file.close()
        rng_file.close()
        
        return

    def populate_confirm_dialog(self, keys_txt, archives):
        """Configures the confirm_dialog for the passed values."""
        print keys_txt
        self.confirm_dialog.set_provider('\n'.join(keys_txt))

        return True
        
    def unpack_apt_file(self, uri):
        """Reads in the .apt file, verifies it, unpacks the data and public
           key."""
           
        # Split the file into lines and check for type.
        lines = []
        for line in open(uri, 'r').readlines():
            lines.append(line)
        if (lines[0].strip() != '#@application/x-apt'):
            utils.dialog_error(None, 'Invalid Apt File', 'Could not determine file type.')
            return None, None, None
            
        # Unpack signature file.
        tmp_sig = tempfile.NamedTemporaryFile()
        state = 0
        for line in lines:
            if (line.strip() == '-----BEGIN PGP SIGNED MESSAGE-----'):
                state = 1
            if (line.strip() == '-----END PGP SIGNATURE-----'):
                state = 2
            if (state == 1 or state == 2):
                tmp_sig.write(line)
            if (state == 2):
                break
        tmp_sig.flush()
        
        # Unpack first public key.
        tmp_key = tempfile.NamedTemporaryFile()
        state = 0
        for line in lines:
            if (line.strip() == '-----BEGIN PGP PUBLIC KEY BLOCK-----'):
                state = 1
            if (line.strip() == '-----END PGP PUBLIC KEY BLOCK-----'):
                state = 2
            if (state == 1 or state == 2):
                tmp_key.write(line)
            if (state == 2):
                break
        tmp_key.flush()
        
        # Import the public key into a temporary keyring file for validation
        # of the other signed file.
        tmp_rng = tempfile.NamedTemporaryFile()        
        cmd = ['/usr/bin/gpg', '--no-default-keyring',
               '--keyring', tmp_rng.name,
               '--import', tmp_key.name]
        p = subprocess.Popen(cmd)
        if (p.wait() != 0):
            utils.dialog_error(None, 'Signature Error', 'Could not import public key.')
            return None, None, None
            
        # Verify the signature on the file.
        cmd = ['/usr/bin/gpg', '--no-default-keyring',
               '--keyring', tmp_rng.name,
               '--verify', tmp_sig.name]
        p = subprocess.Popen(cmd)
        if (p.wait() != 0):
            utils.dialog_error(None, 'Signature Error', 'Could not verify signed .apt file against included key.')
            return None, None, None
            
        # Done.
        tmp_sig.close()
        
        # Unpack signed data. Does GPG have a way to do this?
        tmp_apt = tempfile.NamedTemporaryFile()
        state = 0
        for line in lines:
            if (state == 1):
                tmp_apt.write(line)
            if (line.strip() == '-----BEGIN PGP SIGNED MESSAGE-----'):
                state = 1
            if (line.strip() == '-----BEGIN PGP SIGNATURE-----'):
                break
        tmp_apt.flush()
        
        return tmp_apt, tmp_key, tmp_rng

    def parse_apt_file(self, apt_file):
        """Reads in the .apt file specified by uri, parses it into a list of
           stanzas. Each stanza is a dictionary."""
        obj = []
        
        # apt_pkg lets us parse the basic file format.
        Parse = apt_pkg.ParseTagFile(open(apt_file.name, 'r'))
        while Parse.Step() == 1:
            stanza = { }
            
            # Basic plain text lines.
            stanza['Architecture'] = Parse.Section.get('Architecture')
            stanza['Distribution'] = Parse.Section.get('Distribution')
            stanza['Release']      = Parse.Section.get('Release')
            stanza['Codename']     = Parse.Section.get('Codename')

            # Archive line. Contains one or more archive paths seperated by a
            # new line. properly trim and store these values as a list.
            archive = []
            s = Parse.Section.get('Archive')
            if (s != None):
                for line in s.splitlines():
                    line = line.strip()
                    source = SourceEntry(line)
                    if (source.type != 'deb'):
                        utils.dialog_error(None, 'Invalid Apt File', 'Invalid Archive specified in Apt file.')
                        return None
                    archive.append(line.strip())
                stanza['Archive'] = archive
            else:
                continue

            # Install line specifies packages to install if the user consents
            # to the installation. These are space seperated.
            install = []
            s = Parse.Section.get('Install')
            if (s != None):
                for package in s.split(' '):
                    package = package.strip()
                    if (package != ''):
                        install.append(package)
                stanza['Install'] = install

            # Done. Append to existing obj.
            obj.append(stanza)

        return obj

    def filter_apt_file(self, apt):
        """Removes stanzas which don''t apply to the current system. If a
           stanza contains one of the filterable values, we eliminate it if it
           doesn''t match."""
        new_apt = []

        my_architecture = commands.getoutput('/usr/bin/dpkg --print-architecture')
        my_distribution = commands.getoutput('/bin/lsb_release -i -s')
        my_release      = commands.getoutput('/bin/lsb_release -r -s')
        my_codename     = commands.getoutput('/bin/lsb_release -c -s')

        # Loop over each stanza. If any of the following if statements match,
        # we don't add the stanza to new_apt. Thus it is eliminated.
        for stanza in apt:
            if stanza['Architecture'] != None and stanza['Architecture'] != my_architecture:
                print 'Architecture : `%s` != `%s`' % (stanza['Architecture'], my_architecture)
                continue
            if stanza['Distribution'] != None and stanza['Distribution'] != my_distribution:
                print 'Distribution : `%s` != `%s`' % (stanza['Distribution'], my_distribution)
                continue
            if stanza['Release']      != None and stanza['Release'] != my_release:
                print 'Release      : `%s` != `%s`' % (stanza['Release'], my_release)
                continue
            if stanza['Codename']     != None and stanza['Codename'] != my_codename:
                print 'Codename     : `%s` != `%s`' % (stanza['Codename'], my_codename)
                continue

            # If we've made it this far, either no filters were specified or
            # they all matched.
            new_apt.append(stanza)
        
        return new_apt
        
    def process_apt_file(self, apt):
        """Compiles a resulting tuple of archives and packages which will be
           installed as a result of installing the filtered apt file."""
        archives = []
        packages = []
        
        print apt
        for stanza in apt:
            if (stanza['Archive'] != None):
                for archive in stanza['Archive']:
                    archives.append(archive)
            if (stanza['Install'] != None):
                for package in stanza['Install']:
                    packages.append(package)
                    
        return (archives, packages)
        
    def extract_key_name(self, rng_file):
        """Accepts a GPG keyring file."""

        res = []
        cmd = ['/usr/bin/gpg', '--no-default-keyring',
               '--keyring', rng_file.name,
               '--with-colons', '--batch', '--list-keys']
        print cmd
        p = subprocess.Popen(cmd, stdout=subprocess.PIPE)
        for line in p.stdout.readlines():
            fields = line.split(":")
            if (fields[0] == "pub"):
                name = fields[9]
                res.append("%s" % _(name))
        if (p.wait() != 0):
            utils.dialog_error(None, 'Apt File Error', 'Could not read keys in ring,')
            return None
        
        print 'res:'
        print res
        return res
        
    def install(self, key_file, archives, packages):
        """Proceeds with the installation. Each parameter is saved to a
           temporary file and then passed to gapti-root."""
        
        archives_tmp = self.write_archives(archives)
        packages_tmp = self.write_packages(packages)
        
        cmd = ['/usr/bin/gksu', '--', '%s/gapti-root' % self.datadir]
        cmd.append(key_file.name)
        cmd.append(archives_tmp.name)
        cmd.append(packages_tmp.name)
        print cmd
        p = subprocess.Popen(cmd)
        if (p.wait() != 0):
            return False

        archives_tmp.close()
        packages_tmp.close()
        
        return True
        
    def write_archives(self, archives):
        """Writes the specified archives to a new temporary sources.list."""
        tmp = tempfile.NamedTemporaryFile()
        sources = utils.GAptISourcesList()
        
        # This helpful class helps us parse and verify the lines!
        for line in archives:
            source = SourceEntry(line, tmp.name)
            if (source.invalid == True):
                utils.dialog_error(None, 'Invalid source line in file.')
                return False
            else:
                sources.list.append(source)
        
        sources.save()
        return tmp
        
    def write_packages(self, packages):
        """Writes the packages to a new temporary file."""
        tmp = tempfile.NamedTemporaryFile()
        for package in packages:
            tmp.write('%s install\n' % package)
        tmp.flush()
        return tmp

if __name__ == '__main__':
    gapti = GAptI()
    gtk.main()
