/***************************************************************************
 *                                                                         *
 *   copyright (C) 2004 by Michael Buesch                                  *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include "genpasswd.h"
#include "pwmexception.h"
#include "randomizer.h"

#include <qregexp.h>


/* how often can a char of the same charset be reused in order */
#define FILTER_MAX_CHARSET_REUSE	3
/* re-randomize all charsets on every iteration (0/1) */
#define RERAND_CHARSET			0


struct staticCharsetStruct
{
	const char *lower_consonants;
	const char *lower_vowels;
	const char *upper_consonants;
	const char *upper_vowels;
	const char *num;
	const char *special;
	const char *blank;
};

static struct staticCharsetStruct staticCharset = {
	"bcdfghjklmnpqrstvwxyz",
	"aeiou",
	"BCDFGHJKLMNPQRSTVWXYZ",
	"AEIOU",
	"0123456789",
	"!\"$%&/()=?,.-;:_+",
	" "
};


GenPasswd::GenPasswd()
 : length (8)
 , mod (mod_none)
{
	dynCharset.setAutoDelete(true);
}

void GenPasswd::setCharset(bool lower,
			   bool upper,
			   bool num,
			   bool special,
			   bool blank,
			   QString user)
{
	unsigned int sanityCheck = 0;
	dynCharset_element *tmpElement;
	dynCharset.clear();
	if (lower) {
		tmpElement = new dynCharset_element;
		tmpElement->data = staticCharset.lower_consonants;
		tmpElement->data += staticCharset.lower_vowels;
		dynCharset.append(tmpElement);
		++sanityCheck;
	}
	if (upper) {
		tmpElement = new dynCharset_element;
		tmpElement->data = staticCharset.upper_consonants;
		tmpElement->data += staticCharset.upper_vowels;
		dynCharset.append(tmpElement);
		++sanityCheck;
	}
	if (num) {
		tmpElement = new dynCharset_element;
		tmpElement->data = staticCharset.num;
		dynCharset.append(tmpElement);
		++sanityCheck;
	}
	if (special) {
		tmpElement = new dynCharset_element;
		tmpElement->data = staticCharset.special;
		dynCharset.append(tmpElement);
		++sanityCheck;
	}
	if (blank) {
		tmpElement = new dynCharset_element;
		tmpElement->data = staticCharset.blank;
		dynCharset.append(tmpElement);
	}
	if (!user.isEmpty()) {
		tmpElement = new dynCharset_element;
		tmpElement->data = user;
		dynCharset.append(tmpElement);
		if (likely(user.length() >= 2))
			++sanityCheck;
	}
	BUG_ON(!sanityCheck);
	rndDynCharset();
}

void GenPasswd::rndDynCharset()
{
	QString tmpData;
	unsigned int pos;
	Randomizer rnd;
	QPtrList<dynCharset_element>::iterator i = dynCharset.begin(),
					       end = dynCharset.end();
	while (i != end) {
		tmpData = QString();
		while ((*i)->data.length()) {
			rnd >> pos;
			pos %= (*i)->data.length();
			tmpData.append((*i)->data.at(pos));
			(*i)->data.remove(pos, 1);
		}
		(*i)->data = tmpData;
		++i;
	}
}

QString GenPasswd::gen()
{
	PWM_ASSERT(length > 0);
	switch (mod) {
	case mod_none:
	case mod_filter:
		PWM_ASSERT(!dynCharset.isEmpty());
		return genRegular();
	case mod_pronounce:
		PWM_ASSERT(dynCharset.isEmpty());
		return genPronounceable();
	}
	return QString();
}

QString GenPasswd::genRegular()
{
	dynCharset_element *curCharset;
	QString ret;
	int i;
	for (i = 0; i < length; ++i) {
		curCharset = selectNextCharset();
#if RERAND_CHARSET != 0
		rndDynCharset();
#endif // RERAND_CHARSET
		ret += genNewRandom(curCharset);
	}
	return ret;
}

QString GenPasswd::genPronounceable()
{
	QString ret;
	const int typeRandom = -1;
	const int typeConsonant = 0;
	const int typeVowel = 1;
	int i, nextType;
	unsigned int rndNum;
	dynCharset_element vowels;
	dynCharset_element consonants;
	Randomizer rnd;

	vowels.data = staticCharset.lower_vowels;
	consonants.data = staticCharset.lower_consonants;
	consonants.data.remove(QRegExp("[xyz]"));
	for (i = 0; i < length; ++i) {
		if (i == 1 && consonants.refCnt == 1) {
			// do not start with two consonants.
			nextType = typeVowel;
			consonants.refCnt = 0;
		} else if (vowels.refCnt == 1) {
			nextType = typeConsonant;
			vowels.refCnt = 0;
		} else if (consonants.refCnt == 2) {
			nextType = typeVowel;
			consonants.refCnt = 0;
		} else
			nextType = typeRandom;

	    again:
		switch (nextType) {
		case typeRandom:
			rnd >> rndNum;
			rndNum %= 2;
			PWM_ASSERT(rndNum == 0 || rndNum == 1);
			if (rndNum == 0)
				nextType = typeConsonant;
			else
				nextType = typeVowel;
			goto again;
		case typeConsonant:
			ret += genNewRandom(&consonants);
			consonants.refCnt++;
			break;
		case typeVowel:
			ret += genNewRandom(&vowels);
			vowels.refCnt++;
			break;
		}
	}
	return ret;
}

GenPasswd::dynCharset_element * GenPasswd::selectNextCharset()
{
	dynCharset_element *ret;
	int numCharsets = dynCharset.count();
	PWM_ASSERT(numCharsets > 0);
	if (numCharsets == 1)
		return dynCharset.at(0);
	Randomizer rnd;
	if (mod == mod_filter) {
		// find out which charsets are allowed (filtering)
		QPtrList<dynCharset_element> allowedCharsets;
		QPtrList<dynCharset_element>::iterator i = dynCharset.begin(),
						       end = dynCharset.end();
		while (i != end) {
			if ((*i)->refCnt < FILTER_MAX_CHARSET_REUSE) {
				allowedCharsets.append(*i);
			} else {
				(*i)->refCnt = 0;
			}
			++i;
		}
		int numAllowedCharsets = allowedCharsets.count();
		PWM_ASSERT(numAllowedCharsets > 0);
		// now get a random charset out of the allowed
		unsigned int randomPos;
		rnd >> randomPos;
		randomPos %= numAllowedCharsets;
		ret = allowedCharsets.at(randomPos);
		ret->refCnt++;
		return ret;
	}
	// all charsets are allowed here (no filtering). Get a random.
	unsigned int randomPos;
	rnd >> randomPos;
	randomPos %= numCharsets;
	ret = dynCharset.at(randomPos);
	return ret;
}

QChar GenPasswd::genNewRandom(const dynCharset_element *charset)
{
	Randomizer rnd;
	unsigned int pos;
	rnd >> pos;
	pos %= charset->data.length();
	return charset->data.at(pos);
}
