/***************************************************************************
 *                                                                         *
 *   copyright (C) 2004 by Michael Buesch                                  *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   KWallet DCOP Interface is                                             *
 *    Copyright (c) 2002-2003 George Staikos <staikos@kde.org>             *
 *   KWallet is part of the KDE libraries                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#ifndef PWMANAGER_KWALLETEMU_H_
#define PWMANAGER_KWALLETEMU_H_

#include <qstring.h>
#include <qcstring.h>
#include <qstringlist.h>

#include <dcopobject.h>
#include <dcopclient.h>

#include <vector>

#include "binentrygen.h"

class PwMDoc;
class PwMInit;

#define PWM_KWALLETEMU_DCOPMODULE	"pwmanager_kwalletemu"
#define PWM_KWALLETEMU_DCOPOBJ		"kwalletemu"

/** This is the INTERNAL KWallet Emulation interface.
  * DO NOT try to call any of these DCOP-functions directly. Likely
  * this will not work. Please use the original KDE
  * kwallet interface (kwallet.h), to access PwManager from
  * another application.
  */
class KWalletEmu : public QObject
		 , public DCOPObject
{
	Q_OBJECT
	K_DCOP

protected:
	class ConnectionList
	{
	public:
		struct connectionElement
		{
			QCString	clientName;
			PwMDoc		*document;
			int		handle;
		};

	public:
		ConnectionList(KWalletEmu *_kwalletEmu)
				{ kwalletEmu = _kwalletEmu; }

		void add(const QCString &client, PwMDoc *doc, int handle);
		/** delete the connection with this handle */
		void del(int handle);
		/** delete all connections to this wallet */
		void del(const QString &wallet);
		/** delete all connections from this client */
		void delClient(const QCString &client);
		bool get(int handle, connectionElement *ret);
		bool find(const QCString &client, const QString &wallet,
			  connectionElement *ret);
		/** find a connection to "wallet". Returns the first
		  * element that has the name "wallet"
		  */
		bool find(const QString wallet, connectionElement *ret);

		const std::vector<connectionElement> * getList()
				{ return &connList; }
	
	protected:
		std::vector<connectionElement> connList;
		KWalletEmu *kwalletEmu;
	};

	enum EntryType
	{
		Unknown = 0,
		Password,
		Stream,
		Map,
		Unused=0xffff
	};

public:
	friend class KWalletEmu::ConnectionList;

	KWalletEmu(PwMInit *_init,
		   QObject *parent = 0, const char *name = 0);
	~KWalletEmu();

	/** enable/disable the KWalletEmu interface */
	void setEnabled(bool _enabled)
			{ enabled = _enabled; }
	/** is the KWalletEmu interface enabled? */
	bool isEnabled()
			{ return enabled; }
	/** connects all neccessary signals from "doc" to the
	  * emulation layer. Call this on every new document creation.
	  */
	void connectDocSignals(PwMDoc *doc);

protected:
	/** get a list of all open document-names (titles) */
	QStringList getOpenDocList() const;
	/** generate a new random and unique handle */
	int genHandle();
	/** check if this peer has a connection and
	  * return its document pointer
	  */
	PwMDoc * checkConn(const QCString &peer,
			   int handle);
	/** checks if the function caller (the real DCOP peer,
	  * that called the DCOP function) is KDED
	  */
	bool checkCaller() const;
	/** make a new connection entry for this peer */
	int connectPeer(const QCString &peer, PwMDoc *doc);
	/** load the real KWallet KDED module.
	  * It only loads it if KWallet is enabled in its config
	  */
	int kdedLoadKWallet();
	/** load the KWallet KDED emulation module */
	bool kdedLoadEmu();
	/** unload the real KWallet KDED module */
	bool kdedUnloadKWallet();
	/** unload the KWallet KDED emulation module */
	bool kdedUnloadEmu();
	/** close the document connection with the supplied handle.
	  * This function does _not_ call checkConn()
	  */
	int doClose(int handle,
		    bool force = false);
	/** Write a binary QByteArray entry */
	int writeEntryBinary(const QCString &peer,
			     int handle,
			     const QString& folder,
			     const QString& key,
			     const QByteArray& value,
			     BinEntryGen::DataType requestedType);
	/** Write an entry */
	int doWriteEntry(const QCString &peer,
			 int handle,
			 const QString& folder,
			 const PwMDataItem &value);
	/** Read a binary QByteArray entry */
	QByteArray readEntryBinary(const QCString &peer,
				   int handle,
				   const QString& folder,
				   const QString& key,
				   BinEntryGen::DataType requestedType);
	/** Read an entry */
	bool doReadEntry(const QCString &peer,
			 int handle,
			 const QString &folder,
			 const QString &key,
			 PwMDataItem *ret);
	/** Default KWallet local wallet. */
	QString localWallet() const;
	/** Default KWallet network wallet. */
	QString networkWallet() const;
	/** Returns the PwMDoc pointer for a given
	  * wallet name, or 0 if there is no such doc.
	  */
	PwMDoc * getDocByWalletName(const QString &wallet) const;

protected slots:
	/** the document "doc" changed its data */
	void docChanged(PwMDoc *doc);
	/** the document "doc" got closed */
	void docClosed(PwMDoc *doc);
	/** the document "doc" got just opened */
	void docOpened(PwMDoc *doc);
	/** the document "doc" got created */
	void docCreated(PwMDoc *doc);
	/** an application disconnected from the dcop interface */
	void dcopAppDisconn(const QCString &appId);

protected:
	/* signal sending methods */
	void sig_walletClosed(const QString &wallet);
	void sig_walletClosed(int handle);
	void sig_allWalletsClosed();
	void sig_applicationDisconnected(const QString &wallet,
					 const QCString &application);
	void sig_folderUpdated(const QString &wallet,
			       const QString &folder);
	void sig_folderListUpdated(const QString &wallet);
	void sig_walletListDirty();
	void sig_walletCreated(const QString &wallet);
	void sig_walletOpened(const QString &wallet);
	void sig_walletDeleted(const QString &wallet);

protected:
	/** Interface enabled? */
	bool enabled;
	/** list of all open connections */
	ConnectionList *connList;
	/** pointer to the init class */
	PwMInit *init;

k_dcop:
	bool isEnabled(const QCString &peer) const;
	int open(const QCString &peer,
		 const QString& wallet,
		 uint wId);
	int openPath(const QCString &peer,
		     const QString& path,
		     uint wId);
	void openAsynchronous(const QCString &peer,
			      const QString& wallet,
			      const QCString& returnObject,
			      uint wId);
	int close(const QCString &peer,
		  const QString& wallet,
		  bool force);
	int close(const QCString &peer,
		  int handle,
		  bool force);
	ASYNC sync(const QCString &peer,
		   int handle);
	int deleteWallet(const QCString &peer,
			 const QString& wallet);
	bool isOpen(const QCString &peer,
		    const QString& wallet);
	bool isOpen(const QCString &peer,
		    int handle);
	QStringList users(const QCString &peer,
			  const QString& wallet) const;
	void changePassword(const QCString &peer,
			    const QString& wallet,
			    uint wId);
	QStringList wallets(const QCString &peer) const;
	QStringList folderList(const QCString &peer,
			       int handle);
	bool hasFolder(const QCString &peer,
		       int handle,
		       const QString &folder);
	bool createFolder(const QCString &peer,
			  int handle,
			  const QString& folder);
	bool removeFolder(const QCString &peer,
			  int handle,
			  const QString& folder);
	QStringList entryList(const QCString &peer,
			      int handle,
			      const QString& folder);
	QByteArray readEntry(const QCString &peer,
			     int handle,
			     const QString& folder,
			     const QString& key);
	QByteArray readMap(const QCString &peer,
			   int handle,
			   const QString& folder,
			   const QString& key);
	QString readPassword(const QCString &peer,
			     int handle,
			     const QString& folder,
			     const QString& key);
	int renameEntry(const QCString &peer,
			int handle,
			const QString& folder,
			const QString& oldName,
			const QString& newName);
	int writeEntry(const QCString &peer,
		       int handle,
		       const QString& folder,
		       const QString& key,
		       const QByteArray& value,
		       int entryType);
	int writeEntry(const QCString &peer,
		       int handle,
		       const QString& folder,
		       const QString& key,
		       const QByteArray& value);
	int writeMap(const QCString &peer,
		     int handle,
		     const QString& folder,
		     const QString& key,
		     const QByteArray& value);
	int writePassword(const QCString &peer,
			  int handle,
			  const QString& folder,
			  const QString& key,
			  QString value);
	bool hasEntry(const QCString &peer,
		      int handle,
		      const QString& folder,
		      const QString& key);
	int entryType(const QCString &peer,
		      int handle,
		      const QString& folder,
		      const QString& key);
	int removeEntry(const QCString &peer,
			int handle,
			const QString& folder,
			const QString& key);
	bool disconnectApplication(const QCString &peer,
				   const QString& wallet,
				   const QCString& application);
	void reconfigure(const QCString &peer);
	bool folderDoesNotExist(const QCString &peer,
				const QString& wallet,
				const QString& folder);
	bool keyDoesNotExist(const QCString &peer,
			     const QString& wallet,
			     const QString& folder,
			     const QString& key);
	void closeAllWallets(const QCString &peer);
};

#endif // PWMANAGER_KWALLETEMU_H_
