/***************************************************************************
 *                                                                         *
 *   copyright (C) 2003, 2004 by Michael Buesch                            *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include "randomizer.h"
#include "pwmexception.h"

#include <qfile.h>
#include <qfileinfo.h>

#include <stdlib.h>
#include <time.h>


Randomizer::Use Randomizer::use (Randomizer::use_none);


Randomizer::Randomizer()
 : offset (0)
 , size (0)
{
	if (unlikely(use == use_none))
		probe();
}

void Randomizer::probe()
{
	QFileInfo fi;

	// probe for /dev/urandom
	fi.setFile("/dev/urandom");
	if (fi.isReadable()) {
		printDebug("Randomizer: using /dev/urandom");
		use = use_urandom;
		return;
	}

	// probe for /dev/random
	fi.setFile("/dev/random");
	if (fi.isReadable()) {
		printDebug("Randomizer: using /dev/random");
		use = use_random;
		return;
	}

	// probe for EGD
	char *fn = getenv("RANDFILE");
	if (fn) {
		fi.setFile(fn);
		if (fi.isReadable()) {
			printDebug(string("Randomizer: using $RANDFILE \"")
				   + fn
				   + "\" (aka EGD)");
			use = use_egd;
			return;
		}
	}

	/* no secure randomizer found.
	 * Fall back to stdlib randomizer.
	 */
	use = use_rand;
	srand(time(0));
	printWarn("neither /dev/*random nor EGD found! "
		  "Falling back to insecure rand()!");
}

QFile * Randomizer::get_generic()
{
	QFile *fd = 0;
	switch (use) {
	case use_urandom:
		fd = get_urandom();
		break;
	case use_random:
		fd = get_random();
		break;
	case use_egd:
		fd = get_egd();
		break;
	case use_rand:
		break;
	case use_none:
		PWM_ASSERT(false);
	}
	return fd;
}

QFile * Randomizer::get_generic(const char *fn)
{
	QFile *fd = new QFile(fn);
	bool err = fd->open(IO_ReadOnly);
	BUG_ON(!err);
	return fd;
}

QFile * Randomizer::get_egd()
{
	char *fn = getenv("RANDFILE");
	return get_generic(fn);
}

void Randomizer::put_generic(QFile *f)
{
	if (!f)
		return;
	f->close();
	delete f;
}

Randomizer & Randomizer::operator>> (char *d)
{
	QFile *fd = get_generic();
	PWM_ASSERT(size);

	if (fd) {
		Q_LONG ret = fd->readBlock(d + offset, size);
		BUG_ON(ret != static_cast<Q_LONG>(size));
	} else {
		unsigned int i;
		for (i = 0; i < size; ++i)
			*(d + offset + i) = rand() % 0xFF;
	}

	return cleanup(fd);
}

Randomizer & Randomizer::operator>> (QByteArray &d)
{
	QFile *fd = get_generic();
	unsigned int datasize = d.size();

	if (fd) {
		Q_LONG ret = fd->readBlock(d.data(), datasize);
		BUG_ON(ret != static_cast<Q_LONG>(datasize));
	} else {
		unsigned int i;
		for (i = 0; i < datasize; ++i)
			d[i] = rand() % 0xFF;
	}

	return cleanup(fd);
}

Randomizer & Randomizer::operator>> (unsigned int &d)
{
	QFile *fd = get_generic();

	if (fd) {
		Q_LONG ret = fd->readBlock(reinterpret_cast<char *>(&d), sizeof(d));
		BUG_ON(ret != sizeof(d));
	} else
		d = rand();

	return cleanup(fd);
}

Randomizer & Randomizer::operator>> (unsigned char &d)
{
	QFile *fd = get_generic();

	if (fd) {
		int ret = fd->getch();
		BUG_ON(ret == -1);
		d = ret;
	} else
		d = rand() % 0xFF;

	return cleanup(fd);
}
