/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

//Own
#include "entry.h"

//KDE
#include <kmountpoint.h>
#include <kmessagebox.h>

//Ui
#include "password.h"
#include "map.h"

//KGRUBEditor
#include "kgrubeditor.h"
#include "core/filetransactions.h"

EntryAssistant::EntryAssistant( GRUB::ConfigFile::Entry *entry, QWidget *parent ) : KAssistantDialog( parent )
{
//SETUP UI
	setCaption( i18n( "Entry Assistant" ) );
	setWindowIcon( KIcon( "document-properties" ) );
	setAttribute( Qt::WA_DeleteOnClose );
	setInitialSize( QSize( 650, 450 ) );
	showButton( KDialog::Help, false );

	QWidget *intro = new QWidget;
	ui_intro.setupUi( intro );
	addPage( intro, i18n( "Introduction" ) );
	QWidget *step1 = new QWidget;
	ui_step1.setupUi( step1 );
	titlePage = addPage( step1, i18n( "Step 1/10 - Title" ) );
	QWidget *step2 = new QWidget;
	ui_step2.setupUi( step2 );
	addPage( step2, i18n( "Step 2/10 - Root" ) );
	QWidget *step3 = new QWidget;
	ui_step3.setupUi( step3 );
	addPage( step3, i18n( "Step 3/10 - Kernel" ) );
	QWidget *step4 = new QWidget;
	ui_step4.setupUi( step4 );
	addPage( step4, i18n( "Step 4/10 - Initrd" ) );
	QWidget *step5 = new QWidget;
	ui_step5.setupUi( step5 );
	addPage( step5, i18n( "Step 5/10 - Lock" ) );
	QWidget *step6 = new QWidget;
	ui_step6.setupUi( step6 );
	addPage( step6, i18n( "Step 6/10 - Password" ) );
	QWidget *step7 = new QWidget;
	ui_step7.setupUi( step7 );
	addPage( step7, i18n( "Step 7/10 - MakeActive" ) );
	QWidget *step8 = new QWidget;
	ui_step8.setupUi( step8 );
	addPage( step8, i18n( "Step 8/10 - SaveDefault" ) );
	QWidget *step9 = new QWidget;
	ui_step9.setupUi( step9 );
	addPage( step9, i18n( "Step 9/10 - Chainloader" ) );
	QWidget *step10 = new QWidget;
	ui_step10.setupUi( step10 );
	addPage( step10, i18n( "Step 10/10 - Maps" ) );
	QWidget *summary = new QWidget;
	ui_summary.setupUi( summary );
	summaryPage = addPage( summary, i18n( "Summary" ) );
//PROPERTY SETUP
	tmp_entry = *entry;
	m_entry = entry;
//SHOW ENTRY(IF ANY - ONLY USED IN EDIT MODE)
	ui_step1.klineedit_title->setText( tmp_entry.title() );
	ui_step2.klineedit_root->setText( tmp_entry.root() );
	ui_step3.kurlrequester_kernel->setPath( tmp_entry.kernel().kernel() );
	ui_step3.klineedit_arguments->setText( tmp_entry.kernel().arguments() );
	ui_step4.kurlrequester_initrd->setPath( tmp_entry.initrd() );
	ui_step5.checkBox_lock->setChecked( tmp_entry.lock() );
	showPassword();
	ui_step7.checkBox_makeactive->setChecked( tmp_entry.makeActive() );
	ui_step8.checkBox_savedefault->setChecked( tmp_entry.saveDefault() );
	ui_step9.klineedit_chainloader->setText( tmp_entry.chainLoader() );
	showMaps();
//MISC UI SETUP
	setValid( titlePage, false );
	setTitlePageValid();

	menu_rootSuggestions = new QMenu;
	for ( int i = 0; i < KGRUBEditor::DeviceList.size() ; i++ )
	{
		menu_rootSuggestions->addAction( KGRUBEditor::DeviceList.at( i ).partition() + QString( " (" ) + KGRUBEditor::DeviceList.at( i ).mountPoint() + QString( ")" ) );
	}
	ui_step2.kpushbutton_suggestions->setMenu( menu_rootSuggestions );
	ui_step2.kpushbutton_suggestions->setIcon( KIcon( "tools-wizard" ) );

	menu_kernelSuggestions = new QMenu;
	action_rootDevice = new QAction( i18n( "Root (Device)" ), menu_kernelSuggestions );
	menu_kernelSuggestions->addAction( action_rootDevice );
	action_rootUuid = new QAction( i18n( "Root (UUID)" ), menu_kernelSuggestions );
	menu_kernelSuggestions->addAction( action_rootUuid );
	menu_kernelSuggestions->addSeparator();
	action_ro = new QAction( i18n( "Read-Only" ), menu_kernelSuggestions );
	menu_kernelSuggestions->addAction( action_ro );
	action_quiet = new QAction( i18n( "Quiet" ), menu_kernelSuggestions );
	menu_kernelSuggestions->addAction( action_quiet );
	action_splash = new QAction( i18n( "Splash" ), menu_kernelSuggestions );
	menu_kernelSuggestions->addAction( action_splash );
	ui_step3.kpushbutton_suggestions->setMenu( menu_kernelSuggestions );
	ui_step3.kpushbutton_suggestions->setIcon( KIcon( "tools-wizard" ) );

	ui_step6.kpushbutton_create->setIcon( KIcon( "dialog-password" ) );
	ui_step6.kpushbutton_edit->setIcon( KIcon( "document-properties" ) );
	ui_step6.kpushbutton_delete->setIcon( KIcon( "edit-delete" ) );

	ui_step10.kpushbutton_add->setIcon( KIcon( "list-add" ) );
	ui_step10.kpushbutton_edit->setIcon( KIcon( "document-properties" ) );
	ui_step10.kpushbutton_copy->setIcon( KIcon( "edit-copy" ) );
	ui_step10.kpushbutton_remove->setIcon( KIcon( "list-remove" ) );
//SETUP CONNECTIONS
	connect( ui_step1.klineedit_title, SIGNAL( cursorPositionChanged( int, int ) ), SLOT( setTitlePageValid() ) );
	connect( menu_rootSuggestions, SIGNAL( triggered( QAction * ) ), SLOT( rootSuggestionTriggered( QAction * ) ) );
	connect( ui_step3.kurlrequester_kernel, SIGNAL( urlSelected( const KUrl ) ), SLOT( convertKernelPath( const KUrl ) ) );
	connect( action_rootDevice, SIGNAL( triggered() ), SLOT( appendRootDeviceArgument() ) );
	connect( action_rootUuid, SIGNAL( triggered() ), SLOT( appendRootUuidArgument() ) );
	connect( action_ro, SIGNAL( triggered() ), SLOT( appendReadOnlyArgument() ) );
	connect( action_quiet, SIGNAL( triggered() ), SLOT( appendQuietArgument() ) );
	connect( action_splash, SIGNAL( triggered() ), SLOT( appendSplashArgument() ) );
	connect( ui_step4.kurlrequester_initrd, SIGNAL( urlSelected( const KUrl ) ), SLOT( convertInitrdPath( const KUrl ) ) );
	connect( ui_step6.kpushbutton_create, SIGNAL( pressed() ), SLOT( createPassword() ) );
	connect( ui_step6.kpushbutton_edit, SIGNAL( pressed() ), SLOT( editPassword() ) );
	connect( ui_step6.kpushbutton_delete, SIGNAL( pressed() ), SLOT( deletePassword() ) );
	connect( ui_step10.kpushbutton_add, SIGNAL( pressed() ), SLOT( addMap() ) );
	connect( ui_step10.kpushbutton_edit, SIGNAL( pressed() ), SLOT( editMap() ) );
	connect( ui_step10.kpushbutton_copy, SIGNAL( pressed() ), SLOT( copyMap() ) );
	connect( ui_step10.kpushbutton_remove, SIGNAL( pressed() ), SLOT( removeMap() ) );
	connect( this, SIGNAL( user2Clicked() ), SLOT( nextClicked() ) );
	connect( this, SIGNAL( user1Clicked() ), SLOT( finishClicked() ) );
}
void EntryAssistant::showMaps()
{
	ui_step10.klistwidget_maps->clear();

	if ( !tmp_entry.maps().isEmpty() )
	{
		for ( int i = 0; i < tmp_entry.maps().size() ; i++ )
		{
			ui_step10.klistwidget_maps->addItem( tmp_entry.maps().at( i ).result() );
		}
	}
	ui_step10.kpushbutton_edit->setDisabled( tmp_entry.maps().isEmpty() );
	ui_step10.kpushbutton_copy->setDisabled( tmp_entry.maps().isEmpty() );
	ui_step10.kpushbutton_remove->setDisabled( tmp_entry.maps().isEmpty() );
}
void EntryAssistant::showPassword()
{
	ui_step6.checkBox_md5->setChecked( tmp_entry.password().md5crypted() );
	ui_step6.klineedit_password->setText( tmp_entry.password().password().isEmpty() ? i18n( "<none>" ) : tmp_entry.password().password() );
	ui_step6.klineedit_menu->setText( tmp_entry.password().configFile().isEmpty() ? i18n( "<none>" ) : tmp_entry.password().configFile() );

	ui_step6.kpushbutton_edit->setDisabled( tmp_entry.password().isEmpty() );
	ui_step6.kpushbutton_delete->setDisabled( tmp_entry.password().isEmpty() );
}

void EntryAssistant::setTitlePageValid()
{
	( !ui_step1.klineedit_title->text().isEmpty() ? setValid( titlePage, true ) : setValid( titlePage, false ) );
}
void EntryAssistant::rootSuggestionTriggered( QAction *action )
{
	for ( int i = 0; i < KGRUBEditor::DeviceList.size() ; i++ )
	{
		if ( action->text().startsWith( KGRUBEditor::DeviceList.at( i ).partition() ) )
		{
			ui_step2.klineedit_root->setText( KGRUBEditor::DeviceList.at( i ).grubPartition() );
			return;
		}
	}
}
void EntryAssistant::convertKernelPath( const KUrl path )
{
	if ( path.isEmpty() )
		return;

	QString realKernelPath = FileTransactions::convertToGRUBPath( path.path(), KGRUBEditor::DeviceList );
	if ( realKernelPath.startsWith( ui_step2.klineedit_root->text() ) )
		realKernelPath.replace( ui_step2.klineedit_root->text(), QString() );

	ui_step3.kurlrequester_kernel->setPath( realKernelPath );
}
void EntryAssistant::convertInitrdPath( const KUrl path )
{
	if ( path.isEmpty() )
		return;

	QString realInitrdPath = FileTransactions::convertToGRUBPath( path.path(), KGRUBEditor::DeviceList );
	if ( realInitrdPath.startsWith( ui_step2.klineedit_root->text() ) )
		realInitrdPath.replace( ui_step2.klineedit_root->text(), QString() );

	ui_step4.kurlrequester_initrd->setPath( realInitrdPath );
}
void EntryAssistant::appendRootDeviceArgument()
{
	if ( ui_step3.kurlrequester_kernel->url().isEmpty() )
		return;

	QString realKernelPath;
	if ( ui_step3.kurlrequester_kernel->url().path().startsWith( "/" ) )
		realKernelPath = FileTransactions::convertToGenericPath( ui_step2.klineedit_root->text() + ui_step3.kurlrequester_kernel->url().path(), KGRUBEditor::DeviceList );
	else
		realKernelPath = FileTransactions::convertToGenericPath( ui_step3.kurlrequester_kernel->url().path(), KGRUBEditor::DeviceList );

	//Note that this check is necessary because if the user enters dummy text
	//such as 'jkh' mp would be null and ultimately crash the apllication.
	if ( KMountPoint::Ptr mp = KMountPoint::currentMountPoints().findByPath( realKernelPath ) )
	{
		for ( int i = 0; i < KGRUBEditor::DeviceList.size() ; i++ )
		{
			if ( mp->mountPoint() == KGRUBEditor::DeviceList.at( i ).mountPoint() )
			{
				ui_step3.klineedit_arguments->setText( ui_step3.klineedit_arguments->text() + ( ui_step3.klineedit_arguments->text().isEmpty() ? QString() : QString( " " ) ) + "root=" + KGRUBEditor::DeviceList.at( i ).partition() );
			}
		}
	}
}
void EntryAssistant::appendRootUuidArgument()
{
	if ( ui_step3.kurlrequester_kernel->url().isEmpty() )
		return;

	QString realKernelPath;
	if ( ui_step3.kurlrequester_kernel->url().path().startsWith( "/" ) )
		realKernelPath = FileTransactions::convertToGenericPath( ui_step2.klineedit_root->text() + ui_step3.kurlrequester_kernel->url().path(), KGRUBEditor::DeviceList );
	else
		realKernelPath = FileTransactions::convertToGenericPath( ui_step3.kurlrequester_kernel->url().path(), KGRUBEditor::DeviceList );

	//Note that this check is necessary because if the user enters dummy text
	//such as 'jkh' mp would be null and ultimately crash the apllication.
	if ( KMountPoint::Ptr mp = KMountPoint::currentMountPoints().findByPath( realKernelPath ) )
	{
		for ( int i = 0; i < KGRUBEditor::DeviceList.size() ; i++ )
		{
			if ( mp->mountPoint() == KGRUBEditor::DeviceList.at( i ).mountPoint() )
			{
				ui_step3.klineedit_arguments->setText( ui_step3.klineedit_arguments->text() + ( ui_step3.klineedit_arguments->text().isEmpty() ? QString() : QString( " " ) ) + "root=UUID=" + KGRUBEditor::DeviceList.at( i ).uuid() );
			}
		}
	}
}
void EntryAssistant::appendReadOnlyArgument()
{
	ui_step3.klineedit_arguments->setText( ui_step3.klineedit_arguments->text() + ( ui_step3.klineedit_arguments->text().isEmpty() ? QString() : QString( " " ) ) + "ro" );
}
void EntryAssistant::appendQuietArgument()
{
	ui_step3.klineedit_arguments->setText( ui_step3.klineedit_arguments->text() + ( ui_step3.klineedit_arguments->text().isEmpty() ? QString() : QString( " " ) ) + "quiet" );
}
void EntryAssistant::appendSplashArgument()
{
	ui_step3.klineedit_arguments->setText( ui_step3.klineedit_arguments->text() + ( ui_step3.klineedit_arguments->text().isEmpty() ? QString() : QString( " " ) ) + "splash" );
}
void EntryAssistant::createPassword()
{
	GRUB::ComplexCommand::Password tmp_password;
	PasswordAssistant *passwordAssistant = new PasswordAssistant( &tmp_password, this );
	if ( passwordAssistant->exec() == QDialog::Accepted )
	{
		tmp_entry.setPassword( tmp_password );
		showPassword();
	}
}
void EntryAssistant::editPassword()
{
	GRUB::ComplexCommand::Password tmp_password = tmp_entry.password();
	PasswordAssistant *passwordAssistant = new PasswordAssistant( &tmp_password, this );
	if ( passwordAssistant->exec() == QDialog::Accepted )
	{
		tmp_entry.setPassword( tmp_password );
		showPassword();
	}
}
void EntryAssistant::deletePassword()
{
	if ( KMessageBox::questionYesNo( this, i18n( "Are you sure you want to delete the password?" ), i18n( "Confirmation" ) ) == KMessageBox::Yes )
	{
		tmp_entry.clearPassword();
		showPassword();
	}
}
void EntryAssistant::addMap()
{
	GRUB::ComplexCommand::Map tmp_map;
	MapAssistant *mapAssistant = new MapAssistant( &tmp_map, this );
	if ( mapAssistant->exec() == QDialog::Accepted )
	{
		tmp_entry.addMap( tmp_map );
		showMaps();
	}
}
void EntryAssistant::editMap()
{
	if ( !ui_step10.klistwidget_maps->currentItem() )
		return;

	GRUB::ComplexCommand::Map tmp_map = tmp_entry.maps().at( ui_step10.klistwidget_maps->currentRow() );
	MapAssistant *mapAssistant = new MapAssistant( &tmp_map, this );
	if ( mapAssistant->exec() == QDialog::Accepted )
	{
		tmp_entry.replaceMap( ui_step10.klistwidget_maps->currentRow(), tmp_map );
		showMaps();
	}
}
void EntryAssistant::copyMap()
{
	if ( !ui_step10.klistwidget_maps->currentItem() )
		return;

	tmp_entry.copyMap( ui_step10.klistwidget_maps->currentRow() );
	showMaps();
}
void EntryAssistant::removeMap()
{
	if ( !ui_step10.klistwidget_maps->currentItem() )
		return;

	if ( KMessageBox::questionYesNo( this, i18n( "Are you sure you want to remove this map?" ), i18n( "Confirmation" ) ) == KMessageBox::Yes )
	{
		tmp_entry.removeMap( ui_step10.klistwidget_maps->currentRow() );
		showMaps();
	}
}

void EntryAssistant::nextClicked()
{
	if ( currentPage() == summaryPage )
	{
		tmp_entry.setTitle( ui_step1.klineedit_title->text() );
		tmp_entry.setRoot( ui_step2.klineedit_root->text() );
		tmp_entry.setKernel( GRUB::ComplexCommand::Kernel( ui_step3.kurlrequester_kernel->url().path(), ui_step3.klineedit_arguments->text() ) );
		tmp_entry.setInitrd( ui_step4.kurlrequester_initrd->url().path() );
		tmp_entry.setLock( ui_step5.checkBox_lock->isChecked() );
		//password is already set
		tmp_entry.setMakeActive( ui_step7.checkBox_makeactive->isChecked() );
		tmp_entry.setSaveDefault( ui_step8.checkBox_savedefault->isChecked() );
		tmp_entry.setChainLoader( ui_step9.klineedit_chainloader->text() );
		//maps are already set

		ui_summary.label_title->setText( tmp_entry.title() );
		ui_summary.label_root->setText( tmp_entry.root().isEmpty() ? i18n( "<none>" ) : tmp_entry.root() );
		ui_summary.label_kernel->setText( tmp_entry.kernel().isEmpty() ? i18n( "<none>" ) : tmp_entry.kernel().result() );
		ui_summary.label_initrd->setText( tmp_entry.initrd().isEmpty() ? i18n( "<none>" ) : tmp_entry.initrd() );
		ui_summary.checkBox_lock->setChecked( tmp_entry.lock() );
		ui_summary.label_password->setText( tmp_entry.password().isEmpty() ? i18n( "<none>" ) : tmp_entry.password().result() );
		ui_summary.checkBox_makeactive->setChecked( tmp_entry.makeActive() );
		ui_summary.checkBox_savedefault->setChecked( tmp_entry.saveDefault() );
		ui_summary.label_chainloader->setText( tmp_entry.chainLoader().isEmpty() ? i18n( "<none>" ) : tmp_entry.chainLoader() );
	}
}
void EntryAssistant::finishClicked()
{
	m_entry->setTitle( ui_step1.klineedit_title->text() );
	m_entry->setRoot( ui_step2.klineedit_root->text() );
	m_entry->setKernel( GRUB::ComplexCommand::Kernel( ui_step3.kurlrequester_kernel->url().path(), ui_step3.klineedit_arguments->text() ) );
	m_entry->setInitrd( ui_step4.kurlrequester_initrd->url().path() );
	m_entry->setLock( ui_step5.checkBox_lock->isChecked() );
	m_entry->setPassword( tmp_entry.password() );
	m_entry->setMakeActive( ui_step7.checkBox_makeactive->isChecked() );
	m_entry->setSaveDefault( ui_step8.checkBox_savedefault->isChecked() );
	m_entry->setChainLoader( ui_step9.klineedit_chainloader->text() );
	m_entry->setMaps( tmp_entry.maps() );
}
