/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

//Own
#include "kgrubeditor.h"

//Qt
#include <qtablewidget.h>
#include <qtreewidget.h>
#include <qstringlist.h>
#include <qgraphicsitem.h>
#include <qtimer.h>

//KDE
#include <kapplication.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kaction.h>
#include <kactioncollection.h>
#include <kstandardaction.h>
#include <kfiledialog.h>
#include <kconfigdialog.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>
#include <kio/netaccess.h>
#include <kmountpoint.h>
#include <qsplashscreen.h>
#include <qprocess.h>

//Ui
#include "view.h"
#include "entry.h"
#include "backup.h"
#include "password.h"
#include "map.h"

//KGRUBEditor
#include "settings.h"
#include "core/filetransactions.h"

QVector<GRUB::ConfigFile::Entry> KGRUBEditor::GrubEntries;
QVector<GRUB::Misc::Device> KGRUBEditor::DeviceList;

KGRUBEditor::KGRUBEditor( QWidget *parent ) : KXmlGuiWindow( parent )
{
	setupUi( this );
	setupObjects();
	setupActions();
	setupMenus();
	setupConnections();
	setupGUI();

	refresh( 0, 0, ui_entries.klistwidget_os );
}

//SETUP FUNCTIONS
void KGRUBEditor::setupObjects()
{
	timer_blink = new QTimer;
}
void KGRUBEditor::setupActions()
{
//file
	actionCollection()->addAction( KStandardAction::Open, "open", this, SLOT( open() ) );

	KAction *refreshAction = new KAction( KIcon( "view-refresh" ), i18n( "Refresh" ), this );
	refreshAction->setShortcut( Qt::Key_F5 );
	actionCollection()->addAction( "refresh", refreshAction );
	connect( refreshAction, SIGNAL( triggered( bool ) ), SLOT( refresh() ) );

	KAction *viewAction = new KAction( KIcon( "view-list-text" ), i18n( "View Files" ), this );
	viewAction->setShortcut( Qt::CTRL + Qt::Key_F );
	actionCollection()->addAction( "view", viewAction );
	connect( viewAction, SIGNAL( triggered( bool ) ), SLOT( view() ) );

	actionCollection()->addAction( KStandardAction::Quit, "quit", this, SLOT( close() ) );
//actions
	KAction *addAction = new KAction( KIcon( "list-add" ), i18n( "Add Entry" ), this );
	addAction->setShortcut( Qt::CTRL + Qt::SHIFT + Qt::Key_A );
	actionCollection()->addAction( "add", addAction );
	connect( addAction, SIGNAL( triggered( bool ) ), SLOT( add() ) );

	KAction *editAction = new KAction( KIcon( "document-properties" ), i18n( "Edit Entry" ), this );
	editAction->setShortcut( Qt::CTRL + Qt::SHIFT + Qt::Key_E );
	actionCollection()->addAction( "edit", editAction );
	connect( editAction, SIGNAL( triggered( bool ) ), SLOT( edit() ) );

	KAction *copyAction = new KAction( KIcon( "edit-copy" ), i18n( "Copy Entry" ), this );
	copyAction->setShortcut( Qt::CTRL + Qt::SHIFT + Qt::Key_C );
	actionCollection()->addAction( "copy", copyAction );
	connect( copyAction, SIGNAL( triggered( bool ) ), SLOT( copy() ) );

	KAction *removeAction = new KAction( KIcon( "list-remove" ), i18n( "Remove Entry" ), this );
	removeAction->setShortcut( Qt::Key_Delete );
	actionCollection()->addAction( "remove", removeAction );
	connect( removeAction, SIGNAL( triggered( bool ) ), SLOT( remove() ) );

	KAction *moveTopAction = new KAction( KIcon( "arrow-up-double" ), i18n( "Move To Top" ), this );
	moveTopAction->setShortcut( Qt::CTRL + Qt::SHIFT + Qt::Key_Up );
	actionCollection()->addAction( "moveTop", moveTopAction );
	connect( moveTopAction, SIGNAL( triggered( bool ) ), SLOT( moveTop() ) );

	KAction *moveUpAction = new KAction( KIcon( "arrow-up" ), i18n( "Move Up" ), this );
	moveUpAction->setShortcut( Qt::CTRL + Qt::Key_Up );
	actionCollection()->addAction( "moveUp", moveUpAction );
	connect( moveUpAction, SIGNAL( triggered( bool ) ), SLOT( moveUp() ) );

	KAction *moveDownAction = new KAction( KIcon( "arrow-down" ), i18n( "Move Down" ), this );
	moveDownAction->setShortcut( Qt::CTRL + Qt::Key_Down );
	actionCollection()->addAction( "moveDown", moveDownAction );
	connect( moveDownAction, SIGNAL( triggered( bool ) ), SLOT( moveDown() ) );

	KAction *moveBottomAction = new KAction( KIcon( "arrow-down-double" ), i18n( "Move To Bottom" ), this );
	moveBottomAction->setShortcut( Qt::CTRL + Qt::SHIFT + Qt::Key_Down );
	actionCollection()->addAction( "moveBottom", moveBottomAction );
	connect( moveBottomAction, SIGNAL( triggered( bool ) ), SLOT( moveBottom() ) );
//tools
	KAction *backupAction = new KAction( KIcon( "document-save-all" ), i18n( "Backup" ), this );
	backupAction->setShortcut( Qt::CTRL + Qt::Key_B );
	actionCollection()->addAction( "backup", backupAction );
	connect( backupAction, SIGNAL( triggered( bool ) ), SLOT( backup() ) );

	KAction *restoreBackupAction = new KAction( KIcon( "edit-undo" ), i18n( "Restore" ), this );
	restoreBackupAction->setShortcut( Qt::CTRL + Qt::Key_R );
	actionCollection()->addAction( "restoreBackup", restoreBackupAction );
	connect( restoreBackupAction, SIGNAL( triggered( bool ) ), SLOT( restoreBackup() ) );

	KAction *deleteBackupAction = new KAction( KIcon( "edit-delete" ), i18n( "Delete Backup" ), this );
	deleteBackupAction->setShortcut( Qt::CTRL + Qt::Key_D );
	actionCollection()->addAction( "deleteBackup", deleteBackupAction );
	connect( deleteBackupAction, SIGNAL( triggered( bool ) ), SLOT( deleteBackup() ) );
//settings
	KStandardAction::showMenubar( this, SLOT( toggleMenubar() ), actionCollection() );

	KStandardAction::preferences( this, SLOT( preferences() ), actionCollection() );
//other
	setDefaultAction = new KAction( i18n( "Set As Default" ), menu_entry );
	connect( setDefaultAction, SIGNAL( triggered( bool ) ), SLOT( setDefault() ) );

	setFallbackAction = new KAction( i18n( "Set As Fallback" ), menu_entry );
	connect( setFallbackAction, SIGNAL( triggered( bool ) ), SLOT( setFallback() ) );
}
void KGRUBEditor::setupMenus()
{
	menu_entry->addAction( setDefaultAction );
	menu_entry->addAction( setFallbackAction );
}
void KGRUBEditor::setupConnections()
{
	connect( ui_entries.klistwidget_os, SIGNAL( currentRowChanged( int ) ), SLOT( entryRowChanged( int ) ) );
	connect( ui_entries.klistwidget_os, SIGNAL( customContextMenuRequested( QPoint ) ), SLOT( entryContextMenu() ) );
	connect( ui_entries.klistwidget_os, SIGNAL( doubleClicked( QListWidgetItem *, const QPoint & ) ), SLOT( edit() ) );

	connect( timer_blink, SIGNAL( timeout() ), ui_color.colorPreview, SLOT( update() ) );

	//general
	connect( ui_general.checkBox_hiddenMenu, SIGNAL( clicked() ), SLOT( checkBox_hiddenMenu_clicked() ) );
	connect( ui_general.checkBox_timeout, SIGNAL( clicked() ), SLOT( checkBox_timeout_clicked() ) );
	connect( ui_general.kintspinbox_timeout, SIGNAL( editingFinished() ), SLOT( updateTimeout() ) );
	connect( ui_general.checkBox_default, SIGNAL( clicked() ), SLOT( checkBox_default_clicked() ) );
	connect( ui_general.comboBox_default, SIGNAL( activated( int ) ), SLOT( updateDefault( int ) ) );
	connect( ui_general.checkBox_fallback, SIGNAL( clicked() ), SLOT( checkBox_fallback_clicked() ) );
	connect( ui_general.comboBox_fallback, SIGNAL( activated( int ) ), SLOT( updateFallback( int ) ) );
	//password
	connect( ui_password.kpushbutton_create, SIGNAL( pressed() ), SLOT( createPassword() ) );
	connect( ui_password.kpushbutton_edit, SIGNAL( pressed() ), SLOT( editPassword() ) );
	connect( ui_password.kpushbutton_delete, SIGNAL( pressed() ), SLOT( deletePassword() ) );
	//background
	connect( ui_background.kurlrequester_splash, SIGNAL( urlSelected( const KUrl ) ), SLOT( convertSplashImagePath( const KUrl ) ) );
	connect( ui_background.kpushbutton_create, SIGNAL( pressed() ), SLOT( createSplashImage() ) );
	connect( ui_background.kpushbutton_preview, SIGNAL( pressed() ), SLOT( previewSplashImage() ) );
	connect( ui_background.kpushbutton_clear_splash, SIGNAL( pressed() ), SLOT( clearSplashImagePath() ) );
	connect( ui_background.kurlrequester_gfxmenu, SIGNAL( urlSelected( const KUrl ) ), SLOT( convertGfxmenuPath( const KUrl ) ) );
	connect( ui_background.kpushbutton_clear_gfxmenu, SIGNAL( pressed() ), SLOT( clearGfxmenuPath() ) );
	//color
	connect( ui_color.checkBox_normal, SIGNAL( clicked( bool ) ), SLOT( enableNormal( bool ) ) );
	connect( ui_color.comboBox_normalBackground, SIGNAL( activated( int ) ), SLOT( updateColors() ) );
	connect( ui_color.comboBox_normalForeground, SIGNAL( activated( int ) ), SLOT( updateColors() ) );
	connect( ui_color.checkBox_normalBlink, SIGNAL( clicked() ), SLOT( blinkReset() ) );
	connect( ui_color.checkBox_highlighted, SIGNAL( clicked( bool ) ), SLOT( enableHighlight( bool ) ) );
	connect( ui_color.comboBox_highlightedBackground, SIGNAL( activated( int ) ), SLOT( updateColors() ) );
	connect( ui_color.comboBox_highlightedForeground, SIGNAL( activated( int ) ), SLOT( updateColors() ) );
	connect( ui_color.checkBox_highlightedBlink, SIGNAL( clicked() ), SLOT( blinkReset() ) );
	//maps
	connect( ui_map.kpushbutton_add, SIGNAL( pressed() ), SLOT( addMap() ) );
	connect( ui_map.kpushbutton_edit, SIGNAL( pressed() ), SLOT( editMap() ) );
	connect( ui_map.kpushbutton_copy, SIGNAL( pressed() ), SLOT( copyMap() ) );
	connect( ui_map.kpushbutton_remove, SIGNAL( pressed() ), SLOT( removeMap() ) );
}

//GUI FUNCTIONS
void KGRUBEditor::showInput( const int entryListRow, const int mapListRow, QWidget* focusWidget )
{
//GRUB ENTRIES
	for ( int i = 0; i < KGRUBEditor::GrubEntries.size(); i++ )
	{
		ui_entries.klistwidget_os->addItem( KGRUBEditor::GrubEntries.at( i ).title() );
		ui_general.comboBox_default->addItem( KGRUBEditor::GrubEntries.at( i ).title() );
		ui_general.comboBox_fallback->addItem( KGRUBEditor::GrubEntries.at( i ).title() );

		if ( GrubSettings._default() == i )
		{
			ui_entries.klistwidget_os->item( i )->setBackground( QBrush( Settings::defaultcolor() ) );
			ui_entries.klistwidget_os->item( i )->setText( ui_entries.klistwidget_os->item( i )->text() + " " + i18n( "(Default Entry)" ) );
		}
		else if ( GrubSettings._default() == -2 )
		{
			if ( KGRUBEditor::GrubEntries.at( i ).saveDefault() )
			{
				ui_entries.klistwidget_os->item( i )->setBackground( QBrush( Settings::defaultcolor() ) );
				ui_entries.klistwidget_os->item( i )->setText( ui_entries.klistwidget_os->item( i )->text() + " " + i18n( "(SaveDefault Entry)" ) );
			}
		}
		if ( GrubSettings.fallback() == i )
			ui_entries.klistwidget_os->item( i )->setText( ui_entries.klistwidget_os->item( i )->text() + " " + i18n( "(Fallback Entry)" ) );

		if ( ui_entries.klistwidget_os->item( i )->text().contains( "arch", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/arch.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "debian", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/debian.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "fedora", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/fedora.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "gentoo", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/gentoo.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "kubuntu", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/kubuntu.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "mandriva", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/mandriva.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "mepis", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/mepis.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "mint", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/mint.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "pclinuxos", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/pclinuxos.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "sabayon", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/sabayon.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "slackware", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/slackware.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "suse", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/suse.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "ubuntu", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/ubuntu.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "windows", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/windows.png" ) ) );
		else if ( ui_entries.klistwidget_os->item( i )->text().contains( "xubuntu", Qt::CaseInsensitive ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( KStandardDirs::locate( "appdata", "icons/xubuntu.png" ) ) );
		else if ( !ui_entries.klistwidget_os->item( i )->text().contains( "Other Operating Systems", Qt::CaseInsensitive ) && !ui_entries.klistwidget_os->item( i )->text().contains( QRegExp( "\\-+" ) ) )
			ui_entries.klistwidget_os->item( i )->setIcon( KIcon( "unknown" ) );
	}
//GRUB SETTINGS
	//hiddenmenu
	ui_general.checkBox_hiddenMenu->setChecked( GrubSettings.hiddenMenu() );
	//timeout
	ui_general.checkBox_timeout->setChecked( GrubSettings.timeout() != -1 );
	ui_general.kintspinbox_timeout->setEnabled( GrubSettings.timeout() != -1 );
	if ( GrubSettings.timeout() != -1 )
		ui_general.kintspinbox_timeout->setValue( GrubSettings.timeout() );
	//default
	ui_general.comboBox_default->addItem( i18n( "SaveDefault" ) );
	if ( GrubSettings._default() == -1 )
	{
		ui_general.checkBox_default->setCheckState( Qt::Unchecked );
		ui_general.comboBox_default->setEnabled( false );
		ui_general.comboBox_default->setCurrentIndex( 0 );
	}
	else if ( GrubSettings._default() == -2 )
	{
		ui_general.checkBox_default->setCheckState( Qt::PartiallyChecked );
		ui_general.comboBox_default->setEnabled( false );
		ui_general.comboBox_default->setCurrentIndex( KGRUBEditor::GrubEntries.size() );
	}
	else
	{
		ui_general.checkBox_default->setCheckState( Qt::Checked );
		ui_general.comboBox_default->setEnabled( true );
		ui_general.comboBox_default->removeItem( KGRUBEditor::GrubEntries.size() );
		ui_general.comboBox_default->setCurrentIndex( GrubSettings._default() );
	}
	//fallback
	ui_general.checkBox_fallback->setChecked( GrubSettings.fallback() != -1 );
	ui_general.comboBox_fallback->setEnabled( GrubSettings.fallback() != -1 );
	if ( GrubSettings.fallback() != -1 )
		ui_general.comboBox_fallback->setCurrentIndex( GrubSettings.fallback() );
	//password
	ui_password.checkBox_md5->setChecked( GrubSettings.password().md5crypted() );
	ui_password.klineedit_password->setText( GrubSettings.password().password().isEmpty() ? i18n( "<none>" ) : GrubSettings.password().password() );
	ui_password.klineedit_menu->setText( GrubSettings.password().configFile().isEmpty() ? i18n( "<none>" ) : GrubSettings.password().configFile() );

	ui_password.kpushbutton_edit->setDisabled( GrubSettings.password().isEmpty() );
	ui_password.kpushbutton_delete->setDisabled( GrubSettings.password().isEmpty() );
	//background
	ui_background.kurlrequester_splash->setPath( GrubSettings.splashImage().isEmpty() ? i18n( "<none>" ) : GrubSettings.splashImage() );
	ui_background.kurlrequester_gfxmenu->setPath( GrubSettings.gfxMenu().isEmpty() ? i18n( "<none>" ) : GrubSettings.gfxMenu() );
	//color
	ui_color.colorPreview->setColor( GrubSettings.color() );
	ui_color.colorPreview->setEntries( KGRUBEditor::GrubEntries );
	ui_color.colorPreview->update();
	if ( GrubSettings.color().blinkNormal() || GrubSettings.color().blinkHighlighted() )
		if ( !timer_blink->isActive() )
			timer_blink->start( 500 );
	else
		if ( timer_blink->isActive() )
			timer_blink->stop();

	ui_color.checkBox_normal->setChecked( GrubSettings.color().normalIsEnabled() );
	ui_color.comboBox_normalBackground->setEnabled( GrubSettings.color().normalIsEnabled() );
	ui_color.comboBox_normalForeground->setEnabled( GrubSettings.color().normalIsEnabled() );
	ui_color.checkBox_normalBlink->setEnabled( GrubSettings.color().normalIsEnabled() );
	ui_color.checkBox_highlighted->setEnabled( GrubSettings.color().normalIsEnabled() );
	if ( GrubSettings.color().normalIsEnabled() )
	{
		ui_color.comboBox_normalBackground->setCurrentIndex( colorMap.key( GrubSettings.color().mapOptimisedColor( GrubSettings.color().normalBackground() ) ) );
		ui_color.comboBox_normalForeground->setCurrentIndex( colorMap.key( GrubSettings.color().mapOptimisedColor( GrubSettings.color().normalForeground() ), 15 ) );
		ui_color.checkBox_normalBlink->setChecked( GrubSettings.color().blinkNormal() );
	}
	ui_color.checkBox_highlighted->setChecked( GrubSettings.color().highlightedIsEnabled() );
	ui_color.comboBox_highlightedBackground->setEnabled( GrubSettings.color().highlightedIsEnabled() );
	ui_color.comboBox_highlightedForeground->setEnabled( GrubSettings.color().highlightedIsEnabled() );
	ui_color.checkBox_highlightedBlink->setEnabled( GrubSettings.color().highlightedIsEnabled() );
	if ( GrubSettings.color().highlightedIsEnabled() )
	{
		ui_color.comboBox_highlightedBackground->setCurrentIndex( colorMap.key( GrubSettings.color().mapOptimisedColor( GrubSettings.color().highlightedBackground() ) ) );
		ui_color.comboBox_highlightedForeground->setCurrentIndex( colorMap.key( GrubSettings.color().mapOptimisedColor( GrubSettings.color().highlightedForeground() ), 15 ) );
		ui_color.checkBox_highlightedBlink->setChecked( GrubSettings.color().blinkHighlighted() );
	}
	//maps
	if ( !GrubSettings.maps().isEmpty() )
	{
		for ( int i = 0; i < GrubSettings.maps().size() ; i++ )
		{
			ui_map.klistwidget_maps->addItem( GrubSettings.maps().at( i ).result() );
		}
	}

	ui_map.kpushbutton_edit->setDisabled( GrubSettings.maps().isEmpty() );
	ui_map.kpushbutton_copy->setDisabled( GrubSettings.maps().isEmpty() );
	ui_map.kpushbutton_remove->setDisabled( GrubSettings.maps().isEmpty() );
//GRUB DEVICE NAMING
	QTreeWidgetItem *treeWidgetItem;
	for ( int i = 0; i < KGRUBEditor::DeviceList.size() ; i++ )
	{
		treeWidgetItem = new QTreeWidgetItem( ui_devices.treeWidget_device, QStringList() 
		<< KGRUBEditor::DeviceList.at( i ).mountPoint() 
		<< KGRUBEditor::DeviceList.at( i ).partition() 
		<< KGRUBEditor::DeviceList.at( i ).grubPartition() 
		<< KGRUBEditor::DeviceList.at( i ).device() 
		<< KGRUBEditor::DeviceList.at( i ).grubDevice() 
		<< KGRUBEditor::DeviceList.at( i ).uuid() );

		treeWidgetItem->setFlags( Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled );
	}

	ui_entries.klistwidget_os->setCurrentRow( entryListRow );
	ui_map.klistwidget_maps->setCurrentRow( mapListRow );
	if ( focusWidget )
		focusWidget->setFocus();
}
void KGRUBEditor::gotoMainView()
{
	tabWidget->setCurrentIndex( 0 );
	ui_entries.klistwidget_os->setFocus();
}
//OTHER FUNCTIONS
void KGRUBEditor::updateFile()
{
	FileTransactions::fileOutput( Settings::menulst(), &GrubSettings, &KGRUBEditor::GrubEntries );
	label_update->setText( i18n( " File Updated " ) );
	QTimer::singleShot( 1000, label_update, SLOT( clear() ) );
}

void KGRUBEditor::loadSettings()
{
	refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow() );
}
void KGRUBEditor::setDefault()
{
	GrubSettings.setDefault( ui_entries.klistwidget_os->currentRow() );
	updateFile();
	refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow() );
}
void KGRUBEditor::setFallback()
{
	GrubSettings.setFallback( ui_entries.klistwidget_os->currentRow() );
	updateFile();
	refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow() );
}
//EVENTs
void KGRUBEditor::entryContextMenu()
{
	if ( !ui_entries.klistwidget_os->selectedItems().isEmpty() )
	{
		setDefaultAction->setEnabled( GrubSettings._default() != ui_entries.klistwidget_os->currentRow() );
		setFallbackAction->setEnabled( GrubSettings.fallback() != ui_entries.klistwidget_os->currentRow() );
		menu_entry->exec( QCursor::pos() );
	}
}
void KGRUBEditor::entryRowChanged( const int row )
{
	if ( !ui_entries.klistwidget_os->currentItem() )
		return;

	ui_entries.treeWidget_property->clear();
	QTreeWidgetItem *treeWidgetItem;
	if ( !KGRUBEditor::GrubEntries.at( row ).title().isEmpty() )
	{
		treeWidgetItem = new QTreeWidgetItem( ui_entries.treeWidget_property, QStringList() << i18n( "Title" ) << KGRUBEditor::GrubEntries.at( row ).title() );
	}
	if ( KGRUBEditor::GrubEntries.at( row ).lock() )
	{
		treeWidgetItem = new QTreeWidgetItem( ui_entries.treeWidget_property, QStringList() << i18n( "Lock" ) );
		treeWidgetItem->setCheckState( 1, Qt::Checked );
		treeWidgetItem->setFlags( Qt::ItemIsEnabled | Qt::ItemIsSelectable );
	}
	if ( !KGRUBEditor::GrubEntries.at( row ).password().isEmpty() )
	{
		treeWidgetItem = new QTreeWidgetItem( ui_entries.treeWidget_property, QStringList() << i18n( "Password" ) << KGRUBEditor::GrubEntries.at( row ).password().result() );
	}
	if ( !KGRUBEditor::GrubEntries.at( row ).root().isEmpty() )
	{
		treeWidgetItem = new QTreeWidgetItem( ui_entries.treeWidget_property, QStringList() << i18n( "Root" ) << KGRUBEditor::GrubEntries.at( row ).root() );
	}
	if ( !KGRUBEditor::GrubEntries.at( row ).kernel().isEmpty() )
	{
		treeWidgetItem = new QTreeWidgetItem( ui_entries.treeWidget_property, QStringList() << i18n( "Kernel" ) << KGRUBEditor::GrubEntries.at( row ).kernel().result() );
	}
	if ( !KGRUBEditor::GrubEntries.at( row ).initrd().isEmpty() )
	{
		treeWidgetItem = new QTreeWidgetItem( ui_entries.treeWidget_property, QStringList() << i18n( "Initrd" ) << KGRUBEditor::GrubEntries.at( row ).initrd() );
	}
	if ( !KGRUBEditor::GrubEntries.at( row ).color().isEmpty() )
	{
		treeWidgetItem = new QTreeWidgetItem( ui_entries.treeWidget_property, QStringList() << i18n( "Color" ) << KGRUBEditor::GrubEntries.at( row ).color().result() );
	}
	if ( !KGRUBEditor::GrubEntries.at( row ).maps().isEmpty() )
	{
		for ( int i = 0; i < KGRUBEditor::GrubEntries.at( row ).maps().size() ; i++ )
		{
			treeWidgetItem = new QTreeWidgetItem( ui_entries.treeWidget_property, QStringList() << i18n( "Map" ) << KGRUBEditor::GrubEntries.at( row ).maps().at( i ).result() );
		}
	}
	if ( !KGRUBEditor::GrubEntries.at( row ).chainLoader().isEmpty() )
	{
		treeWidgetItem = new QTreeWidgetItem( ui_entries.treeWidget_property, QStringList() << i18n( "Chainloader" ) << KGRUBEditor::GrubEntries.at( row ).chainLoader() );
	}
	if ( KGRUBEditor::GrubEntries.at( row ).saveDefault() )
	{
		treeWidgetItem = new QTreeWidgetItem( ui_entries.treeWidget_property, QStringList() << i18n( "SaveDefault" ) );
		treeWidgetItem->setCheckState( 1, Qt::Checked );
		treeWidgetItem->setFlags( Qt::ItemIsEnabled | Qt::ItemIsSelectable );
	}
	if ( KGRUBEditor::GrubEntries.at( row ).makeActive() )
	{
		treeWidgetItem = new QTreeWidgetItem( ui_entries.treeWidget_property, QStringList() << i18n( "MakeActive" ) );
		treeWidgetItem->setCheckState( 1, Qt::Checked );
		treeWidgetItem->setFlags( Qt::ItemIsEnabled | Qt::ItemIsSelectable );
	}
}

void KGRUBEditor::checkBox_hiddenMenu_clicked()
{
	GrubSettings.setHiddenMenu( ui_general.checkBox_hiddenMenu->isChecked() );
	updateFile();
}

void KGRUBEditor::checkBox_timeout_clicked()
{
	ui_general.kintspinbox_timeout->setEnabled( ui_general.checkBox_timeout->isChecked() );
	updateTimeout();
}
void KGRUBEditor::updateTimeout()
{
	( ui_general.kintspinbox_timeout->isEnabled() ? GrubSettings.setTimeout( ui_general.kintspinbox_timeout->value() ) : GrubSettings.setTimeout( -1 ) );
	updateFile();
}

void KGRUBEditor::checkBox_default_clicked()
{
	if ( ui_general.checkBox_default->checkState() == Qt::Unchecked )
	{
		ui_general.comboBox_default->setEnabled( false );
		updateDefault( -1 );
	}
	else if ( ui_general.checkBox_default->checkState() == Qt::PartiallyChecked )
	{
		ui_general.comboBox_default->setEnabled( false );
		updateDefault( -2 );
	}
	else
	{
		ui_general.comboBox_default->setEnabled( true );
		updateDefault( 0 );
	}
}
void KGRUBEditor::updateDefault( const int value )
{
	GrubSettings.setDefault( value );
	updateFile();
	refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow() );
}

void KGRUBEditor::checkBox_fallback_clicked()
{
	ui_general.comboBox_fallback->setEnabled( ui_general.checkBox_fallback->isChecked() );
	( ui_general.checkBox_fallback->isChecked() ? updateFallback( ui_general.comboBox_fallback->currentIndex() ) : updateFallback( -1 ) );
}
void KGRUBEditor::updateFallback( const int value )
{
	GrubSettings.setFallback( value );
	updateFile();
	refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow() );
}

void KGRUBEditor::enableNormal( const bool state )
{
	ui_color.comboBox_normalBackground->setEnabled( state );
	ui_color.comboBox_normalForeground->setEnabled( state );
	ui_color.checkBox_normalBlink->setEnabled( state );
	ui_color.checkBox_highlighted->setEnabled( state );
	updateColors();
}
void KGRUBEditor::enableHighlight( const bool state )
{
	ui_color.comboBox_highlightedBackground->setEnabled( state );
	ui_color.comboBox_highlightedForeground->setEnabled( state );
	ui_color.checkBox_highlightedBlink->setEnabled( state );
	updateColors();
}
void KGRUBEditor::blinkReset()
{
	ui_color.colorPreview->resetBlinking();
	updateColors();
}
void KGRUBEditor::updateColors()
{
	GRUB::ComplexCommand::Color m_color;
	if ( ui_color.checkBox_normal->isChecked() )
	{
		m_color.setBlinkNormal( ui_color.checkBox_normalBlink->isChecked() );
		m_color.setNormalBackground( colorMap.value( ui_color.comboBox_normalBackground->currentIndex() ).toLower() );
		m_color.setNormalForeground( colorMap.value( ui_color.comboBox_normalForeground->currentIndex() ).toLower() );

		if ( ui_color.checkBox_highlighted->isChecked() )
		{
			m_color.setBlinkHighlighted( ui_color.checkBox_highlightedBlink->isChecked() );
			m_color.setHighlightedBackground( colorMap.value( ui_color.comboBox_highlightedBackground->currentIndex() ).toLower() );
			m_color.setHighlightedForeground( colorMap.value( ui_color.comboBox_highlightedForeground->currentIndex() ).toLower() );
		}
	}
	GrubSettings.setColor( m_color );
	updateFile();
	refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow(), ui_color.colorPreview );
}

void KGRUBEditor::createPassword()
{
	GRUB::ComplexCommand::Password tmp_password;
	PasswordAssistant *passwordAssistant = new PasswordAssistant( &tmp_password, this );
	if ( passwordAssistant->exec() == QDialog::Accepted )
	{
		GrubSettings.setPassword( tmp_password );
		updateFile();
		refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow() );
	}
}
void KGRUBEditor::editPassword()
{
	GRUB::ComplexCommand::Password tmp_password = GrubSettings.password();
	PasswordAssistant *passwordAssistant = new PasswordAssistant( &tmp_password, this );
	if ( passwordAssistant->exec() == QDialog::Accepted )
	{
		GrubSettings.setPassword( tmp_password );
		updateFile();
		refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow() );
	}
}
void KGRUBEditor::deletePassword()
{
	if ( KMessageBox::questionYesNo( this, i18n( "Are you sure you want to delete the password?" ), i18n( "Confirmation" ) ) == KMessageBox::Yes )
	{
		GrubSettings.clearPassword();
		updateFile();
		refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow() );
	}
}

void KGRUBEditor::convertSplashImagePath( const KUrl path )
{
	if ( path.isEmpty() )
		return;

	QString realSplashImagePath = FileTransactions::convertToGRUBPath( path.path(), KGRUBEditor::DeviceList );

	ui_background.kurlrequester_splash->setPath( realSplashImagePath );
	GrubSettings.setSplashImage( realSplashImagePath );
	updateFile();
}
void KGRUBEditor::createSplashImage()
{
	KFileDialog dlgOpen( KUrl(), QString(), this );
	dlgOpen.setCaption( i18n( "Open Image To Convert" ) );
	dlgOpen.setOperationMode( KFileDialog::Opening );
	if ( !dlgOpen.exec() )
		return;
	KUrl image = dlgOpen.selectedUrl();

	KFileDialog dlgSave( KUrl(), QString( "*.xpm.gz" ), this );
	dlgSave.setCaption( i18n( "Save GRUB Splash Image" ) );
	dlgSave.setOperationMode( KFileDialog::Saving );
	if ( !dlgSave.exec() )
		return;
	KUrl splash = dlgSave.selectedUrl();

	QProcess convert;
	convert.start( "convert", QStringList() << image.path() << "-resize" << "640x480" << "-colors" << "14" << splash.path().remove( ".gz" ) );
	convert.waitForFinished();

	QProcess gzip;
	gzip.start( "gzip", QStringList() << "-f" << splash.path().remove( ".gz" ) );
	gzip.waitForFinished();

	convertSplashImagePath( splash );
}
void KGRUBEditor::previewSplashImage()
{
	if ( ui_background.kurlrequester_splash->url().path() == i18n( "<none>" ) )
		return;

	QString file = FileTransactions::convertToGenericPath( ui_background.kurlrequester_splash->url().path(), KGRUBEditor::DeviceList );

	if ( QFile::exists( "/tmp/qgrubeditor.xpm" ) )
		QFile::remove( "/tmp/qgrubeditor.xpm" );
	if ( QFile::exists( "/tmp/qgrubeditor.xpm.gz" ) )
		QFile::remove( "/tmp/qgrubeditor.xpm.gz" );
	if ( QFile::copy( file, "/tmp/qgrubeditor.xpm.gz" ) )
	{
		QProcess gzip;
		gzip.start( "gzip", QStringList() << "-d" << "/tmp/qgrubeditor.xpm.gz" );
		gzip.waitForFinished();

		QSplashScreen *splash = new QSplashScreen( QPixmap( "/tmp/qgrubeditor.xpm" ) );
		splash->show();
	}
}
void KGRUBEditor::clearSplashImagePath()
{
	GrubSettings.clearSplashImage();
	updateFile();
	refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow(), ui_background.kurlrequester_splash );
}
void KGRUBEditor::convertGfxmenuPath( const KUrl path )
{
	if ( path.isEmpty() )
		return;

	QString realGfxmenuPath = FileTransactions::convertToGRUBPath( path.path(), KGRUBEditor::DeviceList );

	ui_background.kurlrequester_gfxmenu->setPath( realGfxmenuPath );
	GrubSettings.setGfxMenu( realGfxmenuPath );
	updateFile();
}
void KGRUBEditor::clearGfxmenuPath()
{
	GrubSettings.clearGfxMenu();
	updateFile();
	refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow(), ui_background.kurlrequester_gfxmenu );
}

void KGRUBEditor::addMap()
{
	GRUB::ComplexCommand::Map tmp_map;
	MapAssistant *mapAssistant = new MapAssistant( &tmp_map, this );
	if ( mapAssistant->exec() == QDialog::Accepted )
	{
		GrubSettings.addMap( tmp_map );
		updateFile();
		refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->count(), ui_map.klistwidget_maps );
	}
}
void KGRUBEditor::editMap()
{
	if ( !ui_map.klistwidget_maps->currentItem() )
		return;

	GRUB::ComplexCommand::Map tmp_map = GrubSettings.maps().at( ui_map.klistwidget_maps->currentRow() );
	MapAssistant *mapAssistant = new MapAssistant( &tmp_map, this );
	if ( mapAssistant->exec() == QDialog::Accepted )
	{
		GrubSettings.replaceMap( ui_map.klistwidget_maps->currentRow(), tmp_map );
		updateFile();
		refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow(), ui_map.klistwidget_maps );
	}
}
void KGRUBEditor::copyMap()
{
	if ( !ui_map.klistwidget_maps->currentItem() )
		return;

	GrubSettings.copyMap( ui_map.klistwidget_maps->currentRow() );
	updateFile();
	refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->count(), ui_map.klistwidget_maps );
}
void KGRUBEditor::removeMap()
{
	if ( !ui_map.klistwidget_maps->currentItem() )
		return;

	if ( KMessageBox::questionYesNo( this, i18n( "Are you sure you want to remove this map?" ), i18n( "Confirmation" ) ) == KMessageBox::Yes )
	{
		GrubSettings.removeMap( ui_map.klistwidget_maps->currentRow() );
		updateFile();
		refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow() - 1, ui_map.klistwidget_maps );
	}
}

//ACTIONS
//file
void KGRUBEditor::open()
{
	KUrl menulst = KFileDialog::getOpenUrl( Settings::menulst() );
	if ( !menulst.isEmpty() )
	{
		Settings::setMenulst( menulst );
		Settings::self()->writeConfig();
		refresh( 0, 0, ui_entries.klistwidget_os );
	}
}
void KGRUBEditor::refresh( int entryListRow, int mapListRow, QWidget* focusWidget )
{
	FileTransactions::fileInput( Settings::menulst(), Settings::devicemap(), &GrubSettings, &KGRUBEditor::GrubEntries, &KGRUBEditor::DeviceList );

	if ( entryListRow < 0 || entryListRow >= KGRUBEditor::GrubEntries.size() )
		( ui_entries.klistwidget_os->currentRow() != -1 ? entryListRow = ui_entries.klistwidget_os->currentRow() : entryListRow = 0 );
	if ( mapListRow < 0 || mapListRow >= GrubSettings.maps().size() )
		( ui_map.klistwidget_maps->currentRow() != -1 ? mapListRow = ui_map.klistwidget_maps->currentRow() : mapListRow = 0 );

	ui_entries.klistwidget_os->clear();
	ui_entries.treeWidget_property->clear();
	ui_general.comboBox_default->clear();
	ui_general.comboBox_fallback->clear();
	ui_map.klistwidget_maps->clear();
	ui_devices.treeWidget_device->clear();

	showInput( entryListRow, mapListRow, focusWidget );

	if ( Settings::autobackup() && !QFile::exists( Settings::menulst().path() + "_original" ) )
		KIO::NetAccess::file_copy( Settings::menulst(), KUrl( Settings::menulst().path() + "_original" ) );

	label_status->setText( i18n( "File currently opened: %1", Settings::menulst().path() ) );
}
//actions
void KGRUBEditor::add()
{
	gotoMainView();

	GRUB::ConfigFile::Entry tmp_entry;
	EntryAssistant *entryAssistant = new EntryAssistant( &tmp_entry, this );
	if ( entryAssistant->exec() == QDialog::Accepted )
	{
		KGRUBEditor::GrubEntries.append( tmp_entry );
		updateFile();
		refresh( ui_entries.klistwidget_os->count(), ui_map.klistwidget_maps->currentRow(), ui_entries.klistwidget_os );
	}
}
void KGRUBEditor::edit()
{
	if ( !ui_entries.klistwidget_os->currentItem() )
		return;

	gotoMainView();

	GRUB::ConfigFile::Entry tmp_entry = KGRUBEditor::GrubEntries.at( ui_entries.klistwidget_os->currentRow() );
	EntryAssistant *entryAssistant = new EntryAssistant( &tmp_entry, this );
	if ( entryAssistant->exec() == QDialog::Accepted )
	{
		KGRUBEditor::GrubEntries[ ui_entries.klistwidget_os->currentRow() ] = tmp_entry;
		updateFile();
		refresh( ui_entries.klistwidget_os->count(), ui_map.klistwidget_maps->currentRow(), ui_entries.klistwidget_os );
	}
}
void KGRUBEditor::copy()
{
	if ( !ui_entries.klistwidget_os->currentItem() )
		return;

	gotoMainView();

	KGRUBEditor::GrubEntries.append( KGRUBEditor::GrubEntries[ ui_entries.klistwidget_os->currentRow() ] );
	updateFile();
	refresh( ui_entries.klistwidget_os->count(), ui_map.klistwidget_maps->currentRow(), ui_entries.klistwidget_os );
}
void KGRUBEditor::remove()
{
	if ( !ui_entries.klistwidget_os->currentItem() )
		return;

	gotoMainView();

	switch( KMessageBox::questionYesNo( this, i18n( "Are you sure you want to remove the entry named '%1'?", KGRUBEditor::GrubEntries.at( ui_entries.klistwidget_os->currentRow() ).title() ), i18n( "Confirmation" ) ) )
	{
		case KMessageBox::Yes:
			KGRUBEditor::GrubEntries.remove( ui_entries.klistwidget_os->currentRow() );
			updateFile();
			refresh( ui_entries.klistwidget_os->currentRow() - 1, ui_map.klistwidget_maps->currentRow(), ui_entries.klistwidget_os );
		break;
		case KMessageBox::No:
			return;
		break;
	}
}

void KGRUBEditor::moveTop()
{
	if ( !ui_entries.klistwidget_os->currentItem() )
		return;

	gotoMainView();

	if ( ui_entries.klistwidget_os->currentRow() != 0 )
	{
		FileTransactions::moveEntry( ui_entries.klistwidget_os->currentRow(), 0, Settings::menulst(), &GrubSettings, &KGRUBEditor::GrubEntries );
		refresh( 0, ui_map.klistwidget_maps->currentRow(), ui_entries.klistwidget_os );
	}
}
void KGRUBEditor::moveUp()
{
	if ( !ui_entries.klistwidget_os->currentItem() )
		return;

	gotoMainView();

	if ( ui_entries.klistwidget_os->currentRow() != 0 )
	{
		FileTransactions::moveEntry( ui_entries.klistwidget_os->currentRow(), ui_entries.klistwidget_os->currentRow() - 1, Settings::menulst(), &GrubSettings, &KGRUBEditor::GrubEntries );
		refresh( ui_entries.klistwidget_os->currentRow() - 1, ui_map.klistwidget_maps->currentRow(), ui_entries.klistwidget_os );
	}
}
void KGRUBEditor::moveDown()
{
	if ( !ui_entries.klistwidget_os->currentItem() )
		return;

	gotoMainView();

	if ( ui_entries.klistwidget_os->currentRow() != ui_entries.klistwidget_os->count() - 1 )
	{
		FileTransactions::moveEntry( ui_entries.klistwidget_os->currentRow(), ui_entries.klistwidget_os->currentRow() + 1, Settings::menulst(), &GrubSettings, &KGRUBEditor::GrubEntries );
		refresh( ui_entries.klistwidget_os->currentRow() + 1, ui_map.klistwidget_maps->currentRow(), ui_entries.klistwidget_os );
	}
}
void KGRUBEditor::moveBottom()
{
	if ( !ui_entries.klistwidget_os->currentItem() )
		return;

	gotoMainView();

	if ( ui_entries.klistwidget_os->currentRow() != ui_entries.klistwidget_os->count() - 1 )
	{
		FileTransactions::moveEntry( ui_entries.klistwidget_os->currentRow(), ui_entries.klistwidget_os->count() - 1, Settings::menulst(), &GrubSettings, &KGRUBEditor::GrubEntries );
		refresh( ui_entries.klistwidget_os->count() - 1, ui_map.klistwidget_maps->currentRow(), ui_entries.klistwidget_os );
	}
}

void KGRUBEditor::backup()
{
	KUrl tmp_source = Settings::menulst(), tmp_target;
	BackupAssistant *backupAssistant = new BackupAssistant( &tmp_source, &tmp_target, this );
	if ( backupAssistant->exec() == QDialog::Accepted )
		FileTransactions::backup( tmp_source, tmp_target );
}
void KGRUBEditor::restoreBackup()
{
	KUrl tmp_source, tmp_target = Settings::menulst();
	BackupAssistant *backupAssistant = new BackupAssistant( &tmp_source, &tmp_target, this );
	if ( backupAssistant->exec() == QDialog::Accepted )
	{
		FileTransactions::restoreBackup( tmp_source, tmp_target );
		refresh( ui_entries.klistwidget_os->currentRow(), ui_map.klistwidget_maps->currentRow(), ui_entries.klistwidget_os );
	}
}
void KGRUBEditor::deleteBackup()
{
	KUrl backup = KFileDialog::getOpenUrl( Settings::menulst(), QString(), 0, i18n( "Select a backup to delete" ) );
	if ( !backup.isEmpty() )
		FileTransactions::deleteBackup( backup );
}
//tools
void KGRUBEditor::view()
{
	View *view = new View( this );
	view->exec();
}
//settings
void KGRUBEditor::toggleMenubar()
{
	menuBar()->setVisible( !menuBar()->isVisible() );
}
void KGRUBEditor::preferences()
{
	if ( KConfigDialog::showDialog( "Settings" ) )
		return;

	QWidget *generalSettings = new QWidget;
	ui_settings_general.setupUi( generalSettings );
	QWidget *pathsSettings = new QWidget;
	ui_settings_paths.setupUi( pathsSettings );

	KConfigDialog* dialog = new KConfigDialog( this, "Settings", Settings::self() );
	dialog->addPage( generalSettings, i18n( "General" ) )->setIcon( KIcon( "preferences-other" ) );
	dialog->addPage( pathsSettings, i18n( "Paths" ) )->setIcon( KIcon( "folder-txt" ) );
	connect( dialog, SIGNAL( settingsChanged( QString ) ), SLOT( loadSettings() ) );
	dialog->setAttribute( Qt::WA_DeleteOnClose );
	dialog->show();
}

#include "kgrubeditor.moc"
