/*============================================================================
 * BFT version and build information
 *============================================================================*/

/*
  This file is part of the "Base Functions and Types" library, intended to
  simplify and enhance portability, memory and I/O use for scientific codes.

  Copyright (C) 2004-2006  EDF

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/*-----------------------------------------------------------------------------*/

#include "bft_config_defs.h"

/*
 * Standard C library headers
 */

#include <string.h>

/*
 * Optional library and BFT headers
 */

#if defined(HAVE_ZLIB)
#include <zlib.h>
#endif

#include "bft_intl.h"
#include "bft_version.h"

/*-----------------------------------------------------------------------------*/

#ifdef __cplusplus
extern "C" {
#if 0
} /* Fake brace to force Emacs auto-indentation back to column 0 */
#endif
#endif /* __cplusplus */

/*-------------------------------------------------------------------------------
 * Local type definitions
 *-----------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------
 * Local static strings
 *-----------------------------------------------------------------------------*/

#define BFT_VERSION_STRING_LENGTH 128

static char _bft_version_build_cc_string[BFT_VERSION_STRING_LENGTH + 1] = "";

/*-----------------------------------------------------------------------------
 * Local function definitions
 *-----------------------------------------------------------------------------*/

/*============================================================================
 * Public function definitions
 *============================================================================*/

/*!
 * \brief Return BFT version version string
 *
 * \return Pointer to static string BFT version version string.
 */

const char *
bft_version(void)
{
  static const char buf[] = PACKAGE_VERSION;

  return buf;
}

/*!
 * \brief Indicate Zlib version available at run time.
 *
 * It may be useful to compare the Zlib version used at compile
 * and run time in case we use dynamic libraries.
 *
 * \return pointer to string indicating Zlib version in use, or NULL
 *         if Zlib support is not available.
 */

const char *
bft_version_zlib(void)
{
#if defined(HAVE_ZLIB)
  return zlibVersion();
#else
  return NULL;
#endif
}

/*!
 * \brief Indicate Zlib version available at compilation time.
 *
 * It may be useful to compare the Zlib version used at compile
 * and link time in case we use dynamic libraries.
 *
 * \return pointer to string indicating Zlib version at compilation, or NULL
 *         if Zlib support is not available.
 */

const char *
bft_version_build_zlib(void)
{
#if defined(HAVE_ZLIB)
#if defined(ZLIB_VERSION)
  return ZLIB_VERSION;
#else
  return _("unknown");
#endif
#else
  return NULL;
#endif
}

/*!
 * \brief Return some info on C compiler used to build the library if available.
 *
 * \return Pointer to static string containing compiler info extracted
 *         through compiler preprocessor macro definitions
 */

char *
bft_version_build_cc(void)
{
  char * cc_str = _bft_version_build_cc_string;

  /* Reset string */

  _bft_version_build_cc_string[0] = '\0';

  /* GCC */

#if defined(__GNUC__)

#  if defined(__cplusplus)
  strcat(cc_str, _("GNU C++ Compiler"));
#  else
  strcat(cc_str, _("GNU C Compiler"));
#  endif
#  if defined(__GNUC_MINOR__) && defined(__GNUC__PATCHLEVEL__)
  sprintf(cc_str+strlen(cc_str), _(", Version %d.%d.%d"),
          __GNUC__, __GNUC_MINOR__, __GNUC_PATCHLEVEL__);
#  endif
#  if defined(__i386)
  strcat(cc_str, _(", i386 Architecture"));
#  elif defined(__amd64__)
  strcat(cc_str, _(", amd64 Architecture"));
#  elif defined(__x86_64__)
  strcat(cc_str, _(", x86_64 Architecture"));
#  endif

  /* Alpha */

#elif defined(__DECC) && defined(__digital__)

  strcat(cc_str, _("Digital C Compiler"));
#  if defined(__alpha)
  strcat(cc_str, _(", Alpha Processor"));
#  endif
#  if defined(__arch64__)
  strcat(cc_str, _(", 64-Bit Architecture"));
#  endif


  /* IBM XL/C Compiler */

#elif defined(__xlc__)

  sprintf(cc_str, _("IBM XL/C compiler %d"), (int)__xlc__);

  /* IBM Blue Gene/L */
#  if defined(__blrts__)
  strcat(cc_str, _(" for Blue Gene/L"));
#  endif

  /* IBM Blue Gene/P */
#  if defined(__bgp__)
  strcat(cc_str, _(" for Blue Gene/P"));
#  endif

  /* Fujitsu VPP 5000 */

#elif defined(__uxpv__)

  strcat(cc_str, _("UXP/V Compiler"));
#  if defined(__uxpvp__)
  strcat(cc_str, _(", vectorizing"));
#  endif
#  if defined(__uxpch__)
  strcat(cc_str, _(", 64-Bit mode"));
#  endif

  /* IRIX MIPSpro C Compiler */

#elif defined(__sgi) && defined(sgi) && defined(__mips)

  strcat(cc_str, _("SGI MIPS Compiler"));
#  if defined(_ABI32__)
  strcat(cc_str, _(", 32-Bit mode"));
#  elif defined(_ABI64__)
  strcat(cc_str, _(", 64-Bit mode"));
#  endif

  /* HP C Compiler */

#elif defined(__HP_cc)

  strcat(cc_str, _("HP C Compiler"));
#  if defined(__LP32)
  strcat(cc_str, _(", 32-Bit mode"));
#  elif defined(__LP64)
  strcat(cc_str, _(", 64-Bit mode"));
#  endif

  /* Sun Workshop Compiler */

#elif defined(__SUNPRO_C)

  strcat(cc_str, _("Sun WorkShop C Compiler"));

#endif

#if defined(__STDC_VERSION__)
#  if (__STDC_VERSION__ == 199901L)
  strcat(cc_str, _(", C99"));
#  endif
#endif

  if (strlen(cc_str) > 0)
    return cc_str;
  else
    return NULL;
}

/*-----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif /* __cplusplus */
