#include "Tools.h"
#include "SinusTable.h"
#include "MathTools.h"
#include "SDLFrameRate.h"

#include "GameInterface.h"
#include "SoundInterface.h"

#include "ObjectRepository.h"

#include "AllObjects.h"
#include "ObjectVisitors.h"


//============================================================================
// Implementation of MovingObjectBase.h
//============================================================================

//----------------------------------------------------------------------------
MovingObjectBase::MovingObjectBase(const unsigned creatorId)
        : ObjectBase(0, creatorId)
{
}

//----------------------------------------------------------------------------
MovingObjectBase::~MovingObjectBase()
{
}



//============================================================================
// Implementation of ControllableObjectBase.h
//============================================================================

//----------------------------------------------------------------------------
ControllableObjectBase::ControllableObjectBase(const unsigned creatorId)
        : MovingObjectBase(creatorId)
{
    m_angle = SURFACES_ROTATION_STEPS / 2;
    m_fineAngle = 180.0;
    m_rotation = ROT_NONE;
    m_fuelCapacity = 0;
    m_fuel = 0;
    m_thrust = false;
    m_explode = false;

    setPosition(0, 0);
}

//----------------------------------------------------------------------------
ControllableObjectBase::~ControllableObjectBase()
{
}

//----------------------------------------------------------------------------
void ControllableObjectBase::initBottomPixels()
{
    Sint16 x = 0;
    Sint16 y;

    for (y = getSurface()->h-1; y >= 0; y--)
    {
        for (x = 0; x < getSurface()->w; x++)
        {
            if (SDL_TOOLS::getPixel(getSurface(), x, y))
            {
                goto out;
            }
        }
    }

  out:

    m_bottomPixels.x = x;
    m_bottomPixels.y = y;

    for (x = getSurface()->w-1; x >= 0; x--)
    {
        if (SDL_TOOLS::getPixel(getSurface(), x, y))
        {
            break;
        }
    }

    m_bottomPixels.w = x - m_bottomPixels.x + 1;
    m_bottomPixels.h = 1;
}


//----------------------------------------------------------------------------
void ControllableObjectBase::addFuel(unsigned increment)
{
    m_fuel += increment;
    if (m_fuel > m_fuelCapacity)
    {
        m_fuel = m_fuelCapacity;
    }
}

//----------------------------------------------------------------------------
bool ControllableObjectBase::decFuel(unsigned decrement)
{
    if (m_fuel < decrement)
    {
        m_fuel = 0;
        return true;
    }

    m_fuel -= decrement;
    return false;
}

//----------------------------------------------------------------------------
void ControllableObjectBase::setThrust(bool enable)
{
    if (enable && hasFuel())
    {
        SoundInterface::getInstance()->onShipThrustOn();
        m_thrust = true;
    }
    else
    {
        SoundInterface::getInstance()->onShipThrustOff();
        m_thrust = false;
    }
}

//----------------------------------------------------------------------------
void ControllableObjectBase::updateRotation()
{
    double angleStep =
        1.0 * getRotationDegreesPerSecond() / SDLFrameRate::getFrameRate();

    switch (m_rotation)
    {
    case ROT_NONE:
        break;

    case ROT_LEFT:
        m_fineAngle += angleStep;
        break;

    case ROT_RIGHT:
        m_fineAngle -= angleStep;
        break;

    case ROT_ALIGN:
        if (m_fineAngle < 180.0)
        {
            m_fineAngle += angleStep;
        }
        else if (m_fineAngle > 180.0)
        {
            m_fineAngle -= angleStep;
        }
        break;
    }

    // m_fineAngle must always be inside of [0 ... 360).
    if (m_fineAngle < 0.0)
    {
        m_fineAngle += 360.0;
    }
    else if (m_fineAngle >= 360.0)
    {
        m_fineAngle -= 360.0;
    }
    assert(m_fineAngle >= 0.0 && m_fineAngle < 360.0);

    m_angle = (unsigned)rint(m_fineAngle) / SURFACES_ROTATION_ANGLE;

    if (m_rotation == ROT_ALIGN && m_angle == SURFACES_ROTATION_STEPS / 2)
    {
        m_rotation = ROT_NONE;
    }
}

//----------------------------------------------------------------------------
void ControllableObjectBase::updateVelocityAndPosition()
{
    GameInterface *gameInterface = GameInterface::getInstance();

    // The old velocity values are needed.
    MATH_TOOLS::Vector oldFineVelocity = getFineVelocity();

    MATH_TOOLS::Vector fineAccel;

    // Angle-dependend acceleration due thrusting.
    if (isThrust() && hasFuel())
    {
        --m_fuel;

        fineAccel.setExponential(
            1.0 * getThrustForce() / getMass(), 1.0 * getAngle());

        ObjectRepository::getInstance()->addObject(new ThrustParticle(this));
    }

    // Apply the playground's gravity force.
    fineAccel.add(1.0 * gameInterface->getXGravity(getPosition()),
                  1.0 * gameInterface->getYGravity(getPosition()));

    // Friction will reduce the acceleration of this object.
    unsigned frictionScale = 32*32 / (getPosition().w * getPosition().h);
    MATH_TOOLS::Vector frictionSub(
        gameInterface->getXVelocity(getPosition()),
        gameInterface->getYVelocity(getPosition()));
    frictionSub -= oldFineVelocity;
    frictionSub *= 1.0 * gameInterface->getFriction(getPosition());
    frictionSub /= 1.0 * frictionScale * getMass();
    fineAccel += frictionSub;

    // Calculate and add the velocity increment.
    fineAccel /= 1.0 * SDLFrameRate::getFrameRate();
    getFineVelocity() += fineAccel;

    // To be exact, the average of the old and the new velocity values
    // must be used for the division with the frame rate.
    oldFineVelocity += getFineVelocity();
    oldFineVelocity /= 2.0 * SDLFrameRate::getFrameRate();
    getFinePosition() += oldFineVelocity;

    updatePosition();
}

//----------------------------------------------------------------------------
void ControllableObjectBase::updateSurface()
{
    setSurface(getRotationSurface(m_angle));
}



//============================================================================
// Implementation of Missile.h
//============================================================================

//----------------------------------------------------------------------------
Missile::DirectHeadingStrategy Missile::DirectHeadingStrategy::sm_instance;
Missile::SmartHeadingStrategy Missile::SmartHeadingStrategy::sm_instance;


//----------------------------------------------------------------------------
Missile::HeadingStrategy *Missile::HeadingStrategy::getStrategy(
    EnumHeadingStrategy strategy)
{
    switch (strategy)
    {
    case HS_DIRECT:
        return DirectHeadingStrategy::getInstance();
    case HS_SMART:
        return SmartHeadingStrategy::getInstance();
    }

    return NULL;
}


//----------------------------------------------------------------------------
Sint16 Missile::DirectHeadingStrategy::calculateHeadingAngle(
    Sint16 a, Sint16 angle, Sint16 gx, Sint16 gy) const
{
    // Always head directly to the player's ship.

    return angle;
}


//----------------------------------------------------------------------------
Sint16 Missile::SmartHeadingStrategy::calculateHeadingAngle(
    Sint16 a, Sint16 angle, Sint16 gx, Sint16 gy) const
{
    // Calculate the rotation such way, that it flies directly
    // to the player's ship, while compensating gavity.

    double ax = a * SinusTable::sin(angle);
    double ay = a * SinusTable::cos(angle);

    double aa = 1.0 * a * a;
    double axgy_m_aygx = ax*gy - ay*gx;
    double sqrtTerm = 1.0 - axgy_m_aygx * axgy_m_aygx / (aa*aa);

    if (sqrtTerm <= 0.0)
    {
        // If no or only one solution exists,
        // let the missile fly against gravity.
        return MATH_TOOLS::getAngle(-gx, -gy);
    }
    else
    {
        double term1 = (ax*gx + ay*gy) / aa;
        double term2 = sqrt(sqrtTerm);

        // There are two solutions for t: term1+term2 or term1-term2.
        // However, term1+term2 is always the right solution.
        // (Proof by lack of test cases ;-)
        double t = term1 + term2;
        int bx = -gx + (int)round(t*ax);
        int by = -gy + (int)round(t*ay);

        return MATH_TOOLS::getAngle(bx, by);
    }
}



//----------------------------------------------------------------------------
Missile::NormalWarheadStrategy Missile::NormalWarheadStrategy::sm_instance;
Missile::ConeBurstWarheadStrategy Missile::ConeBurstWarheadStrategy::sm_instance;
Missile::StarBurstWarheadStrategy Missile::StarBurstWarheadStrategy::sm_instance;


//----------------------------------------------------------------------------
Missile::WarheadStrategy *Missile::WarheadStrategy::getStrategy(
    EnumWarheadStrategy strategy)
{
    switch (strategy)
    {
    case WH_NORMAL:
        return NormalWarheadStrategy::getInstance();
    case WH_CONEBURST:
        return ConeBurstWarheadStrategy::getInstance();
    case WH_STARBURST:
        return StarBurstWarheadStrategy::getInstance();
    }

    return NULL;
}

//----------------------------------------------------------------------------
void Missile::WarheadStrategy::do_updateWarhead(Missile *missile)
{
    const Ship *player = GameInterface::getInstance()->getPlayerShip();
    if (!player)
    {
        return;
    }

    Sint16 mx, my;
    SDL_TOOLS::getCentre(missile->getPosition(), mx, my);

    Sint16 sx, sy;
    SDL_TOOLS::getCentre(player->getPosition(), sx, sy);

    // Calculate the square of the distance between the missile and the ship.
    int dx = mx - sx;
    int dy = my - sy;
    int d2 = dx*dx + dy*dy;

    if (d2 < 96*96)
    {
        // Signal the PlayGround, that this missile shall explode.
        missile->setExplode();
    }
}


//----------------------------------------------------------------------------
void Missile::NormalWarheadStrategy::updateWarhead(Missile *missile)
{
}

//----------------------------------------------------------------------------
void Missile::NormalWarheadStrategy::onCreateExplosionParticles(const Missile *missile) const
{
}


//----------------------------------------------------------------------------
void Missile::ConeBurstWarheadStrategy::updateWarhead(Missile *missile)
{
    do_updateWarhead(missile);
}

//----------------------------------------------------------------------------
void Missile::ConeBurstWarheadStrategy::onCreateExplosionParticles(const Missile *missile) const
{
    // If the missile crashed or was shot down,
    // no projectiles shall be created.

    if (missile->isExplode())
    {
        int angle = (int)missile->getAngle() - 45;
        for (int i=0; i<=90; i+=15)
        {
            ObjectRepository::getInstance()->addObject(
                new MissileProjectile(missile, angle+i));
        }
    }
}


//----------------------------------------------------------------------------
void Missile::StarBurstWarheadStrategy::updateWarhead(Missile *missile)
{
    do_updateWarhead(missile);
}

//----------------------------------------------------------------------------
void Missile::StarBurstWarheadStrategy::onCreateExplosionParticles(const Missile *missile) const
{
    // If the missile crashed or was shot down,
    // no projectiles shall be created.

    if (missile->isExplode())
    {
        for (int angle=0; angle<360; angle+=45)
        {
            ObjectRepository::getInstance()->addObject(
                new MissileProjectile(missile, angle));
        }
    }
}



//----------------------------------------------------------------------------
Missile::Missile(EnumHeadingStrategy heading,
                 EnumWarheadStrategy warhead,
                 const unsigned fuel,
                 const unsigned creatorId)
        : ControllableObjectBase(creatorId)
{
    setFuelCapacity(fuel);
    setFuel(fuel);

    m_mass = 20;
    m_thrust = true;

    // @todo: The setSurface() call shouldn't be here.
    // But it's currently needed for the internals of initBottomPixels().
    setSurface(getRotationSurface(SURFACES_ROTATION_STEPS / 2));
    initBottomPixels();

    m_headingStrategy = HeadingStrategy::getStrategy(heading);
    m_warheadStrategy = WarheadStrategy::getStrategy(warhead);
}

//----------------------------------------------------------------------------
Missile::~Missile()
{
    m_headingStrategy = NULL;
    m_warheadStrategy = NULL;
}

//----------------------------------------------------------------------------
void Missile::updateHeading()
{
    const Ship *player = GameInterface::getInstance()->getPlayerShip();
    if (!player)
    {
        setRotation(ROT_NONE);
        return;
    }

    Sint16 a = getThrustForce() / getMass();

    int angle = SDL_TOOLS::getAngle(getPosition(), player->getPosition());

    Sint16 gx = GameInterface::getInstance()->getXGravity(getPosition());
    Sint16 gy = GameInterface::getInstance()->getYGravity(getPosition());

    angle = m_headingStrategy->calculateHeadingAngle(a, angle, gx, gy);
    angle -= getAngle();

    // Let the relative angle lie between [-180..180) degrees.
    while (angle >= 180)  angle -= 360;
    while (angle < -180)  angle += 360;

    setRotation(angle > 0 ? ROT_LEFT :
                angle < 0 ? ROT_RIGHT : ROT_NONE);
}

//----------------------------------------------------------------------------
void Missile::updateWarhead()
{
    m_warheadStrategy->updateWarhead(this);
}

//----------------------------------------------------------------------------
void Missile::update()
{
    if (hasFuel())
    {
        updateHeading();
        updateRotation();
    }

    updateVelocityAndPosition();

    updateWarhead();
}

//----------------------------------------------------------------------------
const SDL_Surface *Missile::getRotationSurface(unsigned frame) const
{
    return MissileSurfaces::getInstance()->getSurface(frame);
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Missile)



//============================================================================
// Implementation of Ship.h
//============================================================================

//----------------------------------------------------------------------------
/**
 * If the ship's y-velocity is bigger than SHIP_MIN_LANDING_VELOCITY,
 * the ship will be reflected by the landing platform.
 */
#define SHIP_MIN_LANDING_VELOCITY  20.0

/**
 * If the ship's y-velocity is bigger than SHIP_MAX_LANDING_VELOCITY,
 * the ship will be destroyed.
 */
#define SHIP_MAX_LANDING_VELOCITY  60.0

//----------------------------------------------------------------------------
/**
 * The maximum crate mass, the ship is able to carry.
 */
#define SHIP_MAX_CRATES_MASS  60


//----------------------------------------------------------------------------
Ship::Landed Ship::Landed::sm_instance;
Ship::Flying Ship::Flying::sm_instance;


//----------------------------------------------------------------------------
void Ship::Landed::update(Ship *s)
{
    s->updateFire();

    if (s->isThrust() && s->hasFuel())
    {
        --s->m_fuel;

        // If the thrusters are enabled, raise the ship by one pixel,
        // so the next update won't detect a landing again.

        s->m_platform = NULL;
        s->getFinePosition().subY(1.0);
        s->updatePosition();
        s->setUpdateState(Flying::getInstance());

        GameInterface::getInstance()->onShipTakeoff(s);
    }
}

//----------------------------------------------------------------------------
void Ship::Flying::update(Ship *s)
{
    assert(s->m_mass > 0);

    s->updateRotation();
    s->updateVelocityAndPosition();
    s->updateFire();

    if (s->isThrust() && !s->hasFuel())
    {
        s->setThrust(false);
        GameInterface::getInstance()->onShipOutOfFuel(s);
    }

    // Check if we are landed.
    const Platform *platform =
        GameInterface::getInstance()->isInLandingZone(s);
    if (platform && s->getFineVelocity().getY() > 0.0)
    {
        if (s->getFineVelocity().getY() < SHIP_MIN_LANDING_VELOCITY)
        {
            // We are landing.
            assert(s->m_platform == NULL);
            s->m_platform = platform;
            s->getFineVelocity().set(0.0, 0.0);
            s->setUpdateState(Landed::getInstance());

            SoundInterface::getInstance()->onShipPlatformBump();
            GameInterface::getInstance()->onShipLanded(s, platform);
        }
        else if (s->getFineVelocity().getY() < SHIP_MAX_LANDING_VELOCITY)
        {
            // We are reflecting from the platform.
            s->getFineVelocity().set(
                3.0 * s->getFineVelocity().getX() / 4.0,
                - s->getFineVelocity().getY() / 2.0);

            SoundInterface::getInstance()->onShipPlatformBump();
        }

        // If the ship's y velocity is faster than
        // SHIP_MAX_LANDING_VELOCITY, we do nothing.
        // This will lead to a background collision in the next update.
    }
}



//----------------------------------------------------------------------------
Ship::Ship(ShipSurfaces::Ship shipType) : ControllableObjectBase(0)
{
    // 100 seconds of fuel.
    setFuelCapacity(100 * SDLFrameRate::getFrameRate());

    m_shipType = shipType;
    m_mass = 100;
    m_fire = false;

    m_tipOffset = ShipSurfaces::getInstance()->getTipOffset(m_shipType);

    // @todo: The setSurface() call shouldn't be here.
    // But it's currently needed for the internals of initBottomPixels().
    setSurface(getRotationSurface(SURFACES_ROTATION_STEPS / 2));
    initBottomPixels();

    setUpdateState(Flying::getInstance());
}

//----------------------------------------------------------------------------
Ship::Ship(unsigned id, ShipSurfaces::Ship shipType)
        : ControllableObjectBase(id)
{
    m_shipType = shipType;
    m_mass = 0;
    m_fire = false;
    m_platform = NULL;
    m_tipOffset = 0;

    setUpdateState(NULL);
}

//----------------------------------------------------------------------------
Ship::~Ship()
{
    m_platform = NULL;

    setUpdateState(NULL);
}


//----------------------------------------------------------------------------
bool Ship::hasCapacity(const Crate *crate) const
{
    unsigned sum = 0;
    for (CrateCIter
             iter = m_crates.begin(),
             end = m_crates.end();
         iter != end; ++iter)
    {
        sum += (*iter)->getMass();
    }

    return sum + crate->getMass() <= SHIP_MAX_CRATES_MASS;
}

//----------------------------------------------------------------------------
void Ship::pushCrate(Crate *crate)
{
    m_crates.push_back(crate);
    m_mass += crate->getMass();
}

//----------------------------------------------------------------------------
Crate *Ship::popCrate()
{
    assert(!m_crates.empty());

    Crate *crate = m_crates.front();
    m_crates.pop_front();

    m_mass -= crate->getMass();

    return crate;
}


//----------------------------------------------------------------------------
bool Ship::isInLandingZone(const SDL_Rect &r) const
{
    if (m_angle == SURFACES_ROTATION_STEPS / 2)
    {
        SDL_Rect bottom = m_bottomPixels;
        bottom.x += getPosition().x;
        bottom.y += getPosition().y;

        SDL_Rect i;
        if (SDL_TOOLS::intersect(r, bottom, i))
        {
            if (i.w == bottom.w)
            {
                return true;
            }
        }
    }

    return false;
}


//----------------------------------------------------------------------------
void Ship::resetPosition(Sint16 x, Sint16 y, bool landed)
{
    setPosition(x, y);
    getFineVelocity().set(0.0, 0.0);

    if (landed)
    {
        setUpdateState(Landed::getInstance());
    }
    else
    {
        setUpdateState(Flying::getInstance());
    }

    // @todo The following reset stuff should not be in this method.

    setAngle(180);
    setRotation(ROT_NONE);
    updateRotation();

    setThrust(false);
    m_platform = NULL;

    // Needed to reset the surface itself.
    updateSurface();
}


//----------------------------------------------------------------------------
void Ship::update()
{
    m_updateState->update(this);
}

//----------------------------------------------------------------------------
const SDL_Surface *Ship::getRotationSurface(unsigned frame) const
{
    return ShipSurfaces::getInstance()->getSurface(m_shipType, frame);
}


//----------------------------------------------------------------------------
void Ship::updateFire()
{
    if (m_fire)
    {
        m_fire = false;
        SoundInterface::getInstance()->onShipShoot();
        ObjectRepository::getInstance()->addObject(new ShipProjectile(this));
        GameInterface::getInstance()->onShipFiring(this);
    }
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Ship)



//============================================================================
// Implementation of Particles.h
//============================================================================

//----------------------------------------------------------------------------
ParticleBase::ParticleBase(const ObjectBase *creator)
        : MovingObjectBase(creator->getId())
{
    m_lifeCount = 0;
    m_gravity = false;

    m_gradient = false;
    m_gradientFineIndex = 0;
    m_gradientFineIndexIncrement = 0;
}

//----------------------------------------------------------------------------
ParticleBase::~ParticleBase()
{
}

//----------------------------------------------------------------------------
void ParticleBase::setGradient(unsigned fineSweepSpeed)
{
    m_gradient = true;

    m_gradientFineIndex =
        myRand(ParticleSurfaces::getInstance()->getGradientSize() / 16);
    m_gradientFineIndexIncrement = fineSweepSpeed/2 + (myRand(fineSweepSpeed));
}

//----------------------------------------------------------------------------
void ParticleBase::update()
{
    GameInterface *gameInterface = GameInterface::getInstance();

    // The old velocity values are needed.
    MATH_TOOLS::Vector oldFineVelocity = getFineVelocity();

    MATH_TOOLS::Vector fineAccel;

    if (m_gravity)
    {
        fineAccel.set(
            1.0 * gameInterface->getXGravity(getPosition()),
            1.0 * gameInterface->getYGravity(getPosition()));
    }

    // No friction is calculated for particles.
    fineAccel /= 1.0 * SDLFrameRate::getFrameRate();
    getFineVelocity() += fineAccel;

    // Use the average value of the old and new velocity.
    oldFineVelocity += getFineVelocity();
    oldFineVelocity /= 2.0 * SDLFrameRate::getFrameRate();
    getFinePosition() += oldFineVelocity;

    updatePosition();

    if (m_gradient)
    {
        updateSurface();
        m_gradientFineIndex += m_gradientFineIndexIncrement;

        if (m_gradientFineIndex >= 100*PARTICLES_GRADIENT_SIZE)
        {
            setHidden(true);
        }
    }

    if (--m_lifeCount == 0)
    {
        setHidden(true);
    }
}

//----------------------------------------------------------------------------
void ParticleBase::updateSurface()
{
    setSurface(
        ParticleSurfaces::getInstance()->getSurface(m_gradientFineIndex/100));
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(ParticleBase)


//----------------------------------------------------------------------------
ThrustParticle::ThrustParticle(const ObjectBase *creator)
        : ParticleBase(creator)
{
    setLifeCount(20 + myRand(5));
    setGradient(200);

    init();
}

//----------------------------------------------------------------------------
ThrustParticle::~ThrustParticle()
{
}

//----------------------------------------------------------------------------
void ThrustParticle::init()
{
    const ControllableObjectBase *object =
        static_cast<const ControllableObjectBase*>(
            ObjectRepository::getInstance()->getObject(getCreatorId()));

    // Initialize the position.

    const SDL_Rect &objPosition = object->getPosition();
    Sint16 xm, ym;
    SDL_TOOLS::getCentre(objPosition, xm, ym);

    // Add 2 to the radius to prevent immediate collisions
    // between the object and this particle.
    Uint16 r = objPosition.y + object->getBottomPixels().y - ym + 2;

    setPosition((Uint16)trunc(xm - r * SinusTable::sin(object->getAngle())),
                (Uint16)(ym - r * SinusTable::cos(object->getAngle())));


    // Initialize the velocity.

    MATH_TOOLS::Vector relativeVelocity;
    relativeVelocity.setExponential(
        1.0 * PARTICLE_VELOCITY, 1.0 * object->getAngle());
    relativeVelocity.add(1.0 * (20 - myRand(41)),
                         1.0 * (20 - myRand(41)));

    getFineVelocity() = object->getFineVelocity() - relativeVelocity;
}


//----------------------------------------------------------------------------
ExplosionParticle::ExplosionParticle(const ObjectBase *creator)
        : ParticleBase(creator)
{
    InitializeExplosionParticleConstVisitor v(this);
    creator->accept(v);
}

//----------------------------------------------------------------------------
ExplosionParticle::~ExplosionParticle()
{
}


//----------------------------------------------------------------------------
FountainParticle::FountainParticle(const ParticleFountainBase *f)
        : ParticleBase(f)
{
    setLifeCount(
        f->getParticleLifeTime()*SDLFrameRate::getFrameRate() + myRand(5));
    setGradient(100 / f->getParticleLifeTime());
    setGravity();

    init();
}

//----------------------------------------------------------------------------
FountainParticle::~FountainParticle()
{
}

//----------------------------------------------------------------------------
void FountainParticle::init()
{
    const ParticleFountainBase *f =
        static_cast<const ParticleFountainBase*>(
            ObjectRepository::getInstance()->getObject(getCreatorId()));

    // Initialize the position.

    SDL_Rect position = f->getPosition();
    position.x += 6;
    position.y += 6;
    position.w = 4;
    position.h = 4;

    setPosition(position.x + myRand(position.w),
                position.y + myRand(position.h));


    // Initialize the velocity.

    Sint32 scatter = f->getParticleScatter();
    MATH_TOOLS::Vector relativeVelocity;
    relativeVelocity.set(
        1.0 * (scatter - myRand(2*scatter+1)),
        1.0 * (scatter - myRand(2*scatter+1)));

    getFineVelocity() = f->getParticleInitialVelocity() + relativeVelocity;
}



//============================================================================
// Implementation of Projectiles.h
//============================================================================

//----------------------------------------------------------------------------
ProjectileBase::ProjectileBase(const unsigned creatorId)
        : MovingObjectBase(creatorId)
{
}

//----------------------------------------------------------------------------
ProjectileBase::~ProjectileBase()
{
}

//----------------------------------------------------------------------------
void ProjectileBase::update()
{
    MATH_TOOLS::Vector relPosition = getFineVelocity();
    relPosition /= 1.0 * SDLFrameRate::getFrameRate();

    getFinePosition() += relPosition;

    updatePosition();
}

//----------------------------------------------------------------------------
void ProjectileBase::updateSurface()
{
    setSurface(ProjectileSurfaces::getInstance()->getSurface());
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(ProjectileBase)


//----------------------------------------------------------------------------
ShipProjectile::ShipProjectile(const Ship *shooter) 
        : ProjectileBase(shooter->getId())
{
    // Calculate the ship's rotation point.
    const SDL_Rect &shipPosition = shooter->getPosition();
    Sint16 xm, ym;
    SDL_TOOLS::getCentre(shipPosition, xm, ym);

    Uint16 pw, ph;
    ProjectileSurfaces::getInstance()->getProjectileSize(pw, ph);

    // When rotating, the tip of the ship moves in a circle with a radius,
    // we are calculating here.
    // To keep sure, the projectile won't touch the ship initially,
    // we add the height of the projectile's surface.
    Uint16 r = ym - (shipPosition.y + shooter->getTipOffset()) + ph;

    setPosition(
        (Uint16)(xm - pw/2 + r*SinusTable::sin(shooter->getAngle())),
        (Uint16)(ym - ph/2 + r*SinusTable::cos(shooter->getAngle())));

    // Calculate the projectile's velocity.
    MATH_TOOLS::Vector relativeVelocity;
    relativeVelocity.setExponential(
        1.0 * PROJECTILE_VELOCITY, 1.0 * shooter->getAngle());

    // Add the ship's current velocity to the projectile's velocity.
    getFineVelocity() = shooter->getFineVelocity() + relativeVelocity;
}

//----------------------------------------------------------------------------
ShipProjectile::~ShipProjectile()
{
}


//----------------------------------------------------------------------------
TurretProjectile::TurretProjectile(const TurretBase *turret,
                                   const TurretBarrel *barrel)
        : ProjectileBase(turret->getId())
{
    Uint16 pw, ph;
    ProjectileSurfaces::getInstance()->getProjectileSize(pw, ph);

    // Set the initial position.
    Sint16 x, y;
    switch (turret->getTipPosition(x, y))
    {
    case OrientatingDecorationBase::O_TOP:
        x -= pw / 2;
        break;

    case OrientatingDecorationBase::O_BOTTOM:
        x -= pw / 2;
        y -= ph;
        break;

    case OrientatingDecorationBase::O_LEFT:
        y -= ph / 2;
        break;

    case OrientatingDecorationBase::O_RIGHT:
        x -= pw;
        y -= ph / 2;
        break;
    }

    setPosition(x, y);

    // Set the initial velocity.
    getFineVelocity().setExponential(
        1.0 * barrel->getProjectileSpeed(),
        1.0 * (turret->getCenterAngle() + barrel->getAngle()));
}

//----------------------------------------------------------------------------
TurretProjectile::~TurretProjectile()
{
}


//----------------------------------------------------------------------------
MissileProjectile::MissileProjectile(const Missile *missile, int angle)
        : ProjectileBase(missile->getCreatorId())
{
    Uint16 pw, ph;
    ProjectileSurfaces::getInstance()->getProjectileSize(pw, ph);

    // Set the initial position.
    Sint16 x, y;
    SDL_TOOLS::getCentre(missile->getPosition(), x, y);
    x -= pw / 2;
    y -= ph / 2;

    setPosition(x, y);

    // Calculate the projectile's velocity.
    MATH_TOOLS::Vector relativeVelocity;
    relativeVelocity.setExponential(0.25 * PROJECTILE_VELOCITY, 1.0 * angle);

    // Add the missile's current velocity to the projectile's velocity.
    getFineVelocity() = missile->getFineVelocity() + relativeVelocity;
}

//----------------------------------------------------------------------------
MissileProjectile::~MissileProjectile()
{
}


//----------------------------------------------------------------------------
TankProjectile::TankProjectile(const Tank *tank)
        : ProjectileBase(tank->getId())
{
    Uint16 pw, ph;
    ProjectileSurfaces::getInstance()->getProjectileSize(pw, ph);

    // Set the initial position.
    Sint16 x, y;
    SDL_TOOLS::getCentre(tank->getPosition(), x, y);
    x -= pw / 2;
    y -= ph / 2 + 2;

    x += (Sint16)trunc(9 * SinusTable::sin(tank->getAngle()));
    y += (Sint16)trunc(5 * SinusTable::cos(tank->getAngle()));

    setPosition(x, y);

    // Set the initial velocity.
    getFineVelocity().setExponential(
        1.0 * tank->getProjectileSpeed(), 1.0 * tank->getAngle());
}

//----------------------------------------------------------------------------
TankProjectile::~TankProjectile()
{
}


//----------------------------------------------------------------------------
GrenadeProjectile::GrenadeProjectile(const Grenade *grenade, int angle)
        : ProjectileBase(grenade->getCreatorId())
{
    Uint16 pw, ph;
    ProjectileSurfaces::getInstance()->getProjectileSize(pw, ph);

    // Set the initial position.
    Sint16 x, y;
    SDL_TOOLS::getCentre(grenade->getPosition(), x, y);
    x -= pw / 2;
    y -= ph / 2;

    setPosition(x, y);

    // Calculate the projectile's velocity.
    MATH_TOOLS::Vector relativeVelocity;
    relativeVelocity.setExponential(0.25 * PROJECTILE_VELOCITY, 1.0 * angle);

    // Add the grenade's current velocity to the projectile's velocity.
    getFineVelocity() = grenade->getFineVelocity() + relativeVelocity;
}

//----------------------------------------------------------------------------
GrenadeProjectile::~GrenadeProjectile()
{
}



//============================================================================
// Implementation of Grenade.h
//============================================================================

//----------------------------------------------------------------------------
Grenade::NoWarheadStrategy Grenade::NoWarheadStrategy::sm_instance;
Grenade::StarBurstWarheadStrategy Grenade::StarBurstWarheadStrategy::sm_instance;

//----------------------------------------------------------------------------
Grenade::WarheadStrategy *Grenade::WarheadStrategy::getStrategy(
    EnumWarheadStrategy strategy)
{
    switch (strategy)
    {
    case WH_NONE:
        return NoWarheadStrategy::getInstance();
    case WH_STARBURST:
        return StarBurstWarheadStrategy::getInstance();
    }

    return NULL;
}

//----------------------------------------------------------------------------
void Grenade::NoWarheadStrategy::onCreateExplosionParticles(const Grenade *grenade) const
{
}

//----------------------------------------------------------------------------
void Grenade::StarBurstWarheadStrategy::onCreateExplosionParticles(const Grenade *grenade) const
{
    // If the grenade crashed, no projectiles shall be created.

    if (grenade->isExplode())
    {
        for (int angle=0; angle<360; angle+=45)
        {
            ObjectRepository::getInstance()->addObject(
                new GrenadeProjectile(grenade, angle));
        }
    }
}


//----------------------------------------------------------------------------
Grenade::Grenade(EnumWarheadStrategy warhead, const unsigned creatorId)
        : MovingObjectBase(creatorId)
{
    m_lifeCount = 0;
    m_explode = false;

    m_warheadStrategy = WarheadStrategy::getStrategy(warhead);
}

//----------------------------------------------------------------------------
Grenade::~Grenade()
{
    m_warheadStrategy = NULL;
}

//----------------------------------------------------------------------------
void Grenade::update()
{
    GameInterface *gameInterface = GameInterface::getInstance();

    // The old velocity values are needed.
    MATH_TOOLS::Vector oldFineVelocity = getFineVelocity();

    MATH_TOOLS::Vector fineAccel(
        1.0 * gameInterface->getXGravity(getPosition()),
        1.0 * gameInterface->getYGravity(getPosition()));

    // No friction is calculated for particles.
    fineAccel /= 1.0 * SDLFrameRate::getFrameRate();
    getFineVelocity() += fineAccel;

    // Use the average value of the old and new velocity.
    oldFineVelocity += getFineVelocity();
    oldFineVelocity /= 2.0 * SDLFrameRate::getFrameRate();
    getFinePosition() += oldFineVelocity;

    updatePosition();

    if (--m_lifeCount == 0)
    {
        m_explode = true;
    }
}

//----------------------------------------------------------------------------
void Grenade::updateSurface()
{
    setSurface(GrenadeSurfaces::getInstance()->getSurface());
}

//----------------------------------------------------------------------------
DECLARE_OBJECT_VISITOR_API_BODY(Grenade)
