# GP_EVAL.PY
#
# The code in this file is part of PyXPlot
# <http://www.pyxplot.org.uk>
#
# Copyright (C) 2006-7 Dominic Ford <coders@pyxplot.org.uk>
#
# $Id: gp_eval.py 5 2007-02-22 09:43:42Z dcf21 $
#
# PyXPlot is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# You should have received a copy of the GNU General Public License along with
# PyXPlot; if not, write to the Free Software Foundation, Inc., 51 Franklin
# Street, Fifth Floor, Boston, MA  02110-1301, USA

# ----------------------------------------------------------------------------

# Evaluates expressions using user variables and functions

import sys
import re
from math import *

try: import scipy.integrate
except: SCIPY_ABSENT = True
else: SCIPY_ABSENT = False

from gp_error import *
import gp_spline

# GP_EVAL_INTEGRAND(): Evaluates an integrand, passed to it from the scipy.integrate.quad function
def gp_eval_integrand(x, expression, xname, vars, funcs, iteration):
  vars[xname] = x
  return gp_eval(expression, vars, funcs, False, iteration+1)

# GP_EVAL(): Evaluates an expression, substituting user defined variables and functions
def gp_eval(expression, vars, funcs, verbose=True, iteration=1):
  if (iteration > 20):
   raise OverflowError, "Iteration depth exceeded in function evaluation."
  try:
    # Quick escape route if we're just evaluating a variable name
    if expression in vars: return vars[expression]

    # Evaluate any integral functions
    while 1:
      test = re.match(r"(.*(^|\W))int_d([A-Za-z]\w*)\((.*)$",expression)
      if (test == None): break
      assert not SCIPY_ABSENT, "The integration of functions requires the scipy module for python, which is not installed. Please install and try again."
      func = test.group(3)
      bracketmatch = gp_bracketmatch(expression, test.start(4)-1)
      if (len(bracketmatch) == 0): raise SyntaxError, "Mismatched brackets for integral 'int_d%s'."%func
      if (len(bracketmatch) != 3): raise SyntaxError, "Integral 'int_d%s' should take three parameters -- expression, min, max."%func
      min = gp_eval(expression[ (bracketmatch[0]+1):bracketmatch[0+1] ], vars, funcs, False, iteration+1)
      max = gp_eval(expression[ (bracketmatch[1]+1):bracketmatch[1+1] ], vars, funcs, False, iteration+1)
      func_scope = vars.copy()
      integration_result = scipy.integrate.quad(gp_eval_integrand, min, max, full_output=1, args=(expression[test.start(4):bracketmatch[0]],func,func_scope,funcs,iteration))
      if verbose and (len(integration_result)>3):
        gp_warning("Warning whilst integrating expression %s:\n%s"%(expression[test.start(4):bracketmatch[0]],integration_result[3]))
      expression = test.group(1) + str(integration_result[0]) + expression[bracketmatch[len(bracketmatch)-1]+1:]

    # Evaluate any differential function
    while 1:
      test = re.match(r"(.*(^|\W))diff_d([A-Za-z]\w*)\((.*)$",expression)
      if (test == None): break
      func = test.group(3)
      bracketmatch = gp_bracketmatch(expression, test.start(4)-1)
      if   (len(bracketmatch) == 0):
        raise SyntaxError, "Mismatched brackets for differential 'diff_d%s'."%func
      elif (len(bracketmatch) == 2):
        epsilon1 = epsilon2 = 1e-6
      elif (len(bracketmatch) == 3):
        epsilon1 = gp_eval(expression[ (bracketmatch[1]+1):bracketmatch[1+1] ], vars, funcs, False, iteration+1)
        epsilon2 = 1e-6
      elif (len(bracketmatch) == 4):
        epsilon1 = gp_eval(expression[ (bracketmatch[1]+1):bracketmatch[1+1] ], vars, funcs, False, iteration+1)
        epsilon2 = gp_eval(expression[ (bracketmatch[2]+1):bracketmatch[2+1] ], vars, funcs, False, iteration+1)
      else:
        raise SyntaxError, "Differential 'diff_d%s' should take 2-4 parameters -- expression, point at which to differentiate, epsilon."%func
      func_scope = vars.copy()
      xval = gp_eval(expression[ (bracketmatch[0]+1):bracketmatch[0+1] ], vars, funcs, False, iteration+1)
      epsilon = epsilon1 + xval * epsilon2
      func_scope[func] = xval-epsilon/2.0 ; x1 = gp_eval(expression[ test.start(4):bracketmatch[0]], func_scope, funcs, False, iteration+1)
      func_scope[func] = xval+epsilon/2.0 ; x2 = gp_eval(expression[ test.start(4):bracketmatch[0]], func_scope, funcs, False, iteration+1)
      expression = test.group(1) + str((x2-x1)/epsilon) + expression[bracketmatch[len(bracketmatch)-1]+1:]

    # Evaluate any functions
    for func,fexp in funcs.iteritems():
      while 1:
        test = re.match(r"(.*(^|\W))%s\s*\((.*)$"%func,expression)
        if (test == None): break
        bracketmatch = gp_bracketmatch(expression, test.start(3)-1)
        if (len(bracketmatch) == 0):            raise SyntaxError, "Mismatched brackets for function '%s'."%func
        if (len(bracketmatch) != abs(fexp[0])): raise SyntaxError, "Function '%s' takes %d arguments; %d provided."%(func,abs(fexp[0]),len(bracketmatch))
        args = []
        bracketmatch.insert(0,test.start(3)-1)
        for i in range(abs(fexp[0])):
          args.append(gp_eval(expression[ (bracketmatch[i]+1):bracketmatch[i+1] ], vars, funcs, False, iteration+1))
        if (fexp[0] < 0): # This is a spline
          try:
            value = gp_spline.spline_evaluate(args[0], fexp[1][0][1])
            expression = test.group(1) + str(value) + expression[bracketmatch[len(bracketmatch)-1]+1:]
          except KeyboardInterrupt: raise
          except:
            raise ValueError, "Error evaluating spline %s"%func
        else:             # This is a function
          funcdone = False
          for defno in range(len(fexp[1])):
            if not funcdone:
              j = len(fexp[1]) - 1 - defno
              func_scope = vars.copy()
              inrange = True
              for i in range(fexp[0]): func_scope[fexp[1][j][0][i]] = args[i]
              for i in range(fexp[0]):
                try:
                 if (fexp[1][j][1][i][0] != None): minrange = gp_eval(fexp[1][j][1][i][0],func_scope,funcs,False,iteration+1)
                 else                            : minrange = None
                 if (fexp[1][j][1][i][1] != None): maxrange = gp_eval(fexp[1][j][1][i][1],func_scope,funcs,False,iteration+1)
                 else                            : maxrange = None
                except KeyboardInterrupt: raise
                except:
                 if (verbose):
                  gp_error("Error evaluating range of function '%s'."%func)
                  gp_error("(it may be necessary to delete it with 'f(x)=' and then redefine it)")
                 raise
                if ((minrange != None) and (args[i] < minrange)): inrange = False
                if ((maxrange != None) and (args[i] > maxrange)): inrange = False
              if inrange:
                expression = test.group(1) + str(gp_eval(fexp[1][j][2],func_scope,funcs,False,iteration+1)) + expression[bracketmatch[len(bracketmatch)-1]+1:]
                funcdone   = True
          if not funcdone:
            raise ValueError, "Attempt to evaluate function '%s' with arguments out of their specified ranges."%func

    # All local functions substituted for; now evaluate what is left.
    return float(eval(expression, globals(), vars))
  except KeyboardInterrupt: raise
  except:
   if (verbose):
    gp_error("Error evaluating expression %s"%expression)
    gp_error("Error:" , sys.exc_info()[1], "(" , sys.exc_info()[0] , ")")
   raise

# GP_FUNCTION_DECLARE(): Declare a new function, possibly with a range
def gp_function_declare(line, funcs):
 test = re.match(r'([A-Za-z]\w*)\(([^()]*)\)\s*([^=]*)=(.*)',line)
 if (test == None):
  gp_error("Error: bad function definition '%s' could not be parsed."%line)
  return

 name       = test.group(1) # The name of the function
 arguments  = gp_split(test.group(2),",") # A textual list of its arguments
 arguments2 = [] # A stripped and checked list of its arguments
 ranges     = gp_split(test.group(3),"[")[1:] # List of range strings
 ranges2    = [] # A stripped and checked list of range strings
 expression = test.group(4).strip()

 for argument in arguments:
  test2 = re.match(r'^[A-Za-z]\w*$',argument.strip())
  if (test2 == None): raise SyntaxError, "Error: Function has badly formed argument name '%s'."%argument
  arguments2.append(argument.strip())

 for i in range(len(arguments)):
  if (i >= len(ranges)):
   ranges2.append([None, None]) # No range, if none specified
  else:
   rangestr = ranges[i].strip()
   if (rangestr == "]"): # Case [] means all x
    min = max = None
   else:
    test2 = re.match("([^:\]]*)((:)|( *to *))([^:\]]*)\]", rangestr)
    if (test2 == None): raise SyntaxError, "Error: range of argument %d should take form [min:max] or [min to max], but instead has form '[%s'."%(i+1,rangestr)
    min = test2.group(1).strip()
    max = test2.group(5).strip()
    if (len(min) == 0): min = None
    if (len(max) == 0): max = None
   ranges2.append([min,max])

 if   (len(expression) == 0)                                  :
  try:
   del funcs[name]
  except KeyboardInterrupt: raise
  except:
   raise KeyError, "Attempt to delete a function '%s' which does not exist."%name
 elif ((name in funcs) and (funcs[name][0] == len(arguments))): funcs[name][1].append([arguments2,ranges2,expression])
 else                                                         : funcs[name] = [len(arguments), [[arguments2,ranges2,expression]]]

# GP_BRACKETMATCH(): Find a matching closing bracket for an opening bracket

def gp_bracketmatch(expression, i):
  # Returns a list of comma positions in expression
  # Final item on list is closing )

  bracket_level = 0
  max           = len(expression)
  list          = []

  while( i < max-1 ):
    i = i+1 # Start searching after bracket at position i
    if (expression[i]==')'):
      if (bracket_level == 0):
        list.append(i)
        return list
      else:
        bracket_level = bracket_level - 1
    elif (expression[i]=='('):
        bracket_level = bracket_level + 1
    elif (expression[i]==','):
      if (bracket_level == 0):
        list.append(i)
        continue

# GP_GETQUOTEDSTRING(): Extract a quoted string, finding the end of the string.
# The first character of the passed string is expected to be ('|"). Escaped
# quote characters, \' and \" are unescaped and ignored in the output.

def gp_getquotedstring(string):
  quotetype = string[0]
  if (not quotetype in ["'", '"']): raise SyntaxError, "'Quoted' string does not start with quote character."

  stringend = None
  for i in range(1, len(string)):
    if (string[i] == quotetype) and (string[i-1] != '\\'):
     stringend = i
     break
  if (stringend == None): return [None, None]
  text = string[1:i]
  aftertext = string [i+1:]
  text = re.sub(r"\\'", "'", text)
  text = re.sub(r'\\"', '"', text)
  return [text, aftertext]

# GP_SPLIT(): An intelligent string splitter, which doesn't grab split
# characters embedded in () or ""

def gp_split(expression, splitchar):
  bracket_level = 0
  quote_level   = 0
  apostro_level = 0
  max           = len(expression)
  list          = []
  i             = -1
  word          = ""

  while( i < max-1 ):
    i = i+1
    if   ((expression[i]==')') and (apostro_level == 0) and (quote_level   == 0)): bracket_level = bracket_level - 1
    elif ((expression[i]=='(') and (apostro_level == 0) and (quote_level   == 0)): bracket_level = bracket_level + 1 
    elif ((expression[i]=='"') and (apostro_level == 0)): quote_level   = 1 - quote_level   # Hyphens allowed in quotes and vice-versa
    elif ((expression[i]=="'") and (quote_level   == 0)): apostro_level = 1 - apostro_level
    elif (expression[i]==splitchar):
      if ((bracket_level == 0) and (quote_level == 0) and (apostro_level == 0)):
        list.append(word)
        word=""
        continue
    word = word + expression[i]
  list.append(word)
  return list

# We've hit the end of string without brackets being closed
  return ()

# GP_GETEXPRESSION(): Extracts the longest possible valid algebraic expression
# from the beginning of a string.

# Go through string, spotting atoms from the following list:

# S -- the beginning of the string
# E -- the end of the expression
# B -- a bracketed () series of characters
# D -- a dollar sign -- only allowed in using expressions in the plot command as special variable name
# M -- a minus sign before a numeric value, variable name, or ()
# N -- a numerical value, e.g. 1.2e-34
# O -- an operator, + - * ** / % << >> & | ^ < > <= >= == != <>
# V -- a variable name

# S can be followed by  BDMN V not by E    O
# E
# B can be followed by E    O  not by  BDMN V
# D can be followed by  B  N V not by E DM O
# M can be followed by  BD N V not by E  M O
# N can be followed by E    O  not by  BDMN V
# O can be followed by  BDMN V not by E    O
# V can be followed by EB   O  not by   DMN V

# Returns [ position in string of end of expression, "expecting" error string ]

def gp_getexpression(string, dollar_allowed=False):
  state = "S" # At the beginning
  allowed_next = {"S":["B","D","M","N",    "V"    ],
                  "B":[                "O",    "E"],
                  "D":["B",        "N",    "V"    ],
                  "N":[                "O",    "E"],
                  "M":["B","D","M","N",    "V"    ],
                  "O":["B","D","M","N",    "V"    ],
                  "V":["B",            "O",    "E"]
                  }
  linepos = 0
  while (state != "E"):
   expecting = []
   for type in allowed_next[state]:
     s = string[linepos:]

     if   (type == "E"):
      state = "E" # If we're allowed to end here, and we've got to the end of the allowed_next list, the end we have reached
      expecting = []
      break # We've matched an atom (E)

     elif (type == "B"):
      test = re.match("\s*(\(.*)",s)
      if (test != None):
       bracket_match = gp_bracketmatch(string, linepos+test.start(1))
       if (bracket_match != None):
        linepos = bracket_match[-1]+1 # +1 because we want character after ")"
        state = "B"
        expecting = []
        break # We've matched an atom (B)
       else:
        return [linepos+test.start(1), "a closing bracket to match this one"]
      expecting.append('"("')

     elif (type == "D"):
      if dollar_allowed:
       test = re.match(r"\s*\$(.*)",s)
       if (test != None):
        linepos += test.start(1)
        expecting = []
        break # We've matched an atom (D)
       expecting.append("""$""")

     elif (type == "M"):
      test = re.match("\s*((-)|(not))\s*(.*)",s)
      if (test != None):
       linepos += test.start(4)
       expecting = []
       break # We've matched an atom (M)
      expecting.append("""a minus sign""")

     elif (type == "N"):
      test = re.match(r"\s*[+-]?(\d*)\.?(\d*)([eE][+-]?\d\d*)?\s*(.*)",s)
      if (test != None) and (len(test.group(1))+len(test.group(2)) > 0):
       linepos += test.start(4)
       state = "N"
       expecting = []
       break # We've matched an atom (N)
      expecting.append("a numeric value")

     elif (type == "O"):
      test = re.match(r"\s*((and)|(or)|(<=)|(>=)|(==)|(!=)|(<>)|(<<)|(>>)|(\*\*)|\+|-|\*|/|%|&|\||\^|<|>)\s*(.*)",s)
      if (test != None):
       linepos += test.start(12)
       state = "O"
       expecting = []
       break # We've matched an atom (O)
      expecting.append("""an operator (i.e. +  -  *  **  /  %  <<  >>  &  |  ^  <  >  <=  >=  ==  !=  "and" "or" or  <> )""")

     elif (type == "V"):
      test = re.match(r"\s*[A-Za-z]\w*\s*(.*)",s)
      if (test != None):
       linepos += test.start(1)
       state = "V"
       expecting = []
       break # We've matched an atom (V)
      expecting.append("""a variable or function name""")

     else:
      raise SyntaxError, "Internal Error; shouldn't get here!"

   if (expecting != []): break

  if (state == "E"):
    return [linepos, None]
  else:
    expect_str = ""
    for x in expecting:
      if (expect_str != ""): expect_str += " or "
      expect_str += x
    return [linepos, expect_str]
