/**
 * Copyright Copr. INRIA/INRA
 * Contact :  biomaj_AT_genouest.org
 * 
 * BioMAJ is a workflow engine dedicated to biological bank management. 
 * The Software automates the update cycle and the supervision of the locally 
 * mirrored bank repository. The project is a collaborative effort between two 
 * French Research Institutes INRIA (Institut National de Recherche en
 * Informatique 
 * et en Automatique) & INRA (Institut National de la Recherche Agronomique).
 *
 * Inter Deposit Digital Number : IDDN ...................................
 *
 * This software is governed by the CeCILL-A license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info". 
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability. 
 *
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or 
 * data to be ensured and,  more generally, to use and operate it in the 
 * same conditions as regards security. 
 * 
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL-A license and that you accept its terms.
 */


package org.inria.biomaj.exe.workflow;

import java.util.Collections;
import java.util.Vector;
import org.inria.biomaj.exe.bank.BankSet;
import org.inria.biomaj.exe.bank.BiomajBank;
import org.inria.biomaj.utils.BiomajConst;
import org.inria.biomaj.utils.BiomajException;


public abstract class BankSetWorkflowEngine extends WorkflowEngine {

	//	Les sous workflows executes!
	/**
	 * @uml.property  name="lThread"
	 * @uml.associationEnd  multiplicity="(0 -1)"
	 */
	private WorkflowEngine[] lThread = new WorkflowEngine[0];
	/**
	 * @uml.property  name="withConsole"
	 */
	private boolean withConsole = true;
	/**
	 * @uml.property  name="mode"
	 */
	private int  mode        ;
	
	public BankSetWorkflowEngine(BiomajBank bank) {
		super(bank);
	}

	@Override
	public void setModeConsole(boolean withConsole, int mode) {
		this.withConsole = withConsole;
		this.mode = mode;
	}
	
	protected void initListFils() throws BiomajException{
		WorkflowEngineFactory wef = new WorkflowEngineFactory();

		BankSet bs = (BankSet)getBiomajBank();
		lThread = new WorkflowEngine[bs.getListBanks().size()];
		int i = 0;
		for (BiomajBank b : bs.getListBanks()) {
			WorkflowEngine we ;
			
			we = wef.createWorkflow(b);
			we.setModeConsole(withConsole, mode);
			
			lThread[i++] = we;
		}
	}
	
	synchronized protected void startTargetFils(int target) throws BiomajException {
		initListFils();
		for (int i=0;i<lThread.length;i++) {
			if ((!lThread[i].isAlive())&&(lThread[i].getState()==State.NEW)) {
				lThread[i].setCommand(target);
				lThread[i].start();
			}
		}
	}

	protected void synchronizeFils() throws BiomajException {
		for (int i=0;i<lThread.length;i++) {
			lThread[i].synchr();
		}
		
		for (int i=0;i<lThread.length;i++) {
			while (lThread[i].isErrorOnWorkflow()==null) {} // Wait for this to return something (false or true)
			if (lThread[i].isErrorOnWorkflow())
				throw new BiomajException("kill.application.withoutlog");
		}
		
		for (int i=0;i<lThread.length;i++) {
			WorkflowEngine.removeWE(lThread[i].getBiomajBank().getDbName());
		}
	}
	
	/**
	 * Synchronisation sur un block de bank....
	 * @param bank
	 * @param block
	 * @throws BiomajException
	 */
	protected void synchronizeFils(String bank,String block) throws BiomajException {
		
		WorkflowEngine we = null ;
		for (WorkflowEngine w : lThread) {
			if (w.getBiomajBank().getDbName().compareTo(bank)==0) {
				we = w;
				break;
			}
		}
		
		if (we == null)
			throw new BiomajException("bankset.workflow.error.name.bank.synchro",bank,block);
		
		if (!we.blockExist(block))
			throw new BiomajException("bankset.workflow.error.name.block.synchro",bank,block);
		
		we.synchr(block);
		
		for (int i=0;i<lThread.length;i++) {
			if ((lThread[i].isErrorOnWorkflow()!=null)&&(lThread[i].isErrorOnWorkflow()))
				throw new BiomajException("kill.application");
		}
	}

	@Override
	protected void endWithErrorWorkflow(Exception e) throws BiomajException {
		super.endWithErrorWorkflow(e);
		/*for (int i=0;i<lThread.length;i++) {
			lThread[i].endWithErrorWorkflow(e);
		}*/
	}



	@Override
	public void log(String message, int priority) {
		if (lThread == null) {
			System.out.println(message);
			return;
		}
		
		for (int i=0;i<lThread.length;i++) {
			lThread[i].log(message,priority);
		}
		
	}
	
	
	protected String getBlockBeforeAll() {
		
		String lBlocks = getBiomajBank().getPropertiesFromBankFile().getProperty(BiomajConst.blockPostprocessProperty);
		
		if ((lBlocks == null) || (lBlocks.trim().compareTo("")==0) ) {
			return "";
		}
		
		String[] llB = lBlocks.split(",");
		String result = "";
		boolean stop = true ;
		for (String b : llB) {
			stop = true ;
			if (getBiomajBank().getPropertiesFromBankFile().containsKey(b+".dep")) {
				String dep =  getBiomajBank().getPropertiesFromBankFile().getProperty(b+".dep");
				if (dep.trim().compareTo("before.all")==0)
					{
					result=result+b+",";
					stop = false ;
					}
			} 
			if (stop) {
				if (result.compareTo("")!=0)
					return result.substring(0, result.length()-1);
				
				return result;
			}
				
		}
		return result;
	}
	
	protected String getBlockAfterAll() {
		
		String lBlocks = getBiomajBank().getPropertiesFromBankFile().getProperty(BiomajConst.blockPostprocessProperty);
		
		if ((lBlocks == null) || (lBlocks.trim().compareTo("")==0) ) {
			return "";
		}
		
		String[] llB = lBlocks.split(",");
		String result = "";
		boolean afterAll = false;
		for (String b : llB) {
			if (afterAll||((!getBiomajBank().getPropertiesFromBankFile().containsKey(b+".dep"))
							||(getBiomajBank().getPropertiesFromBankFile().getProperty(b+".dep").trim().compareTo("after.all")==0))) {
				afterAll = true;
				result=result+b+",";
			}
		}
		
		if (result.compareTo("")!=0)
			return result.substring(0, result.length()-1);
		
		return result;
	}

	
	protected String getBankAndBlockChild(int index) {
		
		String lBlocks = getBiomajBank().getPropertiesFromBankFile().getProperty(BiomajConst.blockPostprocessProperty);
		
		if ((lBlocks == null) || (lBlocks.trim().compareTo("")==0) ) {
			return null;
		}
		
		String[] llB = lBlocks.split(",");
		int compteur = 0;
		for (String b : llB) {
			if (getBiomajBank().getPropertiesFromBankFile().containsKey(b+".dep")) {
				String dep =  getBiomajBank().getPropertiesFromBankFile().getProperty(b+".dep");
				if (dep.trim().compareTo("before.all")==0)
					continue;
				if (dep.trim().compareTo("after.all")==0)
					return null;
				if (compteur == index)
					return dep.trim();
				compteur++;
			} else
				return null; // pas de dependance donc after.all
		}
		return null;
	}
	
	protected String getBlockDependsChild(int index) {
		
		String lBlocks = getBiomajBank().getPropertiesFromBankFile().getProperty(BiomajConst.blockPostprocessProperty);
		
		if ((lBlocks == null) || (lBlocks.trim().compareTo("")==0) ) {
			return "";
		}
		
		String[] llB = lBlocks.split(",");
		int compteur = 0;
		for (String b : llB) {
			if (getBiomajBank().getPropertiesFromBankFile().containsKey(b+".dep")) {
				String dep =  getBiomajBank().getPropertiesFromBankFile().getProperty(b+".dep");
				if (dep.trim().compareTo("before.all")==0)
					continue;
				if (dep.trim().compareTo("after.all")==0)
					return null;
				if (compteur == index)
					return b;
				compteur++;
			} else
				return null; // pas de dependnace donc after.all
		}
		return null;
	}

	@Override
	public Vector<WorkflowEngine> getChild() {
		Vector<WorkflowEngine> we = new Vector<WorkflowEngine>();
		Collections.addAll(we, lThread);
		return we;
	}
	
}
