package org.inria.biomaj.sql;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.dbcp.BasicDataSource;
import org.inria.biomaj.singleton.BiomajLogger;
import org.inria.biomaj.utils.BiomajUtils;

/**
 * Class that maps a connection to a standalone in memory HSQL database
 * that is used for testing purpose.
 * 
 * @author rsabas
 *
 */
public class TestConnection implements SQLConnection {

	/**
	 * Directory where the database is stored.
	 */
	private String dbPath;
	
	private String url;
	private String login;
	private String passwd;
	private String driver;
	

	/**
	 * Poolable datasource map that provides the connections.
	 * One datasource per database.
	 */
	private static Map<String, BasicDataSource> dataSources = new HashMap<String, BasicDataSource>();
	
	private static BiomajLogger logger = BiomajLogger.getInstance();

	public TestConnection() {
		dbPath = BiomajUtils.getBiomajRootDirectory() + "/" + SQL_DIR;
		
		url = "jdbc:hsqldb:mem:testdb";
		login = "sa";
		passwd = "";
		driver = "org.hsqldb.jdbcDriver";
		
		createDB();
	}
	
	private Connection getConnection() {
		BasicDataSource bds;
		if ((bds = dataSources.get(url)) == null) {
			bds = new BasicDataSource();
			bds.setDriverClassName(driver);
			bds.setUsername(login);
			bds.setPassword(passwd);
			bds.setUrl(url);
			bds.setValidationQuery("select 1 from INFORMATION_SCHEMA.SYSTEM_USERS");
	
			dataSources.put(url, bds);
		}

		try {
			logger.log("===> " + bds.getNumActive() + " active connections // " + bds.getNumIdle() + " idle connections");
			return bds.getConnection();
		} catch (SQLException e) {
			System.err.println("No connection could be established. Please check that the server is started" +
					" and that the connection parameters are correct.");
			e.printStackTrace();
			return null;
		}
	}
	
	@Override
	public Statement getStatement() {
		try {
			return getConnection().createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY);
		} catch (SQLException e) {
			e.printStackTrace();
			return null;
		}
		
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean createDB() {
		Statement statement = getStatement();
		try {
			String fileName = dbPath + "/hsql.sql";
			try {
				BufferedReader br = new BufferedReader(new FileReader(fileName));
				String line = "";
				StringBuilder sb = new StringBuilder();
				while ((line = br.readLine()) != null) {
					if (!line.trim().isEmpty() && !line.startsWith("--")) {
						sb.append(line);
						if (line.endsWith(";")) {
//							statement.addBatch(sb.toString());
							statement.executeUpdate(sb.toString());

							sb = new StringBuilder();
						}
//						statement.executeUpdate(line);
					}
				}
				br.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
			SQLConnectionFactory.closeConnection(statement);
//			statement.executeBatch();
		} catch (SQLException e) {
			SQLConnectionFactory.closeConnection(statement);
			e.printStackTrace();
			return false;
		}
		return true;
	}
	
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized long getLastInsertedId() {
		long id = -1;
		Statement st = getStatement();
		try {
			ResultSet rs = executeQuery("CALL identity()", st);
			rs.next();
			id = Long.valueOf(rs.getObject(1).toString());
		} catch (SQLException e) {
			e.printStackTrace();
		} catch (NumberFormatException e) {
			System.err.println("Invalid id : NaN");
		}

		SQLConnectionFactory.closeConnection(st);
		return id;
	}

	@Override
	public ResultSet executeQuery(String query, Statement stat) {
		
		long start = new Date().getTime();
		try {
			ResultSet rs = stat.executeQuery(query);
			logger.log((new Date().getTime() - start) + "ms : " + query);
			return rs;
		} catch (SQLException e) {
			logger.log(e);
			e.printStackTrace();
			if (e.getMessage().contains("Connection is closed"))
				System.exit(1);
		}
		return null;
	}

	@Override
	public int executeUpdate(String query, Statement stat) {
		long start = new Date().getTime();
		try {
			int res = stat.executeUpdate(query);
			logger.log((new Date().getTime() - start) + "ms : " + query);
			return res;
		} catch (SQLException e) {
			logger.log(e);
			e.printStackTrace();
			if (e.getMessage().contains("Connection is closed"))
				System.exit(1);
		}
		return -1;
	}
	
	@Override
	public int executeUpdateAndGetGeneratedKey(String query, Statement stat) {
		long start = new Date().getTime();
		try {
			int res = stat.executeUpdate(query, Statement.RETURN_GENERATED_KEYS);
			logger.log((new Date().getTime() - start) + "ms : " + query);
			if (res > 0) {
				ResultSet rs = stat.getGeneratedKeys();
				rs.next();
				return rs.getInt(1);
			}
		} catch (SQLException e) {
			logger.log(e);
			e.printStackTrace();
			if (e.getMessage().contains("Connection is closed"))
				System.exit(1);
		}
		return -1;
	}

}
