
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nickycopeland@hotmail.com> 1996,2008
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <unistd.h>

#include <bristol.h>
#include <bristolmidiapi.h>
#include <bristolmidi.h>

extern void initBristolAudio();
extern void allNotesOff();

int
getsid(Baudio *baudio, int sid)
{
	int tsid;

	if (baudio)
		tsid = getsid(baudio->next, sid);
	else
		return(sid);

/*	printf("getsid(%x, %i): %i\n", baudio, sid, tsid); */

	if (baudio->sid == tsid)
		return(++tsid);

/*	printf("nomatch: %i\n", tsid); */

	return(tsid);
}

int
bristolSystem(audioMain *audiomain, bristolMidiMsg *msg)
{
	int result = 0;

/*	printf("bristolSystem(%i, %i, %i) %x\n", */
/*		msg->params.bristol.operator, */
/*		msg->params.bristol.controller, */
/*		msg->params.bristol.valueMSB, */
/*		msg->params.bristol.valueLSB, audiomain); */

	/*
	 * We are looking for a "hello" message from the GUI, which should include
	 * an algorithm index that we shall be running. We also need to look out 
	 * for exit requests.
	 */
	if (msg->params.bristol.operator != BRISTOL_SYSTEM)
		return(0);

	if (msg->params.bristol.controller == 0)
	{
		/*
		 * These are system global messages - start/stop/hello.
		 */
		int flags = (msg->params.bristol.valueMSB << 7) +
			msg->params.bristol.valueLSB;

		/*
		 * See if we can activate message connections
		 */
		if (flags & BRISTOL_HELLO)
		{
			Baudio *baudio = (Baudio *) bristolmalloc0(sizeof(Baudio));

			if (audiomain->atStatus == BRISTOL_EXIT)
				audiomain->midiflags |= BRISTOL_HELLO;

			/*
			 * We assume that anyone attempting a hello is going to want access
			 * to a bristolAudio channel. We should allocate this here, and give
			 * it an ID for the connection. This ID is returned in an ack back
			 * to the source.
			 */
			baudio->midichannel = msg->params.bristol.channel;
			baudio->mixflags = BRISTOL_HOLDDOWN;
			baudio->samplerate = audiomain->samplerate;
			baudio->samplecount = audiomain->samplecount;

			/*
			 * Put in some default arpeggio values
			 */
			bristolArpeggiatorInit(baudio);

			baudio->sid = result = getsid(audiomain->audiolist, 0);
			if (baudio->sid == -1)
				printf("SID failure....\n");

			baudio->voicecount = (BRISTOL_PARAMMASK & flags);
			if (audiomain->voiceCount <= 0)
			{
				if ((audiomain->voiceCount = (BRISTOL_PARAMMASK & flags))
					< BRISTOL_VOICECOUNT)
					audiomain->voiceCount = BRISTOL_VOICECOUNT;
			}
			if (audiomain->voiceCount > BRISTOL_MAXVOICECOUNT)
				audiomain->voiceCount = BRISTOL_MAXVOICECOUNT;
			if (baudio->voicecount > BRISTOL_MAXVOICECOUNT)
				baudio->voicecount = BRISTOL_MAXVOICECOUNT;

			printf("created %i voices: allocated %i to synth\n",
				audiomain->voiceCount, baudio->voicecount);

			if (audiomain->voiceCount <= 1)
				audiomain->voiceCount = 1;

			baudio->controlid = msg->params.bristol.from;

			baudio->next = audiomain->audiolist;
			if (audiomain->audiolist != NULL)
				audiomain->audiolist->last = baudio;
			audiomain->audiolist = baudio;
		}

		if ((audiomain->midiflags & BRISTOL_HELLO) == 0)
			return(0);

		/*
		 * See what command we have, if any.
		 */
		switch (flags & BRISTOL_COMMASK)
		{
			case BRISTOL_REQSTART:
				if (audiomain->atStatus == BRISTOL_EXIT) {
					audiomain->atReq |= BRISTOL_REQSTART;

					while (audiomain->atStatus != BRISTOL_OK)
						usleep(100000);
				}
				break;
			case BRISTOL_REQSTOP:
				audiomain->atReq = BRISTOL_REQSTOP;
				break;
			case BRISTOL_DEBUGON:
				printf("Debugging enabled\n");
				audiomain->debuglevel = -1;
				break;
			case BRISTOL_DEBUGOF:
				printf("Debugging disabled\n");
				audiomain->debuglevel = 0;
				break;
			case BRISTOL_VOICES:
			{
				Baudio *baudio;

				if (audiomain->voiceCount <= 0)
					audiomain->voiceCount = (BRISTOL_PARAMMASK & flags);
				if (audiomain->voiceCount <= 1)
					audiomain->voiceCount = 1;

				if (audiomain->debuglevel >= 0)
					printf("configured %i voices\n", audiomain->voiceCount);

				baudio = audiomain->audiolist;
				while (baudio != NULL)
				{
					if ((baudio->sid == msg->params.bristol.channel)
							|| (msg->params.bristol.channel == SYSTEM_CHANNEL))
						break;

					baudio = baudio->next;
				}

				/*
				 * If we found an associated bristolAudio then configure its
				 * voices as well, since it may be fewer than the total number
				 * of configured voices.
				 */
				if (baudio != NULL)
				{
					baudio->voicecount = (BRISTOL_PARAMMASK & flags);
					if (audiomain->debuglevel >= 0)
						printf("configured %i channel voices\n",
							baudio->voicecount);
				}

				break;
			}
			case BRISTOL_INIT_ALGO:
			{
				Baudio *baudio;

				if (audiomain->atStatus != BRISTOL_OK)
					break;

				baudio = audiomain->audiolist;
/*printf("init:	%i %i\n", baudio->controlid, msg->params.bristol.from); */
				while (baudio != NULL)
				{
					if (baudio->sid == msg->params.bristol.channel)
						break;

					baudio = baudio->next;
				}

				audiomain->midiflags &= ~BRISTOL_PARAMMASK;
				audiomain->midiflags |= (BRISTOL_PARAMMASK & flags);

				initBristolAudio(audiomain, baudio);
				break;
			}
			case BRISTOL_EXIT_ALGO:
			{
				Baudio *baudio = findBristolAudio(audiomain->audiolist,
					msg->params.bristol.channel, 0);

				if ((baudio == 0) || (baudio->mixflags == 0))
					return(0);

				printf("exit algo: %i, %x, %x\n",
					msg->params.bristol.channel,
						(size_t) baudio, (size_t) baudio->mixflags);

				allNotesOff(audiomain, -1);

				if (baudio != NULL)
					baudio->mixflags = (BRISTOL_HOLDDOWN | BRISTOL_REMOVE);
				sleep(1);

				break;
			}
			/*
			 * This is broken, or we need to find every synth using this
			 * channel?
			 */
			case (BRISTOL_TRANSPOSE & BRISTOL_COMMASK):
			{
				Baudio *baudio = findBristolAudio(audiomain->audiolist,
					msg->params.bristol.channel, 0);

				if (baudio == NULL)
					break;

				/*
				 * We should also consider restating existing frequencies but
				 * that is for later. We should support negative transposes.
				 *
				 * Negative transposes can be done for 2 octaves only - we only
				 * support +ve numbers in messaging so we added in 24 to the
				 * flag that we remove now.
				 */
				baudio->transpose = ((int) (BRISTOL_PARAMMASK & flags)) - 24;

				if (baudio->midiflags & BRISTOL_MIDI_DEBUG1)
					printf("tranpose now %i\n", baudio->transpose);

				break;
			}
			case BRISTOL_HOLD:
			{
				Baudio *baudio = findBristolAudio(audiomain->audiolist,
					msg->params.bristol.channel, 0);

				if (baudio == NULL)
					break;

				if (BRISTOL_PARAMMASK & flags)
				{
					baudio->mixflags |= BRISTOL_KEYHOLD;
				} else {
					baudio->mixflags &= ~BRISTOL_KEYHOLD;
					allNotesOff(audiomain, baudio->midichannel);
				}

				break;
			}
			case BRISTOL_LOWKEY:
			{
				Baudio *baudio = findBristolAudio(audiomain->audiolist,
					msg->params.bristol.channel, 0);

				if (baudio == NULL)
					break;

				baudio->lowkey = (BRISTOL_PARAMMASK & flags);
printf("l/h: %i, %i\n", baudio->lowkey, baudio->highkey);

				break;
			}
			case BRISTOL_HIGHKEY:
			{
				Baudio *baudio = findBristolAudio(audiomain->audiolist,
					msg->params.bristol.channel, 0);

				if (baudio == NULL)
					break;

				baudio->highkey = (BRISTOL_PARAMMASK & flags);
printf("l/h: %i, %i\n", baudio->lowkey, baudio->highkey);

				break;
			}
			case BRISTOL_UNISON:
			{
				Baudio *baudio = findBristolAudio(audiomain->audiolist,
					msg->params.bristol.channel, 0);

				if (msg->params.bristol.valueLSB > 0)
					baudio->mixflags |= BRISTOL_V_UNISON;
				else
					baudio->mixflags &= ~BRISTOL_V_UNISON;
				break;
			}
			case BRISTOL_ALL_NOTES_OFF:
			{
				allNotesOff(audiomain, -1);
				break;
			}
			case BRISTOL_MIDICHANNEL:
			{
				Baudio *baudio;

				allNotesOff(audiomain, -1);

				baudio = audiomain->audiolist;

				while (baudio != NULL)
				{
					if (baudio->sid == msg->params.bristol.channel)
						break;

					baudio = baudio->next;
				}

				/*
				 * We may need to put an all notes off in here.
				 */
				if (baudio != NULL)
					baudio->midichannel = flags & BRISTOL_PARAMMASK;

				break;
			}
			default:
				break;
		}

		/*
		 * return a message, with the controlid assumably at the head
		 * of the playlist - we have just created it.
		 */
		msg->params.bristol.valueLSB = result;
		bristolMidiSendMsg(msg->params.bristol.from, 0, 127, 0, result);
/*			audiomain->audiolist->controlid); */
	}
	return(0);
}

