/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BSCommander; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "Packer.h"
#include "Shared.h"
#include "InfoField.h"
#include "Busy.h"
#include <qlayout.h>
#include <qcombobox.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qlabel.h>
#include <qgroupbox.h>
#include <qbuttongroup.h>
#include <qframe.h>
#include <qcheckbox.h>
#include <qinputdialog.h>
#include <qprocess.h>
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/
const QString Packer::Caption             = QT_TR_NOOP( "Packing" );
const QString Packer::InfoTitel           = QT_TR_NOOP( "Source and destination" );
const QString Packer::SrcLabel            = QT_TR_NOOP( "Source:" );
const QString Packer::DstLabel            = QT_TR_NOOP( "Destination: " );
const QString Packer::CompressLabel       = QT_TR_NOOP( "&Compression" );
const QString Packer::RemoveLabel         = QT_TR_NOOP( "&Remove source files" );
const QString Packer::InputCaption        = QT_TR_NOOP( "New file name" );
const QString Packer::InputPrompt         = QT_TR_NOOP( "Packing file name:" );
const QString Packer::Gzip                = "gzip (gz)";
const QString Packer::Bzip2               = "bzip2 (bz2)";
const QString Packer::Zip                 = "zip (zip)";
const QString Packer::TarExt              = ".tar";
const QString Packer::GzipExt             = ".gz";
const QString Packer::Bzip2Ext            = ".bz2";
const QString Packer::ZipExt              = ".zip";

//*******************************************************************
// OneFileCopier
//*******************************************************************
Packer::Packer( QWidget* const parent, const ViewTable::SelectedItems& in_items, const QString& in_dst_dir )
: QDialog        ( parent )
, d_items        ( in_items )
, d_dst_dir      ( in_dst_dir )
//.........................................................
, d_info_gb      ( new QGroupBox( 2, Qt::Horizontal, tr( InfoTitel ), this ) )
, d_info_frm     ( new QFrame( d_info_gb ) )
, d_info_gl      ( new QGridLayout( d_info_frm, 2, 2, 4 ) )
, d_info_src_lb  ( new QLabel( tr( SrcLabel ), d_info_frm ) )
, d_info_src_fld ( new InfoField( d_info_frm ) )
, d_info_dst_lb  ( new QLabel( tr( DstLabel ), d_info_frm ) )
, d_info_dst_fld ( new InfoField( d_info_frm ) )
//.........................................................
, d_button_layout( new QHBoxLayout )
, d_compress_cb  ( new QCheckBox  ( tr( CompressLabel ), this ) )
, d_remove_cb    ( new QCheckBox  ( tr( RemoveLabel), this ) )
, d_run_btn      ( new QPushButton( tr( Shared::RunBtnLabel ), this ) )
, d_exit_btn     ( new QPushButton( tr( Shared::CloseBtnLabel ), this ) )
//.........................................................
, d_compress_grp ( new QButtonGroup( 4, Qt::Horizontal, QString::null, this ) )
, d_gz_rb        ( new QRadioButton( Gzip, d_compress_grp ) )
, d_bz2_rb       ( new QRadioButton( Bzip2, d_compress_grp ) )
, d_zip_rb       ( new QRadioButton( Zip, d_compress_grp ) )
, d_level_cbox   ( new QComboBox( d_compress_grp ) )
//.........................................................
, d_main_layout  ( new QVBoxLayout( this ) )
, d_pack_fname   ( "" )
, d_pack_fpath   ( "" )
, d_paused       ( FALSE )
, d_started      ( FALSE )
, d_stopped      ( FALSE )
, d_fm           ( font() )
, d_idx          ( 0 )
, d_process      ( 0 )
, d_compress_type( TAR )
{
	setCaption( tr( Caption ) );
	
	d_info_gl->setColStretch( 1, Shared::OverStretch );
	d_info_gl->addWidget( d_info_src_lb, 0, 0, Qt::AlignRight );
	d_info_gl->addWidget( d_info_src_fld, 0, 1 );
	d_info_gl->addWidget( d_info_dst_lb, 1, 0, Qt::AlignRight );
	d_info_gl->addWidget( d_info_dst_fld, 1, 1 );

	d_button_layout->setMargin( Shared::LayoutMargin );
	d_button_layout->setSpacing( Shared::LayoutSpacing );
	d_button_layout->addWidget( d_compress_cb );
	d_button_layout->addWidget( d_remove_cb );
	d_button_layout->addStretch( Shared::OverStretch );
	d_button_layout->addWidget( d_run_btn );
	d_button_layout->addWidget( d_exit_btn );
	d_compress_cb->setChecked( FALSE );
	d_remove_cb->setChecked( FALSE );
	
	d_compress_grp->setEnabled( FALSE );
	d_gz_rb->setChecked( TRUE );
	for( int i = 9; i > 0; --i ) {
		d_level_cbox->insertItem( QString::number( i ), -i + 9 );
	}

	d_main_layout->setMargin( Shared::LayoutMargin  );
	d_main_layout->setSpacing( Shared::LayoutSpacing );
	d_main_layout->addWidget( d_info_gb );
	d_main_layout->addLayout( d_button_layout );
	d_main_layout->addWidget( d_compress_grp );
	
	connect( d_compress_cb, SIGNAL( clicked()       ), this, SLOT( compress_selected()  ) );
	connect( d_gz_rb      , SIGNAL( toggled( bool ) ), this, SLOT( type_changed( bool ) ) );
	connect( d_bz2_rb     , SIGNAL( toggled( bool ) ), this, SLOT( type_changed( bool ) ) );
	connect( d_zip_rb     , SIGNAL( toggled( bool ) ), this, SLOT( type_changed( bool ) ) );
	connect( d_run_btn    , SIGNAL( clicked()       ), this, SLOT( run()                ) );
	connect( d_exit_btn   , SIGNAL( clicked()       ), this, SLOT( break_work()         ) );
}
// end of Packer

//*******************************************************************
// ~Packer
//*******************************************************************
Packer::~Packer()
{
	end_process();
}
// end of ~Packer

//*******************************************************************
// polish                                                    PRIVATE
//-------------------------------------------------------------------
// Zainicjowanie wartosci pola z informacja o startowym pliku,
// od ktorego zacznie sie pakowanie.
// Oczyiwscie pod warunkiem, ze przekazano jakies pliki.
//*******************************************************************
void Packer::polish()
{
	if( FALSE == d_items.empty() ) {
		display( d_items[0]->path() );
	}
}
// end of polish

//*******************************************************************
// break_work                                           PRIVATE slot
//-------------------------------------------------------------------
// Brutalne przerwanie pracy (nacisnieto klawisz 'break').
// 'Ubicie' procesu i opuszczenie dialogu.
//*******************************************************************
void Packer::break_work()
{
	end_process();
	reject();
}
// end of break_work

//*******************************************************************
// compress_selected                                    PRIVATE slot
//-------------------------------------------------------------------
// Uzytkownik zmienil stan checkbox'a 'd_compress_cb'.
//*******************************************************************
void Packer::compress_selected()
{
	d_compress_grp->setEnabled( d_compress_cb->isChecked() );
	display_dst();
}
// end of compress_selected

//*******************************************************************
// type_changed                                         PRIVATE slot
//*******************************************************************
void Packer::type_changed( bool in_check )
{
	if( in_check ) display_dst();
}
// end of type_changed

//*******************************************************************
// display_dst                                               PRIVATE
//*******************************************************************
void Packer::display_dst()
{
	QString str = path2display();
	Shared::clip_path( d_fm, d_info_dst_fld->width(), str );
	d_info_dst_fld->setText( str );
}
// end of display_dst

//*******************************************************************
// path2display                                             PRIVATE
//-------------------------------------------------------------------
// Z docelowego katalogu oraz nazwy pliku tworzymy docelowa sciezke
// okreslajaca spakowany plik wraz z WLASCIWYMI rozszerzeniami.
//*******************************************************************
QString Packer::path2display()
{
	QString result = "";

	const QString fpath = d_dst_dir + "/" + d_pack_fname;
	const QString fpath_tar = fpath + TarExt;
	
	if( d_compress_cb->isChecked() ) {
		if( d_gz_rb->isChecked() ) {
			result = fpath_tar + GzipExt;
			d_compress_type = GZIP;
		}
		else if( d_bz2_rb->isChecked() ) {
			result = fpath_tar + Bzip2Ext;
			d_compress_type = BZIP2;
		}
		else if( d_zip_rb->isChecked() ) {
			result = fpath + ZipExt;
			d_compress_type = ZIP;
		}
	}
	else {
		result = fpath_tar;
		d_compress_type = TAR;
	}

	return result;
}
// end of get_packed_path

//*******************************************************************
// remove_ext                                                PRIVATE
//*******************************************************************
void Packer::remove_ext( QString& in_fname )
{
	static const QString ext[] = { TarExt, GzipExt, Bzip2Ext, ZipExt };
	static const int n = sizeof( ext ) / sizeof( ext[0] );

	bool found = TRUE;
	while( found ) {
		found = FALSE;
		for( int i = 0; i < n; ++i ) {
			if( in_fname.endsWith( ext[i], FALSE ) ) {
				found = TRUE;
				in_fname.truncate( in_fname.length() - ext[i].length() );
			}
		}
	}
}
// end of remove_ext

//*******************************************************************
// display                                                   PRIVATE
//*******************************************************************
void Packer::display( const QString& in_fname )
{
	QString str = in_fname;	
	Shared::clip_path( d_fm, d_info_src_fld->width(), str );
	d_info_src_fld->setText( str );
}
// end of display

//*******************************************************************
// show                                                       PRIVAT
//------
// Reimplementacja funkcji show(). 
// Robimy to w celu wyswietlania wczesniej dialogu edycyjnego,
// w ktorym uzytkownik musi (!) wprowadzic nazwe pliku wynikowego.
//*******************************************************************
void Packer::show()
{
	bool ok;
	d_pack_fname = QInputDialog::getText(	tr( InputCaption ), tr( InputPrompt ),
														QLineEdit::Normal, QString::null,
														&ok, this );
	if( ok ) {
		remove_ext( d_pack_fname );
		display_dst();
		QDialog::show();
	}
	else {
		reject();
	}
}
// end of show

//*******************************************************************
// reject                                               PRIVATE slot
//-------------------------------------------------------------------
// Przeciazenie funkcji 'reject()'.
// Funkcje ta przeciazamy, aby miec mozliwosc usuniecia pliku
// wynikowego.
//*******************************************************************
void Packer::reject()
{
	QFile::remove( d_pack_fpath );
	QDialog::reject();
}
// end of reject

//###################################################################
//#                                                                 #
//#                           THREAD                                #
//#                                                                 #
//###################################################################

//*******************************************************************
// run                                                  PRIVATE slot
//-------------------------------------------------------------------
// Start procesu pakowania.
//*******************************************************************
void Packer::run()
{
	d_process = new QProcess( this );
	if( d_process ) {
		d_compress_cb->setEnabled( FALSE );
		d_remove_cb->setEnabled( FALSE );
		d_run_btn->hide();

		connect( d_process, SIGNAL( readyReadStdout() ), this, SLOT( message() ));
		connect( d_process, SIGNAL( readyReadStderr() ), this, SLOT( error()   ));
		connect( d_process, SIGNAL( processExited()   ), this, SLOT( finish()  ));

		d_idx = 0;
		d_pack_fpath = d_dst_dir + "/" + d_pack_fname;
		d_pack_fpath += ( ( ZIP == d_compress_type ) ? ZipExt : TarExt );
		
		d_process->setWorkingDirectory( d_items[ 0 ]->fi().dirPath( TRUE ) );
		process();
	}
	else {
		reject();
	}
}
// end of run

//*******************************************************************
// process                                                   PRIVATE
//-------------------------------------------------------------------
// Wystartowanie procesu (juz utworzonego) dla kolejnego pliku.
// To, dla ktorego pliku zostanie uruchomiony proces okresla
// globalna zmienne 'd_idx'.
// UWAGA: Nalezy pamietac, ze tylko raz, na poczatku tworzymy plik,
// pozniej dokonujemy tylko 'update' pliku juz istniejacego.
//*******************************************************************
void Packer::process()
{
	d_process->clearArguments();
	
	// TAR, GZIP, BZIP2
	if( d_compress_type != ZIP ) {
		d_process->addArgument( "tar" );
		d_process->addArgument( "--verbose" );
		d_process->addArgument( ( 0 == d_idx ) ? "--create" : "--update" );
		d_process->addArgument( "--directory=" + d_items[ 0 ]->fi().dirPath( TRUE ) );
		d_process->addArgument( "--file=" + d_pack_fpath );
		d_process->addArgument( d_items[ d_idx++ ]->name() );
	}
	// ZIP
	else {
		d_process->addArgument( "zip" );
		d_process->addArgument( "-" + d_level_cbox->currentText() );
		d_process->addArgument( ( 0 == d_idx ) ? "" : "-u" );
		d_process->addArgument( "-r" );
		d_process->addArgument( d_pack_fpath );
		d_process->addArgument( d_items[ d_idx++ ]->name() );
	}
	
	// No to do roboty.
	Busy::set_busy( TRUE );
	const bool retval = d_process->start();
	if( FALSE == retval ) {
		end_process();
		reject();
	}
}
// end of process

//*******************************************************************
// message                                              PRIVATE slot
//*******************************************************************
void Packer::message()
{
	const QString msg = d_process->readLineStdout();
	display( msg );
	Shared::idle();
}
// end of message

//*******************************************************************
// error                                                PRIVATE slot
//-------------------------------------------------------------------
// Funkcja obslugujaca komunikaty o bledach, ktore zwraca 
// pracujacy 'tar'.
//*******************************************************************
void Packer::error()
{
	const QString msg = d_process->readLineStderr();
	Shared::idle();
}
// end of error

//*******************************************************************
// finish                                               PRIVATE slot
//-------------------------------------------------------------------
// Obsluga zakonczenia procesu.
// Jesli zakonczyl sie prawidlowo ale zadanie nie zostalo w
// calosci zakonczone, ponownie zostanie wywolane wykonanie kolejnego
// kroku zadania.
// W przeciwnym przypadku konczymy prace.
//*******************************************************************
void Packer::finish()
{
	const int status = d_process->exitStatus();

	if(( 0 == status ) && ( d_idx != d_items.size())) {
		process();
	}
	else {
		// Tak czy inaczej proces sie zakonczyl.
		end_process();
		Busy::set_busy( FALSE );
		if( 0 == status ) {
			if( ( GZIP  == d_compress_type ) || ( BZIP2 == d_compress_type ) ) {
				pack();
			}
			// Poprawne zakonczenie dialogu.
			accept();
		}
		else {
			// Zakonczenie pracy dialogu z bledem.
			reject();
		}
	}
}
// end of finish

//*******************************************************************
// pack                                                      PRIVATE
//-------------------------------------------------------------------
// Funkcja sluzaca do wywolania programu pakujacego 'gzip'.
// Nie robimy juz nic z nazwami plikow i rozszerzeniami.
// Gzip robi to sam.
// Wybrany jest sredni poziom kompresji (-1 ... -9).
//*******************************************************************
void Packer::pack()
{
	QString cmd;
	
	     if( GZIP  == d_compress_type ) cmd = "gzip";
	else if( BZIP2 == d_compress_type ) cmd = "bzip2";
	cmd += ( " -" + d_level_cbox->currentText() + " " + d_pack_fpath );
	
	Busy::set_busy( TRUE );
	const int retval = system( cmd.ascii() );
	if( -1 == retval ) {
	}
	Busy::set_busy( FALSE );
}
// end of pack

//*******************************************************************
// end_process                                          PRIVATE slot
//-------------------------------------------------------------------
// Zakonczenie procesu i likwidacja klasy 'QProcess'.
// Jesli przez jakis przypadek process jeszcze pracuje zostanie
// on ubity.
//*******************************************************************
void Packer::end_process()
{
	if( d_process ) {
		if( TRUE == d_process->isRunning()) {
			d_process->tryTerminate();
			QTimer::singleShot( 5000, d_process, SLOT( kill() ));
		}
		delete d_process;
		d_process = 0;
	}
}
// end of end_process
