/********************************************************************
 * Copyright (C) 2005 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BsC (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/
#ifndef INCLUDED_VIEWTABLEITEM_H
#define INCLUDED_VIEWTABLEITEM_H

/*------- included files:
-------------------------------------------------------------------*/
#ifndef INCLUDED_SHARED_H
#include "Shared.h"
#endif // INCLUDED_SHARED_H

#ifndef INCLUDED_QLISTVIEW_H
#include <qlistview.h>
#define INCLUDED_QLISTVIEW_H
#endif // INCLUDED_QLISTVIEW_H

#ifndef INCLUDED_QURLINFO_H
#include <qurlinfo.h>
#define INCLUDED_QURLINFO_H
#endif // INCLUDED_QURLINFO_H

#ifndef INCLUDED_QFILEINFO_H
#include <qfileinfo.h>
#define INCLUDED_QFILEINFO_H
#endif // INCLUDED_QFILEINFO_H

/*------- class declaration:
-------------------------------------------------------------------*/
class ViewTableItem : public QListViewItem
{
// ******* CONSTRUCTION *******
public:
	ViewTableItem( QListView* parent,
		QString label1, QString label2, QString label3, QString label4,
		QString label5, QString label6, QString label7, QString label8,
		const bool in_is_ftp = FALSE )
	: QListViewItem( parent,
		label1, label2, label3, label4,
		label5, label6, label7, label8 )
	, d_is_ftp( in_is_ftp )
	, d_is_size_computed( FALSE )
	{}
private:
	ViewTableItem( const ViewTableItem& );
	ViewTableItem& operator=( const ViewTableItem& rhs );

// ******* MEMBERS *******
private:
	static const QColor BG_NoFocusSelected;
	static const QColor BG_NoFocusCurrent;
	static const QColor BG_NoFocusCurrentAndSelected;

	static const QColor FOCUSED_NUMBER_TEXT;
	static const QColor FOCUSED_NUMBER_SELTEXT;
	static const QColor NOFOCUSED_NUMBER_TEXT;
	static const QColor DEFAULT_CURRENT_SELECTED_FG;
	static const QColor DEFAULT_CURRENT_SELECTED_BG;
	static const QColor DEFAULT_CURRENT_FG;
	static const QColor DEFAULT_CURRENT_BG;
	static const QColor DEFAULT_SELECTED_FG;
	static const QColor DEFAULT_SELECTED_BG;

	bool      d_is_ftp;
	bool      d_is_size_computed;
	QUrlInfo  d_ui;
	QFileInfo d_fi;
	
// ******* METHODS *******
public:
	void set_is_size_computed();
	void set_data            ( const QUrlInfo& );
	void set_data            ( const QFileInfo& );
	
	bool      is_lfs           () const;
	bool      is_dir           () const;
	bool      is_readable      () const;
	bool      is_writable      () const;
	bool      is_executable    () const;
	bool      is_parent_dir    () const;
	int       size             () const;
	QDateTime last_modified    () const;
	QString   path             () const;
	QString   name             () const;
	const QFileInfo& fi        () const;
private:
	void paintCell( QPainter*, const QColorGroup&, int, int, int );
	int  compare  ( QListViewItem*, int, bool ) const;
};

// end of compare

//*******************************************************************
// set_is_size_computed                                PUBLIC inline
//*******************************************************************
inline void ViewTableItem::set_is_size_computed()
{
	d_is_size_computed = TRUE;
}
// end of set_is_size_computed

//*******************************************************************
// set_data                                            PUBLIC inline
//*******************************************************************
inline void ViewTableItem::set_data( const QUrlInfo& in_ui )
{
	d_ui = in_ui;
}
// end of set_data

//*******************************************************************
// set_data                                            PUBLIC inline
//*******************************************************************
inline void ViewTableItem::set_data( const QFileInfo& in_fi )
{
	d_fi = in_fi;
}
// end of set_data

//*******************************************************************
// is_lfs                                              PUBLIC inline
//*******************************************************************
inline bool ViewTableItem::is_lfs() const
{
	return ( FALSE == d_is_ftp );
}
// end of is_lfs

//*******************************************************************
// is_dir                                              PUBLIC inline
//*******************************************************************
inline bool ViewTableItem::is_dir() const
{
	return ( d_is_ftp ) ? d_ui.isDir() : d_fi.isDir();
}
// end of is_dir

//*******************************************************************
// is_readable                                         PUBLIC inline
//*******************************************************************
inline bool ViewTableItem::is_readable() const
{
	return ( d_is_ftp ) ? d_ui.isReadable() : d_fi.isReadable();
}
// end of is_readable

//*******************************************************************
// is_writable                                         PUBLIC inline
//*******************************************************************
inline bool ViewTableItem::is_writable() const
{
	return ( d_is_ftp ) ? d_ui.isWritable() : d_fi.isWritable();
}
// end of is_readable

//*******************************************************************
// is_executable                                       PUBLIC inline
//*******************************************************************
inline bool ViewTableItem::is_executable() const
{
	return ( d_is_ftp ) ? d_ui.isExecutable() : d_fi.isExecutable();
}
// end of is_executable

//*******************************************************************
// size                                                PUBLIC inline
//*******************************************************************
inline int ViewTableItem::size() const
{
	return ( d_is_ftp ) ? d_ui.size() : d_fi.size();
}
// end of size

//*******************************************************************
// date_time                                           PUBLIC inline
//*******************************************************************
inline QDateTime ViewTableItem::last_modified() const
{
	return ( d_is_ftp ) ? d_ui.lastModified() : d_fi.lastModified();
}
// end of date_time

//*******************************************************************
// is_parent_dir                                       PUBLIC inline
//*******************************************************************
inline bool ViewTableItem::is_parent_dir() const
{
	return ( d_is_ftp )	? ( Shared::ParentDir == d_ui.name() )
								: ( Shared::ParentDir == d_fi.fileName() );
}
// end of is_executable


//*******************************************************************
// get_abs_path                                       PUBLIC inline
//*******************************************************************
inline QString ViewTableItem::path() const
{
	return ( d_is_ftp ) ? d_ui.name() : d_fi.absFilePath();
}
// end of get_abs_path

//*******************************************************************
// get_fname                                           PUBLIC inline
//*******************************************************************
inline QString ViewTableItem::name() const
{
	return ( d_is_ftp ) ? d_ui.name() : d_fi.fileName();
}
// end of get_fname

//*******************************************************************
// fi                                                  PUBLIC inline
//*******************************************************************
inline const QFileInfo& ViewTableItem::fi() const { return d_fi; }


#endif // INCLUDED_VIEWTABLEITEM_H
