# Written by Henrik Nilsen Omma
# (C) Canonical, Ltd. Licensed under the GPL

import libxml2
import urllib2
import os

from infoFilesDefinitions import clueCondition as clueCondition
from infoFilesDefinitions import bugClue as bugClue

# - The first number has to be bumped, if the schema completely breaks, so an 
#   older version of bughelper can ignore it.
# - Please bump the second number, if you add tags or attributes.
CLUE_FILE_VERSION = "0.2"

class NoClueFoundError(Exception):
    def __init__(self, xml_files):
        self.value = "Checked at %s - no clues found. \
Check README on how to create one using the bugxml(1) command." % xml_files

    def __str__(self):
        return repr(self.value)


class InvalidClueFileError(Exception):
    def __init__(self, xml_files):
        self.value = "%s could not be parsed, please validate with bugxml(1)." % \
                    xml_file

    def __str__(self):
        return repr(self.value)
        
        
def find_schema(path, version):
    schema = os.path.join(path, "schemas/cluefile-%s.rng" % version)
    if os.path.exists(schema):
        return schema
    return ''

def get_clue_file_version(xml_file):
    try:
        doc = libxml2.parseFile(xml_file)
    except:
        return CLUE_FILE_VERSION
        
    ctxt = doc.xpathNewContext()
    clue_version = ctxt.xpathEval("/clues/@version")
    if not clue_version:
        return CLUE_FILE_VERSION

    version = str(clue_version[0].getContent())
    ctxt.xpathFreeContext()
    doc.freeDoc()
    return version


def rng_validate(schema, xml_file):
    rng_parser = libxml2.relaxNGNewParserCtxt(schema)
    rng_context = rng_parser.relaxNGParse().relaxNGNewValidCtxt()
    
    ret = 1
    try:
        xml_doc = libxml2.parseFile(xml_file)
        ret = xml_doc.relaxNGValidateDoc(rng_context)
        xml_doc.freeDoc()
    except:
        pass

    libxml2.relaxNGCleanupTypes()
    
    return ret == 0

def read_clue_files(xml_files):
    clues = set()
    inherits = set()
    dontlist = set()
    for xml_file in xml_files:
        if xml_file:
            (i, d, c) = read_clue_file(xml_file)
        clues.update(c)
        dontlist.update(d)
        inherits.update(i)
    if not inherits and not dontlist and not clues:
        raise NoClueFoundError(xml_files)
    return (inherits, dontlist, clues)


def read_clue_file(xml_file):
    if os.path.exists(xml_file):
        try:
            doc = libxml2.parseFile(xml_file)
        except:
            print "%s could not be parsed, please validate with bugxml(1)." % \
                    xml_file
            return ([], [], [])
    else:
        return ([], [], [])

    if get_clue_file_version(xml_file).split(".")[0] != \
       CLUE_FILE_VERSION.split(".")[0]:
        return ([], [], [])

    ctxt = doc.xpathNewContext()
    clues = set()
    inherits = set([n.content for n in 
                        ctxt.xpathEval("//clues//inherits//inherit/text()")])
    dontlist = set([int(n.content) for n in 
                        ctxt.xpathEval("//clues//dontlist//bug/text()")])
    clue_nodes = ctxt.xpathEval("//clues//clue")
    for clue_node in clue_nodes:
        conditions = set()
        # <info> part
        res = clue_node.xpathEval("info/text()")
        if not res:
            print "No <info> tag for clue file: %s" % xml_file
            return ([],[],[])
        info = [n.content for n in res][0]

        # conditions part
        res = clue_node.xpathEval("contains/*")
        for cond_node in res:
            conditions.add(read_condition_node(cond_node))
        clues.add(bugClue(urllib2.unquote(info), conditions))
    doc.freeDoc()
    ctxt.xpathFreeContext()
    return (inherits, dontlist, clues)


def read_condition_node(cond_node):
    and_cond = None
    or_cond = None
    simple_cond = None
    not_cond = False
    cond_field = None

    nonTextNodeAndRightParent = lambda a: not a.isText() and \
                                          a.parent == cond_node
    if cond_node.name == "and":
        if cond_node.prop("bin") == "not":
            not_cond = True    
        and_cond = [read_condition_node(n) 
                        for n in filter(nonTextNodeAndRightParent, 
                                        cond_node.children)]
    if cond_node.name == "or":
        if cond_node.prop("bin") == "not":
            not_cond = True    
        or_cond = [read_condition_node(n) 
                        for n in filter(nonTextNodeAndRightParent, 
                                        cond_node.children)]
    if cond_node.name == "op":
        if cond_node.prop("bin") == "not":
            not_cond = True
        simple_cond = urllib2.unquote(cond_node.content)
        if cond_node.prop("field") != None:
             cond_field = cond_node.prop("field")

    return clueCondition(and_cond, or_cond, simple_cond, not_cond, cond_field)

    
def xml_add_clue(filename, cond, clue, dontlist):
    libxml2.keepBlanksDefault(0)
    doc = libxml2.parseFile(filename)
    ctxt = doc.xpathNewContext()
    clues_node = ctxt.xpathEval("//clues")[0]

    clue_node = clues_node.newChild(None, "clue", None)
    if dontlist:
        dontlist_node = clue_node.newChild(None, "dontlist", None)
        for bug in dontlist:
            dontlist_node.newChild(None, "bug", bug)
    contains_node = clue_node.newChild(None, "contains", None)
    contains_node.newChild(None, "op", cond)    
    clue_node.newChild(None, "info", clue)

    doc.saveFormatFile(filename, True)
    doc.freeDoc()
    ctxt.xpathFreeContext()

def xml_create_file_with_clue(filename, srcpkg, cond, clue, dontlist):
    doc = libxml2.newDoc("1.0")
    clues_node = doc.newDocNode(None, 'clues', None)
    clues_node.newProp("version", CLUE_FILE_VERSION)
    doc.addChild(clues_node)

    clue_node = clues_node.newChild(None, "clue", None)
    if dontlist:
        dontlist_node = clue_node.newChild(None, "dontlist", None)
        for bug in dontlist:
            dontlist_node.newChild(None, "bug", bug)
    contains_node = clue_node.newChild(None, "contains", None)
    contains_node.newChild(None, "op", cond)    
    clue_node.newChild(None, "info", clue)

    doc.saveFormatFile(filename, True)
    doc.freeDoc()

def add_simple_clue(srcpkg, cond, clue, path, dontlist):
    filename = os.path.join(path, "%s.info" % srcpkg)
    if os.path.exists(filename):
        xml_add_clue(filename, cond, clue, dontlist)
    else:
        xml_create_file_with_clue(filename, srcpkg, cond, clue, dontlist)

