# bzr-avahi - share and browse Bazaar branches with mDNS
# Copyright (C) 2007-2008 James Henstridge
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import avahi
import dbus

from bzrlib import errors
from bzrlib.branch import Branch
from bzrlib.directory_service import directories
from bzrlib.tests import TestCase, TestCaseWithMemoryTransport
from bzrlib.plugins.avahi.browse import AvahiDirectoryService, service_to_uri


class ServiceToUriTests(TestCase):

    def test_service_to_uri(self):
        txt = avahi.string_array_to_txt_array(
            ['garbage', 'g=a=r=b=a=g=e'])
        self.assertEqual(service_to_uri(
                (None, None, 'service name', '_bzr._tcp', '',
                 'foo.local', avahi.PROTO_INET, '1.2.3.4', 0, txt, 0)),
                         'bzr://foo.local/')
        # With txt data:
        txt = avahi.string_array_to_txt_array(
            ['path=/path', 'scheme=bzr+http'])
        self.assertEqual(service_to_uri(
                (None, None, 'service name', '_bzr._tcp', '',
                 'foo.local', avahi.PROTO_INET, '1.2.3.4', 0, txt, 0)),
                         'bzr+http://foo.local/path')
        # Numeric:
        self.assertEqual(service_to_uri(
                (None, None, 'service name', '_bzr._tcp', '',
                 'foo.local', avahi.PROTO_INET, '1.2.3.4', 0, txt, 0), True),
                         'bzr+http://1.2.3.4/path')


class AvahiDirectoryServiceTests(TestCaseWithMemoryTransport):

    def test_look_up(self):
        class MyDS(AvahiDirectoryService):
            def _resolve(self, service_name):
                if service_name == 'testbranch':
                    return 'http://example.com/testbranch'
                else:
                    raise dbus.DBusException('bad branch')
        # We should be redirected to http://example.com
        ds = MyDS()
        self.assertEqual(ds.look_up('testbranch', 'local:testbranch'),
                         'http://example.com/testbranch')

        # With an unknown service, InvalidURL is raised:
        self.assertRaises(errors.InvalidURL, ds.look_up,
                          'unknown', 'local:unknown')

    def test_open_branch(self):
        # Test that we can actually open a branch
        branch = self.make_branch('testbranch')
        class MyDS(AvahiDirectoryService):
            def _resolve(self, service_name):
                if service_name == 'testbranch':
                    return branch.base
                else:
                    raise dbus.DBusException('bad branch')
        old_service = directories.get('local:')
        old_help = directories.get_help('local:')
        old_info = directories.get_info('local:')
        try:
            directories.register('local:', MyDS, override_existing=True)
            branch2 = Branch.open('local:testbranch')
            self.assertEqual(branch.base, branch2.base)
        finally:
            directories.register('local:', old_service, old_help, old_info,
                                 override_existing=True)
