# Written by John Hoffman
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: ServerPortHandler.py 389 2008-06-22 19:23:06Z camrdale-guest $

"""Wrappers to handle multiple torrent downloads.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type default_task_id: C{mutable}
@var default_task_id: the default task ID to use for scheduling all unspecified tasks

"""

from cStringIO import StringIO
#from RawServer import RawServer
from BTcrypto import Crypto
from binascii import b2a_hex
from BT1.Encrypter import protocol_name
import logging

logger = logging.getLogger('DebTorrent.ServerPortHandler')

default_task_id = []

class SingleRawServer:
    """Simplified Server to handle one of many torrents.
    
    This class provides a wrapper around a master L{RawServer.RawServer} 
    instance, processing requests with the same interface and passing them
    on to the master server.
    
    @type info_hash: C{string}
    @ivar info_hash: the torrent infohash this Server is responsible for
    @type identifier: C{string}
    @ivar identifier: the identifier of the torrent
    @type doneflag: C{threading.Event}
    @ivar doneflag: flag to indicate this torrent is being shutdown
    @type protocol: C{string}
    @ivar protocol: the name of the communication protocol
    @type multihandler: L{MultiHandler}
    @ivar multihandler: the collection of all individual simplified servers
    @type rawserver: L{RawServer.RawServer}
    @ivar rawserver: the master Server instance
    @type finished: C{boolean}
    @ivar finished: whether this torrent has been shutdown
    @type running: C{boolean}
    @ivar running: whether this torrent has been started and is running
    @type handler: unknown
    @ivar handler: the data handler to use to process data received on the connection
    @type taskqueue: C{list}
    @ivar taskqueue: unknown
    
    """
    
    def __init__(self, info_hash, identifier, multihandler, doneflag, protocol):
        """Initialize the instance.
        
        @type info_hash: C{string}
        @param info_hash: the torrent infohash this Server is responsible for
        @type identifier: C{string}
        @param identifier: the identifier of the torrent
        @type multihandler: L{MultiHandler}
        @param multihandler: the collection of all individual simplified servers
        @type doneflag: C{threading.Event}
        @param doneflag: flag to indicate this torrent is being shutdown
        @type protocol: C{string}
        @param protocol: the name of the communication protocol
        
        """
        
        self.info_hash = info_hash
        self.identifier = identifier
        self.doneflag = doneflag
        self.protocol = protocol
        self.multihandler = multihandler
        self.rawserver = multihandler.rawserver
        self.finished = False
        self.running = False
        self.handler = None
        self.taskqueue = []

    def shutdown(self):
        """Tell the collection to shutdown this torrent."""
        if not self.finished:
            self.multihandler.shutdown_torrent(self.identifier)

    def _shutdown(self):
        """Shutdown this torrent."""
        if not self.finished:
            self.finished = True
            self.running = False
            self.rawserver.kill_tasks(self.info_hash)
            if self.handler:
                self.handler.close_all()

    def _external_connection_made(self, c, options, already_read,
                                  encrypted = None ):
        """Processes a new socket connection to this torrent.
        
        @type c: unknown
        @param c: the new connection
        @type options: unknown
        @param options: the protocol options the connected peer supports
        @type already_read: C{string}
        @param already_read: the data that has already been read from the connection
        @type encrypted: L{BTcrypto.Crypto}
        @param encrypted: the Crypto instance to use to encrypt this connections
            communication (optional, defaults to None)
        
        """
        
        if self.running:
            c.set_handler(self.handler)
            self.handler.externally_handshaked_connection_made(
                c, options, already_read, encrypted = encrypted)

    ### RawServer functions ###

    def add_task(self, func, delay=0, id = default_task_id):
        """Passes a delayed call to a method on to the master Server.
        
        @type func: C{method}
        @param func: the method to call
        @type delay: C{int}
        @param delay: the number of seconds to delay before calling
        @type id: C{mutable}
        @param id: the ID of the task
        
        """
        
        if id is default_task_id:
            id = self.info_hash
        if not self.finished:
            self.rawserver.add_task(func, delay, id)

#    def bind(self, port, bind = '', reuse = False):
#        pass    # not handled here
        
    def start_connection(self, dns, handler = None):
        """Tell the master Server to start a new connection to a peer.
        
        @type dns: (C{string}, C{int})
        @param dns: the IP address and port number to contact the peer on
        @type handler: unknown
        @param handler: the data handler to use to process data on the connection
            (optional, defaults to the L{handler})
        @rtype: L{SocketHandler.SingleSocket}
        @return: the new connection made to the peer

        """
        
        if not handler:
            handler = self.handler
        c = self.rawserver.start_connection(dns, handler)
        return c

#    def listen_forever(self, handler):
#        pass    # don't call with this
    
    def start_listening(self, handler):
        """Start the Server listening (but not forever).
        
        @type handler: unknown
        @param handler: the default handler to call when data comes in
        @rtype: C{method}
        @return: the method to call to shutdown the torrent download
        
        """
        
        self.handler = handler
        self.running = True
        return self.shutdown    # obviously, doesn't listen forever

    def is_finished(self):
        """Check if the torrent download has been shutdown.
        
        @rtype: C{boolean}
        @return: whether the torrent has been shutdown
        
        """
        
        return self.finished

    def get_exception_flag(self):
        """Get the master Server's exception flag.
        
        @rtype: C{threading.Event}
        @return: the flag used to indicate exceptions
        
        """
        
        return self.rawserver.get_exception_flag()


class NewSocketHandler:
    """Read the handshake and hand a new socket connection off to where it belongs.
    
    This class wraps some of the functionality of the 
    L{BT1.Encrypter.Connection} class. It will receive connections from
    the Server, read the protocol handshake, assign them to the proper 
    torrent server, and pass the connection on to the Encrypter Connection.
    
    @type multihandler: L{MultiHandler}
    @ivar multihandler: the collection of all torrent Servers
    @type connection: L{SocketServer.SingleSocket}
    @ivar connection: the connection to handle
    @type closed: C{boolean}
    @ivar closed: whether the connection has been closed
    @type buffer: C{string}
    @ivar buffer: the buffer of unprocessed data received on the connection
    @type complete: C{boolean}
    @ivar complete: whether the handshake is complete
    @type read: C{method}
    @ivar read: the method to call to read data from the connection
    @type write: C{method}
    @ivar write: the method to call to write data to the connnection
    @type next_len: C{int}
    @ivar next_len: the length of the protocol name header in the connection
    @type next_func: C{method}
    @ivar next_func: the method to call to read the protocol name from the connection
    @type protocol: C{string}
    @ivar protocol: the protocol name used by the connection
    @type encrypted: C{boolean}
    @ivar encrypted: whether the connection is encrypted
    @type encrypter: L{BTcrypto.Crypto}
    @ivar encrypter: the encrypter to use for the connection
    @type _max_search: C{int}
    @ivar _max_search: the number of remaining bytes to search for the pattern
    @type options: C{string}
    @ivar options: the protocol options read from the connection
    
    """
    
    def __init__(self, multihandler, connection):
        """Initialize the instance.
        
        @type multihandler: L{MultiHandler}
        @param multihandler: the collection of all torrent Servers
        @type connection: L{SocketServer.SingleSocket}
        @param connection: the new connection to handle
        
        """
        
        self.multihandler = multihandler
        self.connection = connection
        connection.set_handler(self)
        self.closed = False
        self.buffer = ''
        self.complete = False
        self.read = self._read
        self.write = connection.write
        self.next_len, self.next_func = 1+len(protocol_name), self.read_header
        self.multihandler.rawserver.add_task(self._auto_close, 30)

    def _auto_close(self):
        """Automatically close the connection if it is not fully connected."""
        if not self.complete:
            logger.warning('Connection dropped due to handshake taking too long')
            self.close()
        
    def close(self):
        """Close the connection."""
        if not self.closed:
            self.connection.close()
            self.closed = True

    # copied from Encrypter and modified
    
    def _read_header(self, s):
        """Check if the protocol header matches.
        
        @type s: C{string}
        @param s: the data read from the conection
        @rtype: C{int}, C{method}
        @return: the next length to read and method to call with the data
            (or None if something went wrong)
        
        """
        
        if s == chr(len(protocol_name))+protocol_name:
            self.protocol = protocol_name
            return 8, self.read_options
        return None

    def read_header(self, s):
        """Process the (possibly encrypted) protocol header from the connection.
        
        @type s: C{string}
        @param s: the data read from the conection
        @rtype: C{int}, C{method}
        @return: the next length to read and method to call with the data
            (or None if something went wrong)
        
        """
        
        if self._read_header(s):
            if self.multihandler.config['crypto_only']:
                logger.info('Dropped the connection as it was unencrypted')
                return None
            return 8, self.read_options
        if not self.multihandler.config['crypto_allowed']:
            logger.info('Dropped the connection as it was encrypted')
            return None
        self.encrypted = True
        self.encrypter = Crypto(False)
        self._write_buffer(s)
        return self.encrypter.keylength, self.read_crypto_header

    def read_crypto_header(self, s):
        """Start to read an encrypted header from the connection.
        
        @type s: C{string}
        @param s: the data read from the conection
        @rtype: C{int}, C{method}
        @return: the next length to read and method to call with the data
        
        """
        
        self.encrypter.received_key(s)
        self.write(self.encrypter.pubkey+self.encrypter.padding())
        self._max_search = 520
        return 0, self.read_crypto_block3a

    def _search_for_pattern(self, s, pat):
        """Search for a pattern in the initial connection data.
        
        @type s: C{string}
        @param s: the data read from the conection
        @type pat: C{string}
        @param pat: the data to search for
        @rtype: C{boolean}
        @return: whether the pattern was found
        
        """
        
        p = s.find(pat)
        if p < 0:
            self._max_search -= len(s)+1-len(pat)
            if self._max_search < 0:
                self.close()
                return False
            self._write_buffer(s[1-len(pat):])
            return False
        self._write_buffer(s[p+len(pat):])
        return True

    def read_crypto_block3a(self, s):
        """Find the block3a crypto information in the connection.
        
        @type s: C{string}
        @param s: the data read from the conection
        @rtype: C{int}, C{method}
        @return: the next length to read and method to call with the data
        
        """
        
        if not self._search_for_pattern(s,self.encrypter.block3a):
            return -1, self.read_crypto_block3a     # wait for more data
        return 20, self.read_crypto_block3b

    def read_crypto_block3b(self, s):
        """Process the block3b crypto information in the connection.
        
        Passes the connection off to the appropriate torrent's Server if the
        correct block is found.
        
        @type s: C{string}
        @param s: the data read from the conection
        @rtype: C{boolean}
        @return: whether the crypto block was found
        
        """
        
        for k in self.multihandler.singlerawservers.keys():
            if self.encrypter.test_skey(s,k):
                self.multihandler.singlerawservers[k]._external_connection_made(
                        self.connection, None, self.buffer,
                        encrypted = self.encrypter )
                logger.info('Found the encrypted download ID: '+b2a_hex(k))
                return True
        logger.info('Dropped the encrypted connection to an unknown torrent')
        return None

    def read_options(self, s):
        """Process the protocol options from the connection.
        
        @type s: C{string}
        @param s: the data read from the conection
        @rtype: C{int}, C{method}
        @return: the next length to read and method to call with the data
        
        """
        
        self.options = s
        return 20, self.read_download_id

    def read_download_id(self, s):
        """Read the torrent infohash from the connection.
        
        Passes the connection off to the appropriate torrent's Server.
        
        @type s: C{string}
        @param s: the data read from the conection
        @rtype: C{boolean}
        @return: whether a torrent was found to assign the connection to
        
        """
        
        if self.multihandler.singlerawservers.has_key(s):
            if self.multihandler.singlerawservers[s].protocol == self.protocol:
                self.multihandler.singlerawservers[s]._external_connection_made(
                        self.connection, self.options, self.buffer)
                logger.info('Found the unencrypted download ID: '+b2a_hex(s))
                return True
        logger.info('Dropped the unencrypted connection to an unknown torrent: '+b2a_hex(s))
        return None


    def read_dead(self, s):
        """Do nothing.
        
        @type s: C{string}
        @param s: the data read from the conection
        @rtype: C{none}
        @return: None
        
        """
        
        return None

    def data_came_in(self, garbage, s):
        """Process the read data from the connection.
        
        @type garbage: unknown
        @param garbage: thrown away
        @type s: C{string}
        @param s: the data read from the conection
        
        """
        
        self.read(s)

    def _write_buffer(self, s):
        """Add the read data from the connection back onto the start of the buffer.
        
        @type s: C{string}
        @param s: the data read from the conection
        
        """
        
        self.buffer = s+self.buffer

    def _read(self, s):
        """Process the data read from the connection.
        
        Processes incoming data on the connection. The data is bufferred, then
        the L{next_func} method is called with the L{next_len} amount of the
        data. If it returns None, the connection is closed. If it returns True,
        the connection handshake is complete and the connection is established.
        Otherwise it returns C{int},C{method}, which is the next length to read 
        and method to call with the data. If the length is 0, it will read all
        the available data. If the length is -1 it will wait for more data to
        caome in.
        
        @type s: C{string}
        @param s: the data read from the conection
        
        """
        
        self.buffer += s
        while True:
            if self.closed:
                return
            # self.next_len = # of characters function expects
            # or 0 = all characters in the buffer
            # or -1 = wait for next read, then all characters in the buffer
            if self.next_len <= 0:
                m = self.buffer
                self.buffer = ''
            elif len(self.buffer) >= self.next_len:
                m = self.buffer[:self.next_len]
                self.buffer = self.buffer[self.next_len:]
            else:
                return
            try:
                x = self.next_func(m)
            except:
                logger.exception('Dropped connection due to exception')
                self.next_len, self.next_func = 1, self.read_dead
                raise
            if x is None:
                self.close()
                return
            if x == True:
                self.complete = True
                return
            self.next_len, self.next_func = x
            if self.next_len < 0:  # already checked buffer
                return             # wait for additional data


    def connection_flushed(self, ss):
        """Do nothing.
        
        @type ss: L{SocketServer.SingleSocket}
        @param ss: the connection that was flushed
        
        """
        
        pass

    def connection_lost(self, ss):
        """Close the lost connection.
        
        @type ss: L{SocketServer.SingleSocket}
        @param ss: the connection that was lost
        
        """
        
        self.closed = True

class MultiHandler:
    """Collection of Servers/Port Handlers for multiple torrents.
    
    @type rawserver: L{RawServer.RawServer}
    @ivar rawserver: the master Server
    @type masterdoneflag: C{threading.Event}
    @ivar masterdoneflag: the flag to indicate stopping to the master Server
    @type config: C{dictionary}
    @ivar config: the configuration parameters
    @type singlerawservers: C{dictionary}
    @ivar singlerawservers: keys are torrent infohash strings, values are 
        individual L{SingleRawServer} for the torrents
    @type connections: C{dictionary}
    @ivar connections: not used
    @type taskqueues: C{dictionary}
    @ivar taskqueues: not used
    
    """
    
    def __init__(self, rawserver, doneflag, config):
        """Initialize the instance.
        
        @type rawserver: L{RawServer.RawServer}
        @param rawserver: the master Server
        @type doneflag: C{threading.Event}
        @param doneflag: the flag to indicate stopping to the master Server
        @type config: C{dictionary}
        @param config: the configuration parameters
        
        """
        
        self.rawserver = rawserver
        self.masterdoneflag = doneflag
        self.config = config
        self.singlerawservers = {}
        self.connections = {}
        self.taskqueues = {}

    def newRawServer(self, info_hash, identifier, doneflag, protocol=protocol_name):
        """Create a new Server for the torrent.
        
        @type info_hash: C{string}
        @param info_hash: the torrent's infohash
        @type identifier: C{string}
        @param identifier: the identifier of the torrent
        @type doneflag: C{threading.Event}
        @param doneflag: the flag to indicate stopping to the new Server
        @type protocol: C{string}
        @param protocol: the name to use for the communication protocol
            (optional, defaults to L{DebTorrent.protocol_name})
        @rtype: L{SingleRawServer}
        @return: the new Server that was created
        
        """
        
        new = SingleRawServer(info_hash, identifier, self, doneflag, protocol)
        self.singlerawservers[identifier] = new
        return new

    def shutdown_torrent(self, identifier):
        """Shutdown a single torrent's Server.
        
        @type identifier: C{string}
        @param identifier: the identifier of the torrent

        """
        
        self.singlerawservers[identifier]._shutdown()
        del self.singlerawservers[identifier]

    def listen_forever(self):
        """Call the master server's listen loop.
        
        @rtype: C{boolean}
        @return: whether the server should be restarted
        
        """
        
        restart = self.rawserver.listen_forever(self)
        for srs in self.singlerawservers.values():
            srs.finished = True
            srs.running = False
            srs.doneflag.set()
        
        return restart
        
    ### RawServer handler functions ###
    # be wary of name collisions

    def external_connection_made(self, ss):
        """Handle a new incoming connection from the master Server.
        
        @type ss: unknown
        @param ss: unknown
        
        """
        
        NewSocketHandler(self, ss)
