/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 */

#if !defined (_DEE_H_INSIDE) && !defined (DEE_COMPILATION)
#error "Only <dee.h> can be included directly."
#endif

#ifndef _HAVE_DEE_FILTER_MODEL_H
#define _HAVE_DEE_FILTER_MODEL_H

#include <glib.h>
#include <glib-object.h>

#include <dee-model.h>

G_BEGIN_DECLS

#define DEE_TYPE_FILTER_MODEL (dee_filter_model_get_type ())

#define DEE_FILTER_MODEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
        DEE_TYPE_FILTER_MODEL, DeeFilterModel))

#define DEE_FILTER_MODEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), \
        DEE_TYPE_FILTER_MODEL, DeeFilterModelClass))

#define DEE_IS_FILTER_MODEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
        DEE_TYPE_FILTER_MODEL))

#define DEE_IS_FILTER_MODEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), \
        DEE_TYPE_FILTER_MODEL))

#define DEE_FILTER_MODEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), \
        DBUS_TYPE_FILTER_MODEL, DeeFilterModelClass))

typedef struct _DeeFilterModel DeeFilterModel;
typedef struct _DeeFilterModelClass DeeFilterModelClass;
typedef struct _DeeFilterModelPrivate DeeFilterModelPrivate;
typedef struct _DeeFilter DeeFilter;

/**
 * DeeModelMapFunc:
 * @orig_model: The model containing the original data to filter
 * @filter_model: The model that will contain the filtered results. The
 *                filter func must iterate over @orig_model and add all relevant
 *                rows to @filter_model. This model is guaranteed to be empty
 *                when the filter func is invoked
 * @user_data: User data passed together with the filter func
 *
 * Function used to collect the rows from a model that should be included in
 * a #DeeFilterModel. To add rows to @filter_model use the methods
 * dee_filter_model_append_iter(), dee_filter_model_prepend_iter(),
 * dee_filter_model_insert_iter(), and dee_filter_model_insert_iter_before().
 *
 * The iteration over the original model is purposely left to the map func
 * in order to allow optimized iterations if the the caller has a priori
 * knowledge of the sorting and grouping of the data in the original model.
 */
typedef void (*DeeModelMapFunc) (DeeModel       *orig_model,
                                 DeeFilterModel *filter_model,
                                 gpointer        user_data);

/**
 * DeeModelMapNotify:
 * @orig_model: The model containing the added row
 * @orig_iter: A #DeeModelIter pointing to the new row in @orig_model
 * @filter_model: The model that was also passed to the #DeeModelMapFunc
 *                of the #DeeFilter this functions is a part of
 * @user_data: User data for the #DeeFilter
 *
 * Callback invoked when a row is added to @orig_model. To add rows to
 * @filter_model use the methods dee_filter_model_append_iter(),
 * dee_filter_model_prepend_iter(), dee_filter_model_insert_iter(),
 * and dee_filter_model_insert_iter_before().
 */
typedef void (*DeeModelMapNotify) (DeeModel          *orig_model,
                                   DeeModelIter      *orig_iter,
                                   DeeFilterModel    *filter_model,
                                   gpointer           user_data);

/**
 * DeeFilter:
 * @map_func: The #DeeModelMapFunc used to construct the initial contents of
 *            a #DeeFilterModel
 * @map_notify: Callback invoked when the original model changes
 * @destroy: Callback for freeing the @user_data
 * @user_data: Free form user data associated with the filter. This pointer will
 *             be passed to @map_func and @map_notify
 *
 * Structure encapsulating the mapping logic used to construct a #DeeFilterModel
 */
struct _DeeFilter
{
  DeeModelMapFunc   map_func;
  DeeModelMapNotify map_notify;
  GDestroyNotify    destroy;
  gpointer          user_data;
} ;

/**
 * DeeFilterModel:
 *
 * All fields in the DeeFilterModel structure are private and should never be
 * accessed directly
 */
struct _DeeFilterModel
{
  /*< private >*/
  DeeProxyModel          parent;

  DeeFilterModelPrivate *priv;
};

struct _DeeFilterModelClass
{
  /*< private >*/
  DeeProxyModelClass parent_class;

  /*< private >*/
  void (*_dee_filter_model_1) (void);
  void (*_dee_filter_model_2) (void);
  void (*_dee_filter_model_3) (void);
  void (*_dee_filter_model_4) (void);
};

/**
 * dee_filter_model_get_type:
 *
 * The GType of #DeeFilterModel
 *
 * Return value: the #GType of #DeeFilterModel
 **/
GType                 dee_filter_model_get_type        (void) G_GNUC_CONST;

DeeModel*             dee_filter_model_new             (const DeeFilter *filter,
                                                        DeeModel  *orig_model);

gboolean              dee_filter_model_contains        (DeeFilterModel *self,
                                                        DeeModelIter   *iter);

DeeModelIter*         dee_filter_model_append_iter     (DeeFilterModel *self,
                                                        DeeModelIter   *iter);

DeeModelIter*         dee_filter_model_prepend_iter     (DeeFilterModel *self,
                                                         DeeModelIter   *iter);

DeeModelIter*         dee_filter_model_insert_iter     (DeeFilterModel *self,                                                        
                                                        DeeModelIter   *iter,
                                                        guint           pos);

DeeModelIter*         dee_filter_model_insert_iter_before (DeeFilterModel *self,
                                                           DeeModelIter   *iter,
                                                           DeeModelIter   *pos);

G_END_DECLS

#endif /* _HAVE_DEE_FILTER_MODEL_H */
