/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by
 *              Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *              Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <glib.h>
#include <glib-object.h>
#include <dee.h>

typedef struct 
{
  DeeModel *model;

} ColumnFixture;

static void column_setup          (ColumnFixture *fix, gconstpointer data);
static void column_teardown       (ColumnFixture *fix, gconstpointer data);
static void proxy_column_setup    (ColumnFixture *fix, gconstpointer data);
static void proxy_column_teardown (ColumnFixture *fix, gconstpointer data);

static void test_column_allocation   (ColumnFixture *fix, gconstpointer data);
static void test_column_types        (ColumnFixture *fix, gconstpointer data);
static void test_column_modification (ColumnFixture *fix, gconstpointer data);
static void test_build_col_spec      (ColumnFixture *fix, gconstpointer data);
static void test_column_bad_types    (void);

void
test_model_column_create_suite (void)
{
#define SEQ_DOMAIN "/Model/Sequence/Column"
#define PROXY_DOMAIN "/Model/Proxy/Column"

  g_test_add (SEQ_DOMAIN"/Allocation", ColumnFixture, 0,
              column_setup, test_column_allocation, column_teardown);
  g_test_add (PROXY_DOMAIN"/Allocation", ColumnFixture, 0,
              proxy_column_setup, test_column_allocation, proxy_column_teardown);
  
  g_test_add (SEQ_DOMAIN"/Types", ColumnFixture, 0,
              column_setup, test_column_types, column_teardown);
  g_test_add (PROXY_DOMAIN"/Types", ColumnFixture, 0,
              proxy_column_setup, test_column_types, proxy_column_teardown);
  
  g_test_add (SEQ_DOMAIN"/Modification", ColumnFixture, 0,
              column_setup, test_column_modification, column_teardown);
  g_test_add (PROXY_DOMAIN"/Modification", ColumnFixture, 0,
              proxy_column_setup, test_column_modification, proxy_column_teardown);
  
  g_test_add (SEQ_DOMAIN"/ColSpec", ColumnFixture, 0,
              column_setup, test_build_col_spec, column_teardown);
  g_test_add (PROXY_DOMAIN"/ColSpec", ColumnFixture, 0,
              proxy_column_setup, test_build_col_spec, proxy_column_teardown);
  
  g_test_add_func ("/Model/Column/BadTypes", test_column_bad_types);
}

static void
column_setup (ColumnFixture *fix, gconstpointer data)
{
  fix->model = dee_sequence_model_new (8,
                                       G_TYPE_BOOLEAN,
                                       G_TYPE_UCHAR,
                                       G_TYPE_INT,
                                       G_TYPE_UINT,
                                       G_TYPE_INT64,
                                       G_TYPE_UINT64,
                                       G_TYPE_DOUBLE,
                                       G_TYPE_STRING);

  g_assert (DEE_IS_SEQUENCE_MODEL (fix->model));
  g_assert_cmpint (8, ==, dee_model_get_n_columns (fix->model));
  g_assert_cmpint (G_TYPE_BOOLEAN, ==, dee_model_get_column_type (fix->model, 0));
  g_assert_cmpint (G_TYPE_UCHAR, ==, dee_model_get_column_type (fix->model, 1));
  g_assert_cmpint (G_TYPE_INT, ==, dee_model_get_column_type (fix->model, 2));
  g_assert_cmpint (G_TYPE_UINT, ==, dee_model_get_column_type (fix->model, 3));
  g_assert_cmpint (G_TYPE_INT64, ==, dee_model_get_column_type (fix->model, 4));
  g_assert_cmpint (G_TYPE_UINT64, ==, dee_model_get_column_type (fix->model, 5));
  g_assert_cmpint (G_TYPE_DOUBLE, ==, dee_model_get_column_type (fix->model, 6));
  g_assert_cmpint (G_TYPE_STRING, ==, dee_model_get_column_type (fix->model, 7));
  g_assert_cmpint (0, ==, dee_model_get_n_rows (fix->model));
}

static void
column_teardown (ColumnFixture *fix, gconstpointer data)
{
  g_object_unref (fix->model);
}

static void
proxy_column_setup (ColumnFixture *fix, gconstpointer data)
{
  column_setup (fix, data);
  fix->model = g_object_new (DEE_TYPE_PROXY_MODEL,
                             "back-end", fix->model,
                             NULL);
  
  g_assert (DEE_IS_PROXY_MODEL (fix->model));
}

static void
proxy_column_teardown (ColumnFixture *fix, gconstpointer data)
{
  g_assert (DEE_IS_PROXY_MODEL (fix->model));
  g_object_unref (fix->model);
}

static void
test_column_allocation (ColumnFixture *fix, gconstpointer data)
{
  g_assert (DEE_IS_MODEL (fix->model));
}

static void
test_column_types (ColumnFixture *fix, gconstpointer data)
{
  DeeModel     *model = fix->model;
  DeeModelIter *iter;
  GValue         value = { 0, };

  dee_model_append (fix->model,
                     0, TRUE,
                     1, '1',
                     2, G_MININT,
                     3, G_MAXUINT,
                     4, G_MAXINT64,
                     5, G_MAXUINT64,
                     6, G_MAXDOUBLE,
                     7, "Hello World",
                     -1);

  g_assert (dee_model_get_n_rows (model) == 1);

  iter = dee_model_get_first_iter (model);

  dee_model_get_value (model, iter, 0, &value);
  g_assert_cmpint (0, != , g_value_get_boolean (&value));  
  g_value_unset (&value);
  g_assert_cmpint (0, !=, dee_model_get_bool (model, iter, 0));

  dee_model_get_value (model, iter, 1, &value);
  g_assert_cmpint ('1', ==, g_value_get_uchar (&value));  
  g_value_unset (&value);
  g_assert_cmpint ('1', ==, dee_model_get_uchar (model, iter, 1));

  dee_model_get_value (model, iter, 2, &value);
  g_assert_cmpint (G_MININT, ==, g_value_get_int (&value));
  g_value_unset (&value);
  g_assert_cmpint (G_MININT, ==, dee_model_get_int (model, iter, 2));

  dee_model_get_value (model, iter, 3, &value);
  g_assert (g_value_get_uint (&value) == G_MAXUINT);  
  g_value_unset (&value);
  g_assert_cmpuint (G_MAXUINT, ==, dee_model_get_uint (model, iter, 3));

  dee_model_get_value (model, iter, 4, &value);
  g_assert (g_value_get_int64 (&value) == G_MAXINT64);
  g_value_unset (&value);
  g_assert_cmpint (G_MAXINT64, ==, dee_model_get_int64 (model, iter, 4));

  dee_model_get_value (model, iter, 5, &value);
  g_assert (g_value_get_uint64 (&value) == G_MAXUINT64);
  g_value_unset (&value);
  g_assert_cmpuint (G_MAXUINT64, ==, dee_model_get_uint64 (model, iter, 5));

  dee_model_get_value (model, iter, 6, &value);
  g_assert (g_value_get_double (&value) == G_MAXDOUBLE);
  g_value_unset (&value);
  g_assert_cmpfloat (G_MAXDOUBLE, ==, dee_model_get_double (model, iter, 6));

  dee_model_get_value (model, iter, 7, &value);
  g_assert_cmpstr (g_value_get_string (&value), ==, "Hello World");
  g_value_unset (&value);
  g_assert_cmpstr ("Hello World", ==, dee_model_get_string (model, iter, 7));

  /* Assert that we don't mess up the string copying. The default impl
   * of model.get_string() in DeeAbstractModel uses an ugly/clever stack
   * allocation hack to be able to return a const gchar* in a generic case */
  const gchar *cp1 = dee_model_get_string (model, iter, 7);
  const gchar *cp2 = dee_model_get_string (model, iter, 7);
  g_assert_cmpstr ("Hello World", ==, cp1);
  g_assert_cmpstr ("Hello World", ==, cp2);
  g_assert (cp1 == cp2);
}

static void
test_column_modification (ColumnFixture *fix, gconstpointer data)
{
  DeeModel     *model = fix->model;
  DeeModelIter *iter;
  GValue         value = { 0, };

  dee_model_append (fix->model,
                     0, TRUE,
                     1, '1',
                     2, G_MININT,
                     3, G_MAXUINT,
                     4, G_MAXINT64,
                     5, G_MAXUINT64,
                     6, G_MAXDOUBLE,
                     7, "Hello World",
                     -1);
  
  g_assert (dee_model_get_n_rows (model) == 1);

  iter = dee_model_get_first_iter (model);

  dee_model_set (model, iter, 
                  0, FALSE,
                  1, '2',
                  2, G_MININT+5,
                  3, G_MAXUINT-5,
                  4, G_MAXINT64-5,
                  5, G_MAXUINT64-5,
                  6, G_MAXDOUBLE-5.0,
                  7, "World Hello",
                  -1);

  dee_model_get_value (model, iter, 0, &value);
  g_assert (!g_value_get_boolean (&value));
  g_value_unset (&value);

  dee_model_get_value (model, iter, 1, &value);
  g_assert (g_value_get_uchar (&value) == '2');
  g_value_unset (&value);

  dee_model_get_value (model, iter, 2, &value);
  g_assert (g_value_get_int (&value) == G_MININT+5);
  g_value_unset (&value);

  dee_model_get_value (model, iter, 3, &value);
  g_assert (g_value_get_uint (&value) == G_MAXUINT-5);
  g_value_unset (&value);

  dee_model_get_value (model, iter, 4, &value);
  g_assert (g_value_get_int64 (&value) == G_MAXINT64-5);
  g_value_unset (&value);

  dee_model_get_value (model, iter, 5, &value);
  g_assert (g_value_get_uint64 (&value) == G_MAXUINT64-5);
  g_value_unset (&value);

  dee_model_get_value (model, iter, 6, &value);
  g_assert (g_value_get_double (&value) == G_MAXDOUBLE-5.0);
  g_value_unset (&value);

  dee_model_get_value (model, iter, 7, &value);
  g_assert_cmpstr (g_value_get_string (&value), ==, "World Hello");
  g_assert_cmpstr (dee_model_get_string (model, iter, 7), ==, "World Hello");
  g_value_unset (&value);
}

static void
test_build_col_spec (ColumnFixture *fix, gconstpointer data)
{
  gchar *col_spec;
  
  col_spec = dee_model_build_col_spec (fix->model);
  g_assert_cmpstr ("byiuxtds", ==, col_spec);
  g_free (col_spec);
}

static void
test_column_bad_types (void)
{
  gint           n_types, i;
  /* Some random types which we shouldn't support */
  static const GType types[] =
    {
      G_TYPE_BOXED,
      G_TYPE_CHAR,
      G_TYPE_ENUM,
      G_TYPE_FLAGS,
      G_TYPE_FLOAT,
      G_TYPE_OBJECT,
      G_TYPE_LONG,
    };

  n_types = G_N_ELEMENTS (types);

  for (i = 0; i < n_types; i++)
    {
      /* Make sure we can't create a model with GTypes which we don't support */
      if (g_test_trap_fork (0, G_TEST_TRAP_SILENCE_STDOUT
                               | G_TEST_TRAP_SILENCE_STDERR))
        {
          DeeModel *model = NULL;

          model = dee_shared_model_new ("com.canonical.DbusMenu.Test",
                                         1,
                                         types[i]);
          g_assert (model == NULL);
        }
      g_test_trap_assert_failed ();
    }
}
