# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Eric Casteleijn <eric.casteleijn@canonical.com>

"""testing database/contact.py module"""
import testtools

from desktopcouch.tests import xdg_cache
from desktopcouch.records.server import CouchDatabase
from desktopcouch.records.server_base import row_is_deleted
from desktopcouch.records.record import Record

FAKE_RECORD_TYPE = "http://example.org/test"

js = """
function(doc) {
    if (doc.record_type == '%s') {
        emit(doc._id, null);
    }
}""" % FAKE_RECORD_TYPE


class TestCouchDatabase(testtools.TestCase):
    """tests specific for CouchDatabase"""


    def setUp(self):
        """setup each test"""
        # Connect to CouchDB server
        self.assert_(xdg_cache)
        self.dbname = self._testMethodName
        self.database = CouchDatabase(self.dbname, create=True)
        #create some records to pull out and test
        self.database.put_record(Record({
            "key1_1": "val1_1", "key1_2": "val1_2", "key1_3": "val1_3",
            "record_type": "test.com"}))
        self.database.put_record(Record({
            "key2_1": "val2_1", "key2_2": "val2_2", "key2_3": "val2_3",
            "record_type": "test.com"}))
        self.database.put_record(Record({
            "key13_1": "va31_1", "key3_2": "val3_2", "key3_3": "val3_3",
            "record_type": "test.com"}))

    def tearDown(self):
        """tear down each test"""
        del self.database._server[self.dbname]

    def test_get_records_by_record_type_save_view(self):
        """Test getting mutliple records by type"""
        records = self.database.get_records(
            record_type="test.com",create_view=True)
        self.assertEqual(3,len(records))

    def test_get_record(self):
        """Test getting a record."""
        record = Record({'record_number': 0}, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        retrieved_record = self.database.get_record(record_id)
        self.assertEqual(0, retrieved_record['record_number'])

    def test_put_record(self):
        """Test putting a record."""
        record = Record({'record_number': 0}, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        retrieved_record = self.database._server[self.dbname][record_id]
        self.assertEqual(
            record['record_number'], retrieved_record['record_number'])

    def test_delete_record(self):
        """Test deletion of records."""
        record = Record({'record_number': 0}, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        self.database.delete_record(record_id)
        deleted_record = self.database._server[self.dbname][record_id]
        self.assert_(deleted_record['application_annotations']['Ubuntu One'][
            'private_application_annotations']['deleted'])

    def test_get_deleted_record(self):
        """Test (not) getting a deleted record."""
        record = Record({'record_number': 0}, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        self.database.delete_record(record_id)
        retrieved_record = self.database.get_record(record_id)
        self.assertEqual(None, retrieved_record)

    def test_record_exists(self):
        """Test checking whether a record exists."""
        record = Record({'record_number': 0}, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        self.assert_(self.database.record_exists(record_id))
        self.database.delete_record(record_id)
        self.assert_(not self.database.record_exists(record_id))

    def test_update_fields(self):
        """Test the update_fields method."""
        dictionary = {'record_number': 0, 'field1': 1, 'field2': 2}
        record = Record(dictionary, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        # manipulate the database 'out of view'
        non_working_copy = self.database.get_record(record_id)
        non_working_copy['field2'] = 22
        non_working_copy['field3'] = 3
        self.database.put_record(non_working_copy)
        self.database.update_fields(record_id, {'field1': 11})
        working_copy = self.database.get_record(record_id)
        self.assertEqual(0, working_copy['record_number'])
        self.assertEqual(11, working_copy['field1'])
        self.assertEqual(22, working_copy['field2'])
        self.assertEqual(3, working_copy['field3'])

    def test_view_add_and_delete(self):
        design_doc = "design"
        view1_name = "unit_tests_are_wonderful"
        view2_name = "unit_tests_are_marvelous"
        view3_name = "unit_tests_are_fantastic"

        map_js = """function(doc) { emit(doc._id, null) }"""
        reduce_js = """\
                function (key, values, rereduce) {
                    return sum(values);
                }"""

        # add two and delete two.
        self.assertRaises(
            KeyError, self.database.delete_view, view1_name, design_doc)
        self.assertRaises(
            KeyError, self.database.delete_view, view2_name, design_doc)
        self.database.add_view(view1_name, map_js, reduce_js, design_doc)
        self.database.add_view(view2_name, map_js, reduce_js, design_doc)
        self.database.delete_view(view1_name, design_doc)
        self.assertRaises(
            KeyError, self.database.delete_view, view1_name, design_doc)
        self.database.delete_view(view2_name, design_doc)
        self.assertRaises(
            KeyError, self.database.delete_view, view2_name, design_doc)

    def test_func_get_records(self):
        record_ids_we_care_about = set()
        good_record_type = "http://example.com/unittest/good"
        other_record_type = "http://example.com/unittest/bad"

        for i in range(7):
            record = Record({'record_number': i},
                    record_type=good_record_type)
            if i % 3 == 1:
                record = Record({'record_number': i},
                        record_type=good_record_type)
                record_ids_we_care_about.add(self.database.put_record(record))
            elif i % 3 == 2:
                record = Record({'record_number': i},
                        record_type=good_record_type)
                record_id = self.database.put_record(record)  # correct type,
                self.database.delete_record(record_id)  # but marked deleted!
            else:
                record = Record({'record_number': i},
                        record_type=other_record_type)
                self.database.put_record(record)

        results = self.database.get_records(create_view=True)

        for row in results[good_record_type]:  # index notation
            self.assertTrue(row.id in record_ids_we_care_about)
            record_ids_we_care_about.remove(row.id)
            self.assertFalse(row_is_deleted(row))

        self.assertTrue(len(record_ids_we_care_about) == 0, "expected zero")

    def test_list_views(self):
        design_doc = "d"
        self.assertEqual(self.database.list_views(design_doc), [])

        view_name = "unit_tests_are_fantastic"
        map_js = """function(doc) { emit(doc._id, null) }"""
        self.database.add_view(view_name, map_js, None, design_doc)

        self.assertEqual(self.database.list_views(design_doc), [view_name])
        self.database.delete_view(view_name, design_doc)

        self.assertEqual(self.database.list_views(design_doc), [])

    def test_get_view_by_type_new_but_already(self):
        self.database.get_records(create_view=True)
        self.database.get_records(create_view=True)
        # No exceptions on second run?  Yay.

    def test_get_view_by_type_createxcl_fail(self):
        self.database.get_records(create_view=True)
        self.assertRaises(KeyError, self.database.get_records, create_view=None)
