/////////////////////////////////////////////////////////////////////////////
// Name:        AudioPropDlg.cpp
// Purpose:     The audio properties dialog
// Author:      Alex Thuering
// Created:     23.07.2011
// RCS-ID:      $Id: AudioPropDlg.cpp,v 1.1 2011/08/01 20:30:09 ntalex Exp $
// Copyright:   (c) Alex Thuering
// Licence:     GPL
/////////////////////////////////////////////////////////////////////////////

#include "AudioPropDlg.h"
#include <wxSVG/mediadec_ffmpeg.h>

// begin wxGlade: ::extracode

// end wxGlade

enum {
	DST_CHOICE_ID = 2000
};

BEGIN_EVENT_TABLE(AudioPropDlg, wxDialog)
    // begin wxGlade: AudioPropDlg::event_table
    EVT_CHOICE(DST_CHOICE_ID, AudioPropDlg::OnChangeFormat)
    EVT_BUTTON(wxID_OK, AudioPropDlg::OnOkBt)
    // end wxGlade
END_EVENT_TABLE();

/** Constructor */
AudioPropDlg::AudioPropDlg(wxWindow* parent, Vob* vob, const wxString& audioFile, const wxString& langCode, int streamIdx):
		wxDialog(parent, -1, wxT(""), wxDefaultPosition, wxDefaultSize, wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER) {
	m_vob = vob;
	m_audioFile = audioFile;
	m_langCode = langCode;
	m_stream = vob->GetStreams()[streamIdx];
	
    // begin wxGlade: AudioPropDlg::AudioPropDlg
    m_fileNameLabel = new wxStaticText(this, wxID_ANY, _("File Name:"));
    m_fileNameText = new wxStaticText(this, wxID_ANY, wxEmptyString);
    m_durLabel = new wxStaticText(this, wxID_ANY, _("Duration:"));
    m_durText = new wxStaticText(this, wxID_ANY, wxEmptyString);
    m_srcLabel = new wxStaticText(this, wxID_ANY, _("Source Format:"));
    m_srcText = new wxStaticText(this, wxID_ANY, wxEmptyString);
    m_dstLabel = new wxStaticText(this, wxID_ANY, _("Destination Format:"));
    const wxString *m_dstChoice_choices = NULL;
    m_dstChoice = new wxChoice(this, DST_CHOICE_ID, wxDefaultPosition, wxDefaultSize, 0, m_dstChoice_choices, 0);
    m_langLabel = new wxStaticText(this, wxID_ANY, _("Language:"));
    const wxString *m_langChoice_choices = NULL;
    m_langChoice = new wxChoice(this, wxID_ANY, wxDefaultPosition, wxDefaultSize, 0, m_langChoice_choices, 0);
    m_volumeLabel = new wxStaticText(this, wxID_ANY, _("Volume (256 = normal):"));
    m_volumeCtrl = new wxSpinCtrl(this, wxID_ANY, wxT("256"), wxDefaultPosition, wxDefaultSize, wxSP_ARROW_KEYS, 0, 999);
    m_okBt = new wxButton(this, wxID_OK, _("OK"));
    m_cancelBt = new wxButton(this, wxID_CANCEL, _("Cancel"));

    set_properties();
    do_layout();
    // end wxGlade
    
    wxCommandEvent evt;
    OnChangeFormat(evt);
    
    if (m_vob->GetDoNotTranscode()) {
    	m_dstChoice->Enable(false);
    	m_volumeCtrl->Enable(false);
    	m_langChoice->SetFocus();
    } else
    	m_dstChoice->SetFocus();
}

void AudioPropDlg::set_properties() {
    // begin wxGlade: AudioPropDlg::set_properties
    SetTitle(_("Audio properties"));
    m_okBt->SetDefault();
    // end wxGlade
    
    m_fileNameText->SetLabel(m_audioFile.length() ? m_audioFile : m_vob->GetFilename());
    // duration
    double duration = m_vob->GetDuration();
    if (m_audioFile.length()) {
    	wxFfmpegMediaDecoder ffmpeg;
		if (ffmpeg.Load(m_audioFile) && ffmpeg.GetDuration() > 0)
			duration = ffmpeg.GetDuration();
    }
	wxString s = wxT("N/A");
	if (duration > 0) {
		int secs = (int) duration;
		int mins = secs / 60;
		secs %= 60;
		int hours = mins / 60;
		mins %= 60;
		s = wxString::Format(wxT("%02d:%02d:%02d"), hours, mins, secs);
	}
    m_durText->SetLabel(s);
    m_srcText->SetLabel(m_stream->GetSourceFormat());
    
    m_dstChoice->Append(DVD::GetAudioFormatLabels(true, true));
    m_dstChoice->SetSelection(m_stream->GetAudioFormat());
    
    m_langChoice->Append(DVD::GetLanguageNames());
    map<wxString, wxString>& langMap = DVD::GetLanguageMap();
	for (map<wxString, wxString>::iterator langIt = langMap.begin(); langIt != langMap.end(); langIt++) {
		if (langIt->second == m_langCode) {
			m_langChoice->SetStringSelection(langIt->first);
		}
	}
    
    m_volumeCtrl->SetValue(m_stream->GetAudioVolume());
    m_volumeCtrl->SetMinSize(m_dstChoice->GetBestSize());
}

void AudioPropDlg::do_layout() {
    // begin wxGlade: AudioPropDlg::do_layout
    wxBoxSizer* mainVSizer = new wxBoxSizer(wxVERTICAL);
    wxBoxSizer* buttonSizer = new wxBoxSizer(wxHORIZONTAL);
    wxFlexGridSizer* propSizer = new wxFlexGridSizer(10, 2, 4, 2);
    propSizer->Add(m_fileNameLabel, 0, wxALIGN_CENTER_VERTICAL, 0);
    propSizer->Add(m_fileNameText, 0, wxEXPAND|wxALIGN_CENTER_VERTICAL, 0);
    propSizer->Add(m_durLabel, 0, wxALIGN_CENTER_VERTICAL, 0);
    propSizer->Add(m_durText, 0, wxEXPAND|wxALIGN_CENTER_VERTICAL, 0);
    propSizer->Add(m_srcLabel, 0, wxALIGN_CENTER_VERTICAL, 0);
    propSizer->Add(m_srcText, 0, wxEXPAND|wxALIGN_CENTER_VERTICAL, 0);
    propSizer->Add(m_dstLabel, 0, wxALIGN_CENTER_VERTICAL, 0);
    propSizer->Add(m_dstChoice, 0, wxALIGN_CENTER_VERTICAL, 0);
    propSizer->Add(m_langLabel, 0, wxALIGN_CENTER_VERTICAL, 0);
    propSizer->Add(m_langChoice, 0, 0, 0);
    propSizer->Add(m_volumeLabel, 0, wxALIGN_CENTER_VERTICAL, 0);
    propSizer->Add(m_volumeCtrl, 0, 0, 0);
    propSizer->AddGrowableRow(9);
    propSizer->AddGrowableCol(1);
    mainVSizer->Add(propSizer, 0, wxALL|wxEXPAND, 10);
    mainVSizer->Add(20, 4, 1, wxEXPAND, 0);
    buttonSizer->Add(10, 10, 1, wxEXPAND, 0);
    buttonSizer->Add(m_okBt, 0, 0, 0);
    buttonSizer->Add(8, 8, 0, wxEXPAND, 0);
    buttonSizer->Add(m_cancelBt, 0, 0, 0);
    buttonSizer->Add(10, 10, 1, wxEXPAND, 0);
    mainVSizer->Add(buttonSizer, 0, wxLEFT|wxRIGHT|wxBOTTOM|wxEXPAND, 4);
    SetSizer(mainVSizer);
    mainVSizer->Fit(this);
    Layout();
    Centre();
    // end wxGlade
}

AudioFormat AudioPropDlg::GetAudioFormat() {
	return (AudioFormat) m_dstChoice->GetSelection();
}

wxString AudioPropDlg::GetLangCode() {
	return DVD::GetLanguageMap()[m_langChoice->GetStringSelection()];
}

// wxGlade: add AudioPropDlg event handlers

void AudioPropDlg::OnOkBt(wxCommandEvent &event) {
	m_stream->SetAudioVolume(m_volumeCtrl->GetValue());
	EndModal(wxID_OK);
}

void AudioPropDlg::OnChangeFormat(wxCommandEvent &event) {
	m_volumeCtrl->Enable(GetAudioFormat() != afNONE && GetAudioFormat() != afCOPY);
}
