/*********************************************************************************************************
DVR, Digital Video Recorder - a tool to record movies (audio/video), using realtime compression

It uses libavifile (see http://divx.euro.ru) and some code from kwintv (see wenk@mathematik.uni-kl.de)

This program is free software; you can redistribute it and/or modify it under the terms of the GNU General
Public License as published by the Free Software Foundation, etc.
This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY, etc.

copyright(C) february 2001, Pierre Hbert (pierre.hebert@netcourrier.com)
*********************************************************************************************************/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include "framepool.h"


FramePool::FramePool(int _width, int _height, int _bpp, int _max_frames):
	width(_width), height(_height), bpp(_bpp), opp((_bpp+7)/8), top_margin(0), bottom_margin(0),
	max_frames(_max_frames), previous_frame(NULL), newest(0), oldest(0), nb_frames(0)
{	
	frame_size=width*height*opp;
	frames=new unsigned char[max_frames*frame_size];
  frames_null=new bool[max_frames];

	sem_id=semget(IPC_PRIVATE, 2, IPC_CREAT | 0666 );

	semctl(sem_id, SEM_ID_FRAME_AVAILABLE, SETVAL, 0);
	semctl(sem_id, SEM_ID_MUTEX, SETVAL, 1);
}

FramePool::~FramePool() {
	delete []frames_null;
	delete []frames;
	semctl(sem_id, 0, IPC_RMID, 0);
}

void FramePool::addFrame(unsigned char *src) {
  unsigned char *d=frames+frame_size*newest;
  if(src) {
//#define FLIP_IMAGE
#ifdef FLIP_IMAGE
/*    int line_size=width*opp;
    unsigned char *s=src+(height-bottom_margin-1)*line_size;
    for(int i=0; i<(height-top_margin-bottom_margin);
      s-=line_size,d+=line_size, i++) {
      memcpy(d, s, line_size);
    }*/
    unsigned char *s=src;
    int w=width;
    int h=height-top_margin-bottom_margin;
    for(int i=0; i<h; i++) {
      for(int j=0; j<w; j++) {
        d[j]=s[w-j-1];
      }
      d+=w;
      s+=w;
    }
    w=width/2;
    for(int i=0; i<h/2; i++) {
      for(int j=0; j<w; j++) {
        d[j]=s[w-j-1];
      }
      d+=w;
      s+=w;
    }
    for(int i=0; i<h/2; i++) {
      for(int j=0; j<w; j++) {
        d[j]=s[w-j-1];
      }
      d+=w;
      s+=w;
    }

#else
    memcpy(d, src, width*height*opp);
#endif
    frames_null[newest]=false;
  } else {
    frames_null[newest]=true;
  }

	previous_frame=d;

	struct sembuf op;
	op.sem_flg=0;
	op.sem_num=SEM_ID_MUTEX;op.sem_op=-1;semop(sem_id, &op, 1);
	nb_frames++;
	op.sem_num=SEM_ID_MUTEX;op.sem_op=1;semop(sem_id, &op, 1);
	
	// signal new frame available
	op.sem_num=SEM_ID_FRAME_AVAILABLE;
	op.sem_op=1;
	semop(sem_id, &op, 1);
	
	newest++;
	if(newest==max_frames) newest=0;
}

/*void FramePool::addSameFrame() {
	if(previous_frame) {
		unsigned char *c=frames+frame_size*newest;
		memcpy(c, previous_frame, width*height*opp);
		previous_frame=c;

		struct sembuf op;
		op.sem_flg=0;
		op.sem_num=SEM_ID_MUTEX;op.sem_op=-1;semop(sem_id, &op, 1);
		nb_frames++;
		op.sem_num=SEM_ID_MUTEX;op.sem_op=1;semop(sem_id, &op, 1);
	
		// signal new frame available
		op.sem_num=SEM_ID_FRAME_AVAILABLE;
		op.sem_op=1;
		semop(sem_id, &op, 1);
	
		newest++;
		if(newest==max_frames) newest=0;
	}
}*/

unsigned char *FramePool::getFrame() {
	// wait for a frame to be available     
	struct sembuf op;
	op.sem_num=SEM_ID_FRAME_AVAILABLE;
	op.sem_op=-1;op.sem_flg=0;
	semop(sem_id, &op, 1);
		
	unsigned char *c;

  if(frames_null[oldest]) {
    c=NULL;
  } else {
    c=frames+frame_size*oldest;
  }
	
	oldest++;
	if(oldest==max_frames) oldest=0;

	return c;
}

void FramePool::removeFrame() {
	struct sembuf op;
	op.sem_flg=0;
	op.sem_num=SEM_ID_MUTEX;op.sem_op=-1;semop(sem_id, &op, 1);
	nb_frames--;
	op.sem_num=SEM_ID_MUTEX;op.sem_op=1;semop(sem_id, &op, 1);
}

int FramePool::size() const {
	return nb_frames;
}

int FramePool::max_size() const {
	return max_frames;
}

bool FramePool::full() const {
	return nb_frames==max_frames;
}

void FramePool::setTopMargin(int m) {
	if(m<0) {
		top_margin=0;
	} 
	else if(m>(height-bottom_margin)) {
		top_margin=height-bottom_margin;
	} 
	else {
		top_margin=m;
	}
}

int FramePool::getTopMargin() const {
	return top_margin;
}

void FramePool::setBottomMargin(int m) {
	if(m<0) {
		bottom_margin=0;
	} 
	else if(m>(height-top_margin)) {
		bottom_margin=height-top_margin;
	} 
	else {
		bottom_margin=m;
	}
}

int FramePool::getBottomMargin() const {
	return bottom_margin;
}
