/*
FatRat download manager
http://fatrat.dolezel.info

Copyright (C) 2006-2008 Lubos Dolezel <lubos a dolezel.info>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
version 2 as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include <QString>

#include <fatrat/fatrat.h>
#include <fatrat/Transfer.h>
#include <fatrat/AppTools.h>
#include <fatrat/Settings.h>

#include <QTranslator>
#include <QLocale>
#include <QMessageBox>
#include <QFileDialog>
#include <QFileInfo>
#include <QDir>

#include "config.h"
#include "SubtitlesDlg.h"
#include "SettingsSubtitles.h"

struct MovieFile;
static void searchSubtitles(Transfer* t, Queue* q);
static bool isMovieFile(QString name);
static void searchDirectory(QString absolute, QString relative, QList<MovieFile>& out);

const char* g_movieSuffixes[] = { ".avi", ".mkv", ".mpg", ".mpeg", ".mov", ".divx", 0 };

__attribute__ ((constructor)) void init()
{
	Q_INIT_RESOURCE(opensubtitles);

#ifdef WITH_NLS
	static QTranslator translator;
	{
		QString fname = QString("fatrat-opensubtitles_") + QLocale::system().name();
		translator.load(fname, getDataFileDir("/lang", fname));
		QCoreApplication::installTranslator(&translator);
	}
#endif
	
	MenuAction action;
	
	action.strName = QObject::tr("Search for subtitles...");
	action.lpfnTriggered = searchSubtitles;
	addMenuAction(action);
	
	AppTool at;
	
	at.strName = QObject::tr("Subtitles search");
	at.pfnCreate = SubtitlesDlg::create;
	addAppTool(at);
	
	SettingsItem si;
	
	si.icon = QIcon(":/opensubtitles/opensubtitles.png");
	si.title = QObject::tr("Subtitles search");
	si.lpfnCreate = SettingsSubtitles::create;
	
	addSettingsPage(si);
}

extern "C" __attribute__((visibility("default"))) PluginInfo getInfo()
{
	PluginInfo info;
	
	info.version = "1.1.2";
	info.name = QObject::tr("OpenSubtitles.org search");
	info.author = QString::fromUtf8("Luboš Doležel");
	info.website = "http://fatrat.dolezel.info";
	
	return info;
}

struct MovieFile
{
	QString path;
	qint64 size;
	
	MovieFile(QString npath, qint64 nsize)
	: path(npath), size(nsize)
	{
	}
	bool operator<(const MovieFile& that) const
	{
		return size > that.size;
	}
};

void searchSubtitles(Transfer* t, Queue* q)
{
	QList<MovieFile> files;
	QString path = t->dataPath(true);
	QFileInfo info(path);
	QString chosenFile;
	
	if(info.isDir())
		searchDirectory(path+'/', "/", files);
	else if(!info.isSymLink())
	{
		if(isMovieFile(path))
			files << MovieFile(QString(), info.size());
	}
	
	if(files.isEmpty())
	{
		QMessageBox::warning(getMainWindow(), "FatRat", QObject::tr("No movie files found."));
		return;
	}
	if(files.size() == 1)
		chosenFile = files[0].path;
	else
	{
		QString filter = "(";
		QString chosenFile;
		
		for(size_t i=0;g_movieSuffixes[i];i++)
		{
			filter += '*';
			filter += g_movieSuffixes[i];
			filter += ' ';
		}
		filter += ')';
		
		chosenFile = QFileDialog::getOpenFileName(getMainWindow(), "FatRat", path, filter);
		
		if(chosenFile.isEmpty())
			return;
	}
	
	SubtitlesDlg* dlg = new SubtitlesDlg(getMainWindow());
	
	dlg->search(chosenFile);
	dlg->show();
}


void searchDirectory(QString absolute, QString relative, QList<MovieFile>& out)
{
	QDir dir(absolute);
	QFileInfoList list = dir.entryInfoList(QDir::Dirs | QDir::Files | QDir::NoDotAndDotDot);
	
	foreach(QFileInfo info, list)
	{
		if(info.isDir())
			searchDirectory(absolute+info.fileName()+'/', relative+info.fileName()+'/', out);
		else if(!info.isSymLink())
		{
			if(isMovieFile(info.fileName()))
				out << MovieFile(info.absoluteFilePath(), info.size());
		}
	}
}

bool isMovieFile(QString name)
{
	for(size_t i=0;g_movieSuffixes[i];i++)
	{
		if(name.endsWith(g_movieSuffixes[i], Qt::CaseInsensitive))
			return true;
	}
	return false;
}

