/*
 *  
 *  $Id$
 *  Ginkgo CADx Project
 *
 *  Code based on yasper
 * ==========================
 *
 * yasper - A non-intrusive reference counted pointer.
 *	    Version: 1.04
 *
 *  Many ideas borrowed from Yonat Sharon and
 *  Andrei Alexandrescu.
 *
 * (zlib license)
 * ----------------------------------------------------------------------------------
 * Copyright (C) 2005-2007 Alex Rubinsteyn
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * -----------------------------------------------------------------------------------
 *
 * Send all questions, comments and bug reports to:
 * Alex Rubinsteyn (alex.rubinsteyn {at-nospam} gmail {dot} com)
 */

#pragma once
#include <iostream>
#include <exception>
#include <string>

#define GLOC() (__FILE__ ":" TOSTRING(__LINE__))
#define STRINGIFY(x) #x
#define TOSTRING(x) STRINGIFY(x)

#if defined(_WINDOWS)
#define SUSPEND_WAKEUP()
#define RESUME_WAKEUP()

#else

#include <signal.h>
#include <pthread.h>
#include <errno.h>

#define SUSPEND_WAKEUP()\
siginterrupt(SIGUSR2, 0)

#define RESUME_WAKEUP()\
siginterrupt(SIGUSR2, 1)
#endif

#if defined(_WINDOWS)
#include <windows.h>

class GCriticalSection
{
public:
	GCriticalSection()
	{
		::InitializeCriticalSection(&m_buffer);

	}

	~GCriticalSection()
	{
		::DeleteCriticalSection(&m_buffer);

	}

	inline void Enter()
	{
		::EnterCriticalSection(&m_buffer);

	}

	inline void Leave()
	{
		::LeaveCriticalSection(&m_buffer);
	}

    CRITICAL_SECTION m_buffer;

};
#else

class GCriticalSection
{
public:
	GCriticalSection()
	{
		int err = pthread_mutex_init(&m_mutex, NULL);
		m_isOk = err == 0;
		if ( !m_isOk )
		{
			std::cerr << "pthread_mutex_init() error: " << err << std::endl;
		}
	}

	~GCriticalSection()
	{
		if ( m_isOk )
		{
			int err = pthread_mutex_destroy(&m_mutex);
			if ( err != 0 )
			{
				std::cerr << "pthread_mutex_destroy() error: " << err << std::endl;
			}
		}
		else {
			std::cerr << "pthread_mutex_destroy() error: Mutex no creado correctamente" << std::endl;
		}
	}

	inline void Enter()
	{
		if (m_isOk) {
			int err = pthread_mutex_lock(&m_mutex);
			switch ( err )
			{
				case EDEADLK:
					std::cerr << "pthread_mutex_lock() error: deadlock prevenido" << std::endl;
					break;

				case EINVAL:
					std::cerr << "pthread_mutex_lock() error: No inicializado" << std::endl;
					break;

				case 0:
					break;

				default:
					std::cerr << "pthread_mutex_lock() error: " << err << std::endl;
					break;
			}
		}
		else {
			std::cerr << "pthread_mutex_lock() error: Mutex no creado correctamente" << std::endl;
		}
	}

	inline void Leave()
	{
		if (m_isOk) {
			int err = pthread_mutex_unlock(&m_mutex);
			switch ( err )
			{
				case EPERM:
					std::cerr << "pthread_mutex_unlock() error: No adquirido por el invocador" << std::endl;
					break;

				case EINVAL:
					std::cerr << "pthread_mutex_unlock() error: No inicializado" << std::endl;
					break;

				case 0:
					break;

				default:
					std::cerr << "pthread_mutex_unlock() error: " << err << std::endl;
					break;
			}
		}
		else {
			std::cerr << "pthread_mutex_unlock() error: Mutex no creado correctamente" << std::endl;
		}
	}

private:
    pthread_mutex_t m_mutex;
    bool m_isOk;

};
#endif



class GLocker;

//----------------------------------------------------------------------------------------------------
//region Interfaz de soporte de cerrojos
//----------------------------------------------------------------------------------------------------
class GLockable
{
	//----------------------------------------------------------------------------------------------------
	//region Constructor y destructor
public:

	inline GLockable();

	inline ~GLockable();

	//endregion

	//----------------------------------------------------------------------------------------------------
	//region Interfaz de exclusion mutua
public:

	inline void Lock(const std::string& loc );

	inline void UnLock(const std::string& loc );

	//enregion

	//----------------------------------------------------------------------------------------------------
	//region Interfaz de exclusion mutua desde automatico
private:

	inline void AutoLock(GLocker* pLocker, const std::string& loc = "");

	inline void AutoUnLock(GLocker* pLocker);

	//enregion

	//----------------------------------------------------------------------------------------------------
	//region Interfaz de subscripcion
private:

	GLocker* m_pLocker; // ILocker que ha bloqueado el cerrojo.
	//endregion

	//----------------------------------------------------------------------------------------------------
	//region Atributos
private:

	bool               m_IsLocked;
	std::string        m_LocBloqueo;
	GCriticalSection*  m_pCS;             // El cerrojo real.
	GCriticalSection*  m_pCSRegistro;     // Cerrojo para registrar el ocasionador del bloqueo.
	//endregion

	friend class GLocker;
};
//endregion

//----------------------------------------------------------------------------------------------------
//region Helper de bloqueo automatico
//----------------------------------------------------------------------------------------------------
class GLocker
{
	//----------------------------------------------------------------------------------------------------
	//region Construccion y destruccion
public:

	inline GLocker( GLockable& pLockable, const std::string& loc = "" )
	{
		m_pLockable = &pLockable;
		m_LocInstanciacion = loc;
		//GTRACE("GNC::GCS::ILockable()");
		m_pLockable->AutoLock(this, m_LocInstanciacion);
	}

	inline GLocker( GLockable* pLockable, const std::string& loc = "" )
	{
		m_pLockable = pLockable;
		m_LocInstanciacion = loc;
		//GTRACE("GNC::GCS::ILockable()");
		m_pLockable->AutoLock(this, loc);
	}

	inline ~GLocker()
	{
		m_pLockable->AutoUnLock(this);
		//GTRACE("GNC::GCS::~ILockable()");

	}
	//endregion

	//----------------------------------------------------------------------------------------------------
	//region Atributos
private:

	GLockable* m_pLockable;
	std::string m_LocInstanciacion;
	//endregion

	friend class GLockable;
};
//endregion


//----------------------------------------------------------------------------------------------------
//region Implementacion de GLockable
inline GLockable::GLockable()
{
	m_pLocker = NULL;
	m_IsLocked = false;
	m_pCS = new GCriticalSection();
}

inline GLockable::~GLockable()
{
	if (m_IsLocked) {
		if (m_pLocker != NULL) {
			std::cerr << "Error al destruir GLockable: El cerrojo continua autobloqueado por " << m_pLocker << " instanciado en " << m_pLocker->m_LocInstanciacion.c_str() << std::endl;
		}
		else if (m_LocBloqueo.size() != 0) {
			std::cerr << "Error al destruir GLockable: El cerrojo continua bloqueado por una llamada en " << m_LocBloqueo.c_str() << std::endl;
		}
		else {
			std::cerr << "Error al destruir GLockable: El cerrojo continua bloqueado por una llamada sin registrar" << std::endl;
		}
	}
	if (m_pCS != NULL) {
		delete m_pCS;
	}
	m_pCS = NULL;
}

inline void GLockable::Lock(const std::string& loc )
{
	SUSPEND_WAKEUP();
	//----------------------------------------------------------------------------------------------------
	// Entrada a la seccion Critica
	//----------------------------------------------------------------------------------------------------
	if (m_IsLocked) {
		//GTRACE("GNC::GCS::Lock() en " << loc.c_str() << " Esperando liberacion desde bloqueo previo en " << m_LocBloqueo.c_str());
		m_pCS->Enter();
		m_LocBloqueo = loc;
		m_IsLocked = true;
		//GTRACE("GNC::GCS::Lock() en " << loc.c_str());
	}
	else {
		m_pCS->Enter();
		m_LocBloqueo = loc;
		m_IsLocked = true;
		//GTRACE("GNC::GCS::Lock() en " << loc.c_str());
	}
	RESUME_WAKEUP();
	//----------------------------------------------------------------------------------------------------
	//----------------------------------------------------------------------------------------------------

}

inline void GLockable::UnLock(const std::string& loc )
{
	//----------------------------------------------------------------------------------------------------
	// Salida de la seccion Critica
	//----------------------------------------------------------------------------------------------------
	if (!m_IsLocked) {
		std::cerr << "Error: El cerrojo no estaba bloqueado. (Tratado de liberar en " << loc.c_str() << ")" ;
	}
	else if (m_pLocker != NULL) {
		std::cerr << "Error: El cerrojo estaba auto bloqueado previamente por " << m_pLocker << " instanciado en " << m_pLocker->m_LocInstanciacion.c_str() << std::endl;

	}
	else {
		//GTRACE("GNC::GCS::UnLock() en " << loc.c_str());
		//GTRACE("	>> Liberado bloqueo desde " << m_LocBloqueo.c_str());
		m_LocBloqueo = "";
		m_IsLocked = false;
		m_pCS->Leave();
	}
	//----------------------------------------------------------------------------------------------------
	//----------------------------------------------------------------------------------------------------

}

inline void GLockable::AutoLock(GLocker* pLocker, const std::string& loc)
{
	//----------------------------------------------------------------------------------------------------
	// Entrada a la seccion Critica
	//----------------------------------------------------------------------------------------------------
	SUSPEND_WAKEUP();
	m_pCS->Enter();
	m_IsLocked = true;
	m_pLocker = pLocker;
	m_LocBloqueo = loc;
#if defined(_GINKGO_TRACE)
	if (pLocker != NULL) {
		//GTRACE("GNC::GCS::Lock() automatico por " << pLocker << " instanciado en " << pLocker->m_LocInstanciacion.c_str());
	}
	else {
		//GTRACE("GNC::GCS::Lock() automatico por NULL (¿Error?)");
	}
#endif
	RESUME_WAKEUP();
	//----------------------------------------------------------------------------------------------------
	//----------------------------------------------------------------------------------------------------

}

inline void GLockable::AutoUnLock(GLocker* pLocker)
{
	//----------------------------------------------------------------------------------------------------
	// Salida de la seccion Critica
	//----------------------------------------------------------------------------------------------------
	if (!m_IsLocked) {
		std::cerr << "Error: El cerrojo no estaba bloqueado. (Tratado de liberar automaticamente por " << pLocker;
		if (pLocker != NULL) {
			std::cerr << " instanciado en " << pLocker->m_LocInstanciacion.c_str() << ")" << std::endl;
		}
		else {
			std::cerr << ")" << std::endl;
		}
	}
	else if (m_pLocker != pLocker) {
		std::cerr << "Error: Se ha liberado un bloqueo desde un Locker automatico distinto del que lo inicio: " << std::endl;
		std::cerr << "\tIniciado por " << m_pLocker;
		if (m_pLocker != NULL) {
			std::cerr << " instanciado en " << m_pLocker->m_LocInstanciacion.c_str() << std::endl;
		}
		else {
			std::cerr << std::endl;
		}
		std::cerr << "\tTratado de liberar  por " << pLocker;
		if (pLocker != NULL) {
			std::cerr << " instanciado en " << pLocker->m_LocInstanciacion.c_str() << std::endl;
		}
		else {
			std::cerr << std::endl;
		}
	}
	else {
#if defined (_GINKGO_TRACE)
		if (pLocker != NULL) {
			//GTRACE("GNC::GCS::UnLock() automatico por " << pLocker << " instanciado en " << pLocker->m_LocInstanciacion.c_str());
		}
		else {
			//GTRACE("GNC::GCS::UnLock() automatico por " << pLocker);
		}
#endif
		m_LocBloqueo = "";
		m_IsLocked = false;
		m_pCS->Leave();
	}
	//----------------------------------------------------------------------------------------------------
	//----------------------------------------------------------------------------------------------------

}

//endregion

struct GnkNullPointerException : public std::exception
{
    GnkNullPointerException() throw() {}
    ~GnkNullPointerException() throw() {}

    const char* what() const throw()
    {
        return "[Yasper Exception] Attempted to dereference null pointer";
    }
};

class GnkCounter : public GLockable
{
public:
	GnkCounter(unsigned c = 1) : count(c) {}

	unsigned count;
};

template <typename X>
class GnkPtr : public GLockable
{

public:
    typedef X element_type;

	/*
	 GnkPtr needs to be its own friend so GnkPtr< X > and GnkPtr< Y > can access
	 each other's private data members
	 */
	template <class Y> friend class GnkPtr;
	/*
	 default constructor
	 - don't create GnkCounter
	 */
	GnkPtr() : rawPtr(0), counter(0) { }

	/*
	 Construct from a raw pointer
	 */
	GnkPtr(X* raw, GnkCounter* c = 0) : rawPtr(0), counter(0)
	{
		Lock(GLOC());
		if (raw)
		{
			if (c) {
				c->Lock(GLOC());
				acquire(c);
				rawPtr = raw;
				c->UnLock(GLOC());
			}
			else {
				counter = new GnkCounter;
				rawPtr = raw;
			}
		}
		UnLock(GLOC());
	}

	template <typename Y>
 	explicit GnkPtr(Y* raw, GnkCounter* c = 0) : rawPtr(0), counter(0)
	{
		Lock(GLOC());
		if (raw)
		{
			if (c) {
				c->Lock(GLOC());
				acquire(c);
				rawPtr = static_cast<X*>( raw );
				c->UnLock(GLOC());
			}
			else {
				rawPtr = static_cast<X*>( raw );
				counter = new GnkCounter;
			}
		}
		UnLock(GLOC());
	}


	/*
	 Copy constructor
	 */
	GnkPtr(const GnkPtr< X >& otherPtr)
	{
		Lock(GLOC());
		( (GnkPtr<X>*) &otherPtr)->Lock(GLOC());

		if (otherPtr.counter) {
			otherPtr.counter->Lock(GLOC());

			acquire( otherPtr.counter );
			rawPtr = otherPtr.rawPtr;
			otherPtr.counter->UnLock(GLOC());
		}
		else {
			counter = NULL;
			rawPtr = NULL;
		}

		( (GnkPtr<X>*) &otherPtr)->UnLock(GLOC());
		UnLock(GLOC());
	}

	template <typename Y>
	explicit GnkPtr(const GnkPtr< Y >& otherPtr) : rawPtr(0), counter(0)
	{
		Lock(GLOC());
		( (GnkPtr<Y>*) &otherPtr)->Lock(GLOC());

		if (otherPtr.counter) {
			otherPtr.counter->Lock(GLOC());

			acquire(otherPtr.counter);
			rawPtr = static_cast<X*>( otherPtr.rawPtr );

			otherPtr.counter->UnLock(GLOC());
		}

		( (GnkPtr<Y>*) &otherPtr)->UnLock(GLOC());
		UnLock(GLOC());
	}


	/*
	 Destructor
	 */
	~GnkPtr()
	{
		Lock(GLOC());
		release();
		UnLock(GLOC());
	}

	/*
	 Assignment to another GnkPtr
	 */

	GnkPtr& operator=(const GnkPtr< X >& otherPtr)
	{
		Lock(GLOC());

		( (GnkPtr<X>*) &otherPtr)->Lock(GLOC());

		if (this != &otherPtr)
		{
			release();

			if (otherPtr.counter) {
				otherPtr.counter->Lock(GLOC());

				acquire(otherPtr.counter);
				rawPtr = static_cast<X*> (otherPtr.rawPtr);

				otherPtr.counter->UnLock(GLOC());
			}
			else {
				rawPtr = NULL;
				counter = NULL;
			}


		}

		( (GnkPtr<X>*) &otherPtr)->UnLock(GLOC());

		UnLock(GLOC());

		return *this;
	}

	template <typename Y>
	GnkPtr& operator=(const GnkPtr< Y >& otherPtr)
	{
		Lock(GLOC());
		( (GnkPtr<Y>*) &otherPtr)->Lock(GLOC());

		if ( this != (GnkPtr< X >*) &otherPtr )
		{
			release();

			if (otherPtr.counter) {
				otherPtr.counter->Lock(GLOC());

				acquire(otherPtr.counter);
				rawPtr = static_cast<X*> (otherPtr.rawPtr);

				otherPtr.counter->UnLock(GLOC());
			}
		}

		( (GnkPtr<Y>*) &otherPtr)->UnLock(GLOC());
		UnLock(GLOC());
		return *this;
	}

	/*
	 Assignment to raw pointers is really dangerous business.
	 If the raw pointer is also being used elsewhere,
	 we might prematurely delete it, causing much pain.
	 Use sparingly/with caution.
	 */

	GnkPtr& operator=(X* raw)
	{
		Lock(GLOC());
		if (raw)
		{
			release();

			GnkCounter* c = new GnkCounter;
			c->Lock(GLOC());

			counter = c;
			rawPtr = raw;

			c->UnLock(GLOC());

		}
		UnLock(GLOC());
		return *this;
	}

	template <typename Y>
	GnkPtr& operator=(Y* raw)
	{
		Lock(GLOC());
		if (raw)
		{
			release();

			GnkCounter* c = new GnkCounter();

			c->Lock(GLOC());

			counter = c;
			rawPtr = static_cast<X*>(raw);

			c->UnLock(GLOC());

		}
		UnLock(GLOC());
		return *this;
	}

	/*
	 assignment to long to allow GnkPtr< X > = NULL,
	 also allows raw pointer assignment by conversion.
	 Raw pointer assignment is really dangerous!
	 If the raw pointer is being used elsewhere,
	 it will get deleted prematurely.
	 */
	GnkPtr& operator=(long num)
	{
		Lock(GLOC());
		if (num == 0)  //pointer set to null
		{
			release();
		}

		else //assign raw pointer by conversion
		{
			release();
			GnkCounter* c = new GnkCounter();

			c->Lock(GLOC());

			counter = c;
			rawPtr = reinterpret_cast<X*>(num);

			c->UnLock(GLOC());

		}
		UnLock(GLOC());

		return *this;
	}

	/*
	 Member Access
	 */
	X* operator->() const
	{
		return GetRawPointer();
	}


	/*
	 Dereference the pointer
	 */
	X& operator* () const
	{
		return *GetRawPointer();
	}


	/*
	 Conversion/casting operators
	 */


	operator bool() const
	{
		return IsValid();
	}


	/*
	 implicit casts to base types of the
	 the pointer we're storing
	 */

	template <typename Y>
	operator Y*() const
	{
		Y* r = NULL;

		( (GnkPtr<Y>*) this)->Lock(GLOC());
		r = static_cast<Y*>(rawPtr);
		( (GnkPtr<Y>*) this)->UnLock(GLOC());

		return r;
	}

	template <typename Y>
	operator const Y*() const
	{
		const Y* r = NULL;
		( (GnkPtr<Y>*) this)->Lock(GLOC());
		r = static_cast<const Y*>(rawPtr);
		( (GnkPtr<Y>*) this)->UnLock(GLOC());
		return r;
	}

	template <typename Y>
	operator GnkPtr<Y>()
	{
		//new GnkPtr must also take our counter or else the reference counts
		//will go out of sync
		return GnkPtr<Y>(rawPtr, counter);
	}


	/*
	 Provide access to the raw pointer
	 */

	X* GetRawPointer() const
	{
		X* r = NULL;

		r = rawPtr;

		if (r == 0) {
			throw new GnkNullPointerException;
		}

		return r;
	}


	/*
	 Is there only one reference on the counter?
	 */
	bool IsUnique() const
	{
		bool unique = false;

		Lock(GLOC());
		if (counter) {
			counter->Lock(GLOC());
			unique = (counter->count == 1);
			counter->UnLock(GLOC());
		}
		else {
			unique = true;
		}
		UnLock(GLOC());

		return unique;
	}

	bool IsValid() const
	{
		bool valid = false;

		if (counter) {
			counter->Lock(GLOC());
			valid = (rawPtr != NULL);
			counter->UnLock(GLOC());
		}

		return valid;
	}

	unsigned GetCount() const
	{
		int count = 0;
		Lock(GLOC());

		if (counter) {
			counter->Lock(GLOC());
			count = counter->count;
			counter->UnLock(GLOC());
		}
		UnLock(GLOC());
		return 0;
	}

private:
	X* rawPtr;

	GnkCounter* counter;

	// increment the count
	void acquire(GnkCounter* c)
	{
		counter = c;
		(c->count)++;
	}

	// decrement the count, delete if it is 0
	void release()
	{
		if (counter)
		{
			GnkCounter* c = counter;

			c->Lock(GLOC());

			X* r = rawPtr;

			(c->count)--;

			if (c->count == 0)
			{
				counter = NULL;
				rawPtr = NULL;

				c->UnLock(GLOC());

				delete c;
				delete r;
			}
			else {
				c->UnLock(GLOC());
			}

		}

	}
};


template <typename X, typename Y>
bool operator==(const GnkPtr< X >& lGnkPtr, const GnkPtr< Y >& rGnkPtr)
{
	return lGnkPtr.GetRawPointer() == rGnkPtr.GetRawPointer();
}

template <typename X, typename Y>
bool operator==(const GnkPtr< X >& lGnkPtr, Y* raw)
{
	return lGnkPtr.GetRawPointer() == raw ;
}

template <typename X>
bool operator==(const GnkPtr< X >& lGnkPtr, long num)
{
	if (num == 0 && !lGnkPtr.IsValid())  //both pointer and address are null
	{
		return true;
	}

	else //convert num to a pointer, compare addresses
	{
		return lGnkPtr == reinterpret_cast<X*>(num);
	}

}

template <typename X, typename Y>
bool operator!=(const GnkPtr< X >& lGnkPtr, const GnkPtr< Y >& rGnkPtr)
{
	return ( !operator==(lGnkPtr, rGnkPtr) );
}

template <typename X, typename Y>
bool operator!=(const GnkPtr< X >& lGnkPtr, Y* raw)
{
	return ( !operator==(lGnkPtr, raw) );
}

template <typename X>
bool operator!=(const GnkPtr< X >& lGnkPtr, long num)
{
	return (!operator==(lGnkPtr, num) );
}

template <typename X, typename Y>
bool operator&&(const GnkPtr< X >& lGnkPtr, const GnkPtr< Y >& rGnkPtr)
{
	return lGnkPtr.IsValid() &&  rGnkPtr.IsValid();
}

template <typename X>
bool operator&&(const GnkPtr< X >& lGnkPtr, bool rval)
{
	return lGnkPtr.IsValid() && rval;
}

template <typename X>
bool operator&&(bool lval, const GnkPtr< X >& rGnkPtr)
{
	return lval &&  rGnkPtr.IsValid();
}

template <typename X, typename Y>
bool operator||(const GnkPtr< X >& lGnkPtr, const GnkPtr< Y >& rGnkPtr)
{
	return lGnkPtr.IsValid() || rGnkPtr.IsValid();
}

template <typename X>
bool operator||(const GnkPtr< X >& lGnkPtr, bool rval)
{
	return lGnkPtr.IsValid() || rval;
}

template <typename X>
bool operator||(bool lval, const GnkPtr< X >& rGnkPtr)
{
	return lval || rGnkPtr.IsValid();
}

template <typename X>
bool operator!(const GnkPtr< X >& p)
{
	return (!p.IsValid());
}


/* less than comparisons for storage in containers */
template <typename X, typename Y>
bool operator< (const GnkPtr< X >& lGnkPtr, const GnkPtr < Y >& rGnkPtr)
{
	return lGnkPtr.GetRawPointer() < rGnkPtr.GetRawPointer();
}

template <typename X, typename Y>
bool operator< (const GnkPtr< X >& lGnkPtr, Y* raw)
{
	return lGnkPtr.GetRawPointer() < raw;
}

template <typename X, typename Y>
bool operator< (X* raw, const GnkPtr< Y >& rGnkPtr)
{
	return raw < rGnkPtr.GetRawPointer();
}


