# -*- python -*-

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# This file describes the interface that must be implemented for each
# underlying tool in order for gquilt to provide an interface to for the tool

import os, os.path, shutil, gquilt_pfuns, gquilt_const, gquilt_utils

# run a command and log the result to the provided console
def exec_console_cmd(console, cmd, error=gquilt_const.ERROR):
    if console is not None:
        res, so, se = console.exec_cmd(cmd)
    else:
        res, so, se = gquilt_utils.run_cmd(cmd)
    if res != 0:
        return (error, so, se)
    else:
        return (gquilt_const.OK, so, se)

DEFAULT_NAME_EVARS = ["GIT_AUTHOR_NAME", "GECOS"]
DEFAULT_EMAIL_VARS = ["GIT_AUTHOR_EMAIL", "EMAIL_ADDRESS"]

class interface:
    def __init__(self, name):
        self._name_envars = DEFAULT_NAME_EVARS
        self._email_envars = DEFAULT_EMAIL_VARS
        self.name = name
    def is_available(self):
        return False
    def requires(self):
        assert 0, 'requires() must be provided in child class!!!'
    def new_playground(self, dir=None):
        assert 0, 'new_playground() must be provided in child class!!!'
    def is_playground(self, dir=None):
        assert 0, 'is_playground() must be provided in child class!!!'
    def is_patch_applied(self, patch):
        assert 0, 'is_patch_applied() must be provided in child class!!!'
    def patch_file_name(self, patch):
        assert 0, 'patch_file_name() must be provided in child class!!!'
    def top_patch(self):
        assert 0, 'top_patch() must be provided in child class!!!'
    def next_patch(self):
        assert 0, 'next_patch() must be provided in child class!!!'
    def base_patch(self):
        assert 0, 'next_patch() must be provided in child class!!!'
    def last_patch_in_series(self):
        assert 0, 'last_patch_in_series() must be provided in child class!!!'
    def count_ok_meld(self, count):
        # override in back end if there are restrictions on its use of meld
        return True
    def has_add_files(self):
        # override in back end if the tool does not have an "add file" function
        return True
    def has_finish_patch(self):
        # override in back end if the tool has a "qfinish patch" function
        return False
    def get_description_is_finish_ready(self, patch):
        # make sure we don't get into an infinite loop if this isn't defined
        # in the derived back end
        return True
    def display_files_diff_in_viewer(self, viewer, file, patch=None):
        assert 0, 'display_file_diff_in_viewer() must be provided in child class!!!'
    def get_playground_root(self, dir=None):
        if dir:
            return dir
        return os.getcwd()
    def get_patch_status(self, patch):
        assert 0, 'get_patch_status() must be provided in child class!!!'
    def get_patch_description(self, patch):
        # (almost) back end independent implementation needs to be overridden
        # by a back end if desired
        pfn = self.patch_file_name(patch)
        if os.path.exists(pfn):
            res, lines = gquilt_pfuns.get_patch_descr_lines(pfn)
            if res:
                return (gquilt_const.OK, os.linesep.join(lines) + os.linesep, "")
            else:
                return (gquilt_const.ERROR, "", "Error reading patch description\n")
        else:
            return (gquilt_const.OK, "", "")
    def get_series(self):
        assert 0, 'get_series() must be provided in child class!!!'
    def get_diff(self, filelist=[], patch=None):
        assert 0, 'get_diff() must be provided in child class!!!'
    def get_combined_diff(self, start_patch=None, end_patch=None):
        assert 0, 'get_combined_diff() must be provided in child class!!!'
    def get_patch_files(self, patch=None, withstatus=True):
        assert 0, 'get_patch_files() must be provided in child class!!!'
    def do_set_patch_description(self, console, patch, description):
        # (almost) back end independent implementation needs to be overridden
        # by a back end if desired
        pfn = self.patch_file_name(patch)
        res = gquilt_pfuns.set_patch_descr_lines(pfn, description.splitlines())
        if res:
            res = gquilt_const.OK
            se = ""
        else:
            res = gquilt_const.ERROR
            se = "Error reading patch description\n"
        if console is not None:
            console.log_entry('set description for "' + patch + '"')
        return (res, "", se)
    def do_rename_patch(self, console, patch, newname):
        assert 0, 'do_rename_patch() must be provided in child class!!!'
    def do_pop_patch(self, console, force=False):
        assert 0, 'do_pop_patch() must be provided in child class!!!'
    def do_push_patch(self, console, force=False):
        assert 0, 'do_push_patch() must be provided in child class!!!'
    def do_pop_to_patch(self, console, patch=None):
        assert 0, 'do_pop_to_patch() must be provided in child class!!!'
    def do_push_to_patch(self, console, patch=None):
        assert 0, 'do_push_to_patch() must be provided in child class!!!'
    def do_refresh_patch(self, console, patch=None, force=False):
        assert 0, 'do_refresh_patch() must be provided in child class!!!'
    def do_create_new_patch(self, console, name, force=False):
        assert 0, 'do_create_new_patch() must be provided in child class!!!'
    def do_import_patch(self, console, filename, patchname=None, force=False):
        assert 0, 'do_import_patch() must be provided in child class!!!'
    def do_finish_patch(self, console, patch):
        assert 0, 'do_merge_patch() must be provided in child class!!!'
    def do_merge_patch(self, console, patch):
        assert 0, 'do_merge_patch() must be provided in child class!!!'
    def do_merge_patch_file(self, console, filename):
        assert 0, 'do_merge_patch_file() must be provided in child class!!!'
    def do_delete_patch(self, console, patch):
        assert 0, 'do_delete_patch() must be provided in child class!!!'
    def do_add_files_to_patch(self, console, filelist):
        assert 0, 'do_add_files_to_patch() must be provided in child class!!!'
    def do_copy_file(self, console, file, dest, force=False):
        # back end independent implementation needs to be overridden by a
        # back end if the back end includes the concept of copy
        if os.path.isdir(dest):
            dest = os.path.join(dest, os.path.basename(file))
        if not force and os.path.exists(dest):
            return (gquilt_const.ERROR_FORCE, "", "File \"%s\" already exists. Select \"force\" to overwrite." % dest)
        res, patch_files, so = self.get_patch_files(None, False)
        if not dest in patch_files:
            res, so, se = self.do_add_files_to_patch(console, [dest])
            if res is not gquilt_const.OK:
                return (res, so, se)
        try:
            shutil.copy(file, dest)
            return (gquilt_const.OK, "", "")
        except (IOError, os.error, shutil.Error), why:
            se = "Copy %s to %s failed. %s." % (file, dest, str(why))
            console.append_stderr(se)
            return (gquilt_const.ERROR, "", se)
    def do_move_file(self, console, file, dest, force=False):
        # back end independent implementation needs to be overridden by a
        # back end if the back end includes the concept of move
        if os.path.isdir(dest):
            dest = os.path.join(dest, os.path.basename(file))
        if not force and os.path.exists(dest):
            return (gquilt_const.ERROR_FORCE, "", "File \"%s\" already exists. Select \"force\" to overwrite." % dest)
        res, patch_files, so = self.get_patch_files(None, False)
        if file not in patch_files:
            res, so, se = self.do_add_files_to_patch(console, [file])
            if res is not gquilt_const.OK:
                return (res, so, se)
        if dest not in patch_files:
            res, so, se = self.do_add_files_to_patch(console, [dest])
            if res is not gquilt_const.OK:
                return (res, so, se)
        try:
            os.rename(file, dest)
            return (gquilt_const.OK, "", "")
        except (IOError, os.error, shutil.Error), why:
            se = "Copy %s to %s failed. %s." % (file, dest, str(why))
            console.append_stderr(se)
            return (gquilt_const.ERROR, "", se)
    def do_remove_files_from_patch(self, console, filelist, patch=None):
        assert 0, 'do_remove_files_from_patch() must be provided in child class!!!'
    def do_revert_files_in_patch(self, console, filelist, patch=None):
        assert 0, 'do_revert_files_in_patch() must be provided in child class!!!'
    def do_exec_tool_cmd(self, console, cmd):
        assert 0, 'do_exec_tool_cmd() must be provided in child class!!!'
    def _get_first_in_envar(self, envar_list):
        for envar in envar_list:
            try:
                value = os.environ[envar]
                if value is not "":
                    return value
            except KeyError:
                continue
        return ""
    def get_author_name_and_email(self):
        name = self._get_first_in_envar(self._name_envars)
        if not name:
            name = "UNKNOWN"
        email = self._get_first_in_envar(self._email_envars)
        if not email:
            email = "UNKNOWN"
        return "%s <%s>" % (name, email)

# create a null back end to use when the working directory is not a valid
# playground
class null_interface(interface):
    def __init__(self):
        interface.__init__(self, "null")
    def is_available(self):
        return True
    def requires(self):
        return "Nothing!"
    def _message(self):
        return os.getcwd() + ' is not a valid playground\n'
    def get_patch_files(self, patch=None, withstatus=True):
        return (gquilt_const.OK, "", "")
    def get_series(self):
        return (gquilt_const.OK, ("", []), "")
    def top_patch(self):
        return ""
    def last_patch_in_series(self):
        return ""
    def do_add_files_to_patch(self, console, filelist):
        return (gquilt_const.ERROR, "", self._message())
    def do_copy_file(self, console, file, dest, force=False):
        return (gquilt_const.ERROR, "", self._message())
    def do_move_file(self, console, file, dest, force=False):
        return (gquilt_const.ERROR, "", self._message())
    def do_push_to_patch(self, console, patch=None):
        return (gquilt_const.ERROR, "", self._message())
    def do_pop_to_patch(self, console, patch=None):
        return (gquilt_const.ERROR, "", self._message())
    def do_refresh_patch(self, console, patch=None, force=False):
        return (gquilt_const.ERROR, "", self._message())
    def do_create_new_patch(self, console, name, force=False):
        return (gquilt_const.ERROR, "", self._message())
    def do_import_patch(self, console, filename, patchname=None, force=False):
        return (gquilt_const.ERROR, "", self._message())
    def get_combined_diff(self, start_patch=None, end_patch=None):
        return (gquilt_const.ERROR, "", self._message())
    def do_merge_patch_file(self, console, patch):
        return (gquilt_const.ERROR, "", self._message())
    def do_exec_tool_cmd(self, console, cmd):
        return (gquilt_const.ERROR, "", self._message())

