# -*- coding: iso-8859-1 -*-
#
#-------------------------------------------------------------------------------
#
#     This file is part of the Code_Saturne User Interface, element of the
#     Code_Saturne CFD tool.
#
#     Copyright (C) 1998-2008 EDF S.A., France
#
#     contact: saturne-support@edf.fr
#
#     The Code_Saturne User Interface is free software; you can redistribute it
#     and/or modify it under the terms of the GNU General Public License
#     as published by the Free Software Foundation; either version 2 of
#     the License, or (at your option) any later version.
#
#     The Code_Saturne User Interface is distributed in the hope that it will be
#     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
#     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
#
#     You should have received a copy of the GNU General Public License
#     along with the Code_Saturne Kernel; if not, write to the
#     Free Software Foundation, Inc.,
#     51 Franklin St, Fifth Floor,
#     Boston, MA  02110-1301  USA
#
#-------------------------------------------------------------------------------

"""
This module defines basic classes used for the Pages construction.

This module defines the following classes:
- ComboModel
- IntValidator
- DoubleValidator
- RegExpValidator
"""

#-------------------------------------------------------------------------------
# Library modules import
#-------------------------------------------------------------------------------

import sys
import logging

#-------------------------------------------------------------------------------
# Third-party modules
#-------------------------------------------------------------------------------

from PyQt4 import QtGui, QtCore

#-------------------------------------------------------------------------------
# Application modules import
#-------------------------------------------------------------------------------

from Toolbox import GuiParam

#-------------------------------------------------------------------------------
# log config
#-------------------------------------------------------------------------------

logging.basicConfig()
log = logging.getLogger("QtPage")
log.setLevel(GuiParam.DEBUG)

#-------------------------------------------------------------------------------
# Combobox model
#-------------------------------------------------------------------------------

class ComboModel():
    """
    Class to build a model (QStandardItemModel) used with a QComboBox.

    Main attributes of class are:

    combo: QComboBox passed as arguments. It uses the model
    model: QStandardItemModel which contains items

    dicoV2M: correspondance between strings in QT view and strings in Saturne model
    dicoM2V: correspondance between strings in Saturne model and strings in QT view

    items: tuple which contains all model strings (usefull to get its index in the model)
    """
    def __init__(self, combo, rows=0, columns=0):
        """
        Initialization
        """
        self.combo   = combo

        self.rows    = rows
        self.columns = columns
        self.last    = 0

        self.model   = QtGui.QStandardItemModel()
        self.model.clear()
        self.model.setRowCount(rows)
        self.model.setColumnCount(columns)

        self.dicoV2M = {}
        self.dicoM2V = {}

        self.items   = []
        self.combo.setModel(self.model)


    def addItem(self, str_view, str_model=""):
        """
        Insert an item in the model.

        str_view: string to be displayed in the view.
        For example, 'Eulerian/Lagangian Multi-phase Treatment'

        str_model: correponding string used in the model.
        For example, 'lagrangian'
        """
        item  = QtGui.QStandardItem(QtCore.QString(str(str_view)))
        index = self.last
        self.model.setItem(index, item)

        self.last = index + 1

        if not str_model: str_model = str_view

        self.items.append(str_model)

        self.dicoM2V[str_model] = str_view
        self.dicoV2M[str_view]  = str_model


    def modifyItem(self, old_str_view, new_str_view, new_str_model=""):
        """
        Modify string names.
        """
        if old_str_view in self.items:
            index = self.items.index(old_str_view)
            self.items[index] = new_str_view

            old_str_model = dicoV2M[old_str_view]  
            if new_str_model == "":
                new_str_model = old_str_model

            del self.dicoM2V[str_model]
            del self.dicoV2M[str_view]

            self.dicoM2V[new_str_model] = new_str_view
            self.dicoV2M[new_str_view]  = new_str_model


    def delItem(self,index=None,str_model="",str_view=""):
        """
        Remove the item specified with its index or a string.
        """
        if index is not None:
            self._deleteItem(index)

        elif str_model:
            index = self.items.index(str_model)
            self._deleteItem(index)

        elif str_view:
            str_model = self.dicoV2M[str_view]
            index = self.items.index(str_model)
            self._deleteItem(index)


    def _deleteItem(self,index):
        """
        """
        str_view  = self.items[index]
        str_model = self.dicoV2M[str_view]
        del self.items[index]
        del self.dicoV2M[str_view]
        del self.dicoM2V[str_model]


    def _disableItem(self,index):
        """
        Disable the item specified with its index
        """
        self.model.item(index).setEnabled(False)


    def _enableItem(self,index):
        """
        Enable the item specified with its index
        """
        self.model.item(index).setEnabled(True)


    def disableItem(self,index=None,str_model="",str_view=""):
        """
        Disable the item specified with its index or a string.
        """
        if index is not None:
            self._disableItem(index)

        elif str_model:
            index = self.items.index(str_model)
            self._disableItem(index)

        elif str_view:
            str_model = self.dicoV2M[str_view]
            index = self.items.index(str_model)
            self._disableItem(index)


    def enableItem(self,index=None,str_model="",str_view=""):
        """
        Enable the item specified with its index or a string.
        """
        if index is not None:
            self._enableItem(index)

        elif str_model:
            index = self.items.index(str_model)
            self._enableItem(index)

        elif str_view:
            str_model = self.dicoV2M[str_view]
            index = self.items.index(str_model)
            self._enableItem(index)


    def setItem(self, index=None, str_model="", str_view=""):
        """
        Set item as current.
        """
        if index is not None:
            self.combo.setCurrentIndex(index)

        elif str_model:
            index = self.items.index(str_model)
            self.combo.setCurrentIndex(index)

        elif str_view:
            str_model = self.dicoV2M[str_view]
            index = self.items.index(str_model)
            self.combo.setCurrentIndex(index)


    def getIndex(self, str_model="", str_view=""):
        """
        Get the index for a string.
        """
        if str_model:
            index = self.items.index(str_model)

        elif str_view:
            str_model = self.dicoV2M[str_view]
            index = self.items.index(str_model)

        return index


    def getItems(self):
        """
        Get the tuple of items.
        """
        return self.items

#-------------------------------------------------------------------------------
# Validators
#-------------------------------------------------------------------------------

vmax = sys.maxint
vmax = 2147483647
vmin = -vmax

class IntValidator(QtGui.QIntValidator):
    """
    Validator for integer data.
    """
    def __init__(self, parent, min=vmin, max=vmax):
        """
        Initialization for validator
        """
        QtGui.QIntValidator.__init__(self, parent)
        self.parent = parent
        self.state = QtGui.QValidator.Invalid

        if type(min) != int or type(max) != int:
            raise ValueError, "The given parameters are not integers (warning: long are not allowed)."
        self.setBottom(min)
        self.setTop(max)

        self.exclusiveMin = False
        self.exclusiveMax = False

        self.default = 0
        self.fix = False


    def setExclusiveMin(self, b=True):
        if type(b) != bool:
            raise ValueError, "The given parameter is not a boolean."
        self.exclusiveMin = b


    def setExclusiveMax(self, b=True):
        if type(b) != bool:
            raise ValueError, "The given parameter is not a boolean."
        self.exclusiveMax = b


    def setFixup(self, v):
        if type(v) != int:
            raise ValueError, "The given parameter is not an integer."
        self.default = v
        self.fix = True


    def fixup(self, string):
        if self.fix:
            if string.length() == 0:
                string.truncate(0)
                string += str(self.default)


    def validate(self, string, pos):
        """
        Validation method.

        QValidator.Invalid       0  The string is clearly invalid.
        QValidator.Intermediate  1  The string is a plausible intermediate value during editing.
        QValidator.Acceptable    2  The string is acceptable as a final result; i.e. it is valid.
        """
        state = QtGui.QIntValidator.validate(self, string, pos)[0]

        x, valid = string.toInt()

        if state == QtGui.QValidator.Acceptable:
            if self.exclusiveMin and x == self.bottom():
                state = QtGui.QValidator.Intermediate
            if self.exclusiveMax and x == self.top():
                state = QtGui.QValidator.Intermediate

        palette = self.parent.palette()

        if not valid or state == QtGui.QValidator.Intermediate:
            palette.setColor(QtGui.QPalette.Text, QtGui.QColor("red"))
            self.parent.setPalette(palette)
        else:
            palette.setColor(QtGui.QPalette.Text, QtGui.QColor("black"))
            self.parent.setPalette(palette)

        self.state = state

        return (state, pos)


class DoubleValidator(QtGui.QDoubleValidator):
    """
    Validator for real data.
    """
    def __init__(self, parent, min=-1.e99, max=1.e99):
        """
        Initialization for validator
        """
        QtGui.QDoubleValidator.__init__(self, parent)
        self.parent = parent
        self.state = QtGui.QValidator.Invalid

        self.setNotation(self.ScientificNotation)

        if type(min) != float or type(max) != float:
            raise ValueError, "The given parameters are not floats."
        self.setBottom(min)
        self.setTop(max)

        self.exclusiveMin = False
        self.exclusiveMax = False

        self.default = 0.0
        self.fix = False


    def setExclusiveMin(self, b=True):
        if type(b) != bool:
            raise ValueError, "The given parameter is not a boolean."
        self.exclusiveMin = b


    def setExclusiveMax(self, b=True):
        if type(b) != bool:
            raise ValueError, "The given parameter is not a boolean."
        self.exclusiveMax = b


    def setFixup(self, v):
        if type(v) != float:
            raise ValueError, "The given parameter is not a float."
        self.default = v
        self.fix = True


    def fixup(self, string):
        if self.fix:
            if string.length() == 0:
                string.truncate(0)
                string += str(self.default)


    def validate(self, string, pos):
        """
        Validation method.

        QValidator.Invalid       0  The string is clearly invalid.
        QValidator.Intermediate  1  The string is a plausible intermediate value during editing.
        QValidator.Acceptable    2  The string is acceptable as a final result; i.e. it is valid.
        """
        state = QtGui.QDoubleValidator.validate(self, string, pos)[0]

        x, valid = string.toDouble()

        if state == QtGui.QValidator.Acceptable:
            if self.exclusiveMin and x == self.bottom():
                state = QtGui.QValidator.Intermediate
            if self.exclusiveMax and x == self.top():
                state = QtGui.QValidator.Intermediate

        palette = self.parent.palette()

        if not valid or state == QtGui.QValidator.Intermediate:
            palette.setColor(QtGui.QPalette.Text, QtGui.QColor("red"))
            self.parent.setPalette(palette)
        else:
            palette.setColor(QtGui.QPalette.Text, QtGui.QColor("black"))
            self.parent.setPalette(palette)

        self.state = state

        return (state, pos)


class RegExpValidator(QtGui.QRegExpValidator):
    """
    Validator for regular expression.
    """
    def __init__(self, parent, rx):
        """
        Initialization for validator
        """
        QtGui.QRegExpValidator.__init__(self, parent)
        self.parent = parent
        self.state = QtGui.QRegExpValidator.Invalid

        regExp = QtCore.QRegExp(rx)
        self.__validator = QtGui.QRegExpValidator(regExp, parent)


    def validate(self, string, pos):
        """
        Validation method.

        QValidator.Invalid       0  The string is clearly invalid.
        QValidator.Intermediate  1  The string is a plausible intermediate value during editing.
        QValidator.Acceptable    2  The string is acceptable as a final result; i.e. it is valid.
        """
        state = self.__validator.validate(string, pos)[0]

        palette = self.parent.palette()

        if state == QtGui.QValidator.Intermediate:
            palette.setColor(QtGui.QPalette.Text, QtGui.QColor("red"))
            self.parent.setPalette(palette)
        else:
            palette.setColor(QtGui.QPalette.Text, QtGui.QColor("black"))
            self.parent.setPalette(palette)

        self.state = state

        return (state, pos)

#-------------------------------------------------------------------------------
# End of QtPage
#-------------------------------------------------------------------------------
