# -*- coding: iso-8859-1 -*-
#
#-------------------------------------------------------------------------------
#
#     This file is part of the Code_Saturne User Interface, element of the
#     Code_Saturne CFD tool.
#
#     Copyright (C) 1998-2008 EDF S.A., France
#
#     contact: saturne-support@edf.fr
#
#     The Code_Saturne User Interface is free software; you can redistribute it
#     and/or modify it under the terms of the GNU General Public License
#     as published by the Free Software Foundation; either version 2 of
#     the License, or (at your option) any later version.
#
#     The Code_Saturne User Interface is distributed in the hope that it will be
#     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
#     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
#
#     You should have received a copy of the GNU General Public License
#     along with the Code_Saturne Kernel; if not, write to the
#     Free Software Foundation, Inc.,
#     51 Franklin St, Fifth Floor,
#     Boston, MA  02110-1301  USA
#
#-------------------------------------------------------------------------------

"""
This module defines the Page for the physical properties of the fluid.
These properties can be reference value or initial value

This module contains the following classes and function:
- FluidCharacteristicsView
"""

#-------------------------------------------------------------------------------
# Library modules import
#-------------------------------------------------------------------------------

import logging

#-------------------------------------------------------------------------------
# Third-party modules
#-------------------------------------------------------------------------------

from PyQt4.QtCore import *
from PyQt4.QtGui  import *

#-------------------------------------------------------------------------------
# Application modules import
#-------------------------------------------------------------------------------

from Base.Toolbox import GuiParam
from FluidCharacteristicsForm import Ui_FluidCharacteristicsForm
import Base.QtPage as QtPage
from Pages.FluidCharacteristicsModel import FluidCharacteristicsModel

#-------------------------------------------------------------------------------
# log config
#-------------------------------------------------------------------------------

logging.basicConfig()
log = logging.getLogger("FluidCharacteristicsView")
log.setLevel(GuiParam.DEBUG)

#-------------------------------------------------------------------------------
# Main class
#-------------------------------------------------------------------------------

# self.neta: not used. to delete ?

class FluidCharacteristicsView(QWidget, Ui_FluidCharacteristicsForm):
    """
    Class to open Molecular Properties Page.
    """
    
    def __init__(self, parent, case):
        """
        Constructor
        """
        QWidget.__init__(self, parent)

        Ui_FluidCharacteristicsForm.__init__(self)
        self.setupUi(self)

        self.case = case
        self.mdl = FluidCharacteristicsModel(self.case)

        # Labels

        self.labelRho.setText(QString(self.tr("Reference\nvalue")))
        self.labelNu.setText(QString(self.tr("Reference\nvalue")))
        self.labelCp.setText(QString(self.tr("Reference\nvalue")))
        self.labelAl.setText(QString(self.tr("Reference\nvalue")))

        # Combo models

        self.modelRho = QtPage.ComboModel(self.comboBoxRho, 2, 1)
        self.modelNu  = QtPage.ComboModel(self.comboBoxNu, 2, 1)
        self.modelCp  = QtPage.ComboModel(self.comboBoxCp, 2, 1)
        self.modelAl  = QtPage.ComboModel(self.comboBoxAl, 2, 1)

        self.modelRho.addItem('constant')
        self.modelRho.addItem('variable')
        self.modelNu.addItem('constant')
        self.modelNu.addItem('variable')
        self.modelCp.addItem('constant')
        self.modelCp.addItem('variable')
        self.modelAl.addItem('constant')
        self.modelAl.addItem('variable')

        # Connections

        self.connect(self.comboBoxRho, SIGNAL("activated(const QString&)"), self.slotStateRho)
        self.connect(self.comboBoxNu, SIGNAL("activated(const QString&)"), self.slotStateNu)
        self.connect(self.comboBoxCp, SIGNAL("activated(const QString&)"), self.slotStateCp)
        self.connect(self.comboBoxAl, SIGNAL("activated(const QString&)"), self.slotStateAl)
        self.connect(self.lineEditRho, SIGNAL("textChanged(const QString &)"), self.slotRho)
        self.connect(self.lineEditNu, SIGNAL("textChanged(const QString &)"), self.slotNu)
        self.connect(self.lineEditCp, SIGNAL("textChanged(const QString &)"), self.slotCp)
        self.connect(self.lineEditAl, SIGNAL("textChanged(const QString &)"), self.slotAl)

        # Validators

        validatorRho = QtPage.DoubleValidator(self.lineEditRho, min = 0.0)
        validatorNu = QtPage.DoubleValidator(self.lineEditNu, min = 0.0)
        validatorCp = QtPage.DoubleValidator(self.lineEditCp, min = 0.0)
        validatorAl = QtPage.DoubleValidator(self.lineEditAl, min = 0.0)

        validatorRho.setExclusiveMin(True)
        validatorNu.setExclusiveMin(True)
        validatorCp.setExclusiveMin(True)
        validatorAl.setExclusiveMin(True)

        self.lineEditRho.setValidator(validatorRho)
        self.lineEditNu.setValidator(validatorNu)
        self.lineEditCp.setValidator(validatorCp)
        self.lineEditAl.setValidator(validatorAl)

        # Initialization

        dico = {0:"Rho", 1:"Nu", 2:"Cp", 3:"Al"}

        #for count number of "eta" = variable to print message "usphyv"

        self.neta = [0]*4

        # OptionMenu initialization

        modl_joul, modl_thermo, modl_gas, modl_coal = self.mdl.getThermoPhysicalModel()

        if modl_joul == 'off' and modl_thermo == 'off':
            self.groupBoxAl.hide()
            list = [(0,'density'), (1,'molecular_viscosity'), (2,'specific_heat')]
            self.nbeta = 3
        else:
            list = [ (0,'density'),
                     (1,'molecular_viscosity'),
                     (2,'specific_heat'),
                     (3,'thermal_conductivity')]
            self.nbeta = 4

        for i, name in list:
            _model = getattr(self, "model"    + dico[i])
            _combo = getattr(self, "comboBox" + dico[i])
            _line  = getattr(self, "lineEdit" + dico[i])
            _label = getattr(self, "label"    + dico[i])

            choice = self.mdl.getPropertyMode(name)
            if choice:
                if modl_joul == 'off' and modl_thermo == 'off':
                    _model.setItem(str_model=choice)
                else:
                    _model.setItem(str_model=choice)

            else:
                if modl_joul == 'off' and modl_thermo == 'off':
                    _model.setItem(str_model='constant')
                    self.mdl.setPropertyMode(name, 'constant')
                elif modl_thermo != 'off':
                    _model.setItem(str_model='constant')
                    self.mdl.setPropertyMode(name, 'constant')

            if modl_joul != 'off':
                _model.setItem(str_model='variable')
                _model.disableItem(str_model='constant')
                self.mdl.setPropertyMode(name, 'variable')

            if modl_gas != 'off' or modl_coal != 'off':
                if i == 0:
                    self.groupBoxRho.setDisabled(True)
                    self.mdl.setPropertyMode(name, 'variable')
                    _label.setText(QString("Calculation by\n perfect gas law"))
                else:
                    _model.setItem(str_model='constant')
                    self.mdl.setPropertyMode(name, 'constant')

        # Entry initialization

        if modl_gas != 'off' or modl_coal != 'off':
            rho = ''
        else:
            rho = self.mdl.getInitialValueDensity()
        self.lineEditRho.setText(QString(str(rho)))

        nu = self.mdl.getInitialValueViscosity()
        self.lineEditNu.setText(QString(str(nu)))

        cp = self.mdl.getInitialValueHeat()
        self.lineEditCp.setText(QString(str(cp)))

        if len(list) == 4:
            al = self.mdl.getInitialValueCond()
            self.lineEditAl.setText(QString(str(al)))


    @pyqtSignature("const QString &")
    def slotStateRho(self, text):
        """
        Method to call 'getState' with correct arguements for 'rho'
        """
        self.__getState(0, str(text), 'density', 'IROVAR')


    @pyqtSignature("const QString &")
    def slotStateNu(self, text):
        """
        Method to call 'getState' with correct arguements for 'nu'
        """
        self.__getState(1, str(text), 'molecular_viscosity', 'IVIVAR')


    @pyqtSignature("const QString &")
    def slotStateCp(self, text):
        """
        Method to call 'getState' with correct arguements for 'Cp'
        """
        self.__getState(2, str(text), 'specific_heat', 'ICP')


    @pyqtSignature("const QString &")
    def slotStateAl(self, text):
        """
        Method to call 'getState' with correct arguements for 'Al'
        """
        self.__getState(3, str(text), 'thermal_conductivity', 'IVISLS')


    def __getState(self, i, var, name, key):
        """
        Input variable state
        """
        # In Tk, var is the variable, then its value is var.get()
        # In Qt, var is directly its value (i.e. 'constant' or 'variable')

        # Generic pattern to get appropriates combobox (and its model), and label. 
        dico = {0:"Rho", 1:"Nu", 2:"Cp", 3:"Al"}
        _model = getattr(self, "model"    + dico[i])
        _combo = getattr(self, "comboBox" + dico[i])
        _label = getattr(self, "label"    + dico[i])
        
        modl_joul, modl_thermo, modl_gas, modl_coal = self.mdl.getThermoPhysicalModel()

        if modl_gas != 'off' or modl_coal != 'off':
            if i == 1 or i == 2:
                _model.disableItem(str_model='variable')
                _combo.setDisabled(True)
                self.mdl.setPropertyMode(name, 'constant')
                self.neta[i] = 0

        elif i == 3 and modl_joul == 'off':
            if modl_thermo == 'off':
                pass
            else:
                self.mdl.setChoiceNode(name, var)
                if self.mdl.getPropertyMode(name) == 'variable':
                    self.neta[3] = 1
                    self.mdl.removeListingAndRecordingStatus(name)

        else:
            self.mdl.setPropertyMode(name, var)

        if modl_joul != 'off':
            _model.disableItem(str_model='constant')
            _combo.setDisabled(True)
            self.mdl.setPropertyMode(name, 'variable')
            self.neta[i] = 2

        else:
            if var == 'variable':
                self.neta[i] = 1
                _label.setText(QString("Reference\nvalue"))
            else:
                _label.setText(QString("Reference\nvalue"))
                self.neta[i] = 0

        # we test for printing message to not forget subroutines

        icompt=0
        for n in range(0, self.nbeta):
            if self.neta[n] == 1:
                icompt = 1
##                 self.stbar.set(t.MSG_USPHYV)
            elif self.neta[n] == 2:
                icompt = 2
##                 self.stbar.set(t.MSG_USELPH)
##         if icompt == 0: self.stbar.clear()


    @pyqtSignature("const QString &")
    def slotRho(self, text):
        """
        Update the density
        """
        modl_joul, modl_thermo, modl_gas, modl_coal = self.mdl.getThermoPhysicalModel()
        if modl_gas != 'off' or modl_coal != 'off':
            return

        rho, ok = text.toDouble()
        if self.sender().validator().state == QValidator.Acceptable:
            self.mdl.setInitialValueDensity(rho)


    @pyqtSignature("const QString &")
    def slotNu(self, text):
        """
        Update the molecular viscosity
        """
        nu, ok = text.toDouble()
        if self.sender().validator().state == QValidator.Acceptable:
            self.mdl.setInitialValueViscosity(nu)


    @pyqtSignature("const QString &")
    def slotCp(self, text):
        """
        Update the specific heat
        """
        cp, ok = self.lineEditCp.text().toDouble()
        if self.sender().validator().state == QValidator.Acceptable:
            self.mdl.setInitialValueHeat(cp)


    @pyqtSignature("const QString &")
    def slotAl(self, text):
        """
        Update the thermal conductivity
        """
        modl_joul, modl_thermo, modl_gas, modl_coal = self.mdl.getThermoPhysicalModel()

        if modl_joul == 'off' and modl_thermo == 'off':
            self.groupBoxAl.hide() # self.lf[3].pack_forget()
        else:
            al, ok = text.toDouble()
            if self.sender().validator().state == QValidator.Acceptable:
                self.mdl.setInitialValueCond(al)


    def tr(self, text):
        """
        Translation
        """
        return text 
        

#-------------------------------------------------------------------------------
# Testing part
#-------------------------------------------------------------------------------


if __name__ == "__main__":
    pass


#-------------------------------------------------------------------------------
# End
#-------------------------------------------------------------------------------
