/*
Copyright 1990-2003 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/

#pragma ident	"$Id$"

#include	<stdio.h>
#include	<stdlib.h>
#include	<strings.h>
#include	<X11/Xlib.h>
#include	"xaux_ext_common.h"

typedef struct xaux_ext_hdlrmng_ent_struct {
	struct xaux_ext_hdlrmng_ent_struct	*prev;
	struct xaux_ext_hdlrmng_ent_struct	*next;
	xaux_ext_cb_evhandler_t	handler;
	void			*sodata;
} xaux_ext_hdlrmng_ent_t;

#define	MAXMASK		32

typedef struct xaux_ext_hdlrmng_db_struct {
	xaux_ext_hdlrmng_ent_t	maskable[MAXMASK];
	xaux_ext_hdlrmng_ent_t	nonmaskable;
	long			comb_mask;
	Bool			comb_nonmaskable;
} xaux_ext_hdlrmng_db_t;

/*
 * mask2index
 * 	e.g. PropertyChangeMask is defined as (1L<<22)
 *	mask2index(PropertChangeMask) returns 22.
 *	When bit-or of multiple masks is given, result is undefined.
 */
static int
mask2index(long mask)
{
	int	i;

	for (i = 0; i < MAXMASK; i++) {
		if ((mask & 1L) != 0) {
			return (i);
		} else {
			mask >>= 1;
		}
	}
	return (-1);
}

/*
 * index2mask
 * 	e.g. PropertyChangeMask is defined as (1L<<22)
 *	index2mask(22) returns PropertyChangeMask
 */
static long
index2mask(int idx)
{
	return (1L << idx);
}

/*
 * type2mask - event type to mask
 */
static long
type2mask(
	int	type,
	Bool	*nonmaskable_return)
{
	long	mask;

	switch (type) {
	case DestroyNotify:
		mask = StructureNotifyMask;
		*nonmaskable_return = False;
		break;
	case PropertyNotify:
		mask = PropertyChangeMask;
		*nonmaskable_return = False;
		break;
	case SelectionClear:
	case SelectionNotify:
	case SelectionRequest:
	case ClientMessage:
		mask = NoEventMask;
		*nonmaskable_return = True;
		break;
	default:
		/* event type not interested by xaux_ext */
		mask = NoEventMask;
		*nonmaskable_return = False;
		break;
	}

	return (mask);
}

static Bool
add_to_list(
	xaux_ext_hdlrmng_ent_t	*list,
	xaux_ext_cb_evhandler_t	handler,
	void			*sodata)
{
	xaux_ext_hdlrmng_ent_t	*le = list, *le_prev = NULL;

	while (le != NULL) {
		if ((le->handler == handler) && (le->sodata == sodata)) {
			/* already registered */
			return (True);
		} 
		le_prev = le;
		le = le->next;
	}

	if ((le = le_prev->next = (xaux_ext_hdlrmng_ent_t *)malloc(
			sizeof (xaux_ext_hdlrmng_ent_t))) == NULL) {
		return (False);
	}

	le->prev = le_prev;
	le->next = NULL;
	le->handler = handler;
	le->sodata = sodata;

	return (True);
}

static Bool
rm_from_list(
	xaux_ext_hdlrmng_ent_t	*list,
	xaux_ext_cb_evhandler_t	handler,
	void			*sodata)
{
	xaux_ext_hdlrmng_ent_t	*le = list, *le_prev;
	Bool			rv = False;

	while (le != NULL) {
		if ((le->handler == handler) && (le->sodata == sodata)) {
			/* found */
			le->prev->next = le->next;
			if (le->next != NULL) {
				le->next->prev = le->prev;
			}
			free(le);
			return (True);
		} 
		le = le->next;
	}

	/* not found */
	return (False);
}

static void
free_list(
	xaux_ext_hdlrmng_ent_t	*list)
{
	xaux_ext_hdlrmng_ent_t	*le = list, *le_next;

	while (le) {
		le_next = le->next;
		free(le);
		le = le->next;
	}
	return;
}

static xaux_ext_hdlrmng_handle_t
xaux_ext_hdlrmng_open(void)
{
	xaux_ext_hdlrmng_db_t	*db;
	int	i;

	if ((db = (xaux_ext_hdlrmng_db_t *)malloc(
			sizeof (xaux_ext_hdlrmng_db_t))) == NULL) {
		return (NULL);
	}

	bzero(db, sizeof (xaux_ext_hdlrmng_db_t));

	for (i = 0; i < MAXMASK; i++) {
		db->maskable[i].prev = NULL;
		db->maskable[i].next = NULL;
		db->maskable[i].handler = NULL;
		db->maskable[i].sodata = NULL;
	}

	db->nonmaskable.prev = NULL;
	db->nonmaskable.next = NULL;
	db->nonmaskable.handler = NULL;
	db->nonmaskable.sodata = NULL;

	db->comb_mask = 0L;
	db->comb_nonmaskable = False;

	return ((xaux_ext_hdlrmng_handle_t)db);
}

static Bool
xaux_ext_hdlrmng_add(
	xaux_ext_hdlrmng_handle_t 	handle,
	xaux_ext_cb_evhandler_t	handler,
	void			*sodata,
	long			mask,
	Bool			nonmaskable,
	long			*mask_add_return,
	Bool			*nonmaskable_add_return,
	long			*mask_new_return,
	Bool			*nonmaskable_new_return)
{
	xaux_ext_hdlrmng_db_t	*db = (xaux_ext_hdlrmng_db_t *)handle;
	int			m;
	Bool			rv = False;
	long			mask_new;
	Bool			nonmaskable_new;

#if defined(DEBUG_XAUX)
	fprintf(stderr, "(--) hdlrmng_add: handle=%p handler=%p sodata=%p "
		"mask=0x%x nonmaskable=%d %p %p %p %p\n",
		handle, handler, sodata, mask, nonmaskable,
		mask_add_return, nonmaskable_add_return,
		mask_new_return, nonmaskable_new_return);
#endif /* defined(DEBUG_XAUX) */

	if (handler == NULL) {
		return (False);
	}

	*mask_add_return = NoEventMask;
	*nonmaskable_add_return = False;
	mask_new = db->comb_mask;
	nonmaskable_new = db->comb_nonmaskable;

	for (m = 0; m < MAXMASK; m++) {
		if ((mask & (1L << m)) == 0L) {
			continue;
		}
		if (add_to_list(&(db->maskable[m]), handler, sodata) == True) {
			*mask_add_return |= (1L << m);
			mask_new |= (1L << m);
		}
	}

	if (nonmaskable == True) {
		if (add_to_list(&(db->nonmaskable), handler, sodata) == True) {
			*nonmaskable_add_return = True;
			nonmaskable_new = True;
		}
	}

	if ((*mask_add_return != NoEventMask) ||
		(*nonmaskable_add_return == True)) {
		/* mask and/or nonmaskable has been changed */
		rv = True;
	}

	db->comb_mask = mask_new;
	db->comb_nonmaskable = nonmaskable_new;

	if (mask_new_return != NULL) {
		*mask_new_return = mask_new;
	}
	if (nonmaskable_new_return != NULL) {
		*nonmaskable_new_return = nonmaskable_new;
	}

	return (rv);
}

static Bool
xaux_ext_hdlrmng_remove(
	xaux_ext_hdlrmng_handle_t 	handle,
	xaux_ext_cb_evhandler_t	handler,
	void			*sodata,
	long			mask,
	Bool			nonmaskable,
	long			*mask_rm_return,
	Bool			*nonmaskable_rm_return,
	long			*mask_new_return,
	Bool			*nonmaskable_new_return)
{
	xaux_ext_hdlrmng_db_t	*db = (xaux_ext_hdlrmng_db_t *)handle;
	int			m;
	Bool			rv = False;
	long			mask_new;
	Bool			nonmaskable_new;

	if (handler == NULL) {
		return (False);
	}

	*mask_rm_return = NoEventMask;
	*nonmaskable_rm_return = False;
	mask_new = db->comb_mask;
	nonmaskable_new = db->comb_nonmaskable;

	for (m = 0; m < MAXMASK; m++) {
		if ((mask & (1L << m)) == 0L) {
			continue;
		}
		if (rm_from_list(&(db->maskable[m]), handler, sodata) == True) {
			if (db->maskable[m].next == NULL) {
				*mask_rm_return |= (1L << m);
				mask_new &= ~(1L << m);
			}
		}
	}

	if (nonmaskable == True) {
		if (rm_from_list(&(db->nonmaskable), handler, sodata) == True) {
			if (db->nonmaskable.next == NULL) {
				*nonmaskable_rm_return = True;
				nonmaskable_new = False;
			}
		}
	}

	if ((*mask_rm_return != NoEventMask) ||
		(*nonmaskable_rm_return == True)) {
		/* mask and/or nonmaskable has been changed */
		rv = True;
	}

	db->comb_mask = *mask_new_return;
	db->comb_nonmaskable = *nonmaskable_new_return;

	if (mask_new_return != NULL) {
		*mask_new_return = mask_new;
	}
	if (nonmaskable_new_return != NULL) {
		*nonmaskable_new_return = nonmaskable_new;
	}

	return (rv);
}

static Bool
xaux_ext_hdlrmng_filter(
	xaux_ext_hdlrmng_handle_t 	handle,
	xaux_ext_handle_t	ext_handle,
	XEvent			*ev)
{
	xaux_ext_hdlrmng_db_t	*db = (xaux_ext_hdlrmng_db_t *)handle;
	xaux_ext_hdlrmng_ent_t	*le, *list;
	Bool	nonmaskable = False;
	long	mask;

#if defined(DEBUG_XAUX_DETAIL)
	fprintf(stderr, "(--) filter called ev->type = %d\n", ev->type);
#endif /* defined(DEBUG_XAUX_DETAIL) */

	mask = type2mask(ev->type, &nonmaskable);

	if (mask != NoEventMask) {
		le = db->maskable[mask2index(mask)].next;
		while (le != NULL) {
			if ((*le->handler)(ext_handle,
					(Window)None, ev, le->sodata) == True) {
				return (True);
			}
			le = le->next;
		}
	}

	if (nonmaskable == True) {
		le = db->nonmaskable.next;
		while (le != NULL) {
			if ((*le->handler)(ext_handle,
					(Window)None, ev, le->sodata) == True) {
				return (True);
			}
			le = le->next;
		}
	}

	return (False);
}

static void
xaux_ext_hdlrmng_close(xaux_ext_hdlrmng_handle_t handle)
{
	xaux_ext_hdlrmng_db_t	*db = (xaux_ext_hdlrmng_db_t *)handle;
	int	i;

	if (db == NULL) {
		return;
	}

	for (i = 0; i < MAXMASK; i++) {
		free_list(db->maskable[i].next);
	}
	free_list(db->nonmaskable.next);
	free(db);
	return;
}


xaux_ext_hdlrmng_methods_t xaux_ext_hdlrmng_methods = {
	xaux_ext_hdlrmng_open,
	xaux_ext_hdlrmng_add,
	xaux_ext_hdlrmng_remove,
	xaux_ext_hdlrmng_filter,
	xaux_ext_hdlrmng_close
};
