#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from optparse import OptionParser
import logging
import sys
import os

from twisted.web.server import Site
from twisted.internet import reactor

from imagestore.lib.service import ServiceHub
from imagestore.presenter import Presenter
from imagestore.resources import Root

from imagestore.proxyservice import ProxyService
from imagestore.eucaservice import EucaService
from imagestore.installservice import InstallService
from imagestore.downloadservice import DownloadService
from imagestore.storageservice import StorageService
from imagestore.signatureservice import SignatureService
from imagestore.sqlitestorage import SQLiteStorage


class TooManyArgsError(Exception):
    """Raised when too many arguments are given in the command line."""


def parseOptions(args):
    parser = OptionParser()
    parser.add_option("--debug", action="store_true", default=False,
                      help="Enable debugging options.")
    parser.add_option("--log-file",
                      help="Send logs to the given file rather than stdout.")
    parser.add_option("--api-url",
                      default="https://imagestore.canonical.com/api",
                      help="URL for the upstream image store")
    parser.add_option("--data-dir", default="/var/lib/image-store-proxy",
                      help="Path for the data directory")
    parser.add_option("--eucalyptus-prefix", default="",
                      help="Prefix for Eucalyptus installation, if not in /")
    parser.add_option("--fake-eucalyptus", metavar="SECRET_KEY",
                      help="Use a fake eucalyptus for testing, and use the "
                           "given secret key for checking signatures")
    parser.add_option("--keyring",
                      help="GPG keyring with keys for signature checking "
                           "(defaults to $data_dir/keyring.gpg)")

    options, extraArgs = parser.parse_args(args)
    if extraArgs:
        raise TooManyArgsError("Too many arguments provided in command line.")
    return options


def initializeLogging(debug=False, logFile=None):
    logger = logging.getLogger()
    if debug:
        logger.setLevel(logging.DEBUG)
    else:
        logger.setLevel(logging.INFO)
    if logFile is None:
        handler = logging.StreamHandler(sys.stdout)
    else:
        handler = logging.FileHandler(logFile)
    logger.addHandler(handler)
    format = "%(asctime)s %(levelname)-8s %(message)s"
    handler.setFormatter(logging.Formatter(format))


def buildServiceHub(options):
    logging.debug("Building service hub.")

    serviceHub = ServiceHub()

    databasePath = os.path.join(options.data_dir, "database")
    eucalyptusPath = os.path.join(options.data_dir, "eucalyptus")
    downloadsPath = os.path.join(options.data_dir, "downloads")

    if options.keyring is None:
        keyringPath = os.path.join(options.data_dir, "keyring.gpg")
    else:
        keyringPath = options.keyring

    for path in [options.data_dir, eucalyptusPath, downloadsPath]:
        if not os.path.isdir(path):
            logging.info("Creating directory %s" % (path,))
            try:
                os.makedirs(path)
            except OSError, e:
                message = "Error creating directory %s: %s" % (path, e)
                logging.error(message)
                sys.exit(message)

    proxyService = ProxyService(reactor, endpoint=options.api_url)

    if options.fake_eucalyptus:
        eucaService = EucaService(reactor, eucalyptusPath,
                                  options.eucalyptus_prefix,
                                  fakeSecretKey=options.fake_eucalyptus,
                                  fakeMode=True)
    else:
        eucaService = EucaService(reactor, eucalyptusPath,
                                  options.eucalyptus_prefix)
    downloadService = DownloadService(reactor, downloadsPath)
    storageService = StorageService(reactor, lambda:SQLiteStorage(databasePath))
    signatureService = SignatureService(reactor, keyringPath)
    installService = InstallService(serviceHub)

    serviceHub.addService(proxyService)
    serviceHub.addService(eucaService)
    serviceHub.addService(downloadService)
    serviceHub.addService(storageService)
    serviceHub.addService(signatureService)
    serviceHub.addService(installService)

    logging.debug("Service hub is ready.")

    return serviceHub


def run(args):
    # XXX This needs tests.

    options = parseOptions(args)
    initializeLogging(debug=options.debug, logFile=options.log_file)

    logging.info("Starting up")

    serviceHub = buildServiceHub(options)

    logging.debug("Starting service hub.")
    serviceHub.start()

    presenter = Presenter(serviceHub)
    root = Root(presenter)
    site = Site(root)

    port = 52780
    logging.debug("Trying to listen on port %d." % port)
    reactor.listenTCP(port, site)
    logging.info("Now listening on port %d." % port)

    logging.debug("Running reactor.")
    reactor.run()
