#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import hashlib
import base64
import hmac
import uuid
import time


DEFAULT_TIMEOUT = 60


RFC3986_UNRESERVED = dict.fromkeys("ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                                   "abcdefghijklmnopqrstuvwxyz"
                                   "01234567890-_.~", True)


class Signer(object):

    def __init__(self, secretKey):
        self._secretKey = secretKey

    def check(self, method, host, path, params):
        return self.sign(method, host, path, params,
                         expires=False, nonce=False) == params

    def sign(self, method, host, path, params={}, timeout=None,
             expires=None, nonce=None):
        result = params.copy()
        result.pop("Signature", None)

        for name in result:
            if type(result[name]) is not list:
                result[name] = [result[name]]

        result["SignatureMethod"] = ["HmacSHA256"]
        result["SignatureVersion"] = ["2"]

        if expires is not False:
            now = time.time()
            if expires is None:
                if timeout is None:
                    timeout = DEFAULT_TIMEOUT
                expires = int(now) + timeout
            else:
                expires = int(expires)
            result["Expires"] = [str(expires)]

        if nonce is not False:
            if nonce is None:
                nonce = uuid.uuid4().get_hex()
            result["Nonce"] = [nonce]

        encodedParams = []
        for key, valueList in result.iteritems():
            encodedKey = self._encode(key)
            for value in valueList:
                encodedParams.append((encodedKey, self._encode(value)))
        encodedParams.sort()
        joinedParams = "&".join("%s=%s" % pair for pair in encodedParams)

        payload = "%s\n%s\n%s\n%s" % (method, host, path, joinedParams)
        signature = base64.b64encode(hmac.HMAC(self._secretKey, payload,
                                               hashlib.sha256).digest())

        result["Signature"] = [signature]

        return result

    def _encode(self, value):
        return "".join((c in RFC3986_UNRESERVED and c or ("%%%02X" % ord(c)))
                       for c in value)
