/*
 * Configurable ps-like program.
 * Display device which uses curses for fancy output to the terminal.
 *
 * Copyright (c) 2008 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include <signal.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include <curses.h>

#undef	FALSE
#undef	TRUE

#include "ips.h"


static	BOOL	CursesOpen(DISPLAY *);
static	void	CursesClose(DISPLAY *);
static	void	CursesRefresh(DISPLAY *);
static	void	CursesBeginPage(DISPLAY *);
static	void	CursesPutChar(DISPLAY *, int);
static	void	CursesPutString(DISPLAY *, const char *);
static	void	CursesPutBuffer(DISPLAY *, const char *, int);
static	void	CursesEndPage(DISPLAY *);
static	BOOL	CursesEventWait(DISPLAY *, int);
static	BOOL	CursesInputReady(DISPLAY *);
static	int	CursesReadChar(DISPLAY *);
static	void	CursesRingBell(DISPLAY *);
static	int	CursesGetRows(DISPLAY *);
static	int	CursesGetCols(DISPLAY *);
static	BOOL	CursesDoesScroll(DISPLAY *);


static DISPLAY	cursesDisplay =
{
	CursesOpen, CursesClose, CursesRefresh, CursesBeginPage,
	CursesPutChar, CursesPutString, CursesPutBuffer, CursesEndPage,
	CursesEventWait, CursesInputReady, CursesReadChar, CursesRingBell,
	CursesGetRows, CursesGetCols, CursesDoesScroll
};


/*
 * The main window.
 */
static	WINDOW *	mainWindow;


/*
 * Terminal size data.
 */
static	BOOL	sizeChanged;	/* terminal size has changed */

static  void    HandleResize(int arg);
static  void    GetTerminalSize(void);


/*
 * Return the instance of the curses display device.
 */
DISPLAY *
GetCursesDisplay(void)
{
	return &cursesDisplay;
}


/*
 * Open the display device.
 */
static BOOL
CursesOpen(DISPLAY * display)
{
	SCREEN *	screen;

	screen = newterm(NULL, stdout, stdin);

	if (screen == NULL)
		return FALSE;

	set_term(screen);

	cbreak();
	noecho();

	mainWindow = newwin(0, 0, 0, 0);

	/*
	 * If output is to a terminal, then get its current size and
	 * set up to handle resize signals.
	 */
	if (isatty(STDOUT))
	{
		signal(SIGWINCH, HandleResize);

		GetTerminalSize();
	}

	return TRUE;
}


/*
 * Close the display device.
 */
static void
CursesClose(DISPLAY * display)
{
	refresh();
	endwin();
}


static void
CursesRefresh(DISPLAY * display)
{
	wrefresh(curscr);
}


static void
CursesBeginPage(DISPLAY * display)
{
	wmove(mainWindow, 0, 0);
}


static void
CursesPutChar(DISPLAY * display, int ch)
{
	waddch(mainWindow, ch);
}


static void
CursesPutString(DISPLAY * display, const char * str)
{
	waddstr(mainWindow, str);
}


static void
CursesPutBuffer(DISPLAY * display, const char * str, int len)
{
	while (len-- > 0)
	{
		waddch(mainWindow, *str);
		str++;
	}
}


static void
CursesEndPage(DISPLAY * display)
{
	wclrtobot(mainWindow);
	wmove(mainWindow, 0, 0);
	wrefresh(mainWindow);
}


/*
 * Handle events for the display while waiting for the specified amount
 * of time.  Returns early if there are input characters to be read.
 * Returns TRUE if the window was resized and so needs to be updated soon.
 */
static BOOL
CursesEventWait(DISPLAY * display, int milliSeconds)
{
	struct	timeval	timeOut;
	fd_set		readFds;

	if (milliSeconds <= 0)
		return sizeChanged;

	FD_ZERO(&readFds);
	FD_SET(STDIN, &readFds);

	timeOut.tv_sec = milliSeconds / 1000;
	timeOut.tv_usec = (milliSeconds % 1000) * 1000;

	(void) select(STDIN + 1, &readFds, NULL, NULL, &timeOut);

	return sizeChanged;
}


/*
 * See if input is ready from the terminal.
 */
static BOOL
CursesInputReady(DISPLAY * display)
{
	struct	timeval	timeOut;
	fd_set		readFds;

	FD_ZERO(&readFds);
	FD_SET(STDIN, &readFds);

	timeOut.tv_sec = 0;
	timeOut.tv_usec = 0;

	return (select(STDIN + 1, &readFds, NULL, NULL, &timeOut) > 0);
}


/*
 * Read the next character from the terminal.
 */
static int
CursesReadChar(DISPLAY * display)
{
	char	data;

	if (read(STDIN, &data, 1) < 1)
		return EOF;

	return data & 0xff;
}


static void
CursesRingBell(DISPLAY * display)
{
	fflush(stdout);
	fputc('\007', stderr);
	fflush(stderr);
}


static int
CursesGetRows(DISPLAY * display)
{
	if (sizeChanged)
		GetTerminalSize();

	return LINES;
}


/*
 * Return the number of columns for display.
 * Note: We reduce curses's value by one since it will always
 * auto-line-wrap if the last column is written into, and handling
 * that misfeature for correct output is otherwise painful.
 */
static int
CursesGetCols(DISPLAY * display)
{
	if (sizeChanged)
		GetTerminalSize();

	return COLS - 1;
}


static BOOL
CursesDoesScroll(DISPLAY * display)
{
	return FALSE;
}


/*
 * Signal handler for resizing of window.
 * This only sets a flag so that we can handle the resize later.
 * (Changing the size at unpredictable times would be dangerous.)
 */
static void
HandleResize(int arg)
{
	sizeChanged = TRUE;

	signal(SIGWINCH, HandleResize);
}


/*
 * Routine called to get the new terminal size from the kernel.
 * We inform curses of this change and let it resize its window.
 */
static void
GetTerminalSize(void)
{
	struct	winsize	size;
	int		rows;
	int		cols;

	sizeChanged = FALSE;

	if (ioctl(STDOUT, TIOCGWINSZ, &size) < 0)
		return;

	rows = size.ws_row;
	cols = size.ws_col;

	if (rows <= 0)
		rows = 1;

	if (cols <= 0)
		cols = 1;

	/*
	 * If the values have changed then inform curses.
	 */
	if ((rows != LINES) || (cols != COLS))
		resize_term(rows, cols);
}

/* END CODE */
