/*******************************************************\
* irmp3-ncurses - An ncurses frontend for irmp3 using   *
* the Network Control Module                            *
* (C) 2005 Ross Axe                                     *
*                                                       *
* poll-sigio.c - I/O polling using SIGIO                *
\*******************************************************/


#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include "autosocket.h"

#if HAVE_SYS_IOCTL_H
#  include <sys/ioctl.h>
#endif
#include <signal.h>

#include "irmp3-ncurses.h"

vcid("$Id: poll-sigio.c,v 1.16 2006/01/13 03:49:47 ross Exp $");

#ifndef SIGIO
#define SIGIO SIGPOLL
#endif

static int sigio_sock;


static int refcount = 0;

void block_net(void)
{
    sigset_t sigs;

    if(refcount++ < 0)
	return;

    dbg_printf(2, "Blocking SIGIO");

    if(sigemptyset(&sigs)) {
	sbar_perror("sigemptyset");
	return;
    }
    if(sigaddset(&sigs, SIGIO)) {
	sbar_perror("sigaddset(&sigs, SIGIO)");
	return;
    }
    if(sigprocmask(SIG_BLOCK, &sigs, NULL)) {
	sbar_perror("sigprocmask");
	return;
    }
}

void unblock_net(void)
{
    sigset_t sigs;

    if(--refcount > 0)
	return;

    dbg_printf(2, "Unblocking SIGIO");

    if(sigemptyset(&sigs)) {
	sbar_perror("sigemptyset");
	return;
    }
    if(sigaddset(&sigs, SIGIO)) {
	sbar_perror("sigaddset(&sigs, SIGIO)");
	return;
    }
    if(sigprocmask(SIG_UNBLOCK, &sigs, NULL)) {
	sbar_perror("sigprocmask");
	return;
    }
}


static RETSIGTYPE sigio_handler(int sig)
{
    assert(sig == SIGIO);

    block_all();

    for(;;) {
	fd_set fds;
	struct timeval tv = {0, 0};

	FD_ZERO(&fds);
	FD_SET(sigio_sock, &fds);
	switch(select(sigio_sock + 1, &fds, NULL, NULL, &tv)) {
	case 1:
	    if(!handle_net(sigio_sock)) {
		sbar_printf(_("Server shutdown"));
		unblock_all();
		return (RETSIGTYPE)0;
	    }
	    break;
	case -1:
	    sbar_printf("select: %s", sock_strerror(sock_errno));
	case 0:
	    unblock_all();
	    return (RETSIGTYPE)0;
	default:
	    abort();
	}
    }
}

int main_poll_loop(int sock)
{
    struct sigaction sa;
    int val;

    block_all();

    sigio_sock = sock;

    sa.sa_handler = sigio_handler;
    sa.sa_flags = 0;
    sigemptyset(&sa.sa_mask);

    if(sigaction(SIGIO, &sa, NULL)) {
	sbar_perror("sigaction");
	return 0;
    }

#ifdef SIOCSPGRP
    val = getpid();
    ioctl(sock, SIOCSPGRP, &val);
#endif
    val = true;
    ioctl(sock, FIOASYNC, &val);

    for(;;) {
	if(!key_callback(sock, get_keystroke()))
	    break;
    }

    signal(SIGIO, SIG_IGN);
    unblock_all();
    return 0;
}
