/*******************************************************\
* irmp3-ncurses - An ncurses frontend for irmp3 using   *
* the Network Control Module                            *
* (C) 2003 Ross Axe                                     *
*                                                       *
* scroller.c - Draw scrolling text                      *
\*******************************************************/

#if HAVE_CONFIG_H
#  include "config.h"
#endif

#if HAVE_UNISTD_H
#  include <unistd.h>
#endif
#include <signal.h>

#include "irmp3-ncurses.h"

vcid("$Id: scroller.c,v 1.26 2006/01/13 03:49:47 ross Exp $");


struct scrollerlist {
    struct scrollerlist *next;
    struct scroller *scroller;
} *scrollerlist_head = NULL;

static void dump_scrollerlist(void)
{
    int i;
    struct scrollerlist *scr;

    for(i = 0, scr = scrollerlist_head; scr; i++, scr = scr->next) {
	dbg_printf(2, "msg: %s", scr->scroller->msg);
    }
    dbg_printf(2, "%d items in scrollerlist", i);
}


static int refcount = 0;

void block_scroller(void)
{
    sigset_t sigs;

    if(refcount++ < 0)
	return;

    dbg_printf(2, "Blocking SIGALRM");

    if(sigemptyset(&sigs)) {
	sbar_perror("sigemptyset");
	return;
    }
    if(sigaddset(&sigs, SIGALRM)) {
	sbar_perror("sigaddset(&sigs, SIGALRM)");
	return;
    }
    if(sigprocmask(SIG_BLOCK, &sigs, NULL)) {
	sbar_perror("sigprocmask");
	return;
    }
}

void unblock_scroller(void)
{
    sigset_t sigs;

    if(--refcount > 0)
	return;

    dbg_printf(2, "Unblocking SIGALRM");

    if(sigemptyset(&sigs)) {
	sbar_perror("sigemptyset");
	return;
    }
    if(sigaddset(&sigs, SIGALRM)) {
	sbar_perror("sigaddset(&sigs, SIGALRM)");
	return;
    }
    if(sigprocmask(SIG_UNBLOCK, &sigs, NULL)) {
	sbar_perror("sigprocmask");
	return;
    }
}


/* scroller must be valid by the time this is called */
static void scrollerlist_add(struct scroller *scroller)
{
    struct scrollerlist *scr = malloc(sizeof *scr);

    if(!scr) {
	sbar_printf(_("No memory for scrolling data"));
	return;
    }
    *scr = (struct scrollerlist) {
	.next = scrollerlist_head,
	.scroller = scroller,
    };
    scrollerlist_head = scr;
    dump_scrollerlist();
}

/* scroller must still be valid when this is called */
static void scrollerlist_del(struct scroller *scroller)
{
    struct scrollerlist *scr;

    if(!scrollerlist_head) {
	sbar_printf("scrollerlist is empty");
	return;
    }
    if(scrollerlist_head->scroller == scroller) {
	scr = scrollerlist_head;
	scrollerlist_head = scr->next;
	free(scr);
	dump_scrollerlist();
	return;
    }
    for(scr = scrollerlist_head; scr->next; scr = scr->next) {
	if(scr->next->scroller == scroller) {
	    struct scrollerlist *tmp_scr = scr->next;

	    scr->next = scr->next->next;
	    free(tmp_scr);
	    dump_scrollerlist();
	    return;
	}
    }
    sbar_printf("Tried to delete nonexistant scroller 0x%p", scroller);
}


static RETSIGTYPE sigalarm_handler(int sig)
{
    struct scrollerlist *scr;

    assert(sig == SIGALRM);
    if(!scrollerlist_head)
	return (RETSIGTYPE)0;
    alarm(1);

    block_all();

    for(scr = scrollerlist_head; scr; scr = scr->next) {
	struct scroller *const scroller = scr->scroller;
	unsigned int newpos;
	const unsigned int maxpos = strlen(scroller->msg) - scroller->len;
	int x, y;
	void *hookdata = (scroller->entryhook ? scroller->entryhook(scroller)
			  : NULL);

	assert(scroller->wnd);
	if((scroller->pos + scroller->dir) < 0) {
	    newpos = 0;
	    scroller->dir = -scroller->dir;
	    assert(scroller->dir > 0);
	} else if((unsigned int)(scroller->pos + scroller->dir) > maxpos) {
	    newpos = maxpos;
	    scroller->dir = -scroller->dir;
	    assert(scroller->dir < 0);
	} else
	    newpos = scroller->pos + scroller->dir;

	getyx(*scroller->wnd, y, x);
	mvwprintw(*scroller->wnd, scroller->y, scroller->x, "%.*s",
		  scroller->len, scroller->msg + newpos);
	scroller->pos = newpos;
	wmove(*scroller->wnd, y, x);
	if(scroller->exithook)
	    scroller->exithook(scroller, hookdata);
	wrefresh(*scroller->wnd);
    }

    unblock_all();
    return (RETSIGTYPE)0;
}

void wscroller_hook(WINDOW **wnd, struct scroller *scroller, const char *msg,
		    bool right, void *(*entryhook)(struct scroller *),
		    void (*exithook)(struct scroller *, void *))
{
    unsigned int x, y;
    unsigned int w, h;
    void *hookdata;

    getyx(*wnd, y, x);
    getmaxyx(*wnd, h, w);
    w -= x;
    *scroller = (struct scroller) {
	.wnd = wnd,
	.msg = strdup(msg),
	.x = x,
	.y = y,
	.len = w,
	.pos = right ? strlen(msg) - w : 0,
	.dir = right ? +1 : -1,
	.entryhook = entryhook,
	.exithook = exithook,
    };
    hookdata = entryhook ? entryhook(scroller) : NULL;
    wprintw(*wnd, "%-*.*s", w, w, msg + (scroller->pos > 0
					 ? scroller->pos : 0));
    if(exithook)
	exithook(scroller, hookdata);
    if(w >= strlen(msg)) {
	free((char *)scroller->msg); /* cast is safe since we allocated this
					with strdup() above */
	*scroller = (struct scroller)SCROLLER_INITIALISER;
    } else {
	struct sigaction sa;

	sa.sa_handler = sigalarm_handler;
	sa.sa_flags = 0;
	sigemptyset(&sa.sa_mask);

	block_all();

	scrollerlist_add(scroller);
	if(sigaction(SIGALRM, &sa, NULL)) {
	    sbar_perror("sigaction");
	    unblock_all();
	    return;
	}

	alarm(1);
	unblock_all();
    }
}

void scroller_end(struct scroller *scroller)
{
    if(scroller->msg) {
	scrollerlist_del(scroller);
	free((char *)scroller->msg); /* cast is safe since we allocated this
					with strdup() in wscroller_hook() */
    }
    *scroller = (struct scroller)SCROLLER_INITIALISER;
}

void scroller_endall(void)
{
    signal(SIGALRM, SIG_IGN);
}
