/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_debug.c,v $ -- sample debugging module
 * $Id: mod_debug.c,v 1.10 2004/02/23 20:32:28 boucman Exp $
 *
 * Copyright (C) by Andreas Neuhaus
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

/*
 * This is an example module for IRMP3. It reads stdin and passes all
 * entered text as messages to the other modules. So you can enter
 * commands at the console. It also shows all other messages coming
 * from the other modules.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3mod.h"
#include "mod_debug.h"

/*************************************************************************
 * GLOBALS     
 */

extern int daemonize;	// set by irmp3d.c if we are a daemon
fd_set	mod_debug_fdset;

/*************************************************************************
 * MODULE INFO
 *	This structure is used to pass to mod_register to register
 *	our module with the main program.
 */
mod_t mod_debug = {
	"mod_debug",
	mod_debug_deinit,	// our deinit function
	mod_debug_reload,	// called when got SIGHUP
	&mod_debug_fdset,	// we're watching the stdin
	mod_debug_poll,		// and process the typed text
	NULL,			// we don't need to be called periodically
	mod_debug_message,	// our message handler
	NULL,			// SIGCHLD handler
	mod_debug_init,
	NULL,			// avoid warning
};


/*************************************************************************
 * POLL INPUT DATA
 *	This function will be called whenever data is available at our
 *	watched fd (the standard input). We need to read and process
 *	the new data here.
 */
void mod_debug_poll (int fd)
{
	char s[512];
	readline(fd, s, sizeof(s));
	log_printf(LOG_NOISYDEBUG, "mod_debug_poll(): got input: '%s'\n", s);
	if (s && *s)
		mod_sendmsg(MSGTYPE_INPUT, s);
}


/*************************************************************************
 * RECEIVE MESSAGE
 *	Whenever another module sends out a message via mod_sendmsg(),
 *	this function of all active modules will be called to notify
 *	them about the new message. msgtype shows the type of
 *	the message, msg contains the message.
 */
void mod_debug_message (int msgtype, char *msg,const char __attribute__((unused))*sender)
{
	char *c1,*c2,*c3,*c4,*tmp;
	mod_message_t *answer,*tmp_msg;
	struct stat st;
	// handle input messages
	if (msgtype == MSGTYPE_QUERY) {
		c1 =  strtok(msg, " \t");
		if(!c1) return;
		/* mp3file : deprecated command, replaced by the more general songcmd */
		/* keep backward compatibility here */
		if(!strcasecmp(c1,"mp3file")) {
			answer = mod_query(MSGTYPE_QUERY,"songcmd");
			while(answer) {
				if( answer->msgtype != MSGTYPE_INFO) {
					tmp_msg = answer;
					answer =  answer->next;
					free(tmp_msg->msg);
					free(tmp_msg);
					continue;
				}
				tmp = strdup(answer->msg);
				c1 = strtok(tmp," \t");
				if(!strcasecmp(c1,"songcmd")) {
					c1 = strtok(NULL,"");
					mod_sendmsgf(MSGTYPE_INFO,"file %s",c1);
				} 
				free(tmp);
				tmp_msg = answer;
				answer =  answer->next;
				free(tmp_msg->msg);
				free(tmp_msg);
			}
		} else if (!strcasecmp(c1,"titleguess") || !strcasecmp(c1,"artistguess")){
			// try to get info from filename
			// try format "(artist) title"
			answer = mod_query(MSGTYPE_QUERY,"songcmd");
			if(!answer) return;
			tmp_msg = answer;
			while(tmp_msg && (tmp_msg->msgtype!=MSGTYPE_INFO || strncasecmp(tmp_msg->msg,"songcmd",7))) {
				tmp_msg =tmp_msg->next;
			}
			if(!tmp_msg) {
				free_message(answer);
				return;
			}
			tmp = malloc(strlen(tmp_msg->msg));
			strcpy(tmp,tmp_msg->msg+8); //skip "songcmd "
			free_message(answer);
			// check first if this is a file	
			if(stat(tmp,&st)< 0) {
				free(tmp); 
				return;
			}
			c3 = strrchr(tmp,'/'); //remove trailing directory
			if(c3) strcpy(tmp,c3+1) ;
			c3 = strrchr(tmp,'.'); //remove trailing extension
			if(c3) *c3 = '\0';
				

			c3 = strchr(tmp, '(');
			c4 = strchr(tmp, ')');
			if (c3 && c4 && c3<c4) {
				*c4++ = 0;
				while (*c4==' ') c4++;
				if(!strcasecmp(c1,"titleguess")) {
					mod_sendmsgf(MSGTYPE_INFO, "title %s", c4);
				} else {
					mod_sendmsgf(MSGTYPE_INFO, "artist %s", c3+1);
				}
			} else {
				// try format "artist - title"
				c3 = strchr(tmp, '-');
				if (c3) {
					*c3++ = 0;
					while (*c3==' ') c3++;
					if(!strcasecmp(c1,"titleguess")) {
						mod_sendmsgf(MSGTYPE_INFO, "title %s", c3);
					} else {
						mod_sendmsgf(MSGTYPE_INFO, "artist %s", tmp);
					}
				} else {
					// unable to determine :-(
					if(!strcasecmp(c1,"titleguess")) {
						mod_sendmsgf(MSGTYPE_INFO, "title %s", tmp);
					} else {
						mod_sendmsgf(MSGTYPE_INFO, "artist (unknown artist)");
					}
				}
			}
			free(tmp);
		} else if ( !strcasecmp(c1,"songtypeguess")) {
			c2 =  strtok(NULL, "");
			if(!c2) return;
			if ( strlen(c2)>7 && (!strncasecmp(c2, "http://",7))) {
				mod_sendmsgf(MSGTYPE_INFO,"knowntype http %s",c2);
			} else if (strlen(c2)>6 && (!strncasecmp(c2,"radio ",6))) {
 				mod_sendmsgf(MSGTYPE_INFO,"knowntype radio %s",c2);
			} else if (strlen(c2)>4 && (!strcasecmp(c2+strlen(c2)-4, ".mid"))) {
				mod_sendmsgf(MSGTYPE_INFO,"knowntype midi %s",c2);
			} else if ( strlen(c2)>4 && (!strcasecmp(c2+strlen(c2)-4, ".rcp"))) {
				mod_sendmsgf(MSGTYPE_INFO,"knowntype recompose %s",c2);
			} else if ( strlen(c2)>4 && (!strcasecmp(c2+strlen(c2)-4, ".r36"))) {
				mod_sendmsgf(MSGTYPE_INFO,"knowntype recompose %s",c2);
			} else if ( strlen(c2)>4 && (!strcasecmp(c2+strlen(c2)-4, ".g18"))) {
				mod_sendmsgf(MSGTYPE_INFO,"knowntype recompose %s",c2);
			} else if ( strlen(c2)>4 && (!strcasecmp(c2+strlen(c2)-4, ".g36"))) {
				mod_sendmsgf(MSGTYPE_INFO,"knowntype recompose %s",c2);
			} else if ( strlen(c2)>4 && (!strcasecmp(c2+strlen(c2)-4, ".mod"))) {
				mod_sendmsgf(MSGTYPE_INFO,"knowntype mod %s",c2);
			} else if ( strlen(c2)>4 && (!strcasecmp(c2+strlen(c2)-4, ".kar"))) {
				mod_sendmsgf(MSGTYPE_INFO,"knowntype midi %s",c2);
			} else if ( strlen(c2)>4 && (!strcasecmp(c2+strlen(c2)-4, ".mpg"))) {
				mod_sendmsgf(MSGTYPE_INFO,"knowntype mpg %s",c2);
			} else if ( strlen(c2)>4 && (!strcasecmp(c2+strlen(c2)-4, ".mp2"))) {
				mod_sendmsgf(MSGTYPE_INFO,"knowntype mp2 %s",c2);
			} else if ( strlen(c2)>4 && (!strcasecmp(c2+strlen(c2)-4, ".mp3"))) {
				mod_sendmsgf(MSGTYPE_INFO,"knowntype mp3 %s",c2);
			} else if ( strlen(c2)>4 && (!strcasecmp(c2+strlen(c2)-4, ".ogg"))) {
				mod_sendmsgf(MSGTYPE_INFO,"knowntype ogg %s",c2);
 			} else if ( strlen(c2)>5 && (!strcasecmp(c2+strlen(c2)-5, ".flac"))) {
 				mod_sendmsgf(MSGTYPE_INFO,"knowntype flac %s",c2);
			}
		}
	}
} 


/*************************************************************************
 * MODULE INIT FUNCTION
 *	This is called rigth after starting IRMP3 and should set up global
 *	variables and register your module via mod_register().
 *	It should return NULL if everything went ok, or an error string
 *	with the error description.
 */
char *mod_debug_init (void)
{
	log_printf(LOG_DEBUG, "mod_debug_init(): initializing\n");
	FD_ZERO(&mod_debug_fdset);

	// register our module with the main program so that it knows
	// what we want to handle. We must provide a pointer to a
	// mod_t variable which is valid during the whole runtime, so
	// don't specify a local variable here.
	if (daemonize) {		// don't bother monitoring STDIN
		log_printf(LOG_DEBUG, "mod_debug_init(): in daemon mode... not monitoring stdin.\n");
		mod_debug.poll=NULL;
	} else
		FD_SET(STDIN_FILENO,&mod_debug_fdset);

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 *	This is called right before IRMP3 shuts down. You should clean
 *	up all your used data here to prepare for a clean exit.
 */
void mod_debug_deinit (void)
{
	// nothing to do here

	log_printf(LOG_DEBUG, "mod_debug_deinit(): deinitialized\n");
}


/*************************************************************************
 * MODULE RELOAD FUNCTION
 *	This is called whenever IRMP3 needs to reload all configs (usually
 *	when receiving a SIGHUP). You should reinit your module or check
 *	for changed configuration here so that changes take effect.
 */
char *mod_debug_reload (void)
{
	log_printf(LOG_DEBUG, "mod_debug_reload(): reloading\n");

	// nothing to do here
	return NULL;
}


/*************************************************************************
 * EOF
 */
