#ifndef MIX_SAMPLER_H_
#define MIX_SAMPLER_H_

#include <sampler/Metropolis.h>
#include <graph/StochasticNode.h>

#include <vector>

class Graph;

namespace mix {

/**
 * @short Sampler for normal mixture models
 *
 * MixSampler implements a Metropolis-Hastings sampler for normal
 * mixture models with tempered transitions (Neal, R. Sampling from
 * multimodal distributions using tempered transitions, Statistics and
 * Computing, vol 6, pp 353-355, 1996) as recommended for mixture
 * models by Celeux et al (Celeux G, Hurn, M, Robert CP, Computational
 * and inferential difficulties with mixture posterior distributions,
 * J Am Statist Ass, 95, 957-970, 2000). 
 *
 * A tempered log density at temperature T is derived from the target
 * log density by replacing the contribution L from the log-likelihood
 * by L/T. For large T, the tempered density is flattened, converging
 * to the prior density as T tends to infinity.
 *
 * In a tempered transition, a new Metropolis-Hasting proposal is
 * generated by a sequence of M-H updates that are stationary with
 * respect to a tempered density.  In this sequence, T starts close to
 * 1 increases to a maximum (e.g. 1000) and then decreases to 1
 * again. Tempering allows the proposal to jump to a new local mode of
 * the target density by tunnelling, at high temperature, through a
 * region of low density.  In a mixture model, this tunnelling
 * manifests itself as label switching of the components of the
 * mixture.
 *
 * Three elements are essential for successful tempered updates. The
 * first is a sufficiently high maximum temperature. Without this,
 * tunnelling can never occur between well-separated modes of the
 * target density. The second element is a sufficiently small
 * temperature increment between levels.  If the increase is too
 * large, then the proposal generated by the tempered updates has a
 * low acceptance probability. The third element is a sufficient
 * number of replicates at each temperature level.  In a MixSampler
 * the Metropolis-Hastings updates at each level are tuned to have an
 * acceptance probability between 0.15 and 0.35.  Several replicate
 * updates are therefore required to guarantee that a M-H jump takes
 * place. Without this, the proposal risks getting stuck in a region
 * of low probability. These elements are controlled by the parameters
 * max_level, delta, and nrep respectively.
 */
    class MixSampler : public Metropolis
    {
	const unsigned int _max_level;
	const double _delta; 
	const unsigned int _nrep;
	unsigned int _level;
	double *_lower;
	double *_upper;
	unsigned int _n;
	std::vector<double> _lstep;
	std::vector<double> _pmean;
	bool _temper;
    public:
	/**
	 * Constructor for Mix Sampler with tempered transitions
	 *
	 * @param max_level Maximum number of levels 
	 *
	 * @param delta Increment in the log temperature at each
	 * level. The maximum temperature reached is thus exp(nlevel *
	 * delta).
	 *
	 * @param nrep Number of Metropolis-Hastings updates to do at each level
	 */
	MixSampler(std::vector<StochasticNode*> const &nodes,
		   unsigned int max_level=50, double delta = 0.10, 
		   unsigned int nrep = 4);
	~MixSampler();
	/**
	 * At each iteration, MixSampler alternates nrep standard
	 * Metropolis-Hastings updates, and a single tempered
	 * transition.
	 */
	void update(RNG *rng);
	void rescale(double prob);
	void transform(double const *v, unsigned int length,
		       double *nv, unsigned int nlength) const;
	void untransform(double const *nv, unsigned int nlength,
			 double *v, unsigned int length) const;
	static bool canSample(std::vector<StochasticNode *> const &snodes, 
			      Graph const &graph);
	bool checkAdaptation() const;
	std::string name() const;
    };

}

#endif /* MIX_SAMPLER_H_ */
