#ifndef K3DSDK_GL_H
#define K3DSDK_GL_H

// K-3D
// Copyright (c) 1995-2006, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#if defined K3D_PLATFORM_WIN32

	#include <windows.h>
	#undef min
	#undef max
	#undef interface

#else // K3D_PLATFORM_WIN32

 	#include <GL/glx.h>

#endif // !K3D_PLATFORM_WIN32

#include <GL/gl.h>
#include <GL/glu.h>

#include "algebra.h"
#include "bitmap.h"
#include "color.h"
#include "data.h"
#include "i18n.h"
#include "igl.h"
//#include "property_collection.h"

namespace k3d
{

class bounding_box3;
class idocument;
class plane;

namespace gl
{

/// Converts an OpenGL matrix into a standard K-3D matrix
inline const matrix4 matrix(GLdouble* GLMatrix)
{
	matrix4 result;
	memcpy(&result[0][0], &GLMatrix[0], 16 * sizeof(GLdouble));
	return transpose(result);
}

/// Pushes a matrix onto the OpenGL matrix stack
inline void push_matrix(const matrix4& Matrix)
{
	double glmatrix[16];
	transpose(Matrix).CopyArray(glmatrix);
	glMultMatrixd(glmatrix);
}

/// Passes a k3d::point3 to glVertex3d()
inline void vertex3d(const point3& Vertex)
{
	glVertex3d(Vertex.n[0], Vertex.n[1], Vertex.n[2]);
}

/// Passes a k3d::vector3 to glNormal3d()
inline void normal3d(const vector3& Vector)
{
	glNormal3d(Vector.n[0], Vector.n[1], Vector.n[2]);
}

/// Passes a k3d::normal3 to glNormal3d()
inline void normal3d(const normal3& Vector)
{
	glNormal3d(Vector.n[0], Vector.n[1], Vector.n[2]);
}

/// Passes a k3d::color to glColor3d()
inline void color3d(const color& Color)
{
	glColor3d(Color.red, Color.green, Color.blue);
}

/// Passes a k3d::color to glMaterialfv()
inline void material(GLenum Face, GLenum PName, const color& Color)
{
	GLfloat color[] = { Color.red, Color.green, Color.blue, 1.0 };
	glMaterialfv(Face, PName, color);
}

/// Provides exception- and return-safe RAII behavior for saving / restoring OpenGL attributes
struct store_attributes
{
	store_attributes(const GLbitfield Mask = GL_ALL_ATTRIB_BITS)
	{
		glPushAttrib(Mask);
	}

	~store_attributes()
	{
		glPopAttrib();
	}
};

/// Convenience function for refreshing all OpenGL render engines
void redraw_all(idocument& Document, const irender_engine::redraw_type_t RedrawType);
/// Convenience function for setting-up materials
void setup_material(iunknown* const Material);
/// Draws a 1x1 plane
void draw(const plane& Plane);
/// Draws a box
void draw(const bounding_box3& Box);
/// Draws a standard bounding box to make it easier to visualize an object
void draw_bounding_box(const bounding_box3& Box);

/// Returns true iff the requested OpenGL extension is supported
const bool query_extension(const std::string& Extension);
/// Passes a k3d::bitmap to glTexImage2D(), handling non-power-of-two sizes and translations between image formats
void tex_image_2d(const bitmap& Bitmap);

/**	\brief Provides a boilerplate implementation of k3d::gl::idrawable
	\param base_t Must derive from k3d::transformable
*/
template<typename base_t>
class drawable :
	public base_t,
	public idrawable
{
public:
	drawable(idocument& Document) :
		base_t(Document),
		m_visible(init_owner(*this) + init_name("viewport_visible") + init_label(_("Viewport Visible")) + init_description(_("Controls whether this node will be visibile in the viewport.")) + init_value(true)),
		m_nurbs_renderer(0)
	{
		m_visible.changed_signal().connect(make_async_redraw_slot());
	}

	~drawable()
	{
		if(m_nurbs_renderer)
			gluDeleteNurbsRenderer(m_nurbs_renderer);
	}

	void gl_draw(const render_state& State)
	{
		if(!m_visible.value())
			return;

		store_attributes attributes;

		push_draw();
		on_gl_draw(State);
		pop_draw();
	}

	void gl_select(const render_state& State, const select_state& SelectState)
	{
		if(!m_visible.value())
			return;

		store_attributes attributes;

		push_draw();
		on_gl_select(State, SelectState);
		pop_draw();
	}

protected:
	sigc::slot1<void, iunknown*> make_async_redraw_slot()
	{
		return sigc::mem_fun(*this, &drawable<base_t>::async_redraw);
	}

	void async_redraw(iunknown*)
	{
		k3d::gl::redraw_all(base_t::document(), irender_engine::ASYNCHRONOUS);
	}

	typedef GLUnurbsObj* nurbs_renderer_t;
	nurbs_renderer_t nurbs_renderer(const render_state& State)
	{
		if(!m_nurbs_renderer)
		{
			m_nurbs_renderer = gluNewNurbsRenderer();

			// Important!  We load our own matrices for efficiency (saves round-trips to the server) and to prevent problems with selection
			gluNurbsProperty(m_nurbs_renderer, GLU_AUTO_LOAD_MATRIX, GL_FALSE);
			gluNurbsProperty(m_nurbs_renderer, GLU_CULLING, GL_TRUE);
		}

		GLfloat gl_modelview_matrix[16];
		glGetFloatv(GL_MODELVIEW_MATRIX, gl_modelview_matrix);
		gluLoadSamplingMatrices(m_nurbs_renderer, gl_modelview_matrix, State.gl_projection_matrix, State.gl_viewport);

		return m_nurbs_renderer;
	}

protected:
	/// Set to true iff the object should be visible in OpenGL viewports
	k3d_data(bool, data::immutable_name, data::change_signal, data::with_undo, data::local_storage, data::no_constraint, data::writable_property, data::with_serialization) m_visible;

private:
	void push_draw()
	{
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		push_matrix(base_t::matrix());
	}

	void pop_draw()
	{
		glMatrixMode(GL_MODELVIEW);
		glPopMatrix();
	}

	virtual void on_gl_draw(const render_state& State) = 0;
	virtual void on_gl_select(const render_state& State, const select_state& SelectState) = 0;

	/// OpenGL NURBS renderer
	nurbs_renderer_t m_nurbs_renderer;
};

} // namespace gl

} // namespace k3d

#endif // K3DSDK_GL_H

