/***************************************************************************
 *   Copyright (C) 2004 by Stefano Salvador                                *
 *   salva_ste@tin.it                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
      
#include <qfile.h> 

#include <kgenericfactory.h>
#include <kdebug.h>

#include <config.h>
#include <katalog.h>

#include "kfile_katalog.h"

#define DIR "inode/katalog-directory"
#define KAT "application/x-katalog"
#define ITEM "application/x-katalogitem"

typedef KGenericFactory<katalogPlugin> katalogFactory;

K_EXPORT_COMPONENT_FACTORY(kfile_katalog, katalogFactory( "kfile_katalog" ));

katalogPlugin::katalogPlugin(QObject *parent, const char *name, const QStringList &args)
  : KFilePlugin(parent, name, args)
{
  client = new DCOPClient();
  client->attach();
  appId = client->registerAs("kfile_katalog");
  
  m_info_dir = addMimeTypeInfo( DIR );
  m_info_kat = addMimeTypeInfo( KAT );
  m_info_item = addMimeTypeInfo( ITEM );
}
       

bool katalogPlugin::readInfo( KFileMetaInfo& infoMeta, uint /*what*/)
{
  QDomElement *data = retriveData(infoMeta.url());
  
  if( !data )
    return false;
  
  KFileMimeTypeInfo *infoMime = 0;
  QString mime = infoMeta.mimeType();
  if(mime.compare(DIR) == 0)
    infoMime = m_info_dir;
  else if(mime.compare(KAT) == 0)
    infoMime = m_info_kat;
  else if(mime.compare(ITEM) == 0)
    infoMime = m_info_item;
  
  if(infoMime)
    return readInfo(infoMime, infoMeta, data);
  
  return false;
}

bool katalogPlugin::readInfo(KFileMimeTypeInfo *infoMime, KFileMetaInfo& infoMeta, QDomElement * data)
{
  QDomNode groupNode = data->firstChild();
  
  while ( !groupNode.isNull() )
  {
    if(!groupNode.isElement() || groupNode.nodeName() != "GROUP")
    {
      groupNode = groupNode.nextSibling();
      continue;
    }
    
    QString groupName = groupNode.toElement().attribute("name");
    
    KFileMimeTypeInfo::GroupInfo* groupMime = addGroupInfo(infoMime, groupName, groupName);
    KFileMetaInfoGroup groupMeta = appendGroup(infoMeta, groupName);    
    
    QDomNode itemNode = groupNode.firstChild();
    
    while ( !itemNode.isNull() )
    {
      if(!itemNode.isElement() || itemNode.nodeName() != "ITEM")
      {
        itemNode = itemNode.nextSibling();
        continue;
      }
      
      QDomElement itemEl = itemNode.toElement();
      
      QString itemKey = itemEl.attribute("key");
      
      KFileMimeTypeInfo::ItemInfo* itemMime = 0L;
      
      if(itemEl.hasAttribute("int_value"))
      {
        itemMime = addItemInfo(groupMime, itemKey, itemKey, QVariant::Int);
        appendItem(groupMeta, itemKey, QVariant(itemEl.attribute("int_value").toInt()));
      }
      else if(itemEl.hasAttribute("bool_value"))
      {
        itemMime = addItemInfo(groupMime, itemKey, itemKey, QVariant::Bool);
        appendItem(groupMeta, itemKey, QVariant( itemEl.attribute("bool_value").toInt() != 0 ));
      }
      else if(itemEl.hasAttribute("date_value"))
      {
        QDate d = QDate::fromString(itemEl.attribute("date_value"));
        itemMime = addItemInfo(groupMime, itemKey, itemKey, QVariant::Date);
        appendItem(groupMeta, itemKey, d);
      }
      else if(itemEl.hasAttribute("datetime_value"))
      {
        QDateTime d;
        d.setTime_t(itemEl.attribute("datetime_value").toInt());
        itemMime = addItemInfo(groupMime, itemKey, itemKey, QVariant::DateTime);
        appendItem(groupMeta, itemKey, d);
      }
      else if(itemEl.hasAttribute("double_value"))
      {
        itemMime = addItemInfo(groupMime, itemKey, itemKey, QVariant::Double);
        appendItem(groupMeta, itemKey, QVariant(itemEl.attribute("double_value").toDouble()));
      }
      else if(itemEl.hasAttribute("long_value"))
      {
        itemMime = addItemInfo(groupMime, itemKey, itemKey, QVariant::LongLong);
        appendItem(groupMeta, itemKey, QVariant(itemEl.attribute("long_value").toLongLong()) );
      }
      else if(itemEl.hasAttribute("string_value"))
      {
        itemMime = addItemInfo(groupMime, itemKey, itemKey, QVariant::String);
        appendItem(groupMeta, itemKey, QVariant(itemEl.attribute("string_value")) );
      }
      if(itemMime && itemEl.hasAttribute("unit"))
        setUnit(itemMime, itemEl.attribute("unit").toInt());
      if(itemMime && itemEl.hasAttribute("hint"))
        setHint(itemMime, itemEl.attribute("hint").toInt());
      if(itemMime && itemEl.hasAttribute("attributes"))
        setAttributes(itemMime, itemEl.attribute("attributes").toInt());

      itemNode = itemNode.nextSibling();
    }
    
    groupNode = groupNode.nextSibling();
  }
  return true;
}

QDomElement *katalogPlugin::retriveData(KURL url)
{
  QString path;
  if ( !checkNewFile( url, path ) )
    return 0L;
  
  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);
  arg << *m_katalogURL << path ;
  if (!client->call("katalogdcop", "katalogdcopInterface", "readInfo(KURL, QString)",
            data, replyType, replyData))
  {
    return 0L;
  }
  else {
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "QString")
    {
      QString result;
      reply >> result;
      if(!result)
      {
        return 0L;
      }
      
      int Line;
      int Col;
      QString str;
      QDomDocument doc;
      
      if( !doc.setContent( result, &str, &Line, &Col ) )
        return 0L;

      QDomNode node = doc.firstChild();
      if( node.isNull()  || !node.isElement() )
        return 0L;
      
      QDomElement *el = new QDomElement( node.toElement() );
      return el ;  
    }
    else
    {
      return 0L;
    }
  }
}

bool katalogPlugin::checkNewFile( const KURL & url, QString & path )
{
  wakeDCOP();
  
  QString fullPath = url.path();
  
  // Find where the katalog file is in the full path
  int pos = 0;
  QString katalogFile;
  path = QString::null;
  
  int len = fullPath.length();
  if ( len != 0 && fullPath[ len - 1 ] != '/' )
    fullPath += '/';

  while ( (pos=fullPath.find( '/', pos+1 )) != -1 )
  {
    QString tryPath = fullPath.left( pos );
    struct stat statbuf;
    if ( ::stat( QFile::encodeName(tryPath), &statbuf ) == 0 && !S_ISDIR(statbuf.st_mode) )
    {
      katalogFile = tryPath;
      m_mtime = statbuf.st_mtime;
      path = fullPath.mid( pos + 1 );
      len = path.length();
      if ( len > 1 )
      {
        if ( path[ len - 1 ] == '/' )
          path.truncate( len - 1 );
      }
      else
        path = QString::fromLatin1("/");
      break;
    }
  }
  
  if ( katalogFile.isEmpty() )
    return false;

  // Open new file
  m_katalogURL = new KURL();
  m_katalogURL->setProtocol("file");
  m_katalogURL->setPath(katalogFile);
  
  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);
  arg << *m_katalogURL;
  if (!client->call("katalogdcop", "katalogdcopInterface", "initDocument(KURL)",
            data, replyType, replyData))
  {
    kdDebug() << "there was some error using DCOP." << endl;
    return false;
  }
  else {
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "int") {
      int result;
      reply >> result;
      kdDebug() << "the result is: " << result << endl;
      if(result == 0)
        return true;
    } else
      return false;
  }
  
  return false;
}

void katalogPlugin::wakeDCOP()
{
  if(client->isApplicationRegistered("katalogdcop"))
  {
    return; 
  }
  
  QString app("katalogdcop.desktop");
  
  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);
  QStringList URLs;
  arg << app << URLs;
  client->call( "klauncher", "klauncher", "start_service_by_desktop_path(QString,QStringList)",  
        data, replyType, replyData);
}
