/***************************************************************************

   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>
   Copyright (C) 2008 Christian Weilbach <christian_weilbach@web.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "backend/blogjob.h"
#include "backend/blogjob_p.h"

#include <QMetaType>

#include <kurl.h>
#include <kdebug.h>
#include <kio/global.h>
#include <kblog/blogger1.h>
#include <kblog/metaweblog.h>
#include <kblog/gdata.h>
#include <kblog/movabletype.h>
#include <kblog/wordpressbuggy.h>
#include <kblog/blogpost.h>
#include <kblog/blogmedia.h>
#include <klocalizedstring.h>
#include <kmessagebox.h>

#include "backend/bloglist.h"

#define APPLICATION_NAME "KBlogger"
#define APPLICATION_VER "1.0-alpha3"

Q_DECLARE_METATYPE(KBlog::BlogPost*)
Q_DECLARE_METATYPE(KBlog::BlogMedia*)
namespace KBlogger
{

//TODO check for more member functions for BlogAPI and BlogAction
BlogJob::BlogJob( BlogAPI api, KUrl& url,
                          BlogAction action,
                          QVariant& arg,
                          QObject* parent ): KJob ( parent )
{
    kDebug();
    d_ptr = new BlogJobPrivate();
    d_ptr->q_ptr = this;
    d_ptr->mAction = action;
    d_ptr->mArg = arg;
    d_ptr->mKBlog = 0;
    d_ptr->mUrl = url;
    d_ptr->mApi = api;
}

BlogJob::~BlogJob()
{
    kDebug();
    delete d_ptr;
}

BlogJob::BlogAPI BlogJob::api()
{
    kDebug();
    return d_ptr->mApi;
}

BlogJob::BlogAction BlogJob::action()
{
    kDebug();
    return d_ptr->mAction;
}

void BlogJob::setUsername(const QString& username)
{
    kDebug();
    d_ptr->mUsername = username;
}

void BlogJob::setPassword(const QString& password)
{
    kDebug();
    d_ptr->mPassword = password;
}

void BlogJob::setBlogId(const QString& blogId)
{
    kDebug();
    d_ptr->mBlogId = blogId;
}

void BlogJob::start ()
{
    kDebug();

    d_ptr->createKBlog(d_ptr->mApi);

    connect(d_ptr->mKBlog, SIGNAL( error( KBlog::Blog::ErrorType, const QString& ) ),
            this, SLOT( slotExitError( KBlog::Blog::ErrorType, const QString&) )  );

    connect(d_ptr->mKBlog, SIGNAL( errorPost(KBlog::Blog::ErrorType,
                                      const QString &,
                                      KBlog::BlogPost* )),
            this, SLOT( slotExitError( KBlog::Blog::ErrorType, const QString&) )  );

    connect(d_ptr->mKBlog, SIGNAL( errorMedia(KBlog::Blog::ErrorType,
                                       const QString &,
                                       KBlog::BlogMedia* )),
            this, SLOT( slotExitError( KBlog::Blog::ErrorType, const QString&) )  );

    switch (d_ptr->mAction) {
    case LIST_BLOGS: {
        // NOTE: not all apis, e.g. GData, require authentication, still we use a sane default
        if ( !d_ptr->setKBlogAuthentication() ){
            kError() << "You need to authenticate for this task.";
	    d_ptr->slotExitError( KBlog::Blog::AuthenticationError, "You need to authenticate for this task.");
            return;
        }

        // NOTE: Currently both Blogger1(as a base to all derived classes) and GData support this call
        connect(d_ptr->mKBlog, SIGNAL( listedBlogs( const QList<QMap<QString,QString> >& ) ),
                this, SLOT( slotListBlogs( const QList<QMap<QString,QString> >& ) ) );
        if(qobject_cast<KBlog::Blogger1*>(d_ptr->mKBlog)){
            KBlog::Blogger1* blogger1 = qobject_cast<KBlog::Blogger1*>(d_ptr->mKBlog);
            blogger1->listBlogs();
        }
        else if(qobject_cast<KBlog::GData*>(d_ptr->mKBlog)){ // TODO we need the profile id first
            KBlog::GData* gdata = qobject_cast<KBlog::GData*>(d_ptr->mKBlog);
            gdata->listBlogs();
        }
        else {
            kError() << "Could not convert mKBlog to the fitting API for a blog listing.";
        }
    }
    case LIST_RECENT_POSTS: {
        int postsToDownload;

        if ( ! d_ptr->mArg.canConvert<int>() ) {
            kError() << "Impossible convert QVariant to int.";
            return;
        }

        // NOTE: not all apis, e.g. GData, require authentication, still we use a sane default
        if ( !d_ptr->setKBlogAuthentication() ){
            kError() << "You need to authenticate for this task.";
            d_ptr->slotExitError( KBlog::Blog::AuthenticationError, "You need to authenticate for this task.");
            return;
        }

        postsToDownload = d_ptr->mArg.toInt();
        connect(d_ptr->mKBlog, SIGNAL( listedRecentPosts( const QList<KBlog::BlogPost>& ) ),
                this, SLOT( slotListRecentPosts( const QList<KBlog::BlogPost>& ) ) );
        kDebug() << "Running " << d_ptr->mKBlog
        << "->listRecentPosts(" << postsToDownload << ")" << endl;
        d_ptr->mKBlog->listRecentPosts( postsToDownload );
    }
    break;

    case FETCH_POST: {
        KBlog::BlogPost *post;

        if ( ! d_ptr->mArg.canConvert<KBlog::BlogPost *>() ) {
            kError() << "Impossible convert QVariant to BlogPost";
            return;
        }

        // NOTE: not all apis, e.g. GData, require authentication, still we use a sane default
        if ( !d_ptr->setKBlogAuthentication() ){
            kError() << "You need to authenticate for this task.";
            d_ptr->slotExitError( KBlog::Blog::AuthenticationError, "You need to authenticate for this task.");
            return;
        }

        post = d_ptr->mArg.value<KBlog::BlogPost*>();
        connect(d_ptr->mKBlog, SIGNAL( fetchedPost( KBlog::BlogPost *) ),
                this, SLOT( slotFetchPost( KBlog::BlogPost *)) );
        kDebug() << "Running " << d_ptr->mKBlog << "->fetchPost()" << endl;
        d_ptr->mKBlog->fetchPost(post);
    }
    break;

    case MODIFY_POST: {
        KBlog::BlogPost *post;

        if ( ! d_ptr->mArg.canConvert<KBlog::BlogPost *>() ) {
            kError() << "Impossible convert QVariant to BlogPost";
            return;
        }

        if ( !d_ptr->setKBlogAuthentication() ){
            kError() << "You need to authenticate for this task.";
	    d_ptr->slotExitError( KBlog::Blog::AuthenticationError, "You need to authenticate for this task.");
            return;
        }

        post = d_ptr->mArg.value<KBlog::BlogPost*>();
        connect(d_ptr->mKBlog, SIGNAL( modifiedPost( KBlog::BlogPost *) ),
                this, SLOT( slotModifyPost( KBlog::BlogPost *) ));
        kDebug() << "Running " << d_ptr->mKBlog << "->modifyPost()" << endl;
        d_ptr->mKBlog->modifyPost( post );
    }
    break;

    case CREATE_POST: {
        KBlog::BlogPost *post;

        if ( ! d_ptr->mArg.canConvert<KBlog::BlogPost *>() ) {
            kError() << "Impossible convert QVariant to BlogPost";
            return;
        }
        if ( !d_ptr->setKBlogAuthentication() ){
            kError() << "You need to authenticate for this task.";
	    d_ptr->slotExitError( KBlog::Blog::AuthenticationError, "You need to authenticate for this task.");
            return;
        }

        post = d_ptr->mArg.value<KBlog::BlogPost*>();

        post = d_ptr->mArg.value<KBlog::BlogPost*>();
        connect(d_ptr->mKBlog, SIGNAL( createdPost( KBlog::BlogPost *) ),
                this, SLOT( slotCreatePost( KBlog::BlogPost *) ));
        kDebug() << "Running " << d_ptr->mKBlog << "->createPost()" << endl;
        kDebug() << "Post DateTime" << post->creationDateTime().dateTime().toString() << endl;
        d_ptr->mKBlog->createPost( post );
    }
    break;

    case REMOVE_POST: {
        KBlog::BlogPost *post;

        if ( ! d_ptr->mArg.canConvert<KBlog::BlogPost *>() ) {
            kError() << "Impossible convert QVariant to BlogPost";
            return;
        }

        if ( !d_ptr->setKBlogAuthentication() ){
            kError() << "You need to authenticate for this task.";
	    d_ptr->slotExitError( KBlog::Blog::AuthenticationError, "You need to authenticate for this task.");
            return;
        }

        post = d_ptr->mArg.value<KBlog::BlogPost*>();
        connect(d_ptr->mKBlog, SIGNAL( removedPost( KBlog::BlogPost *) ),
                this, SLOT( slotRemovePost( KBlog::BlogPost *) ));
        kDebug() << "Running " << d_ptr->mKBlog << "->removedPost()" << endl;
        d_ptr->mKBlog->removePost( post );
    }
    break;

    case LIST_CATEGORIES: {
        // NOTE we have to add casts for other KBlog types with category support here, e.g. Atom API when it comes out
        // we can assume MetaWeblog for now, since all others in the Blogger1 hierarchy use it infact
        KBlog::MetaWeblog* mMetaWeblog = qobject_cast<KBlog::MetaWeblog*>(d_ptr->mKBlog);

        if ( !d_ptr->setKBlogAuthentication() ){
            kError() << "You need to authenticate for this task.";
	    d_ptr->slotExitError( KBlog::Blog::AuthenticationError, "You need to authenticate for this task.");
            return;
        }

        connect(mMetaWeblog, SIGNAL( listedCategories( const QList<QMap<QString, QString> >&  )),
                this, SLOT( slotListCategories( const QList<QMap<QString, QString> >&  ) ));
        kDebug() << "Running " << mMetaWeblog << "->listCategories" << endl;
        mMetaWeblog->listCategories();
    }
    break;

    case CREATE_MEDIA: {
        // NOTE we have to add casts for other KBlog types with media support here, e.g. Atom API when it comes out
        // we can assume MetaWeblog for now, since all others in the Blogger1 hierarchy use it infact
        KBlog::MetaWeblog* mMetaWeblog = qobject_cast<KBlog::MetaWeblog*>(d_ptr->mKBlog); 
        KBlog::BlogMedia *media;

        if ( ! d_ptr->mArg.canConvert<KBlog::BlogMedia *>() ) {
            kError() << "Impossible convert QVariant to BlogMedia";
            return;
        }

        if ( !d_ptr->setKBlogAuthentication() ){
            kError() << "You need to authenticate for this task.";
	    d_ptr->slotExitError( KBlog::Blog::AuthenticationError, "You need to authenticate for this task.");
            return;
        }

        media = d_ptr->mArg.value<KBlog::BlogMedia*>();
        connect(mMetaWeblog, SIGNAL( createdMedia( KBlog::BlogMedia* ) ),
                this, SLOT( slotCreateMedia( KBlog::BlogMedia* ) ));
        kDebug() << "Running " << mMetaWeblog << "->createMedia" << endl;
        mMetaWeblog->createMedia(media);
    }
    break;

    }
}



BlogJobPrivate::BlogJobPrivate():
    q_ptr(0),mKBlog(0)
{
    kDebug();
}

BlogJobPrivate::~BlogJobPrivate()
{
    kDebug();
    delete mKBlog;
    mKBlog = 0;
    q_ptr = 0;
}

void BlogJobPrivate::createKBlog(BlogJob::BlogAPI api)
{
    kDebug();

    delete mKBlog;
    if( api == BlogJob::BLOGGER1 )
        mKBlog = new KBlog::Blogger1( mUrl );
    if( api == BlogJob::METAWEBLOG )
        mKBlog = new KBlog::MetaWeblog( mUrl );
    if( api == BlogJob::MOVABLETYPE )
        mKBlog = new KBlog::MovableType( mUrl );
    if( api == BlogJob::WORDPRESSBUGGY )
        mKBlog = new KBlog::WordpressBuggy( mUrl );
    if( api == BlogJob::GDATA )
        mKBlog = new KBlog::GData( mUrl );

     kDebug() << "Backend created with Url: " << mKBlog->url();

}

bool BlogJobPrivate::setKBlogAuthentication()
{
    kDebug();

    if( mBlogId.isEmpty() || mUsername.isEmpty() || mPassword.isEmpty() )
        return false;

    mKBlog->setBlogId ( mBlogId );
    mKBlog->setUsername ( mUsername );
    mKBlog->setPassword ( mPassword );
    kDebug() << "Authentication setup: "
     << "BlogID=" << mKBlog->blogId() << endl
     << "Username=" << mKBlog->username() << endl
     << "Password=" << mKBlog->password() << endl;
    return true;
}

void BlogJobPrivate::slotExitSuccess()
{
    kDebug();
    q_ptr->setError(0);
    q_ptr->emitResult();
}

void BlogJobPrivate::slotExitError( KBlog::Blog::ErrorType type, const QString& message )
{
    kDebug();
    int t = static_cast<int>(type) + 1; //HACK 0 is an error code in ErrorType as well
    q_ptr->setError(t);
    q_ptr->setErrorText( message );
    q_ptr->emitResult();
}

void  BlogJobPrivate::slotListBlogs( const QList<QMap<QString,QString> >& list)
{
    kDebug();
    slotExitSuccess();
}


void  BlogJobPrivate::slotListRecentPosts( const QList<KBlog::BlogPost>& posts)
{
    kDebug();
    emit q_ptr->listedRecentPosts( posts );
    slotExitSuccess();
}

void BlogJobPrivate::slotListCategories( const QList< QMap<QString, QString> >& categories )
{
    kDebug();
    emit q_ptr->listedCategories( categories );
    slotExitSuccess();
}

void BlogJobPrivate::slotCreatePost( KBlog::BlogPost *post)
{
    kDebug();
    slotExitSuccess();
}

void BlogJobPrivate::slotRemovePost( KBlog::BlogPost * post)
{
    kDebug();
    slotExitSuccess();
}

void BlogJobPrivate::slotFetchPost( KBlog::BlogPost *post)
{
    kDebug();
    slotExitSuccess();
}

void BlogJobPrivate::slotModifyPost( KBlog::BlogPost *post)
{
    kDebug();
    slotExitSuccess();
}

void BlogJobPrivate::slotCreateMedia( KBlog::BlogMedia* media)
{
    kDebug();
    slotExitSuccess();
}

}

#include "blogjob.moc"
