/***************************************************************************

   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "backend/blogjobqueue.h"

#include <kdebug.h>
#include <kjob.h>
#include <klocale.h>
#include <kblog/blog.h>

#include "backend/blogjob.h"

namespace KBlogger
{

BlogJobQueue::BlogJobQueue(QObject *parent): QObject(parent),
    mIsRunning(false), mIsStopped(false)
{
    kDebug();
}

BlogJobQueue::~BlogJobQueue()
{
    kDebug();
}

bool BlogJobQueue::addJob( BlogJob* job )
{
    kDebug();

    if (mIsRunning) {
        kError() << "This job cannot be queued while other jobs are running" << endl;
        return false;
    }

    mPendingJobs << job;
    return true;
}

void BlogJobQueue::run()
{
    kDebug();

    if (mIsRunning) {
        kError() << "The job queue runs already." << endl;
        emit error( i18n("The job queue runs already.") );
        return;
    }

    emit jobsStarted(mPendingJobs.count());
    mIsRunning = true;
    runsFirstJob();
}

void BlogJobQueue::stop()
{
    kDebug();
    mIsStopped = true;
}

void BlogJobQueue::clear()
{
    kDebug();
    mPendingJobs.clear();
    mIsRunning = false;
    mIsStopped = false;
}

bool BlogJobQueue::isRunning()
{
    kDebug();
    return mIsRunning;
}

int BlogJobQueue::queueSize()
{
    kDebug();
    return mPendingJobs.size();
}

void BlogJobQueue::runsFirstJob()
{
    kDebug();

    if( mIsStopped ){ // breaks the loop
        mIsStopped = false;
        return;
    }

    if (!mPendingJobs.isEmpty()) {
        mIsRunning = true;
        BlogJob* currentJob = mPendingJobs.takeFirst();
        Q_ASSERT(currentJob);

        QString msg;
        switch( currentJob->action() ){
            case BlogJob::LIST_RECENT_POSTS:
               msg = i18n("Listing recents posts.");
            break;
            case BlogJob::LIST_CATEGORIES:
               msg = i18n("Listing Categories.");
            break;
            case BlogJob::CREATE_POST:
               msg = i18n("Creating post.");
            break;
            case BlogJob::FETCH_POST:
               msg = i18n("Fetching post.");
            break;
            case BlogJob::MODIFY_POST:
               msg = i18n("Modifying post.");
            break;
            case BlogJob::REMOVE_POST:
               msg = i18n("Removing post.");
            break;
            case BlogJob::CREATE_MEDIA:
               msg = i18n("Creating media.");
            break;
        };

        emit waitMessage( msg );
        emit statusMessage( i18np("1 job left", "%1 jobs left", mPendingJobs.count()) );

        connect(currentJob, SIGNAL( result( KJob * ) ),
                this, SLOT( slotResult( KJob * ) ) );
        currentJob->start();
    } else {
        mIsRunning = false;
        emit jobsDone();
    }
}

void BlogJobQueue::slotResult(KJob * job)
{
    kDebug();

    if ( !job ) {
        kError() << "The job unexpectedly died.";
        error( static_cast<KBlog::Blog::ErrorType>(-1), i18n( "The job unexpectedly died." ) );
        return;
    }

    if( job && job->error() ){
        int type = job->error() - 1; // HACK ErrorType has 0 as error, too, error() is ErrorType + 1
        error( static_cast<KBlog::Blog::ErrorType>(type), job->errorText() );
        return;
    }

    emit jobDone(job);
    runsFirstJob(); // recurse through all jobs
}

void BlogJobQueue::error( KBlog::Blog::ErrorType type, const QString& errorMessage )
{
    kError() << errorMessage;

    mIsRunning = false;

    KLocalizedString errType;
    switch (type) {
    case KBlog::Blog::XmlRpc:
        errType = ki18n("XML RPC Error: %1");
        break;
    case KBlog::Blog::Atom:
        errType = ki18n("Atom API Error: %1");
        break;
    case KBlog::Blog::ParsingError:
        errType = ki18n("KBlog Parsing Error: %1");
        break;
    case KBlog::Blog::AuthenticationError:
        errType = ki18n("Authentication Error: %1");
        break;
    case KBlog::Blog::NotSupported:
        errType = ki18n("Not Supported Error: %1");
        break;
    default:
        errType = ki18n("Unknown Error type: %1");
    };
    QString errorString = errType.subs(errorMessage).toString();

    emit error( errorString );
}

}

#include "blogjobqueue.moc"
