/***************************************************************************

   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>
   Copyright (C) 2008 Christian Weilbach <christian_weilbach@web.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "itemsmanager.h"

#include <QUrl>
#include <QFile>

#include <kblog/blogmedia.h>
#include <kdebug.h>


#include "post/postslist.h"
#include "post/post.h"
#include "media/media.h"
#include "media/medialistwidget.h"

#include "backend/backend.h"
#include "mainwidget.h"
#include "media/importmediadialog.h"

namespace KBlogger
{
ItemsManager *ItemsManager::s_self = 0;

ItemsManager *ItemsManager::self(QObject* parent)
{
    if ( !s_self )
        s_self = new ItemsManager(parent);
    return s_self;
}

ItemsManager::ItemsManager(QObject *parent): QObject(parent)
{
    kDebug();
}

ItemsManager::~ItemsManager()
{
    kDebug();
    // TODO we don't delete the pointers in the QMaps here, do we? should we?
    // or are the Post* pointers destroyed elsewhere reliably?
    s_self = 0;
}

void ItemsManager::createNewPost(Post *post)
{
    kDebug();

    if (!localWidgetsList.contains(post->getBlogName())) {
        kError() << "blogname is empty";
        return;
    }
    localWidgetsList.value( post->getBlogName() )->appendPost(post);
}

void ItemsManager::enqueuePostsToUpload( const QString& blogname )
{
    kDebug() << "blogname=" << blogname;

    //Uploads queue
    QList<Post*> entriesUploadList;

    //getting the entries to upload.
    if ( blogname.isEmpty() ) {
        //Upload All posts
        QMapIterator<QString, LocalDraftPostsList*> i_draft(localWidgetsList);
        while (i_draft.hasNext()) {
            i_draft.next();
            //kDebug() << "\tSend:"<< i_draft.key() << ": " << i_draft.value() << endl;
            entriesUploadList += i_draft.value()->getEntriesList();
        }
    } else {
        //Upload all posts of specified blogname.
        entriesUploadList = localWidgetsList[blogname]->getEntriesList();

    }

    //TEST
    if ( !entriesUploadList.size() ) {
        kDebug() << "no post to upload";
    }

    Backend *mBackend = Backend::self();
    for (int i = 0; i < entriesUploadList.size(); ++i) {
        kDebug() << "\ti=" << i << endl;

        mBackend->sendPost( entriesUploadList.at(i) );
    }
}

void ItemsManager::enqueueMediaToUpload( const QString& blogname )
{
    kDebug() << "blogname=" << blogname;;

    //Uploads queue
    QList<Media*> mediaUploadList;

    //Getting the media to upload.
    if ( blogname.isEmpty() ) {
        QMapIterator<QString, MediaList*> i_media(mediaWidgetsList);
        while (i_media.hasNext()) {
            i_media.next();
            mediaUploadList += i_media.value()->getMediaList();
        }
    } else {
        //Upload all media of specified blogname.
        mediaUploadList = mediaWidgetsList[blogname]->getMediaList();
    }

    //TEST
    if ( mediaUploadList.isEmpty() ) {
        kDebug() << "no media to upload";
    }

    //Creating jobs from media
    Backend *mBackend = Backend::self();
    for (int i = 0; i < mediaUploadList.size(); ++i) {
        kDebug() << "\ti=" << i << endl;

        if ( mediaUploadList.at(i)->status() == KBlog::BlogMedia::New )
            mBackend->sendMedia( mediaUploadList.at(i) );
    }
}

void ItemsManager::addInTheWidgetsList( const QString& blogName, LocalDraftPostsList* widget )
{
    kDebug();
    localWidgetsList[blogName] = widget;
}

void ItemsManager::addInTheWidgetsList(const QString& blogName, SentPostsList* widget)
{
    kDebug();
    sentWidgetsList[blogName] = widget;
}

void ItemsManager::addInTheWidgetsList(const QString& blogName, TrashedPostsList* widget)
{
    kDebug();
    trashedWidgetsList[blogName] = widget;
}

void ItemsManager::addInTheWidgetsList( const QString& blogName, MediaList* widget)
{
    kDebug();
    mediaWidgetsList[blogName] = widget;
}

void ItemsManager::removeDraftPost(Post* kbPost)
{
    kDebug();

    if (!kbPost) {
        kWarning() << "currentPost is Null: no problem Houston." << endl;
        return;
    }

    localWidgetsList.value( kbPost->getBlogName() )->removePost(kbPost, true);
}

void ItemsManager::removeTrashedPost(Post* kbPost)
{
    kDebug();

    if (!kbPost) {
        kWarning() << "currentPost is Null, no problem Houston." << endl;
        return;
    }

    trashedWidgetsList.value( kbPost->getBlogName() )->removePost(kbPost, true);
    // this post shouldn't be in the sent list either, important for the stylegetter temporary posts
    sentWidgetsList.value( kbPost->getBlogName() )->removePost(kbPost, true);
    delete kbPost;
}

void ItemsManager::addToSentList(Post* kbPost)
{
    kDebug();
    Q_ASSERT(kbPost);
    kDebug() << "post ID=" << kbPost->postId() << "added in the sent list.";

    //Append post in the Sent Post List
    SentPostsList *spl = sentWidgetsList.value( kbPost->getBlogName() );
    Q_ASSERT(spl);
    spl->appendPost(kbPost);
}

void ItemsManager::trashPost(Post *kbPost)
{
    kDebug();
    Q_ASSERT(kbPost);
    trashedWidgetsList.value( kbPost->getBlogName() )->appendPost(kbPost);
}

void  ItemsManager::moveDraftToSentList(Post *kbPost)
{
    kDebug();
    Q_ASSERT(kbPost);

    localWidgetsList.value( kbPost->getBlogName() )->removePost(kbPost, false);
    addToSentList(kbPost);
}

void ItemsManager::updatePostsSentList(const QString &blogname)
{
    kDebug();
    SentPostsList *mSentEntriesWidget;
    mSentEntriesWidget = sentWidgetsList[blogname];
    Q_ASSERT(mSentEntriesWidget);
    mSentEntriesWidget->updatePostsSentList();
}

bool ItemsManager::isStored(Post *kbPost)
{
    kDebug();
    if ( !kbPost ) {
        return false;
    }

    QList<Post*> postsToScan;
    Post *postToScan;

    postsToScan  << sentWidgetsList[ kbPost->getBlogName() ]->getEntriesList()
    << localWidgetsList[ kbPost->getBlogName() ]->getEntriesList()
    << trashedWidgetsList[ kbPost->getBlogName() ]->getEntriesList();

    for (int i = 0; i < postsToScan.size(); ++i) {
        kDebug() << "I=" << i;
        postToScan = postsToScan.at(i);

        if ( *kbPost == postToScan ) return true;
        if ( kbPost->postId() == postToScan->postId() ) {
            //Exists another post with the same ID. If it's in sent/trash it will be deleted.
            PostsList* containerWidget;
            containerWidget = qobject_cast<PostsList*> ( postToScan->container() );
            if ( containerWidget ) {
                containerWidget->removePost( postToScan );
                return false;
            }
        }
    }
    return false;
}

void ItemsManager::getUncachedMediaInThePosts( const QString &blogname )
{
    kDebug() << blogname;
    Q_ASSERT( !blogname.isEmpty() );
    QList<Post*> PostsList;
    PostsList = sentWidgetsList.value( blogname )->getEntriesList();

    //Get Media Urls:
    QList<Media*> MediaList;
    MediaList = mediaWidgetsList.value( blogname )->getMediaList();

    //Get the list of media present in the kblogger cache.
    QStringList cachedMedia;
    foreach (Media* media, MediaList) {
        QString url;
        url = media->url().url();
        if (!url.isEmpty())
            cachedMedia += url;
    }

    //Get mediaURl from the uploaded posts list
    //And add them to the ImportMediaDialog.
    foreach (Post* post, PostsList) {
        QStringList postsMediaUrl; //list of uploaded media url
        postsMediaUrl = post->mediaUrlList(); //retrive the media url present in the post
        
        foreach( QString url, postsMediaUrl) {
            kDebug()<< "url in the post:" << url;
            //Any url can be added once time to the ImportMediaDialog.
            if ( !cachedMedia.contains(url) ) {
                cachedMedia << url;
                ImportMediaDialog::self()->addUrl(url, post->getBlogName(), post->postId() );
            }
        }
        kDebug() << "url added to the ImportMediaDialog:" << postsMediaUrl;
    }
    ImportMediaDialog::self()->show();
}

void ItemsManager::addMedia( Media *media )
{
    kDebug();
    if (!media) return;
    QString blogname = media->getBlogname();
    if (blogname.isEmpty()) return;
    mediaWidgetsList.value( blogname )->appendMedia(media);
}

QString ItemsManager::url(QString &blogname, quint16 checksum)
{
    kDebug();
    if (blogname.isEmpty()) {
        kError() << "blogname is empty";
        return QString();
    }
    if (!checksum) {
        kDebug() << "checksum is empty";
        return QString();
    }
    QList<Media*> list;
    list = mediaWidgetsList.value( blogname )->getMediaList();
    for (int i = 0; i < list.size(); ++i) {
        if (list.at(i)->checksum() == checksum)
            return list.at(i)->url().url();
    }
    return QString();
}

QByteArray ItemsManager::mediaData(const QUrl &url)
{
    kDebug() << url;

    //Get Blogname
    QString blogname = MainWidget::self()->currentBlogname();
    if (blogname.isEmpty()) return QByteArray();

    ///Search
    //Get Media List
    QList<Media*> MediaList;
    MediaList = mediaWidgetsList.value( blogname )->getMediaList();

    //Is it a local media?
    bool isLocalUrl = false;

    if ( url.toString().left(15) == "kblogger-media-" )
        isLocalUrl = true;

    foreach (Media* media, MediaList) {
        QString mediaUrl;

        if ( isLocalUrl )
            mediaUrl = media->cachedFilename();
        else
            mediaUrl = media->url().url();

        if ( !mediaUrl.isEmpty() && url == mediaUrl ) {
            kDebug() << "Media Data is OK";
            return media->data();
        }
    }

    //No media Found!
    return QByteArray();

}

const QList<Media*> ItemsManager::getMediaList(const QString& blogname)
{
    kDebug() << blogname;
    // blogname is empty when the user doesn't select the blogname!
    // Don't use assert here!
    if ( blogname.isEmpty() ) {
        QList<Media*> emptyList;
        kDebug() << "blognams is empty";
        return emptyList;
    }

    MediaList *ml = mediaWidgetsList.value( blogname );
    Q_ASSERT(ml);

    return ml->getMediaList();
}

void ItemsManager::addToIgnoredMediaUrls(const QUrl& url, const QString& blogname)
{
    kDebug();
    Q_ASSERT ( !blogname.isEmpty() );
    MediaList *ml = mediaWidgetsList.value( blogname );
    Q_ASSERT(ml);

    ml->addToIgnoredMediaUrls( url );
}

QStringList ItemsManager::getIgnoredMediaUrls( const QString& blogname )
{
    kDebug();
    Q_ASSERT ( !blogname.isEmpty() );
    MediaList *ml = mediaWidgetsList.value( blogname );
    Q_ASSERT(ml);

    return ml->getIgnoredMediaUrls();
}

} //namespace
#include "itemsmanager.moc"
