/***************************************************************************

   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "media/media.h"

#include <QFile>
#include <QDataStream>
#include <QByteArray>

#include <kdebug.h>
#include <kstandarddirs.h>
#include <kurl.h>
#include <kfileitem.h>
#include <kicon.h>

#define CACHE_DIR "media-cache/"
#define KBLOGGERMEDIANAME "kblogger-media-"

namespace KBlogger
{

Media::Media(QByteArray& data,
                             const QString& name,
                             const QString& mimetype,
                             const QString& blogname) :
                                       mCachedFileItem(0),
                                       mFullSizeImage(0),
                                       mThumbnail(0)
{
    kDebug()  << name << mimetype << blogname;
    Q_ASSERT( data.count() );
    Q_ASSERT( !name.isEmpty() );
    Q_ASSERT( !mimetype.isEmpty() );
    Q_ASSERT( !blogname.isEmpty() );

    //readData(localFileUrl);
    setData(data);

    //Initialize
    initialize(blogname, name, mimetype, checksum());

    saveDataInACacheFile();
}



Media::Media(const QString& name,
                             const QString& mimetype,
                             const QString& blogname,
                             quint16 checksum) :
                                       mCachedFileItem(0),
                                       mFullSizeImage(0),
                                       mThumbnail(0)
{
    kDebug();

    Q_ASSERT( !name.isEmpty() );
    Q_ASSERT( !mimetype.isEmpty() );
    Q_ASSERT( !blogname.isEmpty() );

    KUrl localFileUrl;

    mChecksum = checksum;
    localFileUrl = cachedFileUrl();

    readData(localFileUrl);

    if ( checksum != mChecksum) {
        kError() << "Data have a different checksum"
        << checksum <<  QString("%1").arg(checksum, 4, 16)
        << mChecksum << QString("%1").arg(mChecksum, 4, 16);
        setChecksum(0); //It's not valid!
        return;
    }

    //Initialize
    initialize(blogname, name, mimetype, mChecksum);
}

void Media::readData(const KUrl& localFileUrl)
{
    kDebug() << localFileUrl.path();

    //Read Data from file
    QByteArray data;
    QFile file(localFileUrl.path());

    if( !file.open(QIODevice::ReadOnly) ) {
        kError() << "cannot open file" << localFileUrl;
        return;
    }

    data = file.readAll();

    Q_ASSERT( data.count() );

    //Set Data and setChecksum from data
    setData(data);

    if (!mChecksum) {
        kError() << "mChecksum is zero";
        return;
    }
}

void Media::initialize(const QString& blogname,
                               const QString& name,
                               const QString& mimetypeString,
                               const quint16 checksum)
{
    kDebug();

    // should not be necessary
    delete mFullSizeImage;
    delete mThumbnail;

    //TEST
    Q_ASSERT( !name.isEmpty() );
    Q_ASSERT( !mimetypeString.isEmpty() );
    Q_ASSERT( !blogname.isEmpty() );

    mBlog = blogname;
    setName(name);
    setMimetype( mimetypeString );
    setStatus( KBlog::BlogMedia::New );
    mChecksum = checksum;
    mCachedFileUrl = cachedFileUrl();
    kDebug() << mCachedFileUrl;
    mCachedFileItem = new KFileItem (mCachedFileUrl, mimetype(), KFileItem::Unknown);
    Q_ASSERT(mCachedFileItem);
}

void Media::saveDataInACacheFile()
{
    kDebug();
    QByteArray mData = data();
    if ( !mData.count() ) {
        kError() << "Data is EMPTY";
        return;
    }
    QFile file( cachedFileUrl() );
    Q_ASSERT( file.open(QIODevice::WriteOnly) );
    if ( file.write(mData) == -1 ) {
        kError() << "Impossible create the file:" << cachedFileUrl();
    }

}

Media::~Media()
{
    kDebug()  << "Blogname:" << getBlogname() 
	      << "Name:" << name();
    // they are only pointers to other images, which must not be deleted here
    mFullSizeImage = 0;
    mThumbnail = 0;
}

void Media::setData(const QByteArray& data)
{
    kDebug() << data.count();

    KBlog::BlogMedia::setData(data);
    setChecksum(data);
}

void Media::setChecksum(const QByteArray& data)
{
    kDebug();
    mChecksum = qChecksum(data.data(), data.count());
}

//Blogname
QString Media::getBlogname()
{
    return mBlog;
}

KIcon Media::icon()
{
    QString iconName;
    iconName = mimetype();
    iconName.replace(QChar('/'), QChar('-'));
    return KIcon(iconName);
}

void Media::setBlogname(const QString &blogname)
{
    mBlog = blogname;
    if ( mCachedFileItem ) {
        delete mCachedFileItem;
        mCachedFileItem = 0;
    }
    mCachedFileItem = new KFileItem(mCachedFileUrl, mimetype(), KFileItem::Unknown);
    Q_ASSERT(mCachedFileItem);
}

//Creation Data
KDateTime Media::creationDateTime() const
{
//     kDebug() << "CREATOR"<< mCachedFileItem->timeString(KFileItem::CreationTime);
//     kDebug() << "MODIF"<< mCachedFileItem->timeString(KFileItem::ModificationTime);
//     kDebug() << "ACCESS"<< mCachedFileItem->timeString(KFileItem::AccessTime);
    return mCachedFileItem->time( KFileItem::ModificationTime );
}

//SIZE
QString Media::sizeString() const
{
    return KIO::convertSize( mCachedFileItem->size() );
}

int Media::sizeInt() const
{
    return KIO::number( mCachedFileItem->size() ).toInt();
}

void Media::run()
{
    mCachedFileItem->run();
}

QString Media::cachedFilename()
{
    QString filename;
    QString hexChecksum;
    hexChecksum = QString("%1").arg(mChecksum, 4, 16);
    return KBLOGGERMEDIANAME + hexChecksum;
}

QString Media::cachedFileUrl()
{
    //mCachedFileUrl = cachePath();
    return cachePath() + cachedFilename();
}

QString Media::cachePath()
{
    return KStandardDirs::locateLocal("appdata", CACHE_DIR , true);
}

bool Media::removeCachedFile()
{
    kDebug() << "Removing:" << cachedFileUrl();
    QFile file( cachedFileUrl() );
    return file.remove();
}

void Media::setChecksum(quint16 checksum)
{
    kDebug();
    mChecksum = checksum;
}

quint16 Media::checksum()
{
    kDebug();
    return mChecksum;
}

bool Media::isCached()
{
    QString url = cachedFileUrl();
    if ( url.isEmpty() ) return false;
    QFile file( url );
    return file.exists();
}

void Media::setUrl(const KUrl& url)
{
    if ( !url.isValid() ) return;
    KBlog::BlogMedia::setStatus( KBlog::BlogMedia::Created );
    KBlog::BlogMedia::setUrl(url);
}

//Thumbail managing
void Media::setFullSizeImage(Media *media)
{
    kDebug();
    mFullSizeImage = media;
}

Media* Media::fullSizeImage()
{
    kDebug();
    return mFullSizeImage;
}

void Media::setThumbnail(Media *media)
{
    kDebug();
    mThumbnail = media;
}

Media* Media::thumbnail()
{
    kDebug();
    return mThumbnail;
}

} //namespace
