/***************************************************************************

   Copyright (C) 2007-2008 Antonio Aloisio <gnuton@gnuton.org>
   Copyright (C) 2008 Christian Weilbach <christian_weilbach@web.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "profiles/profilelist.h"

#include <QFile>
#include <QStringList>

#include <kinputdialog.h>
#include <kconfigdialog.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>

#include "profileconfig.h"
#include "profiles/profileconfigdialog.h"

#define DIR_TO_SAVE_ENTRIES "entries/"
#define DIR_TO_SAVE_MEDIA "media/"

namespace KBlogger
{

ProfileList::ProfileList( QWidget* parent ): QWidget(parent)
{
    kDebug();
    setupUi(this);
    populateList();

    connect(addButton, SIGNAL(clicked()), SLOT(addAccount()));
    connect(modifyButton, SIGNAL(clicked()), SLOT(modifyAccount()));
    connect(delButton, SIGNAL(clicked()), SLOT(delAccount()));
}

ProfileList::~ProfileList()
{
    kDebug();
}

void ProfileList::populateList()
{
    kDebug();
    ProfileConfig *profileConfig = ProfileConfig::prefs("THIS_ACCOUNT_DOESNT_EXIST");
    QStringList groupList = profileConfig->config()->groupList();
    delete profileConfig;
    kDebug () <<  groupList << endl;
    groupList = groupList.filter( QRegExp("^Account_"));
    groupList.replaceInStrings(QRegExp("^Account_"), QString());

    // recreate the list view
    ProfileListTree->clear();
    QList<QTreeWidgetItem *> items;
    for (int i = 0; i < groupList.size() ; ++i)
        items.append(new QTreeWidgetItem(ProfileListTree,
                                         QStringList(groupList.at(i))));
    ProfileListTree->insertTopLevelItems(0, items);
    if (!items.isEmpty()) {
        ProfileListTree->setCurrentItem(items.last());
    }
}

void ProfileList::addAccount()
{
    kDebug();
    QRegExp testRegExp("[.a-z0-9A-Z]+");
    QValidator* validator = new QRegExpValidator(testRegExp, this);

    QString newAccount = KInputDialog::getText(
                     i18n("Add a new profile"),
                     i18n("Insert here your new account name:"),
                     QString(), 0, this, validator);
    kDebug() << newAccount << endl;

    if( newAccount.isEmpty() ) return;

    // check if the account name exists already
    QList<QTreeWidgetItem*> list = ProfileListTree->findItems(newAccount,Qt::MatchExactly);
    if( !list.isEmpty() ){
        kDebug() << "do not try to add accounts twice";
        KMessageBox::error( this, i18n("This profile exists already.") );
        return;
    }

    // add account to the tree list
    QTreeWidgetItem *item = new QTreeWidgetItem(ProfileListTree, QStringList(newAccount));
    ProfileListTree->addTopLevelItem(item);

    //add account to the config file
    ProfileConfig *profileConfig = ProfileConfig::prefs(newAccount.prepend("Account_"));
    profileConfig->writeConfig();
    delete profileConfig;

    // open the ProfileConfigDialog for newly created accounts
    modifyAccount(newAccount);
}

void ProfileList::delAccount()
{
    kDebug();

    //get the account name from the profile list widget
    QTreeWidgetItem *mCurrentTreeItem = ProfileListTree->currentItem();
    if (!mCurrentTreeItem) return;
    QString account = mCurrentTreeItem->text(0);
    kDebug() << account << endl;

    if (KMessageBox::questionYesNo(this,i18n("Do you really want to delete the account \"%1\"? All local changes will be lost.",account),i18n("Confirm deletion"))==KMessageBox::No){
        return;
    }
    // remove stored files for the profile first
    QString entriesFilename = QString(account).prepend(DIR_TO_SAVE_ENTRIES);
    QStringList list;
    list << "-sent" << "-local" << "-trash";
    QStringList::ConstIterator it = list.begin();
    QStringList::ConstIterator end = list.end();
    for( ; it != end; it++ ){
        QString entryFilenameToRemove( entriesFilename );
        entryFilenameToRemove.append( *it );
        QString entryFileToRemove = KStandardDirs::locateLocal("appdata", entryFilenameToRemove , true);
        QFile entryFile( entryFileToRemove );
        if( !entryFile.remove() )
            kError() << "Could not remove: " << entryFileToRemove;
    }
    // remove the media file ( it is a container for all files )
    QString mediaFilename = QString(account);
    mediaFilename.prepend(DIR_TO_SAVE_MEDIA);
    QString mediaFileToRemove = KStandardDirs::locateLocal("appdata", mediaFilename, true);
    QFile mediaFile( mediaFileToRemove );
    if( !mediaFile.remove() )
        kError() << "Could not remove: " << mediaFileToRemove;

    // delete account from the tree list (there must be only one) TODO enable and remove populateList
    // once this code works
//     QTreeWidgetItem *item = ProfileListTree->findItems(account,Qt::MatchExactly).first();
//     ProfileListTree->setCurrentItem(ProfileListTree->itemAbove(item));
//     ProfileListTree->removeItemWidget(item,0);


    // delete account in the config file.
    account.prepend("Account_");
    ProfileConfig *pc = ProfileConfig::prefs(account);
    pc->setDefaults();
    pc->deletePrefs(account);
    pc->writeConfig();
    delete pc;

    // TODO remove when removeItemWidget works reliably
    populateList();
}

void ProfileList::modifyAccount( QString account )
{
    kDebug();
    //If account is empty, get account from selected QListWidget item
    if ( account.isEmpty()) {
        QTreeWidgetItem *currentTreeItem = ProfileListTree->currentItem();
        if (!currentTreeItem) return;
        account = currentTreeItem->text(0);
    }

    //"Account_" string prepend every accountname in the profile file.
    if ( account.left(8) != "Account_" )
        account = account.prepend("Account_");
    kDebug() << account;

    if ( KConfigDialog::showDialog( "profiles" ) )  {
        return;
    }

    ProfileConfig *profileConfig = ProfileConfig::prefs(account);
    KConfigDialog *dialog = new KConfigDialog(this, "profiles", profileConfig);
    ProfileConfigDialog *profileConfigDialog = new ProfileConfigDialog( dialog );
    dialog->setFaceType(KPageDialog::Plain);

    //Read data from ~/.kde4/share/config/kblogger-profilerc file
    //and fills any kfg_ fields in profileConfigDialog
    // replace Account_ with Account: 
    dialog->addPage(profileConfigDialog, account.replace("_", ": ") , "system-users");
    //readPasswordFromWallet use the kfl_url field to get the password from the wallet
    profileConfigDialog->readPasswordFromWallet();
    dialog->setAttribute( Qt::WA_DeleteOnClose );
    dialog->exec();
}

} //namespace kblogger
#include "profilelist.moc"
