#***************************************************************************
#            kalyptusCxxToKimono.pm -  Generates *.cs files for a Custom RealProxy
#										based smoke adaptor
#                             -------------------
#    begin                : Thurs Feb 19 12:00:00 2004
#    copyright            : (C) 2004, Richard Dale. All Rights Reserved.
#    email                : Richard_Dale@tipitina.demon.co.uk
#    author               : Richard Dale, based on the SMOKE generation code
#***************************************************************************/

#/***************************************************************************
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU General Public License as published by  *
# *   the Free Software Foundation; either version 2 of the License, or     *
# *   (at your option) any later version.                                   *
# *                                                                         *
#***************************************************************************/

package kalyptusCxxToKimono;

use File::Path;
use File::Basename;

use Carp;
use Ast;
use kdocAstUtil;
use kdocUtil;
use Iter;
use kalyptusDataDict;

use strict;
no strict "subs";

use vars qw/
	$libname $rootnode $outputdir $opt $debug
	$methodNumber
	%builtins %typeunion %allMethods %allTypes %enumValueToType %typedeflist %maptypeslist %arraytypeslist %mungedTypeMap %csharpImports
	%skippedClasses %partial_classes %operatorNames %new_classidx %interfacemap *CLASS /;

BEGIN
{

# Types supported by the StackItem union
# Key: C++ type  Value: Union field of that type
%typeunion = (
    'void*' => 's_voidp',
    'bool' => 's_bool',
    'char' => 's_char',
    'uchar' => 's_uchar',
    'short' => 's_short',
    'ushort' => 's_ushort',
    'int' => 's_int',
    'uint' => 's_uint',
    'long' => 's_long',
    'ulong' => 's_ulong',
    'float' => 's_float',
    'double' => 's_double',
    'enum' => 's_enum',
    'class' => 's_class'
);

# Mapping for iterproto, when making up the munged method names
%mungedTypeMap = (
     'QString' => '$',
     'QString*' => '$',
     'QString&' => '$',
     'QCString' => '$',
     'QCString*' => '$',
     'QCString&' => '$',
     'QDBusObjectPath' => '$',
     'QDBusObjectPath*' => '$',
     'QDBusObjectPath&' => '$',
     'QDBusSignature' => '$',
     'QDBusSignature*' => '$',
     'QDBusSignature&' => '$',
     'char*' => '$',
     'QCOORD*' => '?',
     'QRgb*' => '?',
     'Q_UINT64' => '$',
     'Q_INT64' => '$',
     'Q_LLONG' => '$',
     'quint64' => '$',
     'qint64' => '$',
     'long long' => '$',
     'qulonglong' => '$',
     'WId' => '$',
     'Q_PID' => '$',
);

# Yes some of this is in kalyptusDataDict's ctypemap
# but that one would need to be separated (builtins vs normal classes)
%typedeflist =
(
   'GLenum' => 'int',
   'GLint' => 'int',
   'GLuint' => 'uint',
   'ksocklen_t' => 'uint',
   'mode_t'  =>  'long',
   'MSG*'  =>  'void*',
   'pid_t' => 'int',
   'QCOORD'  =>  'int',
   'QImageReader::ImageReaderError' => 'int',
   'qint16' => 'short',
   'Q_INT16' => 'short',
   'qint32' => 'int',
   'qint32&' => 'int&',
   'Q_INT32' => 'int',
   'qint8' => 'char',
   'Q_INT8' => 'char',
   'Q_LONG' => 'long',
   'qreal' => 'double',
   'QRgb'  =>  'uint',
   'Qt::HANDLE' => 'uint',
   'QTSMFI'  =>  'int',
   'Qt::WFlags'  =>  'uint',
   'Qt::WState'  =>  'int',
   'quint16' => 'ushort',
   'Q_UINT16' => 'ushort',
   'quint32' => 'uint',
   'Q_UINT32' => 'uint',
   'quint8' => 'uchar',
   'Q_UINT8' => 'uchar',
   'Q_ULONG' => 'long',
   'short int' => 'short',
   'signed char' => 'char',
   'signed' => 'int',
   'signed int' => 'int',
   'signed long int' => 'long',
   'signed long' => 'long',
   'signed short' => 'short',
   'size_t' => 'int',
   'size_type'  =>  'int', # QSqlRecordInfo
   'time_t' => 'int',
   'unsigned char' => 'uchar',
   'unsigned int' => 'uint',
   'unsigned long int' => 'ulong',
   'unsigned long' => 'ulong',
   'unsigned short int' => 'ushort',
   'unsigned short' => 'ushort',
   'unsigned' => 'uint',
   'void(* )()'  =>  'void*',
   'void (*)(void* )'  =>  'void*',
   'WState'  =>  'int',
);

# Some classes need extra info in addition to the autogenerated code.
# So they are split into two sources FooBar.cs and FooBarExtras.cs
# with the 'partial' modifier in the class definition
%partial_classes =
(
   'QAbstractItemModel' => '1',
   'QApplication' => '1',
   'QBrush' => '1',
   'QColor' => '1',
   'QCoreApplication' => '1',
   'QCursor' => '1',
   'QDBusConnectionInterface' => '1',
   'QIcon' => '1',
   'QIconEngineV2' => '1',
   'QKeySequence' => '1',
   'QLineF' => '1',
   'QModelIndex' => '1',
   'QObject' => '1',
   'QPen' => '1',
   'QPointF' => '1',
   'QPolygon' => '1',
   'QPolygonF' => '1',
   'QRectF' => '1',
   'QRegion' => '1',
   'QSizeF' => '1',
   'QSqlQueryModel' => '1',
   'QStringListModel' => '1',
   'QTransform' => '1',
   'Qt' => '1',
   'QUrl' => '1',
   'QVariant' => '1',
);

%operatorNames =
(
    'operator^' => 'op_xor',
    'operator^=' => 'op_xor_assign',
    'operator<' => 'op_lt',
    'operator<<' => 'Write',
    'operator<=' => 'op_lte',
    'operator=' => 'op_assign',
    'operator==' => 'op_equals',
    'operator>' => 'op_gt',
    'operator>=' => 'op_gte',
    'operator>>' => 'Read',
    'operator|' => 'op_or',
    'operator|=' => 'op_or_assign',
    'operator-' => 'op_minus',
    'operator-=' => 'op_minus_assign',
    'operator--' => 'op_decr',
    'operator!' => 'op_not',
    'operator!=' => 'op_not_equals',
    'operator/' => 'op_div',
    'operator/=' => 'op_div_assign',
    'operator()' => 'op_expr',
    'operator[]' => 'op_at',
    'operator*' => 'op_mult',
    'operator*=' => 'op_mult_assign',
    'operator&' => 'op_and',
    'operator&=' => 'op_and_assign',
    'operator+' => 'op_plus',
    'operator+=' => 'op_plus_assign',
    'operator++' => 'op_incr',
);

%maptypeslist =
(
    'QMap<int, QVariant>' => 'Dictionary<int, QVariant>',
    'QMap<int, QVariant>&' => 'Dictionary<int, QVariant>',
    'QMap<QDate, QTextCharFormat>' => 'Dictionary<QDate, QTextCharFormat>',
    'QMap<QDate, QTextCharFormat>&' => 'Dictionary<QDate, QTextCharFormat>',
    'QMap<QString, QString>' => 'Dictionary<string, string>',
    'QMap<QString, QString>&' => 'Dictionary<string, string>',
    'QMap<QString, QVariant>' => 'Dictionary<string, QVariant>',
    'QMap<QString, QVariant>&' => 'Dictionary<string, QVariant>',
    'QMap<QString, QVariant::Type>' => 'Dictionary<string, QVariant.TypeOf>',
);

%arraytypeslist =
(
    'KCompletionMatches' => 'List<string>',
    'KCompletionMatches*' => 'List<string>',
    'KFileItemList' => 'List<KFileItem>',
    'KFileItemList*' => 'List<KFileItem>',
    'KFileItemList&' => 'List<KFileItem>',
    'KService::List' => 'List<KService>',
    'KServiceOfferList' => 'List<KServiceOffer>',
    'KUrl::List' => 'List<KUrl>',
    'KUrl::List*' => 'List<KUrl>',
    'KUrl::List&' => 'List<KUrl>',
    'KUrlList' => 'List<KUrl>',
    'KUrlList&' => 'List<KUrl>',
    'QFileInfoList' => 'List<QFileInfo>',
    'QFileInfoList&' => 'List<QFileInfo>',
    'QList<int>' => 'List<int>',
    'QList<int>&' => 'List<int>',
    'QList<KAboutPerson>' => 'List<KAboutPerson>',
    'QList<KAboutTranslator>' => 'List<KAboutTranslator>',
    'QList<KActionCollection*>&' => 'List<KActionCollection>',
    'QList<KAction*>' => 'List<KAction>',
    'QList<KConfigDialogManager*>&' => 'List<KConfigDialogManager>',
#    'QList<KConfigSkeleton::ItemEnum::Choice>' => 'List<QXmlStreamNotationDeclaration>',,
#    'QList<KConfigSkeleton::ItemEnum::Choice>&' => 'List<QXmlStreamNotationDeclaration>',
    'QList<KDataToolInfo>' => 'List<KDataToolInfo>',
    'QList<KDataToolInfo>&' => 'List<KDataToolInfo>',
    'QList<KConfigDialogManager*>' => 'List<KConfigDialogManager>',
    'QList<KFileItem>' => 'List<KFileItem>',
    'QList<KFileItem>&' => 'List<KFileItem>',
#    'QList<KIO::CopyInfo>&' => 'List<KIO.CopyInfo>',
    'QList<KJob*>&' => 'List<KJob>',
    'QList<KMainWindow*>' => 'List<KMainWindow>',
    'QList<KMainWindow*>&' => 'List<KMainWindow>',
    'QList<KMultiTabBarButton*>' => 'List<KMultiTabBarButton>',
    'QList<KMultiTabBarTab*>' => 'List<KMultiTabBarTab>',
    'QList<KParts::Part*>' => 'List<KParts.Part>',
    'QList<KParts::Plugin*>' => 'List<KParts.Plugin>',
#    'QList<KParts::Plugin::PluginInfo>' => 'List<QXmlStreamNotationDeclaration>',
#    'QList<KParts::Plugin::PluginInfo>&' => 'List<QXmlStreamNotationDeclaration>',
    'QList<KParts::ReadOnlyPart*>' => 'List<KParts.ReadOnlyPart>',
    'QList<KPluginInfo*>&' => 'List<KPluginInfo>',
    'QList<KServiceOffer>&' => 'List<KServiceOffer>',
    'QList<KSSLCertificate*>&' => 'List<KSSLCertificate>',
    'QList<KToolBar*>' => 'List<KToolBar>',
    'QList<KUserGroup>' => 'List<KUserGroup>',
    'QList<KUser>' => 'List<KUser>',
    'QList<KUser>&' => 'List<KUser>',
    'QList<KXMLGUIClient*>' => 'List<KXMLGUIClient>',
    'QList<KXMLGUIClient*>&' => 'List<KXMLGUIClient>',
    'QList<QAbstractButton*>' => 'List<QAbstractButton>',
    'QList<QActionGroup*>' => 'List<QAction>',
    'QList<QAction*>' => 'List<QAction>',
    'QList<QAction*>&' => 'List<QAction>',
    'QList<QByteArray>' => 'List<QByteArray>',
    'QList<QByteArray>*' => 'List<QByteArray>',
    'QList<QByteArray>&' => 'List<QByteArray>',
    'QList<QGraphicsItem*>' => 'List<QGraphicsItem>',
    'QList<QGraphicsItem*>&' => 'List<QGraphicsItem>',
    'QList<QGraphicsView*>' => 'List<QGraphicsView>',
    'QList <QGraphicsView*>' => 'List<QGraphicsView>',
    'QList<QHostAddress>' => 'List<QHostAddress>',
    'QList<QHostAddress>&' => 'List<QHostAddress>',
#    'QList<QImageTextKeyLang>' => 'List<QImageTextKeyLang>',
    'QList<QKeySequence>' => 'List<QKeySequence>',
    'QList<QKeySequence>&' => 'List<QKeySequence>',
    'QList<QListWidgetItem*>' => 'List<QListWidgetItem>',
    'QList<QListWidgetItem*>&' => 'List<QListWidgetItem>',
    'QList<QLocale::Country>' => 'List<QLocale.Country>',
    'QList<QMdiSubWindow*>' => 'List<QMdiSubWindow>',
    'QList<QModelIndex>' => 'List<QModelIndex>',
    'QList<QModelIndex>&' => 'List<QModelIndex>',
    'QList<QNetworkAddressEntry>' => 'List<QNetworkAddressEntry>',
    'QList<QNetworkInterface>' => 'List<QNetworkInterface>',
#	These List types with doubles don't compile:
#    'QList<QPair<qreal, QPointF> >' => 'List<double, QPointF>',
    'QList<QPair<qreal, QPointF> >' => 'List<QPair<double, QPointF>>',
    'QList<QPair<qreal, qreal> >' => 'List<QPair<double, double>>',
#    'QList<QPair<qreal, qreal> >' => 'List<QPair<double, double>>',
    'QList<QPair<QString, QString> >' => 'List<QPair<string, string>>',
    'QList<QPair<QString, QString> >&' => 'List<QPair<string, string>>',
    'QList<QPixmap>' => 'List<QPixmap>',
    'QList<QPolygonF>' => 'List<QPolygonF>',
    'QList<qreal>' => 'List<double>',
    'QList<QRectF>' => 'List<QRectF>',
    'QList<QRectF>&' => 'List<QRectF>',
    'QList<QSslCertificate>' => 'List<QSslCertificate>',
    'QList<QSslCertificate>&' => 'List<QSslCertificate>',
    'QList<QSslCipher>' => 'List<QSslCipher>',
    'QList<QSslCipher>&' => 'List<QSslCipher>',
    'QList<QSslError>&' => 'List<QSslError>',
    'QList<QStandardItem*>' => 'List<QStandardItem>',
    'QList<QStandardItem*>&' => 'List<QStandardItem>',
    'QList<QStringList>' => 'List<List<string>>',
    'QList<QTableWidgetItem*>' => 'List<QTableWidgetItem>',
    'QList<QTableWidgetItem*>&' => 'List<QTableWidgetItem>',
    'QList<QTableWidgetSelectionRange>' => 'List<QTableWidgetSelectionRange>',
    'QList<QTextBlock>' => 'List<QTextBlock>',
    'QList<QTextFrame*>' => 'List<QTextFrame>',
#    'QList<QTextLayout::FormatRange>' => 'List<QTextLayout.FormatRange>',
#    'QList<QTextLayout::FormatRange>&' => 'List<QTextLayout.FormatRange>',
    'QList<QTreeWidgetItem*>' => 'List<QTreeWidgetItem>',
    'QList<QTreeWidgetItem*>&' => 'List<QTreeWidgetItem>',
    'QList<QTreeWidget*>' => 'List<QTreeWidget>',
    'QList<QTreeWidget*>&' => 'List<QTreeWidget>',
    'QList<QUndoStack*>' => 'List<QUndoStack>',
    'QList<QUndoStack*>&' => 'List<QUndoStack>',
    'QList<QUrl>' => 'List<QUrl>',
    'QList<QUrl>&' => 'List<QUrl>',
    'QList<QVariant>' => 'List<QVariant>',
    'QList<QVariant>&' => 'List<QVariant>',
    'QList<QWidget*>' => 'List<QWidget>',
    'QList<QWidget*>&' => 'List<QWidget>',
    'QList<QWizard::WizardButton>&' => 'List<QWizard.WizardButton>',
    'QModelIndexList' => 'List<QModelIndex>',
    'QModelIndexList&' => 'List<QModelIndex>',
    'QObjectList' => 'List<QObject>',
    'QObjectList&' => 'List<QObject>',
    'QStringList' => 'List<string>',
    'QStringList*' => 'List<string>',
    'QStringList&' => 'List<string>',
    'QVariantList&' => 'List<QVariant>',
    'QVector<QColor>' => 'List<QColor>',
    'QVector<QColor>&' => 'List<QColor>',
    'QVector<QLineF>' => 'List<QLineF>',
    'QVector<QLineF>&' => 'List<QLineF>',
    'QVector<QLine>' => 'List<QLine>',
    'QVector<QLine>&' => 'List<QLine>',
    'QVector<QPointF>' => 'List<QPointF>',
    'QVector<QPointF>&' => 'List<QPointF>',
    'QVector<QPoint>' => 'List<QPoint>',
    'QVector<QPoint>&' => 'List<QPoint>',
    'QVector<qreal>' => 'List<double>',
    'QVector<qreal>&' => 'List<double>',
    'QVector<QRectF>' => 'List<QRectF>',
    'QVector<QRectF>&' => 'List<QRectF>',
    'QVector<QRect>' => 'List<QRect>',
    'QVector<QRect>&' => 'List<QRect>',
    'QVector<QRgb>' => 'List<uint>',
    'QVector<QRgb>&' => 'List<uint>',
    'QVector<QTextFormat>' => 'List<QTextFormat>',
    'QVector<QTextFormat>&' => 'List<QTextFormat>',
    'QVector<QTextLength>' => 'List<QTextLength>',
    'QVector<QTextLength>&' => 'List<QTextLength>',
    'QVector<QVariant>' => 'List<QVariant>',
    'QVector<QVariant>&' => 'List<QVariant>',
    'QWidgetList' => 'List<QWidget>',
    'QWidgetList&' => 'List<QWidget>',
    'QXmlStreamEntityDeclarations' => 'List<QXmlStreamEntityDeclaration>',
    'QXmlStreamNamespaceDeclarations' => 'List<QXmlStreamNamespaceDeclaration>',
    'QXmlStreamNotationDeclarations' => 'List<QXmlStreamNotationDeclaration>',
);

%interfacemap = (
'KBookmarkActionInterface' => 'IKBookmarkAction',
'KCompletionBase' => 'IKCompletionBase',
'KDevCore' => 'IKDevCore',
'KDirNotify' => 'IKDirNotify',
'KFileView' => 'IKFileView',
'KIO.SlaveBase' => 'KIO.ISlaveBase',
'KMessageHandler' => 'IKMessageHandler',
'KParts.PartBase' => 'KParts.IPartBase',
'KXMLGUIBuilder' => 'IKXMLGUIBuilder',
'KXMLGUIClient' => 'IKXMLGUIClient',
'PartBase' => 'IPartBase',
'QGraphicsItem' => 'IQGraphicsItem',
'QLayoutItem' => 'IQLayoutItem',
'QMimeSource' => 'IQMimeSource',
'QPaintDevice' => 'IQPaintDevice',
'QwAbsSpriteFieldView' => 'IQwAbsSpriteFieldView',
'QwtAbstractScale' => 'IQwtAbstractScale',
'QwtDoubleRange' => 'IQwtDoubleRange',
'QwtEventPattern' => 'IQwtEventPattern',
'QwtPlotDict' => 'IQwtPlotDict',
'QXmlContentHandler' => 'IQXmlContentHandler',
'QXmlDeclHandler' => 'IQXmlDeclHandler',
'QXmlDTDHandler' => 'IQXmlDTDHandler',
'QXmlEntityResolver' => 'IQXmlEntityResolver',
'QXmlErrorHandler' => 'IQXmlErrorHandler',
'SlaveBase' => 'ISlaveBase',
);

}

sub csharpImport($)
{
	my ( $classname ) = @_;
	my $classname_ptr = $classname . "*";
	if ( cplusplusToCSharp($classname_ptr) eq "" or $classname eq $main::globalSpaceClassName ) {
		return "";
	} elsif ( cplusplusToCSharp($classname_ptr) eq "ArrayList" ) {
		return "System.Collections";
	} elsif ( cplusplusToCSharp($classname_ptr) =~ /^List</ ) {
		return "System.Collections.Generic";
	} elsif ( cplusplusToCSharp($classname_ptr) eq "StringBuilder" ) {
		return "";
	} elsif ( cplusplusToCSharp($classname_ptr) eq "string" ) {
		return "";
	} elsif ( cplusplusToCSharp($classname_ptr) eq "string[][]" ) {
		return "";
#	} elsif ( cplusplusToCSharp($classname_ptr) eq "string[]" ) {
#		return "";
	} elsif ( cplusplusToCSharp($classname_ptr) =~ /^[a-z]/ ) {
		return "";
	} 
	return "";
}
	
sub cplusplusToCSharp
{
	my ( $cplusplusType )  = @_;
	my $isConst = ($cplusplusType =~ /const / or $cplusplusType !~ /[*&]/ ? 1 : 0);
	$cplusplusType =~ s/const //;
	$cplusplusType =~ s/^signed//;
	my $className = $cplusplusType;
	$className =~ s/[*&]//;
	
	if ( $cplusplusType =~ /void\*|KDateTime::Spec&?$|K3Icon|KHTMLPart::PageSecurity|EditMode|QNetworkProtocolFactoryBase|QDomNodePrivate|QSqlDriverCreatorBase|QSqlFieldInfoList|QObjectUserData|QUObject|QTextParag|QWidgetMapper|QMemArray<int>|QLayoutIterator|QAuBucket|QUnknownInterface|QConnectionList/ ) {
		return ""; # Unsupported type
	} elsif ( $cplusplusType =~ /bool/ && kalyptusDataDict::ctypemap($cplusplusType) eq "int" ) {
		return "bool";
	} elsif ( $cplusplusType =~ /bool\s*[*&]/ ) {
#		return "bool";
		return "ref bool";
	} elsif ( $cplusplusType =~ /^long$|^qint64$/) {
		return "long";
	} elsif ( $cplusplusType =~ /^ulong$|^quint64$/) {
		return "ulong";
	} elsif ( $cplusplusType eq 'qlonglong') {
		return "long";
	} elsif ( $cplusplusType eq 'qulonglong') {
		return "ulong";
	} elsif ( $cplusplusType eq 'QPair<QSslKey, QSslKey>') {
		return "QPair<QSslKey, QSslKey>";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^void\s*\*/ ) {
		return "int";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^qt_QIntValueList\*/ )
	{
		return "int[]";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*(unsigned )?int\s*\*/
				|| $cplusplusType =~ /^int[*&]$/ )
	{
#		return "int";
		return "ref int";
	} elsif ( $cplusplusType =~ /qreal[*&]$/ )
	{
#		return "double";
		return "ref double";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*double\s*\*/ ) {
#		return "double";
		return "ref double";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*(unsigned )?short\s*\*/ ) {
#		return "short";
		return "ref short";
	} elsif ( $cplusplusType =~ /KCmdLineOptions/ ) {
		return "string[][]";	
	} elsif ( $maptypeslist{$cplusplusType} ) {
		return $maptypeslist{$cplusplusType};
	} elsif ( $arraytypeslist{$cplusplusType} ) {
		return $arraytypeslist{$cplusplusType};
	} elsif ( $cplusplusType =~ /uchar\s*\*/ ) {
		return "char[]";
	} elsif ( $cplusplusType =~ /QC?String/ and !$isConst ) {
		return "StringBuilder"
	} elsif ( $cplusplusType =~ /(DOM::)?DOMString/ 
			|| $cplusplusType =~ /^[^<]*QString/ 
			|| $cplusplusType =~ /QCString/ 
			|| $cplusplusType =~ /^(const )?char\s*\*$/ 
			|| kalyptusDataDict::ctypemap($cplusplusType) =~ /^(const )?char\s*\*/ ) {
		return "string"
	} elsif ( $cplusplusType =~ /QChar\s*[&\*]?/ || $cplusplusType =~ /^char$/ ) {
		return "char"
	} elsif ( $cplusplusType =~ /QDBusObjectPath/ ) {
		return "QDBusObjectPath"
	} elsif ( $cplusplusType =~ /QDBusSignature/ ) {
		return "QDBusSignature"
	} elsif ( $cplusplusType =~ /QDBusVariant/ ) {
		return "QDBusVariant"
	} elsif ( $cplusplusType =~ /QPaintDevice/ ) {
		return "IQPaintDevice"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /unsigned char/ ) {
		return "ushort";
	} elsif ( $typedeflist{$cplusplusType} =~ /ulong/ || $cplusplusType eq 'ulong' ) {
		return "ulong";
	} elsif ( $typedeflist{$cplusplusType} =~ /long/ || $cplusplusType eq 'long' ) {
		return "long";
	} elsif ( $typedeflist{$cplusplusType} =~ /uint/ || $cplusplusType eq 'uint' ) {
		return "uint";
	} elsif ( $typedeflist{$cplusplusType} =~ /int/  || $cplusplusType eq 'int' ) {
		return "int";
	} elsif ( $typedeflist{$cplusplusType} =~ /ushort/  || $cplusplusType eq 'ushort' ) {
		return "ushort";
	} elsif ( $typedeflist{$cplusplusType} =~ /short/ || $cplusplusType eq 'short') {
		return "short";
	} elsif ( $typedeflist{$cplusplusType} =~ /float/ || $cplusplusType eq 'float' ) {
		return "float";
	} elsif ( $typedeflist{$cplusplusType} =~ /double/ || $cplusplusType eq 'double') {
		return "double";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /(unsigned )(.*)/ ) {
		return "u" . $2;
	} else {
		my $node;
		my $item;
		if ($className =~ /^(\w+)::(\w+)$/) {
			$node = kdocAstUtil::findRef( $rootnode, $1 );
			$item = kdocAstUtil::findRef( $node, $2 ) if defined $node;
			if (defined $item && $item->{NodeType} eq 'enum') {
				if ($2 eq 'Type') {
					return "$1.TypeOf";
				} else {
					return "$1.$2";
				}
				
			} elsif (defined $item && ($item->{NodeType} eq 'class' || $item->{NodeType} eq 'struct')) {
				return $skippedClasses{$className} ? "" : "$1.$2";
			}
		}

		if ($className =~ /^\w+$/) {
			$item = kdocAstUtil::findRef( $rootnode, $className );
			if (defined $item && ($item->{NodeType} eq 'class' || $item->{NodeType} eq 'struct')) {
				return $skippedClasses{$className} ? "" : $className;
			}
		}
		return kalyptusDataDict::ctypemap($cplusplusType);
	}

}

sub writeDoc
{
	( $libname, $rootnode, $outputdir, $opt ) = @_;

	print STDERR "Starting writeDoc for $libname...\n";

	$debug = $main::debuggen;

	mkpath( $outputdir ) unless -f $outputdir;

	print STDERR "Preparsing...\n";

	# Preparse everything, to prepare some additional data in the classes and methods
	Iter::LocalCompounds( $rootnode, sub { preParseFlags( shift ); } );

	# Preparse everything, to prepare some additional data in the classes and methods
	Iter::LocalCompounds( $rootnode, sub { preParseClass( shift ); } );

	# Have a look at each class again, to propagate CanBeCopied
	Iter::LocalCompounds( $rootnode, sub { propagateCanBeCopied( shift ); } );

	# Write out smokedata.cpp
	writeSmokeDataFile($rootnode);

	print STDERR "Writing *.cs...\n";

my @classlist;
push @classlist, ""; # Prepend empty item for "no class"
my %enumclasslist;
Iter::LocalCompounds( $rootnode, sub {
	my $classNode = $_[0];
	my $className = join( "::", kdocAstUtil::heritage($classNode) );
	
	return if $classNode->{NodeType} eq 'namespace';
	
	push @classlist, $className;
	$enumclasslist{$className}++ if keys %{$classNode->{enumerations}};
	$classNode->{ClassIndex} = $#classlist;
#	addImportForClass( $classNode, \%allImports, undef );
} );

%new_classidx = do { my $i = 0; map { $_ => $i++ } @classlist };

	# Generate *cs file for each class
	Iter::LocalCompounds( $rootnode, sub { 
		writeClassDoc( shift ); 
	} );

	print STDERR "Done.\n";
}

=head2 preParseFlags
	Called for each class, looks for Q_DECLARE_FLAGS, and maps them to uints
=cut
sub preParseFlags
{
	my( $classNode ) = @_;
	my $className = join( "::", kdocAstUtil::heritage($classNode) );

    Iter::MembersByType ( $classNode, undef,
			sub {	
				my( $classNode, $m ) = @_;
				
				if ( $m->{NodeType} eq 'flags' ) {
	    			my $fullFlagsName = $className."::".$m->{astNodeName};

                    if (exists $typedeflist{$fullFlagsName}) {
						print("typemap for $fullFlagsName exists\n");
					}

					$typedeflist{$fullFlagsName} = 'uint';
	    			registerType( $fullFlagsName );
				}
			}, undef );

}

=head2 preParseClass
	Called for each class
=cut
sub preParseClass
{
	my( $classNode ) = @_;
	my $className = join( "::", kdocAstUtil::heritage($classNode) );

	if (	$classNode->{Deprecated} 
			|| $classNode->{NodeType} eq 'union' 
			|| $#{$classNode->{Kids}} < 0
			|| $classNode->{Access} eq "private"
			|| $classNode->{Access} eq "protected"  # e.g. QPixmap::QPixmapData
			|| $className =~ /.*Private$/  # Ignore any classes which aren't for public consumption
			|| $className =~ /.*Impl$/ 
			|| $className =~ /.*Internal.*/ 
			|| exists $classNode->{Tmpl}
			|| $className eq 'KAccelGen'
			|| $className eq 'KDateTime::Spec'
			|| $className eq 'KDE'
			|| $className eq 'KDEDModule'
			|| $className eq 'KDialogButtonBox'
			|| $className eq 'KDirOperator'
			|| $className eq 'KDirSelectDialog'
			|| $className eq 'KEditListBox::CustomEditor'
			|| $className eq 'KEncodingFileDialog::Result'
			|| $className eq 'KFileFilterCombo'
			|| $className eq 'KFileMetaInfo'
			|| $className eq 'KFileMetaInfoGroup'
			|| $className eq 'KFileTreeBranch'
			|| $className eq 'KFileView'
			|| $className eq 'KFileViewSignaler'
			|| $className eq 'KGlobalSettings::KMouseSettings'
			|| $className eq 'khtml'
			|| $className eq 'khtml::DrawContentsEvent'
			|| $className eq 'khtml::MouseDoubleClickEvent'
			|| $className eq 'khtml::MouseEvent'
			|| $className eq 'khtml::MouseMoveEvent'
			|| $className eq 'khtml::MousePressEvent'
			|| $className eq 'khtml::MouseReleaseEvent'
			|| $className eq 'KIconTheme'
			|| $className eq 'KIO::NetRC'
			|| $className eq 'KMimeTypeChooserDialog'
			|| $className eq 'KParts::ComponentFactory'
			|| $className eq 'KParts::MainWindow'
			|| $className eq 'KParts::Plugin::PluginInfo'
			|| $className eq 'KProtocolInfo::ExtraField'
			|| $className eq 'KServiceTypeProfile'
			|| $className eq 'KSettings::PluginPage'
			|| $className eq 'KTimeZone::Transition'
			|| $className eq 'KTipDatabase'
			|| $className eq 'KTzfileTimeZoneData'
			|| $className eq 'KUrl::List'
			|| $className eq 'KXMLGUIClient::StateChange'
			|| $className eq 'QAbstractTextDocumentLayout::PaintContext'
			|| $className eq 'QAbstractTextDocumentLayout::Selection'
			|| $className eq 'QAbstractUndoItem'
			|| $className eq 'QAccessibleBridgePlugin'
			|| $className eq 'QBrushData'
			|| $className eq 'QDBusObjectPath'
			|| $className eq 'QDBusSignature'
			|| $className eq 'QDBusVariant'
			|| $className eq 'QDebug'
			|| $className eq 'QImageTextKeyLang'
			|| $className eq 'QInputMethodEvent::Attribute'
			|| $className eq 'QIPv6Address'
			|| $className eq 'QLatin1String'
			|| $className eq 'QMap::const_iterator'
			|| $className eq 'QMapData'
			|| $className eq 'QMapData::Node'
			|| $className eq 'QMap::iterator'
			|| $className eq 'QMutex'
			|| $className eq 'QMutexLocker'
			|| $className eq 'QObjectData'
			|| $className eq 'QPainterPath::Element'
			|| $className eq 'QProxyModel'
			|| $className eq 'QReadLocker'
			|| $className eq 'QReadWriteLock'
			|| $className eq 'QSemaphore'
			|| $className eq 'QSharedData'
			|| $className eq 'QString'
			|| $className eq 'QStringList'
			|| $className eq 'QStyleOptionQ3DockWindow'
			|| $className eq 'QStyleOptionQ3ListView'
			|| $className eq 'QStyleOptionQ3ListViewItem'
			|| $className eq 'QSysInfo'
			|| $className eq 'QTextCodec::ConverterState'
			|| $className eq 'QTextLayout::FormatRange'
			|| $className eq 'QTextStreamManipulator'
			|| $className eq 'QThread'
			|| $className eq 'QThreadStorageData'
			|| $className eq 'QUpdateLaterEvent'
			|| $className eq 'QVariant::Handler'
			|| $className eq 'QVariant::PrivateShared'
			|| $className eq 'QVariantComparisonHelper'
			|| $className eq 'QVectorData'
			|| $className eq 'QWaitCondition'
			|| $className eq 'QWidgetData'
			|| $className eq 'QWriteLocker'
			|| $className eq 'QX11Info' )
	{
	    print STDERR "Skipping $className\n" if ($debug);
	    print STDERR "Skipping union $className\n" if ( $classNode->{NodeType} eq 'union');
	    $skippedClasses{$className} = 1;
	    delete $classNode->{Compound}; # Cheat, to get it excluded from Iter::LocalCompounds
	    return;
	}

	my $signalCount = 0;
	my $eventHandlerCount = 0;
	my $defaultConstructor = 'none'; #  none, public, protected or private. 'none' will become 'public'.
	my $constructorCount = 0; # total count of _all_ ctors
	# If there are ctors, we need at least one public/protected one to instanciate the class
	my $hasPublicProtectedConstructor = 0;
	# We need a public dtor to destroy the object --- ### aren't protected dtors ok too ??
	my $hasPublicDestructor = 1; # by default all classes have a public dtor!
	#my $hasVirtualDestructor = 0;
	my $hasDestructor = 0;
	my $hasPrivatePureVirtual = 0;
	my $hasCopyConstructor = 0;
	my $hasPrivateCopyConstructor = 0;
	# Note: no need for hasPureVirtuals. $classNode{Pure} has that.

        my $doPrivate = $main::doPrivate;
	$main::doPrivate = 1;
	# Look at each class member (looking for methods and enums in particular)
	Iter::MembersByType ( $classNode, undef,
		sub {

	my( $classNode, $m ) = @_;
	my $name = $m->{astNodeName};

	if( $m->{NodeType} eq "method" ) {
	    if ( $m->{ReturnType} eq 'typedef' # QFile's EncoderFn/DecoderFn callback, very badly parsed
	       ) {
		$m->{NodeType} = 'deleted';
		next;
	    }

	    print STDERR "preParseClass: looking at $className\::$name  $m->{Params}\n" if ($debug);

	    if ( $name eq $classNode->{astNodeName} ) {
		if ( $m->{ReturnType} =~ /~/  ) {
		    # A destructor
		    $hasPublicDestructor = 0 if $m->{Access} ne 'public';
		    #$hasVirtualDestructor = 1 if ( $m->{Flags} =~ "v" && $m->{Access} ne 'private' );
		    $hasDestructor = 1;
		} else {
		    # A constructor
		    $constructorCount++;
		    $defaultConstructor = $m->{Access} if ( $m->{Params} eq '' );
		    $hasPublicProtectedConstructor = 1 if ( $m->{Access} ne 'private' );

		    # Copy constructor?
		    if ( $#{$m->{ParamList}} == 0 ) {
			my $theArgType = @{$m->{ParamList}}[0]->{ArgType};
			if ($theArgType =~ /$className\s*\&/) {
			    $hasCopyConstructor = 1;
			    $hasPrivateCopyConstructor = 1 if ( $m->{Access} eq 'private' );
			}
		    }
		    # Hack the return type for constructors, since constructors return an object pointer
		    $m->{ReturnType} = $className."*";
		}
	    }

	    if ( $name =~ /~$classNode->{astNodeName}/ && $m->{Access} ne "private" ) { # not used
		$hasPublicDestructor = 0 if $m->{Access} ne 'public';
		#$hasVirtualDestructor = 1 if ( $m->{Flags} =~ "v" );
		$hasDestructor = 1;
	    }

	    if ( $m->{Flags} =~ "p" && $m->{Access} =~ /private/ ) {
                $hasPrivatePureVirtual = 1; # ouch, can't inherit from that one
	    }

	    # All we want from private methods is to check for virtuals, nothing else
	    next if ( $m->{Access} =~ /private/ );

		# Don't generate code for deprecated methods, 
		# or where the code won't compile/link for obscure reasons. Or even obvious reasons..
		if ( $m->{Deprecated} 
			# Assume only Qt classes have tr() and trUtf8() in their Q_OBJECT macro
			|| ($classNode->{astNodeName} !~ /^Q/ and $name eq 'tr')
			|| ($classNode->{astNodeName} !~ /^Q/ and $name eq 'trUtf8')
			|| $m->{ReturnType} =~ /template/ 
			|| $m->{ReturnType} =~ /QT3_SUPPORT/ 
			|| $name eq 'qt_metacast' 
			|| $name eq 'virtual_hook' 
			|| $name eq 'handle' 
			|| ($name eq 'qt_metacall')
			|| ($name eq 'metaObject')
			|| $name eq 'qWarning' 
			|| $name eq 'qCritical' 
			|| $name eq 'qDebug' 
			|| $name eq 'finalize' 
			|| ($m->{ReturnType} =~ /iterator/)
			|| ($classNode->{astNodeName} eq 'QApplication' and $name eq 'QApplication')
			|| ($classNode->{astNodeName} eq 'QCoreApplication' and $name eq 'QCoreApplication')
			|| ($classNode->{astNodeName} eq 'QBoxLayout' and $name eq 'spacing')
			|| ($classNode->{astNodeName} eq 'QBoxLayout' and $name eq 'setSpacing')
			|| ($classNode->{astNodeName} eq 'QGridLayout' and $name eq 'setSpacing')
			|| ($classNode->{astNodeName} eq 'QGridLayout' and $name eq 'spacing')
			|| ($classNode->{astNodeName} eq 'QMessageBox' and $name eq 'setWindowTitle')
			|| ($classNode->{astNodeName} eq 'TextEvent' and $name eq 'data')
			|| ($classNode->{astNodeName} eq 'KConfigGroup' and $name eq 'groupImpl')
			|| ($classNode->{astNodeName} eq 'KConfigGroup' and $name eq 'setReadDefaults')
			|| ($name eq 'operator<<' and $m->{ParamList}[0]->{ArgType} =~ /QDebug/ )
			|| ($name eq 'operator<<' and $m->{ParamList}[0]->{ArgType} =~ /QDataStream/ and $m->{ParamList}[1]->{ArgType} =~ /KDateTime::Spec/ )
			|| ($name eq 'operator>>' and $m->{ParamList}[0]->{ArgType} =~ /QDataStream/ and $m->{ParamList}[1]->{ArgType} =~ /KDateTime::Spec/ )
			|| ($name eq 'operator<<' and $m->{ParamList}[0]->{ArgType} =~ /QDataStream/ and $m->{ParamList}[1]->{ArgType} =~ /const KDateTime/ )
			|| ($name eq 'operator>>' and $m->{ParamList}[0]->{ArgType} =~ /QDataStream/ and $m->{ParamList}[1]->{ArgType} =~ /KDateTime/ )
			|| ($classNode->{astNodeName} eq 'KInputDialog' and $name eq 'getDouble')
			|| ($classNode->{astNodeName} eq 'KInputDialog' and $name eq 'getInteger')
			|| ($classNode->{astNodeName} eq 'KIO' and $name eq 'buildHTMLErrorString')
			|| ($classNode->{astNodeName} eq 'KJob' and $name eq 'description')
			|| ($classNode->{astNodeName} eq 'KJob' and $name eq 'KJob')
			|| ($classNode->{astNodeName} eq 'KShortcutsEditor' and $name eq 'checkGlobalShortcutsConflict')
			|| ($classNode->{astNodeName} eq 'KShortcutsEditor' and $name eq 'checkStandardShortcutsConflict')
			|| ($classNode->{astNodeName} eq 'KStandardShortcut' and $name eq 'insert')
			|| ($classNode->{astNodeName} eq 'KTzfileTimeZoneSource' and $name eq 'location')
			|| ($classNode->{astNodeName} eq 'Wallet' and $name eq 'Wallet')
			|| ($classNode->{astNodeName} eq 'KMD5' and $name eq 'transform') ) 
		{
		    $m->{NodeType} = 'deleted';
			next;
		}
			    
		my $argId = 0;
	    my $firstDefaultParam;
	    foreach my $arg ( @{$m->{ParamList}} ) {
		# Look for first param with a default value
		if ( defined $arg->{DefaultValue} && !defined $firstDefaultParam ) {
		    $firstDefaultParam = $argId;
		}

		if (	$arg->{ArgType} eq '...' # refuse a method with variable arguments
				|| $arg->{ArgType} eq 'const QTextItem&' # ref to a private class
				|| $arg->{ArgType} eq 'DecoderFn' # QFile's callback
				|| $arg->{ArgType} eq 'EncoderFn' # QFile's callback
				|| $arg->{ArgType} eq 'FILE*' ) # won't be able to handle that I think
		{
		    $m->{NodeType} = 'deleted';
		}
		else
		{
		    # Resolve type in full, e.g. for QSessionManager::RestartHint
		    # (QSessionManagerJBridge doesn't inherit QSessionManager)
		    $arg->{ArgType} = kalyptusDataDict::resolveType($arg->{ArgType}, $classNode, $rootnode);
		    registerType( $arg->{ArgType} );
		    $argId++;
		}
	    }
	    $m->AddProp( "FirstDefaultParam", $firstDefaultParam );
	    $m->{ReturnType} = kalyptusDataDict::resolveType($m->{ReturnType}, $classNode, $rootnode) if ($m->{ReturnType});
	    registerType( $m->{ReturnType} );
	}
	elsif( $m->{NodeType} eq "enum" ) {
		if ( ! $m->{astNodeName} ) {
			$m->{Access} = 'protected';
		}
		my $fullEnumName = $className."::".$m->{astNodeName};
		if ( ($fullEnumName eq 'KMimeType::Format' and $name eq 'compression')
				|| $fullEnumName eq 'QDataStream::ByteOrder'
				|| $m->{Deprecated} ) {
		    $m->{NodeType} = 'deleted';
			next;
		}
	    
		$classNode->{enumerations}{$m->{astNodeName}} = $fullEnumName;
#		if $m->{astNodeName} and $m->{Access} ne 'private';
#		if $m->{astNodeName} ;

	    # Define a type for this enum
	    registerType( $fullEnumName );

	    # Remember that it's an enum
	    findTypeEntry( $fullEnumName )->{isEnum} = 1;
 	} elsif( $m->{NodeType} eq 'property' ) {
		if ( ($classNode->{astNodeName} eq 'QWidget' and $name eq 'Q_PROPERTY_height')
			|| ($classNode->{astNodeName} eq 'QWidget' and $name eq 'Q_PROPERTY_minimumSizeHint')
			|| ($classNode->{astNodeName} eq 'QWidget' and $name eq 'Q_PROPERTY_sizeHint')
			|| ($classNode->{astNodeName} eq 'QWidget' and $name eq 'Q_PROPERTY_visible')
			|| ($classNode->{astNodeName} eq 'QWidget' and $name eq 'Q_PROPERTY_width')
			|| ($classNode->{astNodeName} eq 'QStackedLayout' and $name eq 'Q_PROPERTY_count') )
		{
		    $m->{NodeType} = 'deleted';
			next;
		}
		# Don't generate C# code for the property's read and write methods
		my $method;
		if ( defined $m->{READ} && $m->{READ} ne '') {
			$method = kdocAstUtil::findRef( $classNode, $m->{READ} );
			if (	defined $method 
					&& $#{$method->{ParamList}} == -1 
					&& $method->{Flags} !~ 'v'
					&& $method->{Access} !~ /slots|signals/ ) 
		{
		    	$method->{NodeType} = 'deleted';
			}
		}

		if ( defined $m->{WRITE} && $m->{WRITE} ne '') {
			$method = kdocAstUtil::findRef( $classNode, $m->{WRITE} );
			if (	defined $method 
					&& $#{$method->{ParamList}} == 0 
					&& $method->{Flags} !~ 'v'
					&& $method->{Access} !~ /slots|signals/ ) 
			{
		    	$method->{NodeType} = 'deleted';
			}
		}
	} elsif( $m->{NodeType} eq 'var' ) {
		if ($name eq 'staticMetaObject') {
		    $m->{NodeType} = 'deleted';
			next;
		}
	    my $varType = $m->{Type};
	    # We are interested in public static vars, like QColor::blue
	    if ( $varType =~ s/static\s+// && $m->{Access} ne 'private'
			&& $className."::".$m->{astNodeName} ne "KSpell::modalListText" )
	    {
		$varType =~ s/const\s+(.*)\s*&/$1/;
		$varType =~ s/\s*$//;
		print STDERR "var: $m->{astNodeName} '$varType'\n" if ($debug);

		# Register the type
		registerType( $varType );

	    } else {
		# To avoid duplicating the above test, we just get rid of any other var
		$m->{NodeType} = 'deleted';
	    }
	}
		},
		undef
	);
	$main::doPrivate = $doPrivate;

	print STDERR "$className: ctor count: $constructorCount, hasPublicProtectedConstructor: $hasPublicProtectedConstructor, hasCopyConstructor: $hasCopyConstructor:, defaultConstructor: $defaultConstructor, hasPublicDestructor: $hasPublicDestructor, hasPrivatePureVirtual:$hasPrivatePureVirtual\n" if ($debug);

	# Note that if the class has _no_ constructor, the default ctor applies. Let's even generate it.
	if ( !$constructorCount && $defaultConstructor eq 'none' && !$hasPrivatePureVirtual ) {
	    # Create a method node for the constructor
	    my $methodNode = Ast::New( $classNode->{astNodeName} );
	    $methodNode->AddProp( "NodeType", "method" );
	    $methodNode->AddProp( "Flags", "" );
	    $methodNode->AddProp( "Params", "" );
            $methodNode->AddProp( "ParamList", [] );
	    kdocAstUtil::attachChild( $classNode, $methodNode );

	    # Hack the return type for constructors, since constructors return an object pointer
	    $methodNode->AddProp( "ReturnType", $className."*" );
	    registerType( $className."*" );
	    $methodNode->AddProp( "Access", "public" ); # after attachChild
	    $defaultConstructor = 'public';
	    $hasPublicProtectedConstructor = 1;
	}

	# Also, if the class has no explicit destructor, generate a default one.
	if ( !$hasDestructor && !$hasPrivatePureVirtual ) {
	    my $methodNode = Ast::New( "$classNode->{astNodeName}" );
	    $methodNode->AddProp( "NodeType", "method" );
	    $methodNode->AddProp( "Flags", "" );
	    $methodNode->AddProp( "Params", "" );
	    $methodNode->AddProp( "ParamList", [] );
	    kdocAstUtil::attachChild( $classNode, $methodNode );

	    $methodNode->AddProp( "ReturnType", "~" );
	    $methodNode->AddProp( "Access", "public" );
	}

	# If we have a private pure virtual, then the class can't be instanciated (e.g. QCanvasItem)
	# Same if the class has only private constructors (e.g. QInputDialog)
	$classNode->AddProp( "CanBeInstanciated", $hasPublicProtectedConstructor 
#												&& !$hasPrivatePureVirtual
												&& (!$classNode->{Pure} or $classNode->{astNodeName} eq 'QValidator')
												&& !($classNode->{NodeType} eq 'namespace')
												&& ($classNode->{astNodeName} !~ /^DrawContentsEvent$|^MouseEvent$|^MouseDoubleClickEvent$|^MouseMoveEvent$|^MouseReleaseEvent$|^MousePressEvent$/)
												&& ($classNode->{astNodeName} !~ /QMetaObject|QDragObject|Slave|CopyJob|KMdiChildFrm|KNamedCommand/) );

	# We will derive from the class only if it has public or protected constructors.
	# (_Even_ if it has pure virtuals. But in that case the *.cpp class can't be instantiated either.)
	$classNode->AddProp( "BindingDerives", $hasPublicProtectedConstructor );

	# We need a public dtor to destroy the object --- ### aren't protected dtors ok too ??
	$classNode->AddProp( "HasPublicDestructor", $hasPublicDestructor );

	# Hack for QAsyncIO. We don't implement the "if a class has no explicit copy ctor,
	# then all of its member variables must be copiable, otherwise the class isn't copiable".
	$hasPrivateCopyConstructor = 1 if ( $className eq 'QAsyncIO' );

	# Remember if this class can't be copied - it means all its descendants can't either
	$classNode->AddProp( "CanBeCopied", !$hasPrivateCopyConstructor );
	$classNode->AddProp( "HasCopyConstructor", $hasCopyConstructor );
	if ($classNode->{astNodeName} =~ /Abstract/
		|| $classNode->{astNodeName} eq 'QAccessibleInterface'
		|| $classNode->{astNodeName} eq 'QAccessibleApplication'
		|| $classNode->{astNodeName} eq 'QAccessibleObjectEx'
		|| $classNode->{astNodeName} eq 'QAccessibleWidgetEx'
		|| $classNode->{astNodeName} eq 'QAccessibleObject' ) 
 	{
		$classNode->AddProp( "Pure", 1 );
	}
}

sub propagateCanBeCopied($)
{
	my $classNode = shift;
	my $className = join( "::", kdocAstUtil::heritage($classNode) );
	my @super = superclass_list($classNode);
	# A class can only be copied if none of its ancestors have a private copy ctor.
	for my $s (@super) {
	    if (!$s->{CanBeCopied}) {
		$classNode->{CanBeCopied} = 0;
		print STDERR "$classNode->{astNodeName} cannot be copied\n" if ($debug);
		last;
	    }
	}

	# Prepare the {case} dict for the class
	prepareCaseDict( $classNode );
}

sub generateClass($$$$$)
{
	my( $node, $packagename, $namespace, $indent, $addImport ) = @_;
	my $className = join( "::", kdocAstUtil::heritage($node) );
	my $csharpClassName = $node->{astNodeName};
	my $classCode = "";
	my %csharpMethods = ();
#	my %addImport = ();
	
	my @ancestors = ();
	my @ancestor_nodes = ();
	Iter::Ancestors( $node, $rootnode, undef, undef, sub { 
				my ( $ances, $name, $type, $template ) = @_;
				if ( 	$name ne "khtml::KHTMLWidget"
						and $name !~ /QList</ and $name ne 'QList' and $name !~ /QVector/ 
                        and $name !~ /QMap/ and $name !~ /QHash/ and $name ne '' ) {
					if (defined $ances) {
						push @ancestor_nodes, $ances;
						my $ancestorName = join( ".", kdocAstUtil::heritage($ances) );
						push @ancestors, $ancestorName;
					}
				}
			},
			undef
		);
        
	my ($methodCode, $staticMethodCode, $interfaceCode, $proxyInterfaceCode, $signalCode, $extraCode, $notConverted) = generateAllMethods( $node, $#ancestors + 1, 
																		\%csharpMethods, 
																		$node, 
																		1, 
																		$addImport );

	my $tempMethodNumber = $methodNumber;
	
	# Add method calls for the interfaces implemented by the class
	foreach my $ancestor_node ( @ancestor_nodes ) {
		if ( defined $interfacemap{$ancestor_node->{astNodeName}} && ($#ancestors > 0) ) {
			my ($meth, $static, $interf, $proxyInterf, $sig, $extra, $notconv) = generateAllMethods( $ancestor_node, 0, \%csharpMethods, $node, 0, $addImport );
			$methodCode .= $meth;
			$staticMethodCode .= $static;
			$extraCode .= $extra;
			$interfaceCode .= $interf;
			$proxyInterfaceCode .= $proxyInterf;
			$notConverted .= $notconv;
		}
	}
	
	my $globalSpace = kdocAstUtil::findRef( $rootnode, $main::globalSpaceClassName );
	my ($meth, $static, $interf, $proxyInterf, $sig, $extra, $notconv) = generateAllMethods( $globalSpace, 0, \%csharpMethods, $node, 0, $addImport );
	$methodCode .= $meth;
	$staticMethodCode .= $static;
	$extraCode .= $extra;
	$interfaceCode .= $interf;
	$proxyInterfaceCode .= $proxyInterf;
	$notConverted .= $notconv;
	$methodNumber = $tempMethodNumber;
		
	if ( $className eq 'Qt' ) {
		;
	} else {
		if ( $className eq 'QListViewItem' or $className eq 'QUriDrag' or $className eq 'KDE' ) {
			# Special case these two classes as they have methods that use ArrayList added as 'extras'
	    	$classCode .= "$indent\tusing System.Collections.Generic;\n";
		}

		if ( $className eq 'QObject' ) {
	    	$classCode .= "$indent\tusing System.Reflection;\n";
		}
	}

	if ( defined $interfacemap{$csharpClassName} ) {
		$classCode .= "\n$indent\tpublic interface " . $interfacemap{$csharpClassName} . " {\n";
		$classCode .= $interfaceCode;
		$classCode .= "$indent\t}\n";
	}
	
	my $classdec = "";
	my $parentClassName = "";

	if ($node->{NodeType} eq 'namespace') {
		$classdec .= "\t[SmokeClass(\"$className\")]\n";
#		$classdec .= "\tnamespace $className {\n";

		if ( $partial_classes{$csharpClassName} ) {
			$classdec .= "\tpublic partial class $csharpClassName : Object {\n";
		} else {
			$classdec .= "\tpublic class $csharpClassName : Object {\n";
		}
		$classdec .= "\t\tprotected SmokeInvocation interceptor = null;\n";
	} elsif ( $#ancestors < 0 ) {
		$classdec .= "\t[SmokeClass(\"$className\")]\n";

		if ( $csharpClassName eq 'QObject' ) {
			$classdec .= "\tpublic partial class QObject : Qt, IDisposable {\n";
			$classdec .= "\t\tprivate IntPtr smokeObject;\n";
			$classdec .= "\t\tprotected Object Q_EMIT = null;\n";
			$classdec .= "\t\tprotected $csharpClassName(Type dummy) {\n";
			$classdec .= "\t\t\ttry {\n";
			$classdec .= "\t\t\t\tType proxyInterface = Qyoto.GetSignalsInterface(GetType());\n";
			$classdec .= "\t\t\t\tSignalInvocation realProxy = new SignalInvocation(proxyInterface, this);\n";
			$classdec .= "\t\t\t\tQ_EMIT = realProxy.GetTransparentProxy();\n";
			$classdec .= "\t\t\t}\n";
			$classdec .= "\t\t\tcatch {\n";
			$classdec .= "\t\t\t\tConsole.WriteLine(\"Could not retrieve signal interface\");\n";
			$classdec .= "\t\t\t}\n";
			$classdec .= "\t\t}\n";
			$classdec .= "\t\t[SmokeMethod(\"metaObject()\")]\n";
			$classdec .= "\t\tpublic virtual QMetaObject MetaObject() {\n";
			$classdec .= "\t\t\tif (SmokeMarshallers.IsSmokeClass(GetType())) {\n";
			$classdec .= "\t\t\t\treturn (QMetaObject) interceptor.Invoke(\"metaObject\", \"metaObject()\", typeof(QMetaObject));\n";
			$classdec .= "\t\t\t} else {\n";
			$classdec .= "\t\t\t\treturn Qyoto.GetMetaObject(this);\n";
			$classdec .= "\t\t\t}\n";
			$classdec .= "\t\t}\n";
		} else {
			if ( $node->{Pure} ) {
				$classdec .= "\tpublic abstract ";
			} else {
				$classdec .= "\tpublic ";
			}

			if ( $partial_classes{$csharpClassName} ) {
				$classdec .= "partial class $csharpClassName : Object";
			} else {
				$classdec .= "class $csharpClassName : Object";
			}
			if ( defined $interfacemap{$csharpClassName} ) {
				$classdec .= ", " . $interfacemap{$csharpClassName};
			}
			
			if ($node->{CanBeInstanciated} and $node->{HasPublicDestructor} and !$node->{Pure}) {
				$classdec .= ", IDisposable";
			}
			
			$classdec .= " {\n\t\tprotected SmokeInvocation interceptor = null;\n";
			$classdec .= "\t\tprivate IntPtr smokeObject;\n";
			$classdec .= "\t\tprotected $csharpClassName(Type dummy) {}\n";
		}
	} else {
		$classdec .= "\t[SmokeClass(\"$className\")]\n";
		if ( $partial_classes{$csharpClassName} ) {
			if ( $node->{Pure} 
				|| $csharpClassName eq 'QAccessibleInterface'
				|| $csharpClassName eq 'QAccessibleApplication'
				|| $csharpClassName eq 'QAccessibleObjectEx'
				|| $csharpClassName eq 'QAccessibleWidgetEx'
				|| $csharpClassName eq 'QAccessibleObject' ) 
			{
				$classdec .= "\tpublic abstract partial class $csharpClassName : ";
			} else {
				$classdec .= "\tpublic partial class $csharpClassName : ";
			}
		} else {
			if ( $node->{Pure} ) {
				$classdec .= "\tpublic abstract class $csharpClassName : ";
			} else {
				$classdec .= "\tpublic class $csharpClassName : ";
			}
		}
		my $ancestor;
		foreach $ancestor ( @ancestors ) {
			if ( !defined $interfacemap{$ancestor} or $ancestor eq @ancestors[$#ancestors] ) {
				$parentClassName .= "$ancestor";
				$classdec .= "$ancestor";
				last;
			}
		}

		my @implements = ();
		if ( $#ancestors >= 1 ) {
			$classdec .= ", ";
			foreach $ancestor ( @ancestors ) {
				if ( defined $interfacemap{$ancestor} ) {
					push(@implements, $interfacemap{$ancestor});
				}
			}
	    }
	
		if ($#implements >= 0) {
			$classdec .= join(", ", @implements);
		}
		
		if ($node->{CanBeInstanciated} and $node->{HasPublicDestructor} and !$node->{Pure}) {
			$classdec .= ", IDisposable";
		}
		
		$classdec .= " {\n";
		$classdec .= " \t\tprotected $csharpClassName(Type dummy) : base((Type) null) {}\n";
	}
		
	$classCode .= "\n";
	if ( $csharpClassName !~ /^Q/ or $signalCode ne '' ) {
		my $signalLink = '';
 		if ( $signalCode ne '' ) {
			$signalLink = " See <see cref=\"I$csharpClassName" . "Signals\"></see> for signals emitted by $csharpClassName\n";
		}
		my $docnode = $node->{DocNode};
		if ( defined $docnode ) { 
			my $comment = printCSharpdocComment( $docnode, "", "$indent\t/// ", $signalLink )  . "\n"; 
			$classCode .= $comment;
		} else {
			$classCode .= "$indent\t///$signalLink";
		}
	}
	
	$classCode .= indentText($indent, $classdec);

    Iter::MembersByType ( $node, undef,
		sub {	my ($node, $subclassNode ) = @_;
			if ( $subclassNode->{NodeType} eq 'class' && $subclassNode->{Compound} ) {
				$classCode .= generateClass($subclassNode, $packagename, $namespace, $indent . "\t", $addImport);
			}
		}, undef );
    
	if ($methodCode ne '') {
		if ( $#ancestors < 0 ) {
			$classCode .= "$indent\t\tprotected void CreateProxy() {\n";
		} else {
			$classCode .= "$indent\t\tprotected new void CreateProxy() {\n";
		}
		$classCode .= "$indent\t\t\tinterceptor = new SmokeInvocation(typeof($csharpClassName), this);\n$indent\t\t}\n";
	}

	if ($proxyInterfaceCode ne '') {
		$classCode .= "$indent\t\tprivate static SmokeInvocation staticInterceptor = null;\n";
		$classCode .= "$indent\t\tstatic $csharpClassName() {\n";
		$classCode .= "$indent\t\t\tstaticInterceptor = new SmokeInvocation(typeof($csharpClassName), null);\n";
		$classCode .= "$indent\t\t}\n";
	}

	$classCode .= indentText($indent, $extraCode);	
	$classCode .= indentText($indent, $notConverted);	
	$classCode .= indentText($indent, $methodCode);	
	$classCode .= indentText($indent, $staticMethodCode);	

 	if ( is_kindof($node, "QObject") ) {
		if ( $csharpClassName eq 'QObject' ) {
			$classCode .= "$indent\t\tprotected I" . $csharpClassName . "Signals Emit {\n";
		} else {
			$classCode .= "$indent\t\tprotected new I" . $csharpClassName . "Signals Emit {\n";
		}

		$classCode .= "$indent\t\t\tget { return (I" . $csharpClassName . "Signals) Q_EMIT; }\n";
		$classCode .= "$indent\t\t}\n";
    	$classCode .= "$indent\t}\n";

		$classCode .= "\n$indent\tpublic interface I$csharpClassName" . "Signals";
		if ($parentClassName =~ /(.*)[.](.*)/) {
			$classCode .= " : $1.I" . $2 . "Signals" unless $csharpClassName eq "QObject";
		} else {
			$classCode .= " : I" . $parentClassName . "Signals" unless $csharpClassName eq "QObject";
		}
		$classCode .= " {\n";
		$classCode .= $signalCode;
		$classCode .= "$indent\t}\n";
	} else {
    	$classCode .= "$indent\t}\n";
	}

	return $classCode;
}

=head2 writeClassDoc

	Called by writeDoc for each class to be written out

=cut

sub writeClassDoc
{
	my( $node ) = @_;
	my $className = join( "::", kdocAstUtil::heritage($node) );
	my $csharpClassName = $node->{astNodeName};
	# Makefile doesn't like '::' in filenames, so use __
	my $fileName  = $className;
	$fileName =~ s/::/_/g;
#	my $fileName  = join( "__", kdocAstUtil::heritage($node) );
	print "Enter: $className\n" if $debug;

	my $packagename;
	if ($className =~ /^Qsci/) {
		$packagename = "QScintilla";
	} elsif ($className =~ /^Qwt/) {
		$packagename = "Qwt";
	} elsif ($className =~ /^Q/) {
		$packagename = "Qyoto";
	} else {
		$packagename = "Kimono";
	}

	my $namespace;
	if ($className =~ /([^:]*)::.*/) {
		$namespace = $1;
	}

	my %addImport = ();

	# Write out the *.csharp file
	my $classFile = "$outputdir/$fileName.cs";
	open( CLASS, ">$classFile" ) || die "Couldn't create $classFile\n";
	print STDERR "Writing $fileName.csharp\n" if ($debug);
	
	print CLASS "//Auto-generated by kalyptus. DO NOT EDIT.\n";
#	print CLASS "//Auto-generated by $0. DO NOT EDIT.\n";
	
	print CLASS "namespace $packagename {\n\n";
	if (defined $namespace) {
		print CLASS "\tnamespace $namespace {\n\n";
	}

	print CLASS "\tusing System;\n";

    my $classCode = generateClass($node, $packagename, $namespace, "", \%addImport);

	foreach my $imp (keys %addImport) {
		die if $imp eq '';
		# Ignore any imports for classes in the same package as the current class
		if ($imp !~ /$packagename/) {
			print CLASS "\tusing $imp;\n";
		}
	}	

	print CLASS $classCode;

	if (defined $namespace) {
		print CLASS "\t}\n";
	}

	print CLASS "}\n";

	close CLASS;
}


# Generate the prototypes for a method (one per arg with a default value)
# Helper for makeprotos
sub iterproto($$$$$) {
    my $classidx = shift; # to check if a class exists
    my $method = shift;
    my $proto = shift;
    my $idx = shift;
    my $protolist = shift;

    my $argcnt = scalar @{ $method->{ParamList} } - 1;
    if($idx > $argcnt) {
	push @$protolist, $proto;
	return;
    }
    if(defined $method->{FirstDefaultParam} and $method->{FirstDefaultParam} <= $idx) {
	push @$protolist, $proto;
    }

    my $arg = $method->{ParamList}[$idx]->{ArgType};

    my $typeEntry = findTypeEntry( $arg );
    my $realType = $typeEntry->{realType};

    # A scalar ?
    $arg =~ s/\bconst\b//g;
    $arg =~ s/\s+//g;
    if($typeEntry->{isEnum} || $allTypes{$realType}{isEnum} || exists $typeunion{$realType} || exists $mungedTypeMap{$arg})
    {
	my $id = '$'; # a 'scalar
	$id = '?' if $arg =~ /[*&]{2}/;
	$id = $mungedTypeMap{$arg} if exists $mungedTypeMap{$arg};
	iterproto($classidx, $method, $proto . $id, $idx + 1, $protolist);
	return;
    }

    # A class ?
    if(exists $classidx->{$realType}) {
	iterproto($classidx, $method, $proto . '#', $idx + 1, $protolist);
	return;
    }

    # A non-scalar (reference to array or hash, undef)
    iterproto($classidx, $method, $proto . '?', $idx + 1, $protolist);
    return;
}

# Generate the prototypes for a method (one per arg with a default value)
sub makeprotos($$$) {
    my $classidx = shift;
    my $method = shift;
    my $protolist = shift;
    iterproto($classidx, $method, $method->{astNodeName}, 0, $protolist);
}

# Return the string containing the signature for this method (without return type).
# If the 2nd arg is not the size of $m->{ParamList}, this method returns a
# partial signature (this is used to handle default values).
sub argsSignature($$) {
    my $method = shift;
    my $last = shift;
#    my $sig = $method->{astNodeName};
    my $sig = "";
    my @argTypeList;
    my $argId = 0;
    foreach my $arg ( @{$method->{ParamList}} ) {
	last if $argId > $last;
	push @argTypeList, $arg->{ArgType};
	$argId++;
    }
    $sig .= "(". join(", ",@argTypeList) .")";
    $sig .= " const" if $method->{Flags} =~ "c";
    return $sig;
}

# Return the string containing the signature for this method (without return type).
# If the 2nd arg is not the size of $m->{ParamList}, this method returns a
# partial signature (this is used to handle default values).
sub methodSignature($$) {
    my $method = shift;
    my $last = shift;
    my $sig = $method->{astNodeName};
    my @argTypeList;
    my $argId = 0;
    foreach my $arg ( @{$method->{ParamList}} ) {
	last if $argId > $last;
	push @argTypeList, $arg->{ArgType};
	$argId++;
    }
    $sig .= "(". join(", ",@argTypeList) .")";
    $sig .= " const" if $method->{Flags} =~ "c";
    return $sig;
}

# Return the string containing the signature for this method (without return type).
# If the 2nd arg is not the size of $m->{ParamList}, this method returns a
# partial signature (this is used to handle default values).
sub mungedSignature($$) {
    my $method = shift;
    my $last = shift;
#    my $sig = $method->{astNodeName};
    my $sig = "";
    my $argId = 0;
    foreach my $arg ( @{$method->{ParamList}} ) {
		last if $argId > $last;
		$sig .= mungedArgType($method, $arg->{ArgType});
		$argId++;
    }
    return $sig;
}

# Return the string containing the csharp signature for this method (without return type).
# If the 2nd arg is not the size of $m->{ParamList}, this method returns a
# partial signature (this is used to handle default values).
sub csharpMethodSignature($$) {
    my $method = shift;
    my $last = shift;
    my $sig = $method->{astNodeName};
    my @argTypeList;
    my $argId = 0;
    foreach my $arg ( @{$method->{ParamList}} ) {
	$argId++;
	last if $argId > $last;
	push @argTypeList, cplusplusToCSharp( $arg->{ArgType} );
    }
    $sig .= "(". join(", ",@argTypeList) .")";
    return $sig;
}

sub smokeInvocation($$$$$$) {
	my ( $target, $argtypes, $returnType, $mungedMethod, $signature, $addImport ) = @_;

	my $methodCode = "\t\t\t";
	if ($returnType ne 'void') {
		$methodCode .= "return ($returnType) ";
	}

	$methodCode .= "$target.Invoke(\"$mungedMethod\", \"$signature\", typeof($returnType)";

	my $arglist = "";
	foreach my $arg ( @{$argtypes} ) {
        $arg =~ /^(ref )?(.*)\s(.*)$/;
		if ($1 ne '') {
			return smokeRefInvocation($target, $argtypes, $returnType, $mungedMethod, $signature, $addImport);
		}
        $arglist .= ", typeof($2), $3";
	}

	return $methodCode . $arglist  . ");\n";
}

sub smokeRefInvocation($$$$$$) {
	my ( $target, $argtypes, $returnType, $mungedMethod, $signature, $addImport ) = @_;

	my $preMethodCode = "";
	my $methodCode = "";
	my $postMethodCode = "";

   $preMethodCode = "\t\t\tStackItem[] stack = new StackItem[" . (scalar(@{$argtypes}) + 1) . "];\n";
   $methodCode .= "\t\t\t$target.Invoke(\"$mungedMethod\", \"$signature\", stack);\n";

	my $arglist = "";
	my $argNo = 1;
	foreach my $arg ( @{$argtypes} ) {
        $arg =~ /^(ref )?(.*)\s(.*)$/;
		my $argtype = $2;
		my $argname = $3;

		if ($1 ne '') {
			$preMethodCode .= "\t\t\tstack[$argNo].s_$argtype = $argname;\n";
			$postMethodCode .= "\t\t\t$argname = stack[$argNo].s_$argtype;\n";
		} elsif ($argtype =~ /^int$|^uint$|^bool$|^double$|^float$|^long$|^ulong$|^short$|^ushort$/ ) {
			$preMethodCode .= "\t\t\tstack[$argNo].s_$argtype = $argname;\n";
		} elsif ($argtype =~ /\./ ) {
			$preMethodCode .= "\t\t\tstack[$argNo].s_int = (int) $argname;\n";
		} else {
			$addImport->{"System.Runtime.InteropServices"} = 1;
			$preMethodCode .= "#if DEBUG\n";
			$preMethodCode .= "\t\t\tstack[$argNo].s_class = (IntPtr) DebugGCHandle.Alloc($argname);\n";
			$preMethodCode .= "#else\n";
			$preMethodCode .= "\t\t\tstack[$argNo].s_class = (IntPtr) GCHandle.Alloc($argname);\n";
			$preMethodCode .= "#endif\n";

			$postMethodCode .= "#if DEBUG\n";
			$postMethodCode .= "\t\t\tDebugGCHandle.Free((GCHandle) stack[$argNo].s_class);\n";
			$postMethodCode .= "#else\n";
			$postMethodCode .= "\t\t\t((GCHandle) stack[$argNo].s_class).Free();\n";
			$postMethodCode .= "#endif\n";
		}

		$argNo++;
#        $arglist .= ", typeof($2), $3";
	}

	if ($returnType eq 'void' ) {
		$postMethodCode .= "\t\t\treturn;\n";
	} elsif ($returnType =~ /^int$|^uint$|^bool$|^double$|^float$|^long$|^ulong$|^short$|^ushort$/ ) {
		$postMethodCode .= "\t\t\treturn stack[0].s_$returnType;\n";
	} elsif ($returnType =~ /\./ ) {
		$postMethodCode .= "\t\t\treturn ($returnType) Enum.ToObject(typeof($returnType), stack[0].s_int);\n";
	} else {
		$addImport->{"System.Runtime.InteropServices"} = 1;
		$postMethodCode .= "\t\t\tobject returnValue = ((GCHandle) stack[0].s_class).Target;\n";

		$postMethodCode .= "#if DEBUG\n";
		$postMethodCode .= "\t\t\tDebugGCHandle.Free((GCHandle) stack[0].s_class);\n";
		$postMethodCode .= "#else\n";
		$postMethodCode .= "\t\t\t((GCHandle) stack[0].s_class).Free();\n";
		$postMethodCode .= "#endif\n";

		$postMethodCode .= "\t\t\treturn ($returnType) returnValue;\n";
	}

	return $preMethodCode . $methodCode . $postMethodCode;
}

sub smokeInvocationArgList($) {
	my $argtypes = shift;

	my $arglist = "";
	foreach my $arg ( @{$argtypes} ) {
        if ( $arg =~ /^(ref )?(.*)\s(.*)$/ ) {
        	$arglist .= ", typeof($2), $3";
		}
	}
	return $arglist;
}

sub coerce_type($$$$) {
    #my $m = shift;
    my $union = shift;
    my $var = shift;
    my $type = shift;
    my $new = shift; # 1 if this is a return value, 0 for a normal param

    my $typeEntry = findTypeEntry( $type );
    my $realType = $typeEntry->{realType};

    my $unionfield = $typeEntry->{typeId};
#    die "$type" unless defined( $unionfield );
	if ( ! defined( $unionfield ) ) {
		print STDERR "type field not defined: $type\n";
		return "";
	}
	
    $unionfield =~ s/t_/s_/;

    $type =~ s/\s+const$//; # for 'char* const'
    $type =~ s/\s+const\s*\*$/\*/; # for 'char* const*'

    my $code = "$union.$unionfield = ";
    if($type =~ /&$/) {
	$code .= "(void*)&$var;\n";
    } elsif($type =~ /\*$/) {
	$code .= "(void*)$var;\n";
    } else {
	if ( $unionfield eq 's_class' 
		or ( $unionfield eq 's_voidp' and $type ne 'void*' )
		or $type eq 'QString' ) { # hack
	    $type =~ s/^const\s+//;
	    if($new) {
	        $code .= "(void*)new $type($var);\n";
	    } else {
	        $code .= "(void*)&$var;\n";
	    }
	} else {
	    $code .= "$var;\n";
	}
    }

    return $code;
}

# Generate the list of args casted to their real type, e.g.
# (QObject*)x[1].s_class,(QEvent*)x[2].s_class,x[3].s_int
sub makeCastedArgList
{
    my @castedList;
    my $i = 1; # The args start at x[1]. x[0] is the return value
    my $arg;
    foreach $arg (@_) {
	my $type = $arg;
	my $cast;

	my $typeEntry = findTypeEntry( $type );
	my $unionfield = $typeEntry->{typeId};
#	die "$type" unless defined( $unionfield );
	if ( ! defined( $unionfield ) ) {
		print STDERR "type field not defined: $type\n";
		return "";
	}
	$unionfield =~ s/t_/s_/;

	$type =~ s/\s+const$//; # for 'char* const'
	$type =~ s/\s+const\s*\*$/\*/; # for 'char* const*'

	my $v .= " arg$i";
	if($type =~ /&$/) {
	    $cast = "*($type *)";
	} elsif($type =~ /\*$/) {
	    $cast = "($type)";
        } elsif($type =~ /\(\*\)\s*\(/) { # function pointer ... (*)(...)
            $cast = "($type)";
	} else {
	    if ( $unionfield eq 's_class'
		or ( $unionfield eq 's_voidp' and $type ne 'void*' )
		or $type eq 'QString' ) { # hack
	        $cast = "*($type *)";
	    } else {
	        $cast = "($type)";
	    }
	}
	push @castedList, "$type$v";
	$i++;
    }
    return @castedList;
}


# Adds the import for node $1 to be imported in $2 if not already there
# Prints out debug stuff if $3
sub addImportForClass($$$)
{
    my ( $node, $addImport, $debugMe ) = @_;
    my $importname = csharpImport( $node->{astNodeName} );
#	print "  Importing $importname for node name: " . $node->{astNodeName} . "\n";
	# No import needed, so return
    return if ( $importname eq '' );
    unless ( defined $addImport->{$importname} ) {
	print "  Importing $importname\n" if ($debugMe);
	$addImport->{$importname} = 1;
    }
    else { print "  $importname already imported.\n" if ($debugMe); }
}

sub checkImportsForObject($$$)
{
    my $type = shift;
    my $addImport = shift;
	my $classNode;
	if ($type eq '') {
		return;
	}

	$type = kalyptusDataDict::resolveType($type, $classNode, $rootnode);
	my $csharptype = cplusplusToCSharp($type);
	if ( $csharptype eq 'ArrayList' ) {
		$addImport->{"System.Collections"} = 1;
	} elsif ( $csharptype =~ /^List</ ) {
		$addImport->{"System.Collections.Generic"} = 1;
	} elsif ( $csharptype =~ /^Dictionary</ ) {
		$addImport->{"System.Collections.Generic"} = 1;
	} elsif ( $csharptype =~ /StringBuilder/ ) {
		$addImport->{"System.Text"} = 1;
	} 
}

sub generateVirtualMethod($$$$$)
{
    # Generating methods for $class.
    # $m: method node. $methodClass: the node of the class in which the method is really declared
    # (can be different from $class when the method comes from a super class)
    # This is important because of $allMethods, which has no entry for class::method in that case.

    my( $classNode, $signature, $m, $methodClass, $addImport ) = @_;
    my $methodCode = '';                    # output
    my $returnType = $m->{ReturnType};
    return ('', '') if $returnType eq '~'; # skip destructors

    my $className = $classNode->{astNodeName};
    my $flags = $m->{Flags};
    my @argList = @{$m->{ParamList}};

    print "generateVirtualMethod $className: $signature  ($m->{Access})\n" if ($debug);

    # Detect objects returned by value
    checkImportsForObject( $returnType, $addImport, $classNode ) if ($returnType ne 'void');

    # Generate a matching virtual method in the x_ class
    $methodCode .= "\t\tvirtual $returnType $m->{astNodeName}(";
    my $i = 0;
    foreach my $arg ( @argList ) {
        $methodCode .= ", " if $i++;
        $methodCode .= $arg->{ArgType};
        $methodCode .= " x$i";

	# Detect objects passed by value
	checkImportsForObject( $arg->{ArgType}, $addImport, $classNode );
    }
    $methodCode .= ") ";
    $methodCode .= "const " if ($flags =~ "c");
    $methodCode .= "\{\n";

    # Now the code of the method
    my $this = $classNode->{BindingDerives} > 0 ? "this" : "xthis";

    $i++; # Now the number of args
    $methodCode .= "\tSmoke::StackItem x[$i];\n";
    $i = 1;
    for my $arg (@argList) {
	$methodCode .= "\t";
	$methodCode .= coerce_type("x[$i]", "x$i", $arg->{ArgType}, 0);
	$i++;
    }

    my $sig = $methodClass->{astNodeName} . "::" . $signature;
    my $idx = $allMethods{$sig};
#    die "generateVirtualMethod: $className: No method found for $sig\n" if !defined $idx;
	if ( !defined $idx ) {
		print STDERR "generateVirtualMethod: $className: No method found for $sig\n";
		return "";
	}
	
    if($flags =~ "p") { # pure virtual
	$methodCode .= "\t${libname}_Smoke->binding->callMethod($idx, (void*)$this, x, true /*pure virtual*/);\n";
    } else {
	$methodCode .= "\tif(${libname}_Smoke->binding->callMethod($idx, (void*)$this, x)) ";
    }

    $returnType = undef if ($returnType eq 'void');
    if($returnType) {
	my $arg = $returnType;
	my $it = $arg;
	my $cast;
	my $v = "x[0]";
	my $indent = ($flags =~ "p") ? "\t" : "";
	if($it and exists $typeunion{$it}) {
	    $v .= ".$typeunion{$it}";
	    $cast = "($arg)";
	    $methodCode .= "${indent}return $cast$v;\n";
	} else {
	    $v .= ".s_class";
	    if($arg =~ s/&//) {
		$cast = "*($arg *)";
		$methodCode .= "${indent}return $cast$v;\n";
	    } elsif($arg !~ /\*/) {
		unless($flags =~ "p") {
		    $indent = "\t    ";
		    $methodCode .= "{\n";
		}
		# we assume it's a new thing, and handle it
		$methodCode .= "${indent}$arg *xptr = ($arg *)$v;\n";
		$methodCode .= "${indent}$arg xret(*xptr);\n";
		$methodCode .= "${indent}delete xptr;\n";
		$methodCode .= "${indent}return xret;\n";
		$methodCode .= "\t}\n" unless $flags =~ "p";
	    } else {
		$cast = "($arg)";
		$methodCode .= "${indent}return $cast$v;\n";
	    }
	}
    } else {
	$methodCode .= "\t" if $flags =~ "p";
	$methodCode .= "return;\n";
    }
    if($flags =~ "p") {
	$methodCode .= "\t// ABSTRACT\n";
	$methodCode .= "    }\n";
	return ( $methodCode );
    }
    $methodCode .= "\t";
    if($returnType) {
	$methodCode .= "return ";
    }
    $methodCode .= "$this\->$methodClass->{astNodeName}\::$m->{astNodeName}(";
    $i = 0;
    for my $arg (@argList) {
	$methodCode .= ", " if $i++;
	$methodCode .= "x$i";
    }
    $methodCode .= ");\n";
    $methodCode .= "\t}\n";
    return ( $methodCode );
}


sub mungedArgType($$) {
    my $method = shift;
    my $arg = shift;

#    my $arg = $method->{ParamList}[$idx]->{ArgType};

    my $typeEntry = findTypeEntry( $arg );
    my $realType = $typeEntry->{realType};

    # A scalar ?
    $arg =~ s/\bconst\b//g;
    $arg =~ s/\s+//g;
#print($method->{astNodeName} . " realType: " . $realType  . " arg: $arg\n");

    if($typeEntry->{isEnum} || $allTypes{$realType}{isEnum} || exists $typeunion{$realType} || exists $mungedTypeMap{$arg})
    {
		my $id = '$'; # a 'scalar
		$id = '?' if $arg =~ /[*&]{2}/;
		$id = $mungedTypeMap{$arg} if exists $mungedTypeMap{$arg};
		return $id;
    }

    # A class ?
    if(exists $new_classidx{$realType}) {
		return '#';
    }

    # A non-scalar (reference to array or hash, undef)
    return '?';
}

sub generateMethod($$$$$$$$$)
{
    my( $virtualMethods, $overridenMethods, $classNode, $m, $addImport, $ancestorCount, $csharpMethods, $mainClassNode, $generateConstructors ) = @_;	# input
    my $methodCode = '';	# output
    my $staticMethodCode = '';	# output
    my $interfaceCode = '';	# output
    my $proxyInterfaceCode = '';	# output
    my $signalCode = '';	# output
    my $notConverted = '';	# output
	
    my $name = $m->{astNodeName}; # method name
    
	my @heritage = kdocAstUtil::heritage($classNode);
    my $className  = join( "::", @heritage );
    
	@heritage = kdocAstUtil::heritage($mainClassNode);
	my $mainClassName  = join( "::", @heritage );

	# The csharpClassName might be 'QWidget', while currentClassName is 'QRangeControl'
	# and the QRangeControl methods are being copied into QWidget.    
	my $csharpClassName  = $mainClassNode->{astNodeName};
    my $currentClassName  = $classNode->{astNodeName};
	
	my $firstUnknownArgType = 99;
    my $returnType = $m->{ReturnType};

    # Don't use $className here, it's never the fully qualified (A::B) name for a ctor.
    my $isConstructor = ($name eq $classNode->{astNodeName} );
    my $isDestructor = ($returnType eq '~');

	my $isStatic = $m->{Flags} =~ "s" || $classNode->{NodeType} eq 'namespace';
	my $isPure = $m->{Flags} =~ "p";
	my $fullSignature = methodSignature( $m, $#{$m->{ParamList}} );

    # Don't generate anything for destructors, or constructors for namespaces
    return if $isDestructor 
			or ($classNode->{NodeType} eq 'namespace' and $isConstructor)
#			or (!$mainClassNode->{CanBeInstanciated} and $m->{Access} =~ /protected/)
			or $name =~ /^operator\s*(=|(\[\])|([|&^+-]=)|(!=))\s*$/
			or (!$isStatic and $name =~ /^operator\s*((\+\+)|(--))$/ and $#{$m->{ParamList}} == 0)
			or ($name =~ /^operator\s*\*$/ and $#{$m->{ParamList}} == -1);

	my $item = kdocAstUtil::findRef( $classNode, "Q_PROPERTY_" . $name );
	if ( defined $item 
			&& $item->{NodeType} eq 'property' 
			&& ! $isStatic 
			&& $#{$m->{ParamList}} == -1 
			&& $m->{Flags} !~ 'v'
			&& $m->{Access} !~ /slots|signals/) {
        # If there is a property with the same name, don't bother
		return;
	}
    my $propertyName = $name;
    if ( @{$m->{ParamList}} == 1 && $propertyName =~ /^set(.)(.*)/ ) {
		$propertyName = "Q_PROPERTY_" . lc($1) . $2;
		$item = kdocAstUtil::findRef( $classNode, $propertyName );
		if (	defined $item 
				&& $item->{NodeType} eq 'property' 
				&& ! $isStatic 
				&& $m->{Flags} !~ 'v'
				&& $m->{Access} !~ /slots|signals/ ) 
		{
        	# If there is a property with the same name, don't bother
			return;
		}
	}

	if ($classNode->{astNodeName} eq $main::globalSpaceClassName) {
		my $sourcename = $m->{Source}->{astNodeName};
		# Only put Global methods which came from sources beginning with q into class Qt
		if ($csharpClassName eq 'Qt' and ( $sourcename !~ /\/q[^\/]*$/ or $sourcename =~ /string.h$/ or $sourcename =~ /qwt/ )) {
			return;
		}
		# ..and any other global methods into KDE
		if ($csharpClassName eq 'KDE' and $m->{Source}->{astNodeName} =~ /\/q[^\/]*$/) {
			return;
		}

		if ( $sourcename !~ s!.*(kio/|kparts/|dom/|kabc/|ksettings/|kjs/|ktexteditor/|kdeprint/|kdesu/)(.*)!$1$2!m ) {
			$sourcename =~ s!.*/(.*)!$1!m;
		}
		if ( $sourcename eq '' ) {
			return;
		}
	}
	
    if ($returnType eq 'void') {
    	$returnType = undef;
	} else {
	    # Detect objects returned by value
    	checkImportsForObject( $returnType, $addImport, $classNode );
	}
	
	my $hasDuplicateSignature = 0;
	
    return if ( $m->{SkipFromSwitch} && $m->{Flags} !~ "p" ); 

    my $argId = 0;

    my @argTypeList=();
    my @csharpArgTypeList=();
    my @csharpArgTypeOnlyList = ();
	
    foreach my $arg ( @{$m->{ParamList}} ) {
		$argId++;
	
		if ( $arg->{ArgName} =~ /^ref$|^super$|^int$|^params$|^env$|^cls$|^obj$|^byte$|^event$|^base$|^object$|^in$|^out$|^checked$|^delegate$|^string$|^interface$|^override$|^lock$/ ) {
			$arg->{ArgName} = "";
		}

		if ( $arg->{ArgName} =~ /^short$|^long$/ ) {
			# Oops looks like a parser error
			$arg->{ArgType} = $arg->{ArgName};
			$arg->{ArgName} = "";
		}
	
	print STDERR "  Param ".$arg->{astNodeName}." type: ".$arg->{ArgType}." name:".$arg->{ArgName}." default: ".$arg->{DefaultValue}." csharp: ".cplusplusToCSharp($arg->{ArgType})."\n" if ($debug);
		
	my $argType = $arg->{ArgType};
	my $csharpArgType;
	my $csharpArg;
	my $argName;
	
	if ( cplusplusToCSharp($argType) eq "" && $firstUnknownArgType > $argId ) {
		$firstUnknownArgType = $argId;
	}
	
	$csharpArg = ($arg->{ArgName} eq "" ? "arg" . $argId : $arg->{ArgName});
	$csharpArgType = cplusplusToCSharp($argType);

#	if ( $csharpArgType =~ /StringBuilder/ && $classNode->{astNodeName} ne $main::globalSpaceClassName) {
#		$addImport->{"System.Text"} = 1;
#	}
	if ( $classNode->{astNodeName} eq 'Qt' or $classNode->{astNodeName} eq 'KDE' ) {
		$addImport->{"System.Collections.Generic"} = 1;
	}

	push @argTypeList, $argType;
	push @csharpArgTypeOnlyList, $csharpArgType;
	push @csharpArgTypeList, $csharpArgType  . " " . $csharpArg;
	
	# Detect objects passed by value
	if ($classNode->{astNodeName} ne $main::globalSpaceClassName) {
		checkImportsForObject( $argType, $addImport, $classNode );
	}
    }

	if ( $name eq 'QApplication' or ($csharpClassName eq 'KCmdLineArgs' and $name eq 'init' and scalar(@csharpArgTypeList) > 1) ) {
		# Junk the 'int argc' parameter
		shift @csharpArgTypeList;
		shift @csharpArgTypeOnlyList;
	}

    my @castedArgList = makeCastedArgList( @argTypeList );

    # We iterate as many times as we have default params
    my $firstDefaultParam = $m->{FirstDefaultParam};
    $firstDefaultParam = scalar(@argTypeList) unless defined $firstDefaultParam;
    my $iterationCount = scalar(@argTypeList) - $firstDefaultParam;
	
	my $csharpReturnType = cplusplusToCSharp($m->{ReturnType});
	$csharpReturnType =~ s/^(out) |(ref) //;
	$csharpReturnType =~ s/StringBuilder/string/;
	
	if ($m->{ReturnType} =~ /^int\&/) {
		$csharpReturnType = 'int';
	}
	
	if ($csharpReturnType eq "") {
		$firstUnknownArgType = 0;
	}

    print STDERR "  ". ($iterationCount+1). " iterations for $name\n" if ($debug);
	
	my $csharpSignature = csharpMethodSignature( $m, @argTypeList );
	
	if ( defined $csharpMethods->{$csharpSignature} ) {
		$hasDuplicateSignature = 1;
	}
	
	my $docnode = $m->{DocNode};
	if ( $firstUnknownArgType >= 0 && $m->{Access} !~ /signals/ && ! $hasDuplicateSignature
		&& $classNode->{astNodeName} ne $main::globalSpaceClassName
		&& defined $docnode && ($generateConstructors || !$isConstructor) ) 
	{
		my $csharpdocComment = printCSharpdocComment( $docnode, "", "\t\t/// ", "" );
		if ($isStatic) {
			$staticMethodCode .=  $csharpdocComment unless $csharpdocComment =~ /^\s*$/;
		} else {
			$methodCode .=  $csharpdocComment unless $csharpdocComment =~ /^\s*$/;
		}
	}

    while($iterationCount >= 0) {
	
	$csharpMethods->{$csharpSignature} = 1;

	local($") = ",";
	my $argsSignature = argsSignature( $m, $#argTypeList );
	my $signature = methodSignature( $m, $#argTypeList );
	my $mungedSignature = mungedSignature( $m, $#argTypeList );
	my $mungedMethod = $m->{astNodeName} . $mungedSignature;
	my $csharpparams = join( ", ", @csharpArgTypeList );
	
	# Ignore any methods in QGlobalSpace except those that are operator methods
	# with at least one arg that is the type of the current class
	if (	$classNode->{astNodeName} eq $main::globalSpaceClassName
			&& $csharpClassName ne 'Qt'
			&& $csharpClassName ne 'KDE'
			&& !(	$name =~ /^operator.*/ 
					&& $name ne 'operator<<' 
					&& $name ne 'operator>>'
					&& $csharpparams =~ /$csharpClassName / ) ) 
	{
		;
	} elsif ($firstUnknownArgType <= scalar(@argTypeList) || $hasDuplicateSignature || ($name =~ /^qObject$/) || $m->{Access} =~ /dcop/ || $name =~ ?[|&*/+^-]=? ) {
		if ( $firstUnknownArgType <= scalar(@argTypeList) || $m->{Access} =~ /dcop/  || $name =~ ?[|&*/+^-]=? ) {
			my $failedConversion = "\t\t// " . $m->{ReturnType} . " $name(@castedArgList[0..$#argTypeList]); >>>> NOT CONVERTED\n";
			if ( $m->{Access} =~ /signals/ ) {
				$signalCode .= $failedConversion;
			} else {
				$notConverted .= $failedConversion;
			}
		}
	} else {

		if ( $csharpReturnType =~ s/string\[\]/ArrayList/ ) {
			$addImport->{"System.Collections"} = 1;
		}

		if ( $csharpReturnType =~ s/string\[\]/List<string>/ ) {
			$addImport->{"System.Collections.Generic"} = 1;
		}
	    
		my $cplusplusparams;
		my $i = 0;
		for my $arg (@argTypeList) {
			$cplusplusparams .= "," if $i++;
			$cplusplusparams .= "arg" . $i;
		}

		my $access = $m->{Access};
		$access =~ s/_slots//;

		if ($isConstructor) {
			if ( $generateConstructors ) {
#				$proxyInterfaceCode .= "\t\t\tvoid new$csharpClassName($csharpparams);\n";
				$methodCode .= "\t\tpublic $csharpClassName($csharpparams) : this((Type) null) {\n";
				$methodCode .= "\t\t\tCreateProxy();\n";
				$methodCode .= smokeInvocation("interceptor", \@csharpArgTypeList, "void", $mungedMethod, $signature, $addImport);
				$methodCode .= "\t\t}\n";
			}
		} elsif ( $mainClassNode->{Pure} && $isPure ) {
			if ( $#argTypeList == $#{$m->{ParamList}} ) {
				if ($name =~ /^([a-z])(.*)/) {
					$name = uc($1) . $2;
				}
	
				
				if ($access eq 'public' && ! $isStatic) {
					$interfaceCode .= "\t\t$csharpReturnType $name($csharpparams);\n";
				}
	
				# Only change the method name to start with an upper case letter
				# if it doesn't clash with an enum with the same name
				my $item = kdocAstUtil::findRef( $classNode, $name );
				if ( defined $item && $item->{NodeType} eq 'enum' && $name =~ /^([A-Z])(.*)/) {
					$name = lc($1) . $2;
				}

				$methodCode .= "\t\t\[SmokeMethod(\"$signature\")]\n";

				if (	defined $virtualMethods->{$fullSignature}{method}
						|| defined $overridenMethods->{$signature}{method}
						|| (defined $overridenMethods->{$name}{method}) )
				{
					if ($csharpClassName eq 'QLayout' && $name eq 'SetGeometry') {
						$methodCode .= "\t\t" . $access . " abstract ";
					} else {
						$methodCode .= "\t\t" . $access . " new abstract ";
					}
				} else {
					$methodCode .= "\t\t" . $access . " abstract ";
				}

				$methodCode .= $csharpReturnType;
				$methodCode .= " $name($csharpparams);\n";
			}
		} elsif ( $name =~ /^operator.*/ && $name ne 'operator<<' && $name ne 'operator>>') {
			$name =~ s/ //;
			$name =~ s!([|&*/+^-])=!$1!;
			if ( $csharpSignature =~ s!([|&*/+^-])=!$1! ) {
				if ( $csharpMethods->{$csharpSignature} ) {
					print("dup method found: $csharpSignature\n");
				}
				$csharpMethods->{$csharpSignature} = 1;
			}
			if (!$isStatic) {
				# In C# operator methods must be static, so if the C++ version isn't 
				# static, then add another arg 'lhs', the value of 'this'.
	    		$csharpparams = "$csharpClassName lhs" . ($csharpparams eq "" ? "" : ", ") . $csharpparams;
				unshift @csharpArgTypeList, "$csharpClassName lhs";
				unshift @csharpArgTypeOnlyList, $csharpClassName;
				$csharpSignature =~ s/\(/($csharpClassName, /;
				$csharpSignature =~ s/, \)/)/;
				if ( $csharpMethods->{$csharpSignature} ) {
					print("dup method found: $csharpSignature\n");
				}
				$csharpMethods->{$csharpSignature} = 1;
			}

			if ( $classNode->{astNodeName} ne $main::globalSpaceClassName
				|| (@csharpArgTypeOnlyList[0] eq $csharpClassName || @csharpArgTypeOnlyList[1] eq $csharpClassName) )
			{
				$proxyInterfaceCode .= "\t\t\t$csharpReturnType $operatorNames{$name}($csharpparams);\n";
	    		
				$staticMethodCode .= "\t\t" . $access . " static ";
				$staticMethodCode .= $csharpReturnType;

	    		$staticMethodCode .= " $name($csharpparams) \{\n";
	    		$staticMethodCode .= smokeInvocation("staticInterceptor", \@csharpArgTypeList, $csharpReturnType, $mungedMethod, $signature, $addImport);
		    	$staticMethodCode .= "\t\t}\n";
			}
	    		
			if (	$name =~ /operator==/ 
					&& ( @csharpArgTypeOnlyList[0] eq $csharpClassName || @csharpArgTypeOnlyList[1] eq $csharpClassName )
					&& $csharpClassName ne 'Qt' 
					&& $csharpClassName ne 'KDE' )
			{
				# Add a 'operator!=' method defined in terms of 'operator=='
				$staticMethodCode .= "\t\t" . $access . " static bool";
	    		$staticMethodCode .= " operator!=($csharpparams) \{\n";
	    		
				$staticMethodCode .= "\t\t\treturn ";
	    		$staticMethodCode .= "!(bool) staticInterceptor.Invoke(\"$mungedMethod\", \"$signature\", typeof($csharpReturnType)". smokeInvocationArgList(\@csharpArgTypeList) . ");\n";
		    	$staticMethodCode .= "\t\t}\n";

				if (!defined $mainClassNode->{HasEquals}) {
					$methodCode .= "\t\tpublic override bool Equals(object o) \{\n";
					$methodCode .= "\t\t\tif (!(o is $csharpClassName)) { return false; }\n";
					
					$methodCode .= "\t\t\treturn this == ($csharpClassName) o;\n";
					$methodCode .= "\t\t}\n";
					
					$methodCode .= "\t\tpublic override int GetHashCode() \{\n";
					$methodCode .= "\t\t\treturn interceptor.GetHashCode();\n";
					$methodCode .= "\t\t}\n";
                    $mainClassNode->{HasEquals} = 1;
				}
			}

			if (	$name =~ /^operator\s*<$/ 
					&& $classNode->{astNodeName} ne $main::globalSpaceClassName )
			{
				my $item = kdocAstUtil::findRef( $classNode, "operator>" );
				if (! defined $item || $item->{Parent}->{astNodeName} eq 'Global') {
					$staticMethodCode .= "\t\t" . $access . " static bool";
					$staticMethodCode .= " operator>($csharpparams) \{\n";
					
					$staticMethodCode .= "\t\t\treturn ";
					$staticMethodCode .= "!(bool) staticInterceptor.Invoke(\"$mungedMethod\", \"$signature\", typeof($csharpReturnType)". smokeInvocationArgList(\@csharpArgTypeList) . ")\n";
					$staticMethodCode .= "\t\t\t\t\t\t&& !(bool) staticInterceptor.Invoke(\"operator==$mungedSignature\", \"operator==$argsSignature\", typeof($csharpReturnType)". smokeInvocationArgList(\@csharpArgTypeList) . ");\n";
					$staticMethodCode .= "\t\t}\n";
				}
			}
	    } else {
			if ( $access eq 'public' or $access eq 'protected' ) {
				if (	($csharpClassName eq 'QHeaderView' && $name =~ /^scrollTo$|^indexAt$|^visualRect$/)
						|| ($csharpClassName eq 'QSvgGenerator' && $name =~ /^paintEngine$/)
						|| ($csharpClassName eq 'QGridLayout' && $name =~ /^addItem$/) )
				{
					# These methods are public in some places, but protected in others,
					# so make them all public.
					$access = "public";
				}

				if ($name eq 'operator<<' || $name eq 'operator>>') {
					# 'operator<<' and 'operator>>' can only have int types as the second
					# arg in C#, so convert them to op_read() and op_write() calls
					$name = $operatorNames{$name}
				} elsif ($name =~ /^([a-z])(.*)/) {
					if ($name ne 'type') {
						$name = uc($1) . $2;
					}

					# Only constructors can have the same name as the class
					if ( $name eq $csharpClassName || $name eq 'Transition' ) {
						$name = lc($1) . $2;
					}

					# Only change the method name to start with an upper case letter
					# if it doesn't clash with an enum with the same name
					my $item = kdocAstUtil::findRef( $classNode, $name );
					if ( defined $item && $item->{NodeType} eq 'enum' && $name =~ /^([A-Z])(.*)/) {
						$name = lc($1) . $2;
					}

					$item = kdocAstUtil::findRef( $classNode, "Q_PROPERTY_" . $m->{astNodeName} );
					if ( defined $item && $item->{NodeType} eq 'property' ) {
        				# If there is a property with the same name, start with lower case 
						$name = lc($1) . $2;
					}

					if ($classNode->{astNodeName} eq 'QIODevice' and $name eq 'State') {
						$name = 'state';
					}
				}
				
				if ($access eq 'public' && ! $isStatic) {
	    			$interfaceCode .= "\t\t$csharpReturnType $name($csharpparams);\n";
				}
					
				if (($isStatic or $classNode->{NodeType} eq 'namespace')) {
	    			$proxyInterfaceCode .= "\t\t\t$csharpReturnType $name($csharpparams);\n";
				}

				if ( $m->{Access} =~ /_slots/ && !$isStatic )  {
					$methodCode .= "\t\t[Q_SLOT(\"". $m->{ReturnType} . " $signature" . "\")]\n";
				}

				my $overridenVirtualMethod = $virtualMethods->{$signature}{method};
				my $overridenMethod = $overridenMethods->{$signature}{method};

                if (defined $overridenVirtualMethod || $m->{Flags} =~ "v") {
					$methodCode .= "\t\t\[SmokeMethod(\"$signature\")]\n";
				}

				if ($isStatic or $classNode->{NodeType} eq 'namespace') {
					$staticMethodCode .= "\t\t$access static ";
				} else {
					$methodCode .= "\t\t$access ";		
				}
	
				if (	($name eq 'ToString' && $csharpparams eq '')
						|| (	($csharpClassName eq 'QGraphicsSvgItem' || $csharpClassName eq 'QGraphicsTextItem')
								&& ($name eq 'Children' || $name eq 'InputMethodQuery') ) )
				{
					# Tricky. QGraphicsSvgItem and QGraphicsTextItem inherit a 'children()' method from both
					# of their parents, and so is it resolved on the return type in C++?
					$methodCode .= "new ";
				} elsif ( defined $overridenVirtualMethod ) {
					# Only change the method name to start with an upper case letter
					# if it doesn't clash with an enum with the same name
					my $overrideClass = $virtualMethods->{$fullSignature}{class};

					# Special case looking for QStyle name clashes with methods/enums
					if ($csharpClassName =~ /Style/) {
						$overrideClass = kdocAstUtil::findRef( $rootnode, "QStyle" );
					}

					my $item = kdocAstUtil::findRef( $overrideClass, $name );
					if ( defined $item && $item->{NodeType} eq 'enum' && $name =~ /^([A-Z])(.*)/) {
						$name = lc($1) . $2;
					}

					$item = kdocAstUtil::findRef( $overrideClass, "Q_PROPERTY_" . $m->{astNodeName} );
					if ( defined $item && $item->{NodeType} eq 'property' ) {
        				# If there is a property with the same name, start with lower case 
						$name = lc($1) . $2;
					}

					if (	$overridenVirtualMethod->{Flags} !~ "p"
							&& (	($overridenVirtualMethod->{Access} =~ /public/ && $m->{Access} =~ /protected/)
									|| ($overridenVirtualMethod->{Access} =~ /protected/ && $m->{Access} =~ /public/) ) ) 
					{
						$methodCode .= "new virtual ";
					} elsif (	!defined $overridenVirtualMethod->{FirstDefaultParam} 
							&& $#argTypeList < $#{$overridenVirtualMethod->{ParamList}} )
					{
						;
					} elsif (	defined $overridenVirtualMethod->{FirstDefaultParam} 
								&& $overridenVirtualMethod->{Flags} =~ "p"
								&& $#argTypeList < $#{$overridenVirtualMethod->{ParamList}} )
					{
						$methodCode .= "virtual ";
					} elsif ( $csharpClassName eq 'QAbstractListModel' && $name eq 'Index' && $#argTypeList == 0) {
						;
					} elsif (	$ancestorCount == 0
								|| (	$ancestorCount > 1
										&& defined $interfacemap{$overridenVirtualMethod->{Parent}->{astNodeName}} ) )
					{
						$methodCode .= "virtual ";
					} elsif ($overridenVirtualMethod->{Flags} =~ "p") {
						$methodCode .= "override ";
					} else {
 						$methodCode .= "override ";
					}
				} elsif ($m->{Flags} =~ "v") {
					if (defined $overridenMethods->{$name}{method}) {
						$methodCode .= "new virtual ";
					} else {
						$methodCode .= "virtual ";
					}
				} elsif ( defined $overridenMethod ) {
					if ( $isStatic ) {
						$staticMethodCode .= "new ";
					} elsif (	$ancestorCount == 1 
								|| !defined $interfacemap{$overridenMethod->{Parent}->{astNodeName}} ) 
					{
						$methodCode .= "new ";
					}
				} elsif ( defined $overridenMethods->{$name}{method} ) {
					$methodCode .= "new ";
				}

				if (	$name eq 'Exec' 
						&& ($csharpClassName eq 'QApplication' || $csharpClassName eq 'QCoreApplication') ) 
				{
					$staticMethodCode .= $csharpReturnType;
	    			$staticMethodCode .= " $name($csharpparams) \{\n";
    				$staticMethodCode .= "\t\t\tint result = (int) staticInterceptor.Invoke(\"exec\", \"exec()\", typeof(int));\n";
    				$staticMethodCode .= "\t\t\tQyoto.SetApplicationTerminated();\n";
    				$staticMethodCode .= "\t\t\treturn result;\n";
		    		$staticMethodCode .= "\t\t}\n";
				} elsif ($isStatic or $classNode->{NodeType} eq 'namespace') {
					$staticMethodCode .= $csharpReturnType;
	    			$staticMethodCode .= " $name($csharpparams) \{\n";
	    			$staticMethodCode .= smokeInvocation("staticInterceptor", \@csharpArgTypeList, $csharpReturnType, $mungedMethod, $signature, $addImport);
		    		$staticMethodCode .= "\t\t}\n";
				} else {
					$methodCode .= $csharpReturnType;
	    			$methodCode .= " $name($csharpparams) \{\n";
	    			$methodCode .= smokeInvocation("interceptor", \@csharpArgTypeList, $csharpReturnType, $mungedMethod, $signature, $addImport);
		    		$methodCode .= "\t\t}\n";
				}
			} else {
				if ( $access =~ /signals/ )  {
					if ($name =~ /^([a-z])(.*)/) {
						$name = uc($1) . $2;
					}
					my $docnode = $m->{DocNode};
					if ( defined $docnode ) {
						my $csharpdocComment = printCSharpdocComment( $docnode, "", "\t\t/// ", "" );
						$signalCode .=  $csharpdocComment unless $csharpdocComment =~ /^\s*$/;
					}
					$signalCode .= "\t\t[Q_SIGNAL(\"" . $m->{ReturnType} . " $signature" . "\")]\n";
	    			$signalCode .= "\t\tvoid $name($csharpparams);\n";
				}
			}
	    }
    }

	pop @argTypeList;
	pop @csharpArgTypeList;
	pop @csharpArgTypeOnlyList;
	
	$csharpSignature = csharpMethodSignature( $m, @argTypeList );
	$hasDuplicateSignature = (defined $csharpMethods->{$csharpSignature} ? 1 : 0);
	
	$methodNumber++;
	$iterationCount--;
    } # Iteration loop

    return ( $methodCode, $staticMethodCode, $interfaceCode, $proxyInterfaceCode, $signalCode, $notConverted );
}


sub generateEnum($$$)
{
    my( $classNode, $m, $generateAnonymous ) = @_;	# input
    my $methodCode = '';	# output

    my @heritage = kdocAstUtil::heritage($classNode);
    my $className  = join( "::", @heritage );
    my $csharpClassName  = $classNode->{astNodeName};

	if ( ($generateAnonymous and $m->{astNodeName} ) or (! $generateAnonymous and ! $m->{astNodeName}) ) {
		return;
	}
	
	if ( defined $m->{DocNode} ) {
		my $csharpdocComment = printCSharpdocComment( $m->{DocNode}, "", "\t\t/// ", "" );
		$methodCode .=  $csharpdocComment unless $csharpdocComment =~ /^\s*$/;
	}
	
	# In C# enums must have names, so anonymous C++ enums become constants
	if (! $m->{astNodeName}) {
		return generateConst($classNode, $m, $generateAnonymous);
	}
	
	$m->{astNodeName} =~ /(.)(.*)/;

	my $item = kdocAstUtil::findRef( $classNode, lc($1) . $2 );
	if ( $m->{astNodeName} eq 'Type') {
		# Enums and capitalized method names share the same namespace in C#, so add
		# 'E_' to the front to avoid a clash.
		$methodCode .= "\t\tpublic enum TypeOf {\n";
	} elsif ( $m->{astNodeName} eq 'ConversionFlag'
				|| ($classNode->{astNodeName} eq 'QTextOption' and $m->{astNodeName} eq 'Flag')
				|| ($classNode->{astNodeName} eq 'Qt' and $m->{astNodeName} eq 'Modifier')
				|| ($classNode->{astNodeName} eq 'QGraphicsItem' and $m->{astNodeName} eq 'Extension')
				|| ($classNode->{astNodeName} eq 'QAccessible' and $m->{astNodeName} eq 'StateFlag')
				|| ($classNode->{astNodeName} eq 'Qt' and $m->{astNodeName} eq 'KeyboardModifier') )
	{
		$methodCode .= "\t\tpublic enum " . $m->{astNodeName} . " : long {\n";
	} else {
		$methodCode .= "\t\tpublic enum " . $m->{astNodeName} . " {\n";
	}
	
	my @enums = split(",", $m->{Params});
	my $enumCount = 0;
	foreach my $enum ( @enums ) {
		# C++ enums share a global namespace, but C# names are
		# unique within the enum, so special case these ones
		$enum =~ s/Qt::DecorationRole/Qt.ItemDataRole.DecorationRole/;
		$enum =~ s/Qt::UserRole/Qt.ItemDataRole.UserRole/;
		$enum =~ s/Qt::MetaModifier/Qt.KeyboardModifier.MetaModifier/;
		$enum =~ s/Qt::ShiftModifier/Qt.KeyboardModifier.ShiftModifier/;
		$enum =~ s/Qt::ControlModifier/Qt.KeyboardModifier.ControlModifier/;
		$enum =~ s/Qt::AltModifier/Qt.KeyboardModifier.AltModifier/;
		if ($m->{astNodeName} ne 'KeyboardModifier') {
			$enum =~ s/KeyboardModifierMask/Qt.KeyboardModifier.KeyboardModifierMask/;
		}
		if ($m->{astNodeName} ne 'PolicyFlag') {
			$enum =~ s/GrowFlag/PolicyFlag.GrowFlag/;
			$enum =~ s/ExpandFlag/PolicyFlag.ExpandFlag/;
			$enum =~ s/ShrinkFlag/PolicyFlag.ShrinkFlag/;
			$enum =~ s/IgnoreFlag/PolicyFlag.IgnoreFlag/;
		}

		$enum =~ s/\s//g;
		$enum =~ s/::/./g;
		$enum =~ s/::([a-z])/./g;
		$enum =~ s/\(mode_t\)//;
		$enum =~ s/internal/_internal/;
		$enum =~ s/fixed/_fixed/;
		$enum =~ s/sizeof\(void\*\)/4/;
		if ( $enum =~ /(.*)=([-0-9]+)$/ ) {
			$methodCode .= "\t\t\t$1 = $2,\n";
			$enumCount = $2;
			$enumCount++;
		} elsif ( $enum =~ /(.*)=(.*)/ ) {
			my $name = $1;
			my $value = $2;
			$value =~ s/^MAX_INT$/2147483647/;
			$value =~ s/^SO_/QStyleOption.OptionType.SO_/;
			if ($classNode->{astNodeName} eq 'QStyleHintReturn' || $classNode->{astNodeName} eq 'QStyleHintReturnMask') {
				$value =~ s/^SH_/QStyleHintReturn.HintReturnType.SH_/;
			} elsif ($classNode->{astNodeName} eq 'QStyle') {
				$value =~ s/^SH_/QStyle.StyleHint.SH_/;
			}

			$methodCode .= "\t\t\t$name = $value,\n";
			if ($value =~ /(0xf0000000)|(0xffffffff)/) {
				$methodCode =~ s/enum ((E_)?[^\s]*)/enum $1 : uint/;
			}
		} else {
			$methodCode .= "\t\t\t$enum = $enumCount,\n";
			$enumCount++;
		}
	}
		
	$methodCode .= "\t\t}\n";
	$methodNumber++;
		
    return ( $methodCode );
}

sub generateConst($$$)
{
    my( $classNode, $m, $generateAnonymous ) = @_;	# input
    my $methodCode = '';	# output

    my @heritage = kdocAstUtil::heritage($classNode);
    my $className  = join( "::", @heritage );
    my $csharpClassName  = $classNode->{astNodeName};
	
	my @enums = split(",", $m->{Params});
	my $enumCount = 0;
	foreach my $enum ( @enums ) {
		$enum =~ s/\s//g;
		$enum =~ s/::/./g;
		$enum =~ s/\(mode_t\)//;
		$enum =~ s/internal/_internal/;
		$enum =~ s/fixed/_fixed/;
		$enum =~ s/IsActive/_IsActive/;

		if ($enum =~ /(.*)=(0[xX][fF][0-9a-fA-F]{7})$/) {
			$methodCode .= "\t\tpublic const long $1 = $2;\n";
		} elsif ( $enum =~ /(.*)=([-0-9]+)$/ ) {
			if (	$1 eq 'Type' 
					&& $classNode->{astNodeName} ne 'QGraphicsItem' 
					&& $classNode->{astNodeName} ne 'QGraphicsTextItem' 
					&& $classNode->{astNodeName} ne 'QGraphicsSvgItem' )
  			{
				$methodCode .= "\t\tpublic new const int $1 = $2;\n";
			} else {
				$methodCode .= "\t\tpublic const int $1 = $2;\n";
			}
			$enumCount = $2;
			$enumCount++;
		} elsif ( $enum =~ /(.*)=.*static_cast<.*>\((.*)\)/ ) {
			$methodCode .= "\t\tpublic const int $1 = $2;\n";
		} elsif ( $enum =~ /(.*)=(.*)/ ) {
			my $name = $1;
			my $value = $2;
			$value =~ s/^MAX_INT$/2147483647/;
			$value =~ s/^SO_/(int) QStyleOption.OptionType.SO_/;
			$value =~ s/^SH_/(int) QStyleHintReturn.HintReturnType.SH_/;
			$methodCode .= "\t\tpublic const int $name = $value;\n";
		} else {
			$methodCode .= "\t\tpublic const int $enum = $enumCount;\n";
			$enumCount++;
		}
	}
		
	$methodCode .= "\n";
	$methodNumber++;
		
    return ( $methodCode );
}

sub generateVar($$$)
{
    my( $classNode, $m, $addImport ) = @_;	# input
    my $methodCode = '';	# output
    my $interfaceCode = '';	# output
    my $proxyInterfaceCode = '';	# output

    my @heritage = kdocAstUtil::heritage($classNode);
    my $className  = join( "::", @heritage );
    my $csharpClassName  = $classNode->{astNodeName};

    my $name = $m->{astNodeName};
    my $varType = $m->{Type};
    $varType =~ s/static\s//;
    $varType =~ s/const\s+(.*)\s*&/$1/;
    $varType =~ s/\s*$//;
    my $fullName = "$className\::$name";
	$varType = cplusplusToCSharp($varType);
	if (!defined $varType) {
    	return ( $methodCode, $interfaceCode, $proxyInterfaceCode );
	}

    checkImportsForObject( $varType, $addImport, $classNode );

    my $propertyName = $name;
	if ( $propertyName =~ /^([a-z])(.*)/) {
		$propertyName = uc($1) . $2;
	}

	if ( $m->{Flags} =~ "s" ) {	
    	$methodCode .= "\t\tpublic static $varType $propertyName() {\n";
    	$methodCode .= "\t\t\treturn ($varType) staticInterceptor.Invoke(\"$name\", \"$name()\", typeof($varType));\n";
    	$methodCode .= "\t\t}\n";
		$proxyInterfaceCode .= "\t\t\t$varType $name();\n";
	} else {
    	$methodCode .= "\t\tprivate $varType $name() {\n";
    	$methodCode .= "\t\t\treturn ($varType) interceptor.Invoke(\"$name\", \"$name()\", typeof($varType));\n";
    	$methodCode .= "\t\t}\n";
		$methodCode .= "\t\tpublic $varType $propertyName {\n";
		$methodCode .= "\t\t\tget { return $name(); }\n";
		$methodCode .= "\t\t}\n";
		$interfaceCode .= "\t\t". cplusplusToCSharp($varType) . " $name();\n";
	}

    $methodNumber++;
    return ( $methodCode, $interfaceCode, $proxyInterfaceCode );
}

sub generateProperty($$$$)
{
    my( $overridenMethods, $classNode, $m, $addImport ) = @_;	# input
    my $methodCode = '';	# output

    my @heritage = kdocAstUtil::heritage($classNode);
    my $className  = join( "::", @heritage );
    my $csharpClassName  = $classNode->{astNodeName};

    my $name = $m->{astNodeName};

	my $resolvedType = kalyptusDataDict::resolveType($m->{Type}, $classNode, $rootnode);
    my $propertyType = cplusplusToCSharp( $resolvedType );
    if ( ! defined $propertyType ) {
    	return ( $methodCode );
	}

	checkImportsForObject( $m->{Type}, $addImport, $classNode );

    $name =~ s/Q_PROPERTY_//;
    my $propertyName = $name;
	if ( $propertyName =~ /^([a-z])(.*)/ && $propertyName ne 'icon') {
		$propertyName = uc($1) . $2;
	}

	# Only change the method name to start with an upper case letter
	# if it doesn't clash with an enum with the same name
	my $uppercaseItem = kdocAstUtil::findRef( $classNode, $propertyName );
	if (defined $uppercaseItem && $uppercaseItem->{NodeType} eq 'enum' && $propertyName =~ /^([A-Z])(.*)/) {
		$propertyName = lc($1) . $2;
	}

	$methodCode .= "\t\t[Q_PROPERTY(\"$resolvedType\", \"$name\")]\n";

	if ( defined $overridenMethods->{$propertyName}{method} ) {
		$methodCode .= "\t\tpublic new $propertyType $propertyName {\n";
	} else {
		$methodCode .= "\t\tpublic $propertyType $propertyName {\n";
	}

	if ( defined $m->{READ} ) {
		$methodCode .= "\t\t\tget { return ($propertyType) interceptor.Invoke(\"$m->{READ}\", \"$m->{READ}()\", typeof($propertyType)); }\n";
	}

	if ( defined $m->{WRITE} ) {
		$methodCode .= "\t\t\tset { interceptor.Invoke(\"$m->{WRITE}" . mungedArgType($m, $resolvedType) . "\", \"$m->{WRITE}($resolvedType)\", typeof(void), typeof($propertyType), value); }\n";
	}

	$methodCode .= "\t\t}\n";

    return ( $methodCode );
}

## Called by writeClassDoc
sub generateAllMethods($$$$$$)
{
    my ($classNode, $ancestorCount, $csharpMethods, $mainClassNode, $generateConstructors, $addImport) = @_;
    my $methodCode = '';
    my $staticMethodCode = '';
    my $interfaceCode = '';
    my $proxyInterfaceCode = '';
    my $signalCode = '';
    my $notConverted = '';
    my $extraCode = '';
    $methodNumber = 0;
    #my $className = $classNode->{astNodeName};
    my $className = join( "::", kdocAstUtil::heritage($classNode) );
	my $csharpClassName = $mainClassNode->{astNodeName};
	# If the C++ class had multiple inheritance, then the code for all but one of the
	# parents must be copied into the code for csharpClassName. Hence, for QWidget current
	# classname might be QPaintDevice, as its methods are needed in QWidget.
	my $currentClassName = join( ".", kdocAstUtil::heritage($classNode) );

    my $sourcename = $classNode->{Source}->{astNodeName};
    
	if ( $sourcename !~ s!.*(kio/|kparts/|dom/|kabc/|ksettings/|kjs/|ktexteditor/|kdeprint/|kdesu/)(.*)!$1$2!m ) {
    	$sourcename =~ s!.*/(.*)!$1!m;
	}
    die "Empty source name for $classNode->{astNodeName}" if ( $sourcename eq '' );
	
	if ($classNode->{astNodeName} ne $main::globalSpaceClassName) {
#		my $s;
#        for my $sn( @{$classNode->{Sources}} ) {
#    	if ( ($s = $sn->{astNodeName}) !~ s!.*(kio/|kparts/|dom/|kabc/|ksettings/|kjs/|ktexteditor/|kdeprint/|kdesu/)(.*)!$1$2!m ) {
#    		$s =~ s!.*/(.*)!$1!m;
#		}
#		$addInclude->{$s} = 1;
#        }
	}
	
    $addImport->{"Qyoto"} = 1;

	my %virtualMethods;
	allVirtualMethods( $classNode, \%virtualMethods, $classNode );

	my %overridenMethods;
	allOverridenMethods( $classNode, \%overridenMethods, $classNode );
				
    Iter::MembersByType ( $classNode, undef,
			  sub {	my ($classNode, $methodNode ) = @_;
				
	if ( $methodNode->{NodeType} eq 'enum' and $currentClassName eq $csharpClassName ) {
	    my ($code) = generateEnum( $classNode, $methodNode, 0 );
	    $extraCode .= $code;
	}
				}, undef );
	
    # Do all enums first, anonymous ones and then named enums
    Iter::MembersByType ( $classNode, undef,
			  sub {	my ($classNode, $methodNode ) = @_;
				
	if ( $methodNode->{NodeType} eq 'enum' and $currentClassName eq $csharpClassName ) {
	    my ($code) = generateEnum( $classNode, $methodNode, 1 );
	    $extraCode .= $code;
	}
				}, undef );

    # Then all static vars
    Iter::MembersByType ( $classNode, undef,
			  sub {	my ($classNode, $methodNode ) = @_;
				
	if ( $methodNode->{NodeType} eq 'var' and $currentClassName eq $csharpClassName ) {
	    my ($code, $interface, $proxyInterface) = generateVar( $classNode, $methodNode, $addImport );
	    $extraCode .= $code;
	    $interfaceCode .= $interface;
	    $proxyInterfaceCode .= $proxyInterface;
	}
				}, undef );

    # Then all properties
    Iter::MembersByType ( $classNode, undef,
			  sub {	my ($classNode, $methodNode ) = @_;
				
	if ( $methodNode->{NodeType} eq 'property' and $currentClassName eq $csharpClassName ) {
	    my ($code, $interface) = generateProperty( \%overridenMethods, $classNode, $methodNode, $addImport );
	    $extraCode .= $code;
	}
				}, undef );

    # Then all methods
    Iter::MembersByType ( $classNode, undef,
			  sub {	my ($classNode, $methodNode ) = @_;

        if ( $methodNode->{NodeType} eq 'method' ) {
	    	my ($meth, $static, $interface, $proxyInterface, $signals, $notconv) = generateMethod( \%virtualMethods, \%overridenMethods, $classNode, $methodNode, $addImport, $ancestorCount, $csharpMethods, $mainClassNode, $generateConstructors );
	    	$methodCode .= $meth;
	    	$staticMethodCode .= $static;
	    	$interfaceCode .= $interface;
	    	$proxyInterfaceCode .= $proxyInterface;
			$signalCode .= $signals;
			$notConverted .= $notconv;
		}
			      }, undef );

    # Virtual methods
#    if ($classNode->{BindingDerives}) {
#	my %virtualMethods;
#	allVirtualMethods( $classNode, \%virtualMethods );

#	for my $sig (sort keys %virtualMethods) {
#            my ($meth) = generateVirtualMethod( $classNode, $sig, $virtualMethods{$sig}{method}, $virtualMethods{$sig}{class}, \%addImport );
#	    $methodCode .= $meth;
#	}
#    }

    # Destructor
    # "virtual" is useless, if the base class has a virtual destructor then the x_* class too.
    #if($classNode->{HasVirtualDestructor} and $classNode->{HasDestructor}) {
    #	$methodCode .= "    virtual ~$bridgeClassName() {}\n";
    #}
    # We generate a dtor though, because we might want to add stuff into it

    if (	$classNode->{astNodeName} eq $csharpClassName 
			&& $classNode->{HasPublicDestructor} 
			&& $classNode->{CanBeInstanciated}
			&& !$classNode->{Pure} ) 
	{
    	if ( $generateConstructors && $csharpClassName ne 'Qt' ) {
			$methodCode .= "\t\t~$csharpClassName() {\n";
            if ($csharpClassName eq 'QModelIndex') {
				$methodCode .= "\t\t\tQAbstractItemModel.DerefIndexHandle(InternalPointer());\n";
			}
			$methodCode .= "\t\t\tinterceptor.Invoke(\"~$classNode->{astNodeName}\", \"~$classNode->{astNodeName}()\", typeof(void));\n\t\t}\n";
			
			my $overridenMethod = $overridenMethods{~$classNode->{astNodeName}}{method};
			if (	defined $overridenMethod
					&& $classNode->{astNodeName} ne 'QObject'
				 	&& (	$ancestorCount == 1 
							|| !defined $interfacemap{$overridenMethod->{Parent}->{astNodeName}} ) )
			{
				$methodCode .= "\t\tpublic new ";
			} else {
				$methodCode .= "\t\tpublic ";
			}
			$methodCode .= "void Dispose() {\n";
            if ($csharpClassName eq 'QModelIndex') {
				$methodCode .= "\t\t\tQAbstractItemModel.DerefIndexHandle(InternalPointer());\n";
			}
			$methodCode .= "\t\t\tinterceptor.Invoke(\"~$classNode->{astNodeName}\", \"~$classNode->{astNodeName}()\", typeof(void));\n\t\t}\n";
		}
#	die "$className destructor: methodNumber=$methodNumber != case entry=".$classNode->{case}{"~$className()"}."\n"
#	     if $methodNumber != $classNode->{case}{"~$className()"};
			$methodNumber++;
    }
                                                               
    return ( $methodCode, $staticMethodCode, $interfaceCode, $proxyInterfaceCode, $signalCode, $extraCode, $notConverted );
}

# Return 0 if the class has no virtual dtor, 1 if it has, 2 if it's private
sub hasVirtualDestructor($$)
{
    my ( $classNode, $startNode ) = @_;
    my $className = join( "::", kdocAstUtil::heritage($classNode) );
    return if ( $skippedClasses{$className} || defined $interfacemap{$className} );

    my $parentHasIt;
    # Look at ancestors, and (recursively) call hasVirtualDestructor for each
    # It's enough to have one parent with a prot/public virtual dtor
    Iter::Ancestors( $classNode, $rootnode, undef, undef, sub {
                     my $vd = hasVirtualDestructor( $_[0], $_[1] );
                     $parentHasIt = $vd unless $parentHasIt > $vd;
                    } );
    return $parentHasIt if $parentHasIt; # 1 or 2

    # Now look in $classNode - including private methods
    my $doPrivate = $main::doPrivate;
    $main::doPrivate = 1;
    my $result;
    Iter::MembersByType ( $classNode, undef,
		sub {	my ($classNode, $m ) = @_;
			return unless( $m->{NodeType} eq "method" && $m->{ReturnType} eq '~' );

			if ( $m->{Flags} =~ /[vp]/ && $classNode != $startNode) {
			    if ( $m->{Access} =~ /private/ ) {
				$result=2; # private virtual
			    } else {
				$result=1; # [protected or public] virtual
			    }
			}
		},
		undef
	);
    $main::doPrivate = $doPrivate;
    $result=0 if (!defined $result);
    return $result;
}

=head2 allVirtualMethods

	Parameters: class node, dict

	Adds to the dict, for all method nodes that are virtual, in this class and in parent classes :
        {method} the method node, {class} the class node (the one where the virtual is implemented)

=cut

sub allVirtualMethods($$$)
{
    my ( $classNode, $virtualMethods, $startNode ) = @_;
    my $className = join( "::", kdocAstUtil::heritage($classNode) );
    return if ( $skippedClasses{$className} );

    # Look at ancestors, and (recursively) call allVirtualMethods for each
    # This is done first, so that virtual methods that are reimplemented as 'private'
    # can be removed from the list afterwards (below)
    Iter::Ancestors( $classNode, $rootnode, undef, undef, sub {
			 allVirtualMethods( @_[0], $virtualMethods, $startNode );
		     }, undef
		   );

    # Now look for virtual methods in $classNode - including private ones
    my $doPrivate = $main::doPrivate;
    $main::doPrivate = 1;
    Iter::MembersByType ( $classNode, undef,
		sub {	my ($classNode, $m ) = @_;
			# Only interested in methods, and skip destructors
			return unless( $m->{NodeType} eq "method" && $m->{ReturnType} ne '~' );

			my @args = @{ $m->{ParamList} };
			my $last = $m->{FirstDefaultParam};
			$last = scalar @args unless defined $last;
			my $iterationCount = scalar(@args) - $last;
			while($iterationCount >= 0) {
				my $signature = methodSignature( $m, $#args );
				print STDERR $signature . " ($m->{Access})\n" if ($debug);

				# A method is virtual if marked as such (v=virtual p=pure virtual)
				# or if a parent method with same signature was virtual
				if ( $m->{Flags} =~ /[vp]/ or defined $virtualMethods->{$signature} ) {
					if ( $m->{Access} =~ /private/ ) {
						if ( defined $virtualMethods->{$signature} ) { # remove previously defined
							delete $virtualMethods->{$signature};
						}
						# else, nothing, just ignore private virtual method
					} elsif ( $classNode->{astNodeName} ne $startNode->{astNodeName} ) {
						$virtualMethods->{$signature}{method} = $m;
						$virtualMethods->{$signature}{class} = $classNode;
					}
				}

				pop @args;
				$iterationCount--;
			}
		},
		undef
	);
    $main::doPrivate = $doPrivate;
}

sub allOverridenMethods($$$)
{
    my ( $classNode, $overridenMethods, $startNode ) = @_;
    my $className = join( "::", kdocAstUtil::heritage($classNode) );
    return if ( $skippedClasses{$className} );

	my $qtnode;
	if ( $classNode->{astNodeName} eq 'QObject' ) {
		$qtnode = kdocAstUtil::findRef( $rootnode, "Qt" );
		if ( defined $qtnode ) {
			allOverridenMethods( $qtnode, $overridenMethods, $startNode );
		}
	}

    Iter::Ancestors( $classNode, $qtnode, undef, undef, sub {
			 allOverridenMethods( @_[0], $overridenMethods, $startNode );
		     }, undef
		   );

    # Look at ancestors, and (recursively) call allVirtualMethods for each
    # This is done first, so that virtual methods that are reimplemented as 'private'
    # can be removed from the list afterwards (below)
    Iter::Ancestors( $classNode, $rootnode, undef, undef, sub {
			 allOverridenMethods( @_[0], $overridenMethods, $startNode );
		     }, undef
		   );

    # Now look for virtual methods in $classNode - including private ones
    Iter::MembersByType ( $classNode, undef,
		sub {	my ($classNode, $m ) = @_;
			# Only interested in methods, and skip destructors
			return unless( $m->{NodeType} eq "method" or $m->{NodeType} eq "enum" or $m->{NodeType} eq "property" );

			my $signature = methodSignature( $m, $#{$m->{ParamList}} );
			print STDERR $signature . " ($m->{Access})\n" if ($debug);

			if ( $classNode->{astNodeName} ne $startNode->{astNodeName} && $classNode->{astNodeName} ne 'Global' ) {
				if ( $m->{NodeType} eq "enum" ) {
					$overridenMethods->{$m->{astNodeName}}{method} = $m;
					$overridenMethods->{$m->{astNodeName}}{class} = $classNode;
				} elsif ( $m->{NodeType} eq "property" ) {
					my $name = $m->{astNodeName};
					if ( $name =~ s/Q_PROPERTY_([a-z])(.*)// ) {
						$name = uc($1) . $2;
					}

					$overridenMethods->{$name}{method} = $m;
					$overridenMethods->{$name}{class} = $classNode;
				} elsif ( $m->{ReturnType} eq '~' ) {
					if ( ! exists $classNode->{Pure} ) {
						$overridenMethods->{~$startNode->{astNodeName}}{method} = $m;
						$overridenMethods->{~$startNode->{astNodeName}}{class} = $classNode;
					}
				} elsif ( $m->{Access} =~ /private/ ) {
					if ( defined $overridenMethods->{$signature} ) { # remove previously defined
						delete $overridenMethods->{$signature};
					}
					# else, nothing, just ignore private virtual method
				} else {
					my @args = @{ $m->{ParamList} };
					my $last = $m->{FirstDefaultParam};
					$last = scalar @args unless defined $last;
					my $iterationCount = scalar(@args) - $last;
					while($iterationCount >= 0) {
	    				my $signature = methodSignature( $m, $#args );

    					my $propertyName = $m->{astNodeName};
    					if ( @{$m->{ParamList}} == 1 && $propertyName =~ /^set(.)(.*)/ ) {
							$propertyName = "Q_PROPERTY_" . lc($1) . $2;
							my $item = kdocAstUtil::findRef( $classNode, $propertyName );
							if ( defined $item && $item->{NodeType} eq 'property' ) {
        						# If there is a property with the same name, don't bother
								$signature = "";
							}
						}


						if ($signature ne "") {
							$overridenMethods->{$signature}{method} = $m;
							$overridenMethods->{$signature}{class} = $classNode;
						}

						pop @args;
						$iterationCount--;
					}
				}
			}
		},
		undef
	);
}

# Known typedef? If so, apply it.
sub applyTypeDef($)
{
    my $type = shift;
    # Parse 'const' in front of it, and '*' or '&' after it
    my $prefix = $type =~ s/^const\s+// ? 'const ' : '';
    my $suffix = $type =~ s/\s*([\&\*]+)$// ? $1 : '';

    if (exists $typedeflist{$type}) {
	return $prefix.$typedeflist{$type}.$suffix;
    }
    return $prefix.$type.$suffix;
}

# Register type ($1) into %allTypes if not already there
sub registerType($$) {
    my $type = shift;
    #print "registerType: $type\n" if ($debug);

    $type =~ s/\s+const$//; # for 'char* const'
    $type =~ s/\s+const\s*\*$/\*/; # for 'char* const*'

    return if ( $type eq 'void' or $type eq '' or $type eq '~' );
    die if ( $type eq '...' );     # ouch

    # Let's register the real type, not its known equivalent
    #$type = applyTypeDef($type);

    # Enum _value_ -> get corresponding type
    if (exists $enumValueToType{$type}) {
	$type = $enumValueToType{$type};
    }

    # Already in allTypes
    if(exists $allTypes{$type}) {
        return;
    }

    die if $type eq 'QTextEdit::UndoRedoInfo::Type';
    die if $type eq '';

    my $realType = $type;

    # Look for references (&) and pointers (* or **)  - this will not handle *& correctly.
    # We do this parsing here because both the type list and iterproto need it
    if($realType =~ s/&$//) {
	$allTypes{$type}{typeFlags} = 'Smoke::tf_ref';
    }
    elsif($realType ne 'void*' && $realType =~ s/\*$//) {
	$allTypes{$type}{typeFlags} = 'Smoke::tf_ptr';
    }
    else {
	$allTypes{$type}{typeFlags} = 'Smoke::tf_stack';
    }

    if ( $realType =~ s/^const\s+// ) { # Remove 'const'
	$allTypes{$type}{typeFlags} .= ' | Smoke::tf_const';
    }

    # Apply typedefs, and store the resulting type.
    # For instance, if $type was Q_UINT16&, realType will be ushort
    $allTypes{$type}{realType} = applyTypeDef( $realType );

    # In the first phase we only create entries into allTypes.
    # The values (indexes) are calculated afterwards, once the list is full.
    $allTypes{$type}{index} = -1;
    #print STDERR "Register $type. Realtype: $realType\n" if($debug);
}

# Get type from %allTypes
# This returns a hash with {index}, {isEnum}, {typeFlags}, {realType}
# (and {typeId} after the types array is written by writeSmokeDataFile)
sub findTypeEntry($) {
    my $type = shift;
    my $typeIndex = -1;
    $type =~ s/\s+const$//; # for 'char* const'
    $type =~ s/\s+const\s*\*$/\*/; # for 'char* const*'

    return undef if ( $type =~ '~' or $type eq 'void' or $type eq '' );

    # Enum _value_ -> get corresponding type
    if (exists $enumValueToType{$type}) {
	$type = $enumValueToType{$type};
    }

	if ( ! defined $allTypes{$type} ) {
    	print("type not known: $type\n");
		return undef;
	}

#    die "type not known: $type" unless defined $allTypes{$type};
    return $allTypes{ $type };
}

# List of all csharp super-classes for a given class, via single inheritance. 
# Excluding any which are mapped onto interfaces to avoid multiple inheritance.
sub direct_superclass_list($)
{
    my $classNode = shift;
    my @super;
	my $has_ancestor = 0;
	my $direct_ancestor = undef;
	my $name;
	
    Iter::Ancestors( $classNode, $rootnode, undef, undef, sub {
			( $direct_ancestor, $name ) = @_;
			if ($name =~ /QMemArray|QSqlFieldInfoList/) {
				# Template classes, give up for now..
				$has_ancestor = 1;
			} elsif (!defined $interfacemap{$name}) {
				push @super, $direct_ancestor;
				push @super, direct_superclass_list( $direct_ancestor );
				$has_ancestor = 1;
			}
		     }, undef );
	
	if (! $has_ancestor and defined $direct_ancestor) {
		push @super, $direct_ancestor;
    	push @super, direct_superclass_list( $direct_ancestor );
	}
	
	return @super;
}

# List of all super-classes for a given class
sub superclass_list($)
{
    my $classNode = shift;
    my @super;
    Iter::Ancestors( $classNode, $rootnode, undef, undef, sub {
			push @super, @_[0];
			push @super, superclass_list( @_[0] );
		     }, undef );
    return @super;
}

sub is_kindof($$)
{
    my $classNode = shift;
	my $className = shift;
	
	if ($classNode->{astNodeName} eq $className) {
		return 1;
	}
	
	my @superclasses = superclass_list($classNode);
	foreach my $ancestor (@superclasses) {
		if ($ancestor->{astNodeName} eq $className) {
			return 1;
		}
	}
	
	return 0;
}

# Store the {case} dict in the class Node (method signature -> index in the "case" switch)
# This also determines which methods should NOT be in the switch, and sets {SkipFromSwitch} for them
sub prepareCaseDict($) {

     my $classNode = shift;
     my $className = join( "::", kdocAstUtil::heritage($classNode) );
     $classNode->AddProp("case", {});
     my $methodNumber = 0;

     # First look at all enums for this class
     Iter::MembersByType ( $classNode, undef,
		sub {	my ($classNode, $m ) = @_;

	next unless $m->{NodeType} eq 'enum';
	foreach my $val ( @{$m->{ParamList}} ) {
	    my $fullEnumName = "$className\::".$val->{ArgName};
	    print STDERR "Enum: $fullEnumName -> case $methodNumber\n" if ($debug);
	    $classNode->{case}{$fullEnumName} = $methodNumber;
	    $enumValueToType{$fullEnumName} = "$className\::$m->{astNodeName}";
	    $methodNumber++;
	}
		      }, undef );

     # Check for static vars
     Iter::MembersByType ( $classNode, undef,
		sub {	my ($classNode, $m ) = @_;

	    next unless $m->{NodeType} eq 'var';
	    my $name = "$className\::".$m->{astNodeName};			
	    print STDERR "Var: $name -> case $methodNumber\n" if ($debug);
	    $classNode->{case}{$name} = $methodNumber;
	    $methodNumber++;

		      }, undef );


     # Now look at all methods for this class
     Iter::MembersByType ( $classNode, undef,
		sub {	my ($classNode, $m ) = @_;

	next unless $m->{NodeType} eq 'method';
	my $name = $m->{astNodeName};
        my $isConstructor = ($name eq $classNode->{astNodeName} );
	if ($isConstructor and ($m->{ReturnType} eq '~')) # destructor
	{
	    # Remember whether we'll generate a switch entry for the destructor
	    $m->{SkipFromSwitch} = 1 unless ($classNode->{CanBeInstanciated} and $classNode->{HasPublicDestructor});
	    next;
	}

        # Don't generate bindings for protected methods (incl. signals) if
        # we're not deriving from the C++ class. Only take public and public_slots
        my $ok = ( $classNode->{BindingDerives} or $m->{Access} =~ /public/ ) ? 1 : 0;

        # Don't generate bindings for pure virtuals - we can't call them ;)
        $ok = 0 if ( $ok && $m->{Flags} =~ "p" );

        # Bugfix for Qt-3.0.4: those methods are NOT implemented (report sent).
        $ok = 0 if ( $ok && $className eq 'QLineEdit' && ( $name eq 'setPasswordChar' || $name eq 'passwordChar' ) );
        $ok = 0 if ( $ok && $className eq 'QWidgetItem' && $name eq 'widgetSizeHint' );

        if ( !$ok )
        {
	    #print STDERR "Skipping $className\::$name\n" if ($debug);
	    $m->{SkipFromSwitch} = 1;
	    next;
	}

	my @args = @{ $m->{ParamList} };
	my $last = $m->{FirstDefaultParam};
	$last = scalar @args unless defined $last;
	my $iterationCount = scalar(@args) - $last;
	while($iterationCount >= 0) {
	    my $sig = methodSignature( $m, $#args );
	    $classNode->{case}{$sig} = $methodNumber;
	    #print STDERR "prepareCaseDict: registered case number $methodNumber for $sig in $className()\n" if ($debug);
	    pop @args;
	    $iterationCount--;
	    $methodNumber++;
	}
		    }, undef );

    # Add the destructor, at the end
    if ($classNode->{CanBeInstanciated} and $classNode->{HasPublicDestructor}) {
        $classNode->{case}{"~$className()"} = $methodNumber;
	# workaround for ~Sub::Class() being seen as Sub::~Class()
	$classNode->{case}{"~$classNode->{astNodeName}()"} = $methodNumber;
	#print STDERR "prepareCaseDict: registered case number $methodNumber for ~$className()\n" if ($debug);
    }
}

sub writeSmokeDataFile($) {
    my $rootnode = shift;

    # Make list of classes
    my %allImports; # list of all header files for all classes
    my @classlist;
    push @classlist, ""; # Prepend empty item for "no class"
    my %enumclasslist;
    Iter::LocalCompounds( $rootnode, sub {
	my $classNode = $_[0];
	my $className = join( "::", kdocAstUtil::heritage($classNode) );
	
	return if $classNode->{NodeType} eq 'namespace';
	
	push @classlist, $className;
	$enumclasslist{$className}++ if keys %{$classNode->{enumerations}};
	$classNode->{ClassIndex} = $#classlist;
	addImportForClass( $classNode, \%allImports, undef );
    } );

    my %classidx = do { my $i = 0; map { $_ => $i++ } @classlist };

    my $file = "$outputdir/smokedata.cpp";
#    open OUT, ">$file" or die "Couldn't create $file\n";

#    foreach my $incl (sort{ 
#                           return 1 if $a=~/qmotif/;  # move qmotif* at bottom (they include dirty X11 headers)
#                           return -1 if $b=~/qmotif/;
#			   return -1 if substr($a,0,1) eq 'q' and substr($b,0,1) ne 'q'; # move Qt headers on top
#			   return 1 if substr($a,0,1) ne 'q' and substr($b,0,1) eq 'q';			   
#                           $a cmp $b
#                          } keys %allIncludes) {
#	die if $imp eq '';
#	print OUT "import $imp;\n";
#    }	

#    print OUT "\n";

    print STDERR "Writing ${libname}_cast function\n" if ($debug);

    # Prepare descendants information for each class
    my %descendants; # classname -> list of descendant nodes
    Iter::LocalCompounds( $rootnode, sub {
	my $classNode = shift;
	# Get _all_ superclasses (up any number of levels)
	# and store that $classNode is a descendant of $s
	my @super = superclass_list($classNode);
	for my $s (@super) {
	    my $superClassName = join( "::", kdocAstUtil::heritage($s) );
	    Ast::AddPropList( \%descendants, $superClassName, $classNode );
	}
    } );

    # Iterate over all classes, to write the xtypecast function
    Iter::LocalCompounds( $rootnode, sub {
	my $classNode = shift;
	my $className = join( "::", kdocAstUtil::heritage($classNode) );
	# @super will contain superclasses, the class itself, and all descendants
	my @super = superclass_list($classNode);
	push @super, $classNode;
        if ( defined $descendants{$className} ) {
	    push @super, @{$descendants{$className}};
	}
	my $cur = $classidx{$className};
	
	return if $classNode->{NodeType} eq 'namespace';

#	print OUT "      case $cur:\t//$className\n";
#	print OUT "\tswitch(to) {\n";
#	$cur = -1;
#	my %casevalues;
#	for my $s (@super) {
#		my $superClassName = join( "::", kdocAstUtil::heritage($s) );
#		next if !defined $classidx{$superClassName}; # inherits from unknown class, see below
#		next if $classidx{$superClassName} == $cur;    # shouldn't happen in Qt
#		next if $s->kdocAstUtil::inheritsAsVirtual($classNode); # can't cast from a virtual base class
#		$cur = $classidx{$superClassName}; # KDE has MI with diamond shaped cycles (cf. KXMLGUIClient)
#		next if $casevalues{$cur};         # ..so skip any duplicate parents
#		print OUT "\t  case $cur: return (void*)($superClassName*)($className*)xptr;\n";
#		$casevalues{$cur} = 1;
#	}
#	print OUT "\t  default: return xptr;\n";
#	print OUT "\t}\n";
    } );
#    print OUT "      default: return xptr;\n";
#    print OUT "    }\n";
#    print OUT "}\n\n";


    # Write inheritance array
    # Imagine you have "Class : public super1, super2"
    # The inheritlist array will get 3 new items: super1, super2, 0
    my %inheritfinder;  # key = (super1, super2) -> data = (index in @inheritlist). This one allows reuse.
    my %classinherit;   # we store that index in %classinherit{className}
    # We don't actually need to store inheritlist in memory, we write it
    # directly to the file. We only need to remember its current size.
    my $inheritlistsize = 1;

#    print OUT "// Group of class IDs (0 separated) used as super class lists.\n";
#    print OUT "// Classes with super classes have an index into this array.\n";
#    print OUT "static short ${libname}_inheritanceList[] = {\n";
#    print OUT "\t0,\t// 0: (no super class)\n";
    Iter::LocalCompounds( $rootnode, sub {
	my $classNode = shift;
	my $className = join( "__", kdocAstUtil::heritage($classNode) );
	
	return if $classNode->{NodeType} eq 'namespace';
	
	print STDERR "inheritanceList: looking at $className\n" if ($debug);

	# Make list of direct ancestors
	my @super;
	Iter::Ancestors( $classNode, $rootnode, undef, undef, sub {
			     my $superClassName = join( "::", kdocAstUtil::heritage($_[0]) );
			     push @super, $superClassName;
		    }, undef );
	# Turn that into a list of class indexes
	my $key = '';
	foreach my $superClass( @super ) {
	    if (defined $classidx{$superClass}) {
		$key .= ', ' if ( length $key > 0 );
		$key .= $classidx{$superClass};
	    }
	}
	if ( $key ne '' ) {
	    if ( !defined $inheritfinder{$key} ) {
		print OUT "\t";
		my $index = $inheritlistsize; # Index of first entry (for this group) in inheritlist
		foreach my $superClass( @super ) {
		    if (defined $classidx{$superClass}) {
			print OUT "$classidx{$superClass}, ";
			$inheritlistsize++;
		    }
		}
		$inheritlistsize++;
		my $comment = join( ", ", @super );
		print OUT "0,\t// $index: $comment\n";
		$inheritfinder{$key} = $index;
	    }
	    $classinherit{$className} = $inheritfinder{$key};
	} else { # No superclass
	    $classinherit{$className} = 0;
	}
    } );
#    print OUT "};\n\n";


#    print OUT "// These are the xenum functions for manipulating enum pointers\n";
    for my $className (keys %enumclasslist) {
	my $c = $className;
	$c =~ s/::/__/g;
#	print OUT "void xenum_$c\(Smoke::EnumOperation, Smoke::Index, void*&, long&);\n";
    }
#    print OUT "\n";
#    print OUT "// Those are the xcall functions defined in each x_*.cpp file, for dispatching method calls\n";
    my $firstClass = 1;
    for my $className (@classlist) {
	if ($firstClass) {
	    $firstClass = 0;
	    next;
	}
	my $c = $className;   # make a copy
	$c =~ s/::/__/g;
#	print OUT "void xcall_$c\(Smoke::Index, void*, Smoke::Stack);\n";
    }
#    print OUT "\n";

    # Write class list afterwards because it needs offsets to the inheritance array.
#    print OUT "// List of all classes\n";
#    print OUT "// Name, index into inheritanceList, method dispatcher, enum dispatcher, class flags\n";
#    print OUT "static Smoke::Class ${libname}_classes[] = {\n";
    my $firstClass = 1;
    Iter::LocalCompounds( $rootnode, sub {
	my $classNode = shift;
	my $className = join( "__", kdocAstUtil::heritage($classNode) );
	
	return if $classNode->{NodeType} eq 'namespace';

	if ($firstClass) {
	    $firstClass = 0;
	    print OUT "\t{ 0L, 0, 0, 0, 0 }, \t// 0 (no class)\n";
	}
	my $c = $className;
	$c =~ s/::/__/g;
	my $xcallFunc = "xcall_$c";
	my $xenumFunc = "0";
	$xenumFunc = "xenum_$c" if exists $enumclasslist{$className};
	# %classinherit needs Foo__Bar, not Foo::Bar?
	die "problem with $className" unless defined $classinherit{$c};

	my $xClassFlags = 0;
	$xClassFlags .= "|Smoke::cf_constructor" if $classNode->{CanBeInstanciated}; # correct?
	$xClassFlags .= "|Smoke::cf_deepcopy" if $classNode->{CanBeCopied}; # HasCopyConstructor would be wrong (when it's private)
	$xClassFlags .= "|Smoke::cf_virtual" if hasVirtualDestructor($classNode, $classNode) == 1;
	# $xClassFlags .= "|Smoke::cf_undefined" if ...;
	$xClassFlags =~ s/0\|//; # beautify
#	print OUT "\t{ \"$className\", $classinherit{$c}, $xcallFunc, $xenumFunc, $xClassFlags }, \t//$classidx{$className}\n";
    } );
#    print OUT "};\n\n";


#    print OUT "// List of all types needed by the methods (arguments and return values)\n";
#    print OUT "// Name, class ID if arg is a class, and TypeId\n";
#    print OUT "static Smoke::Type ${libname}_types[] = {\n";
    my $typeCount = 0;
    $allTypes{''}{index} = 0; # We need an "item 0"
    for my $type (sort keys %allTypes) {
	$allTypes{$type}{index} = $typeCount;      # Register proper index in allTypes
	if ( $typeCount == 0 ) {
#	    print OUT "\t{ 0, 0, 0 },\t//0 (no type)\n";
	    $typeCount++;
	    next;
	}
	my $isEnum = $allTypes{$type}{isEnum};
	my $typeId;
	my $typeFlags = $allTypes{$type}{typeFlags};
	my $realType = $allTypes{$type}{realType};
	die "$type" if !defined $typeFlags;
#	die "$realType" if $realType =~ /\(/;
	# First write the name
#	print OUT "\t{ \"$type\", ";
	# Then write the classId (and find out the typeid at the same time)
	if(exists $classidx{$realType}) { # this one first, we want t_class for QBlah*
	    $typeId = 't_class';
#	    print OUT "$classidx{$realType}, ";
	}
	elsif($type =~ /&$/ || $type =~ /\*$/) {
	    $typeId = 't_voidp';
#	    print OUT "0, "; # no classId
	}
	elsif($isEnum || $allTypes{$realType}{isEnum}) {
	    $typeId = 't_enum';
	    if($realType =~ /(.*)::/) {
		my $c = $1;
		if($classidx{$c}) {
#		    print OUT "$classidx{$c}, ";
		} else {
#		    print OUT "0 /* unknown class $c */, ";
		}
	    } else {
#		print OUT "0 /* unknown $realType */, "; # no classId
	    }
	}
	else {
	    $typeId = $typeunion{$realType};
	    if (defined $typeId) {
		$typeId =~ s/s_/t_/; # from s_short to t_short for instance
	    }
	    else {
		# Not a known class - ouch, this happens quite a lot
		# (private classes, typedefs, template-based types, etc)
		if ( $skippedClasses{$realType} ) {
#		    print STDERR "$realType has been skipped, using t_voidp for it\n";
		} else {
		    unless( $realType =~ /</ ) { # Don't warn for template stuff...
			print STDERR "$realType isn't a known type (type=$type)\n";
		    }
		}
		$typeId = 't_voidp'; # Unknown -> map to a void *
	    }
#	    print OUT "0, "; # no classId
	}
	# Then write the flags
	die "$type" if !defined $typeId;
#	print OUT "Smoke::$typeId | $typeFlags },";
#	print OUT "\t//$typeCount\n";
	$typeCount++;
	# Remember it for coerce_type
	$allTypes{$type}{typeId} = $typeId;
    }
#    print OUT "};\n\n";


    my %arglist; # registers the needs for argumentList (groups of type ids)
    my %methods;
    # Look for all methods and all enums, in all classes
    # And fill in methods and arglist. This loop writes nothing to OUT.
    Iter::LocalCompounds( $rootnode, sub {
	my $classNode = shift;
	my $className = join( "::", kdocAstUtil::heritage($classNode) );
	print STDERR "writeSmokeDataFile: arglist: looking at $className\n" if ($debug);

	Iter::MembersByType ( $classNode, undef,
		sub {	my ($classNode, $m ) = @_;

	my $methName = $m->{astNodeName};
	# For destructors, get a proper signature that includes the '~'
	if ( $m->{ReturnType} eq '~' )
	{
	    $methName = '~' . $methName ;
	    # Let's even store that change, otherwise we have to do it many times
	    $m->{astNodeName} = $methName;
	}
	
	if( $m->{NodeType} eq "enum" ) {

	    foreach my $enum ( @{$m->{ParamList}} ) {
		my $enumName = $enum->{ArgName};
	        $methods{$enumName}++;
	    }

        } elsif ( $m->{NodeType} eq 'var' ) {

	    $methods{$m->{astNodeName}}++;

	} elsif( $m->{NodeType} eq "method" ) {

	    $methods{$methName}++;
	    my @protos;
	    makeprotos(\%classidx, $m, \@protos);

	    #print "made @protos from $className $methName $m->{Signature})\n" if ($debug);
	    for my $p (@protos) {
			$methods{$p}++;
			my $argcnt = 0;
			$argcnt = length($1) if $p =~ /([\$\#\?]+)/;
			my $sig = methodSignature($m, $argcnt-1);
			# Store in a class hash named "proto", a proto+signature => method association
			$classNode->{proto}{$p}{$sig} = $m;
			#$classNode->{signature}{$sig} = $p;
			# There's probably a way to do this better, but this is the fastest way
			# to get the old code going: store classname into method
			$m->{class} = $className;
	    }

	    my $firstDefaultParam = $m->{FirstDefaultParam};
	    $firstDefaultParam = scalar(@{ $m->{ParamList} }) unless defined $firstDefaultParam;
	    my $argNames = '';
	    my $args = '';
	    for(my $i = 0; $i < @{ $m->{ParamList} }; $i++) {
		$args .= ', ' if $i;
		$argNames .= ', ' if $i;
		my $argType = $m->{ParamList}[$i]{ArgType};
		my $typeEntry = findTypeEntry( $argType );
		$args .= defined $typeEntry ? $typeEntry->{index} : 0;
		$argNames .= $argType;

		if($i >= ($firstDefaultParam - 1)) {
		    #print "arglist entry: $args\n";
		    $arglist{$args} = $argNames;
		}
		
	    }
	    # create an entry for e.g. "arg0,arg1,arg2" where argN is index in allTypes of type for argN
	    # The value, $argNames, is temporarily stored, to be written out as comment
	    # It gets replaced with the index in the next loop.
	    #print "arglist entry : $args\n";
	    $arglist{$args} = $argNames;
	}
		    }, # end of sub
	undef
       );
    });


    $arglist{''} = 0;
    # Print arguments array
#    print OUT "static Smoke::Index ${libname}_argumentList[] = {\n";
    my $argListCount = 0;
    for my $args (sort keys %arglist) {
	my @dunnohowtoavoidthat = split(',',$args);
	my $numTypes = $#dunnohowtoavoidthat;
	if ($args eq '') {
#	    print OUT "\t0,\t//0  (void)\n";
	} else {
	    # This is a nice trick : args can be written in one go ;)
#	    print OUT "\t$args, 0,\t//$argListCount  $arglist{$args}  \n";
	}
	$arglist{$args} = $argListCount;      # Register proper index in argList
	$argListCount += $numTypes + 2;       # Move forward by as much as we wrote out
    }
#    print OUT "};\n\n";

    $methods{''} = 0;
    my @methodlist = sort keys %methods;
    my %methodidx = do { my $i = 0; map { $_ => $i++ } @methodlist };

#    print OUT "// Raw list of all methods, using munged names\n";
#    print OUT "static const char *${libname}_methodNames[] = {\n";
    my $methodNameCount = $#methodlist;
    for my $m (@methodlist) {
#	print OUT qq(    "$m",\t//$methodidx{$m}\n);
    }
#    print OUT "};\n\n";

#    print OUT "// (classId, name (index in methodNames), argumentList index, number of args, method flags, return type (index in types), xcall() index)\n";
#    print OUT "static Smoke::Method ${libname}_methods[] = {\n";
    my @methods;
    %allMethods = ();
    my $methodCount = 0;
    # Look at all classes and all enums again
    Iter::LocalCompounds( $rootnode, sub {
	my $classNode = shift;
	my $className = join( "::", kdocAstUtil::heritage($classNode) );
	return if $classNode->{NodeType} eq 'namespace';
	
	my $classIndex = $classidx{$className};
	print STDERR "writeSmokeDataFile: methods: looking at $className\n" if ($debug);

	Iter::MembersByType ( $classNode, undef,
		sub {	my ($classNode, $m ) = @_;

	if( $m->{NodeType} eq "enum" ) {

	    foreach my $enum ( @{$m->{ParamList}} ) {
		my $enumName = $enum->{ArgName};
		my $fullEnumName = "$className\::$enumName";
		my $sig = "$className\::$enumName\()";
		my $xmethIndex = $methodidx{$enumName};
		die "'Method index' for enum $sig not found" unless defined $xmethIndex;
		my $typeId = findTypeEntry( $fullEnumName )->{index};
		die "enum has no {case} value in $className: $fullEnumName" unless defined $classNode->{case}{$fullEnumName};
#		print OUT "\t{$classIndex, $xmethIndex, 0, 0, Smoke::mf_static, $typeId, $classNode->{case}{$fullEnumName}},\t//$methodCount $fullEnumName (enum)\n";
		$allMethods{$sig} = $methodCount;
		print STDERR "Added entry for " . $sig . " into \$allMethods\n" if ($debug);
		$methods[$methodCount] = {
				c => $classIndex,
				methIndex => $xmethIndex,
				argcnt => '0',
				args => 0,
				retTypeIndex => 0,
				idx => $classNode->{case}{$fullEnumName}
			       };
		$methodCount++;
	    }

	} elsif( $m->{NodeType} eq 'var' ) {

	    my $name = $m->{astNodeName};
	    my $fullName = "$className\::$name";
	    my $sig = "$fullName\()";
	    my $xmethIndex = $methodidx{$name};
	    die "'Method index' for var $sig not found" unless defined $xmethIndex;
	    my $varType = $m->{Type};
	    $varType =~ s/static\s//;
	    $varType =~ s/const\s+(.*)\s*&/$1/;
	    $varType =~ s/\s*$//;
	    my $typeId = findTypeEntry( $varType )->{index};
	    die "var has no {case} value in $className: $fullName" unless defined $classNode->{case}{$fullName};
#	    print OUT "\t{$classIndex, $xmethIndex, 0, 0, Smoke::mf_static, $typeId, $classNode->{case}{$fullName}},\t//$methodCount $fullName (static var)\n";
            $allMethods{$sig} = $methodCount;
	    print STDERR "Added entry for " . $sig . " into \$allMethods\n" if ($debug);
	    $methods[$methodCount] = {
				c => $classIndex,
				methIndex => $xmethIndex,
				argcnt => '0',
				args => 0,
				retTypeIndex => 0,
				idx => $classNode->{case}{$fullName}
			       };
	    $methodCount++;


	} elsif( $m->{NodeType} eq "method" ) {

	    # We generate a method entry only if the method is in the switch() code
	    # BUT: for pure virtuals, they need to have a method entry, even though they
	    # do NOT have a switch code.
	    return if ( $m->{SkipFromSwitch} && $m->{Flags} !~ "p" );

	    # No switch code for destructors if we didn't derive from the class (e.g. it has private ctors only)
    	    return if ( $m->{ReturnType} eq '~' && ! ( $classNode->{BindingDerives} and $classNode->{HasPublicDestructor}) );

            # Is this sorting really important?
	    #for my $m (sort {$a->{name} cmp $b->{name}} @{ $self->{$c}{method} }) {

	    my $methName = $m->{astNodeName};
	    my $def = $m->{FirstDefaultParam};
	    $def = scalar(@{ $m->{ParamList} }) unless defined $def;
	    my $last = scalar(@{ $m->{ParamList} }) - 1;
	    #print STDERR "writeSmokeDataFile: methods: generating for method $methName, def=$def last=$last\n" if ($debug);

	    while($last >= ($def-1)) {
		last if $last < -1;
		my $args = [ @{ $m->{ParamList} }[0..$last] ];
		my $sig = methodSignature($m, $last);
		#my $methodSig = $classNode->{signature}{$sig}; # Munged signature
		#print STDERR "writeSmokeDataFile: methods: sig=$className\::$sig methodSig=$methodSig\n" if ($debug);
		#my $methodIndex = $methodidx{$methodSig};
		#die "$methodSig" if !defined $methodIndex;

		my $methodIndex = $methodidx{$methName};
		die "$methName" if !defined $methodIndex;
		my $case = $classNode->{case}{$sig};
		my $typeEntry = findTypeEntry( $m->{ReturnType} );
		my $retTypeIndex = defined $typeEntry ? $typeEntry->{index} : 0;

		my $i = 0;
		my $t = '';
		for my $arg (@$args) {
		    $t .= ', ' if $i++;
		    my $typeEntry = findTypeEntry( $arg->{ArgType} );
		    $t .= defined $typeEntry ? $typeEntry->{index} : 0;
		}
		my $arglist = $t eq '' ? 0 : $arglist{$t};
		die "arglist for $t not found" unless defined $arglist;
		if ( $m->{Flags} =~ "p" ) {
		    # Pure virtuals don't have a {case} number, that's normal
		    die if defined $case;
		    $case = -1; # This remains -1, not 0 !
		} else {
			;
#		    die "$className\::$methName has no case number for sig=$sig" unless defined $case;
		}
		my $argcnt = $last + 1;
		my $methodFlags = '0';
		$methodFlags .= "|Smoke::mf_static" if $m->{Flags} =~ "s";
		$methodFlags .= "|Smoke::mf_const" if $m->{Flags} =~ "c"; # useful?? probably not
		$methodFlags =~ s/0\|//; # beautify
		
#		print OUT "\t{$classIndex, $methodIndex, $arglist, $argcnt, $methodFlags, $retTypeIndex, $case},\t//$methodCount $className\::$sig";
#		print OUT " [pure virtual]" if ( $m->{Flags} =~ "p" ); # explain why $case = -1 ;)
#		print OUT "\n";
		
		$allMethods{$className . "::" . $sig} = $methodCount;
		$methods[$methodCount] = {
					  c => $classIndex,
					  methIndex => $methodIndex,
					  argcnt => $argcnt,
					  args => $arglist,
					  retTypeIndex => $retTypeIndex,
					  idx => $case
					 };
		$methodCount++;
		$last--;
	    } # while
	} # if method
      } ); # Method Iter
    } ); # Class Iter
#    print OUT "};\n\n";

    my @protos;
    Iter::LocalCompounds( $rootnode, sub {
	my $classNode = shift;
	my $className = join( "::", kdocAstUtil::heritage($classNode) );
	
	return if $classNode->{NodeType} eq 'namespace';
	
	my $classIndex = $classidx{$className};
	print STDERR "writeSmokeDataFile: protos: looking at $className\n" if ($debug);

	Iter::MembersByType ( $classNode, undef,
		sub {	my ($classNode, $m ) = @_;

	if( $m->{NodeType} eq "enum" ) {
	    foreach my $enum ( @{$m->{ParamList}} ) {
		my $enumName = $enum->{ArgName};
		my $sig = "$className\::$enumName\()";
		my $xmeth = $allMethods{$sig};
		die "'Method' for enum $sig not found" unless defined $xmeth;
		my $xmethIndex = $methodidx{$enumName};
		die "'Method index' for enum $enumName not found" unless defined $xmethIndex;
		push @protos, {
			       methIndex => $xmethIndex,
			       c => $classIndex,
			       over => {
					$sig => {
						 sig => $sig,
						}
				       },
			       meth => $xmeth
			      };
	    }

	} elsif( $m->{NodeType} eq 'var' ) {

	    my $name = $m->{astNodeName};
	    my $fullName = "$className\::$name";
	    my $sig = "$fullName\()";
	    my $xmeth = $allMethods{$sig};
	    die "'Method' for var $sig not found" unless defined $xmeth;
	    my $xmethIndex = $methodidx{$name};
	    die "'Method index' for var $name not found" unless defined $xmethIndex;
	    push @protos, {
			       methIndex => $xmethIndex,
			       c => $classIndex,
			       over => {
					$sig => {
						 sig => $sig,
						}
				       },
			       meth => $xmeth
			  };

	}
		    });

	for my $p (keys %{ $classNode->{proto} }) {
	    # For each prototype
	    my $scratch = { %{ $classNode->{proto}{$p} } }; # sig->method association
	    # first, grab all the superclass voodoo
	    for my $supNode (superclass_list($classNode)) {
		my $i = $supNode->{proto}{$p};
		next unless $i;
		for my $k (keys %$i) {
		    $scratch->{$k} = $i->{$k} unless exists $scratch->{$k};
		}
	    }

	    # Ok, now we have a full list
	    #if(scalar keys %$scratch > 1) {
		#print STDERR "Overload: $p (@{[keys %$scratch]})\n" if ($debug);
	    #}
	    my $xmethIndex = $methodidx{$p};
	    my $classIndex = $classidx{$className};
	    for my $sig (keys %$scratch) {
		#my $xsig = $scratch->{$sig}{class} . "::" . $sig;
		my $xsig = $className . "::" . $sig;
		$scratch->{$sig}{sig} = $xsig;
		delete $scratch->{$sig}
		    if $scratch->{$sig}{Flags} =~ "p" # pure virtual
			or not exists $allMethods{$xsig};
	    }
	    push @protos, {
		methIndex => $xmethIndex,
		c => $classIndex,
		over => $scratch
	    } if scalar keys %$scratch;
	}
    });

    my @protolist = sort { $a->{c} <=> $b->{c} || $a->{methIndex} <=> $b->{methIndex} } @protos;
#for my $abc (@protos) {
#print "$abc->{methIndex}.$abc->{c}\n";
#}

    print STDERR "Writing methodmap table\n" if ($debug);
    my @resolve = ();
#    print OUT "// Class ID, munged name ID (index into methodNames), method def (see methods) if >0 or number of overloads if <0\n";
    my $methodMapCount = 1;
#    print OUT "static Smoke::MethodMap ${libname}_methodMaps[] = {\n";
#    print OUT "\t{ 0, 0, 0 },\t//0 (no method)\n";
    for my $cur (@protolist) {
	if(scalar keys %{ $cur->{over} } > 1) {
#	    print OUT "\t{$cur->{c}, $cur->{methIndex}, -@{[1+scalar @resolve]}},\t//$methodMapCount $classlist[$cur->{c}]\::$methodlist[$cur->{methIndex}]\n";
	    $methodMapCount++;
	    for my $k (keys %{ $cur->{over} }) {
	        my $p = $cur->{over}{$k};
	        my $xsig = $p->{class} ? "$p->{class}\::$k" : $p->{sig};
	        push @resolve, { k => $k, p => $p, cur => $cur, id => $allMethods{$xsig} };
	    }
	    push @resolve, 0;
	} else {
	    for my $k (keys %{ $cur->{over} }) {
	        my $p = $cur->{over}{$k};
	        my $xsig = $p->{class} ? "$p->{class}\::$k" : $p->{sig};
#	        print OUT "\t{$cur->{c}, $cur->{methIndex}, $allMethods{$xsig}},\t//$methodMapCount $classlist[$cur->{c}]\::$methodlist[$cur->{methIndex}]\n";
	        $methodMapCount++;
	    }
	}
    }
#    print OUT "};\n\n";


    print STDERR "Writing ambiguousMethodList\n" if ($debug);
#    print OUT "static Smoke::Index ${libname}_ambiguousMethodList[] = {\n";
#    print OUT "    0,\n";
    for my $r (@resolve) {
	unless($r) {
#	    print OUT "    0,\n";
	    next;
	}
	my $xsig = $r->{p}{class} ? "$r->{p}{class}\::$r->{k}" : $r->{p}{sig};
	die "ambiguousMethodList: no method found for $xsig\n" if !defined $allMethods{$xsig};
#	print OUT "    $allMethods{$xsig},  // $xsig\n";
    }
#    print OUT "};\n\n";

#    print OUT "extern \"C\" { // needed?\n";
#    print OUT "    void init_${libname}_Smoke();\n";
#    print OUT "}\n";
#    print OUT "\n";
#    print OUT "Smoke* qt_Smoke = 0L;\n";
#    print OUT "\n";
#    print OUT "// Create the Smoke instance encapsulating all the above.\n";
#    print OUT "void init_${libname}_Smoke() {\n";
#    print OUT "    qt_Smoke = new Smoke(\n";
#    print OUT "        ${libname}_classes, ".$#classlist.",\n";
#    print OUT "        ${libname}_methods, $methodCount,\n";
#    print OUT "        ${libname}_methodMaps, $methodMapCount,\n";
#    print OUT "        ${libname}_methodNames, $methodNameCount,\n";
#    print OUT "        ${libname}_types, $typeCount,\n";
#    print OUT "        ${libname}_inheritanceList,\n";
#    print OUT "        ${libname}_argumentList,\n";
#    print OUT "        ${libname}_ambiguousMethodList,\n";
#    print OUT "        ${libname}_cast );\n";
#    print OUT "}\n";
#    close OUT;

#print "@{[keys %allMethods ]}\n";
}

sub indentText($$)
{
	my ( $indent, $text ) = @_;

	if ( $indent eq "" || $text eq "" ) {
		return $text;
	}

	$text =~ s/\n(.)/\n$indent$1/g;
	return $indent . $text;
}

=head2 printCSharpdocComment

	Parameters: docnode filehandle

	Converts a kdoc comment to csharpdoc format.
	@ref's are converted to <see>'s; @p's and @em's are converted
	to inline HTML.

=cut

sub printCSharpdocComment($$$$)
{
	my( $docnode, $name, $indent, $signalLink ) = @_;

	my $node;
	my $returntext = '<remarks>';
	foreach $node ( @{$docnode->{Text}} ) {
		next if $node->{NodeType} ne "DocText" and $node->{NodeType} ne "ListItem" 
		and $node->{NodeType} ne "Param";
		my $line = '';
		
		if ($node->{NodeType} eq "Param") {
			if ($node->{Name} !~ /argc/) {
				$line = "<param> name=\"" . $node->{Name} . "\" " . $node->{astNodeName} . "</param>";
			}
		} else {
			$line = $node->{astNodeName};
		}
		$line =~ s/argc, ?argv/args/g;
		$line =~ s/int argc, ?char ?\* ?argv(\[\])?/string[] args/g;
		$line =~ s/int argc, ?char ?\*\* ?argv/string[] args/g;
#		if ($node->{NodeType} eq "Param") {
#			$line =~ s/(const )?QC?StringList(\s*&)?/string[]/g;
#		} else {
			$line =~ s/(const )?QC?StringList(\s*&)?/List<string>/g;
#		}
		$line =~ s/NodeList/ArrayList/g;
		$line =~ s/KTrader::OfferList/ArrayList/g;
		$line =~ s/QString::null/null/g;
		$line =~ s/(const )?QC?String(\s*&)?/string/g;
		$line =~ s/(const )?KCmdLineOptions\s*(\w+)\[\]/string[][] $2/;
		$line =~ s/KCmdLineLastOption//g;
		$line =~ s/virtual //g;
		$line =~ s/~\w+\(\)((\s*{\s*})|;)//g;
		$line =~ s/0L/null/g;
		$line =~ s/(\([^\)]*\))\s*:\s*\w+\([^\)]*\)/$1/g;
		$line =~ s/\(void\)//g;
		$line =~ s/const char/string/g;
		$line =~ s/const (\w+)\&/$1/g;
		$line =~ s/bool/bool/g;
		$line =~ s/SLOT\(\s*([^\)]*)\) ?\)/SLOT("$1)")/g;
		$line =~ s/SIGNAL\(\s*([^\)]*)\) ?\)/SIGNAL("$1)")/g;
		$line =~ s/Q_OBJECT\n//g;
		$line =~ s/public\s*(slots)?:\n/public /g;
		$line =~ s/([^0-9"]\s*)\*(\s*[^0-9"-])/$1$2/g;
		$line =~ s/^(\s*)\*/$1/g;
		$line =~ s/\n \*/\n /g;
		$line =~ s!\@ref\s+([\w]+)::([\w]+)\s*(\([^\)]*\))(\.)?!<see cref=\"$1#$2\"></see>$4!g;
		$line =~ s!\@ref\s+#([\w:]+)(\(\))?!<see cref=\"#$1\"</see>!g;
		$line =~ s!\@ref\s+([\w]+)\s*(\([^\)]*\))!<see cref=\"#$1\"></see>!g;
		$line =~ s!\@ref\s+([\w]+)::([\w]+)!<see cref=\"$1#$2\"></see>!g;
		$line =~ s!\@ref\s+([a-z][\w]+)!<see cref=\"#$1\"></see>!g;
		$line =~ s!\@ref\s+([\w]+)!<see cref=\"$1\"></see>!g;
		while ($line =~ /\@c\s+([\w#\\\.<>]+)/ ) {
			my $code = $1;
			$code =~ s!<!&lt;!g;
			$code =~ s!>!&gt;!g;
			$code =~ s!\\#!#!g;
			$line =~ s!\@c\s+([\w#\\\.<>]+)!<code>$code</code>!;
		}
		$line =~ s!\@em\s+(\w+)!<b>$1</b>!g;
		$line =~ s!\@p\s+([\w\._]*)!<code>$1</code>!g;
		$line =~ s!\\paragraph\s+[\w]+\s([\w]+)!<li><b>$1</b></li>!g;
		$line =~ s!\\b\s+([\w -]+)\\n!<li><b>$1</b></li>!g;
		$line =~ s!\\c\s+([\w\@&\\?;-]+)!<code>$1</code>!g;
		$line =~ s!\\p\s+([\w\@]+)!<pre>$1</pre>!g;
		$line =~ s!\\li\s+([\w\@]+)!<li>$1</li>!g;
		$line =~ s!<b>([\w\t \(\)-]*:?)</b>\\n!<li><b>$1</b></li>!g;
		$line =~ s!static_cast<\s*([\w\.]*)\s*>!($1)!g;
#		if ($name ne "") {
#			$line =~ s/\@link #/\@link $name\#/g;
#		}
		
		if ($node->{NodeType} eq "ListItem") {
			$line =~ s/^/\n<li>\n/;
			$line =~ s!$!\n</li>!;
#			$line =~ s/\n/\n$indent\t/g;
		} else {
#			$line =~ s/^/$indent/;
#			$line =~ s/\n/\n$indent/g;
		}
		
#		$line =~ s/\n/\n$indent/g;
		$returntext .= $line;
	}
	
	$returntext .= "$signalLink</remarks>";
	
	if ( defined $docnode->{Returns} ) {
		my $text = $docnode->{Returns};
		$text =~ s/QString::null/null/g;
		$returntext .=  "\t\t<return> $text</return>\n";
	}
		
	if ( defined $docnode->{Author} ) { 
		$returntext .= "\t\t<author> " . $docnode->{Author} . "</author>\n" 
	}
		
	if ( defined $docnode->{Version} ) {
		my $versionStr = $docnode->{Version};
		$versionStr =~ s/\$\s*Id:([^\$]*) Exp \$/$1/;
		$returntext .= "\t\t<version> $versionStr</version>\n";
	}
		
	if ( defined $docnode->{ClassShort} ) { 
		my $shortText = $docnode->{ClassShort};
		$shortText =~ s![\*\n]! !g;
		$returntext .= "\t\t<short> $shortText</short>\n";
	}
	
	if ( defined $docnode->{See} ) {
		foreach my $text ( @{$docnode->{See}} ) {
			next if ($text =~ /QString|^\s*and\s*$|^\s*$|^[^\w]*$/);
			$text =~ s/KIO:://g;
			$text =~ s/KParts:://g;
			while ($text =~ /((::)|(->))(.)/) {
				my $temp = uc($4);
				$text =~ s/$1$4/.$temp/;
			}
			$text =~ s/\(\)//g;
			$text =~ s/^\s*([a-z].*)/$1/g;
			$returntext .= "\t\t<see> $text</see>\n";
		}
	}	

	$returntext =~ s!\\link!<see>!g;
	$returntext =~ s!\\endlink!</see>!g;
	$returntext =~ s/DOM#([A-Z])/$1/g;
	$returntext =~ s/KIO#([A-Z])/$1/g;
	$returntext =~ s/KParts#([A-Z])/$1/g;
	$returntext =~ s/const\s+(\w+)\s*\&/$1/g;
	$returntext =~ s/QChar/char/g;
	$returntext =~ s/QStringList/List<string>/g;
	$returntext =~ s/([Aa]) ArrayList/$1n ArrayList/g;
	$returntext =~ s/QString/string/g;
	$returntext =~ s/KCmdLineOptions/string[][]/;
	$returntext =~ s!\\note!<b>Note:<\b>!g;
	$returntext =~ s!\\(code|verbatim)!<pre>!g;
	$returntext =~ s!\\(endcode|endverbatim)!</pre>!g;
	$returntext =~ s!\\addtogroup\s+[\w]+\s+"([^"\@]+)"\s+\@{!<li><b>$1</b></li>!g;
	$returntext =~ s![\\\@]relates\s+([a-z][\w]*)!<see cref=\"$1\"></see>!g;
	$returntext =~ s![\\\@]relates\s+(\w+)::(\w+)!<see cref=\"$1.$2\"></see>!g;
	$returntext =~ s![\\\@]relates\s+(#?\w+)!<see cref=\"$1\"></see>!g;
	$returntext =~ s!\\c\s+([\w\@&\\?";-]+)!<code>$1</code>!g;
	$returntext =~ s!\@p\s+([\w\._]*)!<code>$1</code>!g;
	$returntext =~ s!\@a\s+([:\w]+)!<b>$1</b>!g;
	$returntext =~ s![\@\\]b\s+[:\w]!<b>$1</b>!g;
	$returntext =~ s/};/}/g;

	while ($returntext =~ /((::)|(->))(.)/) {
		my $temp = uc($4);
		$returntext =~ s/$1$4/.$temp/;
	}
	
	$returntext =~ s/\s*$//;
	if ($returntext =~ /^<remarks>\s*<\/remarks>$/) {
		return "";
	} else {
		$returntext =~ s/\n/\n$indent/g;
		$returntext =~ s/^/$indent/;
		return $returntext . "\n";
	}
}

1;
