/*
**
** Copyright (C) 1998-2001 by Matthias Hölzer-Klüpfel <hoelzer@kde.org>
**	Maintainence has ceased - send questions to kde-devel@kde.org.
**
*/

/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program in a file called COPYING; if not, write to
** the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
** MA 02110-1301, USA.
*/

/*
** Bug reports and questions can be sent to kde-devel@kde.org
*/

#include <stdlib.h>
#include <stdio.h>

#include <QtGui/QLabel>
#include <QtGui/QLayout>
#include <QtGui/QPainter>
#include <QtGui/QPixmap>
#include <QtGui/QWidget>
#include <QtGui/QVBoxLayout>
#include <QtCore/QByteArray>
#include <QtGui/QDesktopWidget>

#include <kapplication.h>
#include <kconfig.h>
#include <kconfiggroup.h>
#include <kaboutdata.h>
#include <kcmdlineargs.h>
#include <klocale.h>
#include <kdebug.h>
#include <kglobal.h>

#include "main.moc"
#include "mapwidget.h"
#include "zoneclock.h"

WorldWideWatch::WorldWideWatch(bool restore, QWidget *parent, const char *name)
  : KMainWindow(parent)
{
  KGlobal::locale()->insertCatalog("timezones4"); // For time zone translation
  setObjectName(name);

  QWidget *w = new QWidget(this);
  setCentralWidget(w);

  setPlainCaption(i18n("KDE World Clock"));

  QVBoxLayout *vbox = new QVBoxLayout(w);
  vbox->setSpacing(0);
  vbox->setMargin(0);

  _map = new MapWidget(false, restore, w);
  vbox->addWidget(_map,1);

  _clocks = new ZoneClockPanel(w);
  vbox->addWidget(_clocks);

  connect(_map, SIGNAL(addClockClicked(const QString &)), _clocks, SLOT(addClock(const QString &)));
  connect(_map, SIGNAL(saveSettings()), this, SLOT(doSave()));

  if (restore)
    load(KGlobal::config()->group( QString() ));
}


void WorldWideWatch::load(const KConfigGroup &config)
{
  _map->load(config);
  _clocks->load(config);

  resize(config.readEntry("Width", 320),
  config.readEntry("Height", 200));
}


void WorldWideWatch::save(KConfigGroup &config)
{
  _map->save(config);
  _clocks->save(config);

  config.writeEntry("Width", width());
  config.writeEntry("Height", height());
}


void WorldWideWatch::doSave()
{
    KConfigGroup grp = KGlobal::config()->group( QString() );
    save( grp );
}


void WatchApplication::dumpMap()
{
  // guess some default parameters
  QSize mapSize(kapp->desktop()->size());

  KCmdLineArgs *args = KCmdLineArgs::parsedArgs();

  QString themeName = args->getOption("theme");
  QString outName = args->getOption("o");

  QString ssize = args->getOption("size");
  if (!ssize.isEmpty())
    {
      int w,h;
      if (sscanf(ssize.toLocal8Bit().data(), "%dx%d", &w, &h) == 2)
        mapSize = QSize(w,h);
    }

  kDebug() << "theme=" << themeName << " out=" << outName << " size=" << mapSize.width() << "x" << mapSize.height();

  MapWidget *w = new MapWidget(false, true, 0);
  w->setTheme(themeName);
  w->setSize(mapSize.width(), mapSize.height());

  QPixmap pm = w->calculatePixmap();
  QPainter p(&pm);
  w->paintContents(&p);
  pm.save(outName, "PPM");

  delete w;
}


int WatchApplication::newInstance()
{
  // dump mode, used for background drawing
  KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
  if (args->isSet("dump"))
    {
      dumpMap();
      return 0;
    }

  if (!restoringSession())
  {
    WorldWideWatch *www = new WorldWideWatch(true);
    www->show();
  }

  return 0;
}

static void listThemes()
{
  QList<MapTheme*> _themes = MapLoader::themes();
  QListIterator<MapTheme*> it(_themes);
  while(it.hasNext())
  {
     printf("%s\n", it.next()->tag().toLocal8Bit().data());
  }
}


int main(int argc, char *argv[])
{
  KAboutData about("kworldclock", 0, ki18n("KDE World Clock"), "1.5");
  KCmdLineArgs::init(argc, argv, &about);

  KCmdLineOptions options;
  options.add("dump", ki18n("Write out a file containing the actual map"));
  options.add("theme <file>", ki18n("The name of the theme to use"), "depths");
  options.add("themes", ki18n("List available themes"));
  options.add("o <file>", ki18n("The name of the file to write to"), "dump.ppm");
  options.add("size <WxH>", ki18n("The size of the map to dump"));
  KCmdLineArgs::addCmdLineOptions(options);
  KUniqueApplication::addCmdLineOptions();

  KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
  if (args->isSet("themes"))
  {
    KComponentData app(&about);
    listThemes();
    return 0;
  }

  if (!KUniqueApplication::start())
    return 0;

  WatchApplication app;
  if (args->isSet("dump"))
      app.disableSessionManagement();

  if (app.isSessionRestored())
    RESTORE(WorldWideWatch)

  return app.exec();
}
